/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: SongEditor.cpp,v 1.74 2004/07/19 09:20:11 comix Exp $
 *
 */


#include "SongEditor.h"
#include "config.h"
#include "PatternEditorPanel.h"
#include "HydrogenApp.h"
#include "Button.h"
#include "PatternPropertiesDialog.h"
#include "SongPropertiesDialog.h"

#include "lib/Song.h"
#include "lib/Hydrogen.h"

#include "qtooltip.h"
#include "qpainter.h"
#include "qwidget.h"
#include "qvbox.h"
#include "qcursor.h"

SongEditorPanel::SongEditorPanel(QWidget *parent)
 : QWidget(parent)
 , Object( "SongEditPanel" )
{
//	cout << "SongEditorPanel INIT" << endl;
	m_nInitialWidth = 600;
	m_nInitialHeight = 250;

	const uint nMinWidth = 100;
	const uint nMinHeight = 50;

	resize( QSize(m_nInitialWidth, m_nInitialHeight) );
	setMinimumSize( nMinWidth, nMinHeight );
//	setMaximumSize( width, m_nHeight );

	setCaption( trUtf8( "Song Editor" ) );
	setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );


	// new pattern button
	string newPat_on_path = string( IMG_PATH ) + string( "/img/songEditor/newPatternBtn_on.png" );
	string newPat_off_path = string( IMG_PATH ) + string( "/img/songEditor/newPatternBtn_off.png" );
	string newPat_over_path = string( IMG_PATH ) + string( "/img/songEditor/newPatternBtn_over.png" );
	Button *newPatBtn = new Button(this, QSize(20, 20), newPat_on_path, newPat_off_path, newPat_over_path);
	newPatBtn->move(5, 5);
	QToolTip::add( newPatBtn, trUtf8("Create new pattern") );
	connect( newPatBtn, SIGNAL( clicked( Button* ) ), this, SLOT( newPatBtnClicked( Button* ) ) );

	// down button
	string m_pDownBtn_on_path = string( IMG_PATH ) + string( "/img/songEditor/downBtn_on.png" );
	string m_pDownBtn_off_path = string( IMG_PATH ) + string( "/img/songEditor/downBtn_off.png" );
	string m_pDownBtn_over_path = string( IMG_PATH ) + string( "/img/songEditor/downBtn_over.png" );
	m_pDownBtn = new Button(this, QSize(20, 20), m_pDownBtn_on_path, m_pDownBtn_off_path, m_pDownBtn_over_path);
	m_pDownBtn->move( 30, 5 );
	QToolTip::add( m_pDownBtn, trUtf8("Move the selected pattern down") );
	connect( m_pDownBtn, SIGNAL( clicked( Button* ) ), this, SLOT( downBtnClicked( Button* ) ) );

	// up button
	string m_pUpBtn_on_path = string( IMG_PATH ) + string( "/img/songEditor/upBtn_on.png" );
	string m_pUpBtn_off_path = string( IMG_PATH ) + string( "/img/songEditor/upBtn_off.png" );
	string m_pUpBtn_over_path = string( IMG_PATH ) + string( "/img/songEditor/upBtn_over.png" );
	m_pUpBtn = new Button(this, QSize(20, 20), m_pUpBtn_on_path, m_pUpBtn_off_path, m_pUpBtn_over_path);
	m_pUpBtn->move( 55, 5 );
	QToolTip::add( m_pUpBtn, trUtf8("Move the selected pattern up") );
	connect( m_pUpBtn, SIGNAL( clicked( Button* ) ), this, SLOT( upBtnClicked( Button* ) ) );


	// song button
	string m_pSongBtn_on_path = string( IMG_PATH ) + string( "/img/songEditor/songFuncBtn_on.png" );
	string m_pSongBtn_off_path = string( IMG_PATH ) + string( "/img/songEditor/songFuncBtn_off.png" );
	string m_pSongBtn_over_path = string( IMG_PATH ) + string( "/img/songEditor/songFuncBtn_over.png" );
	m_pSongBtn = new Button(this, QSize(20, 20), m_pSongBtn_on_path, m_pSongBtn_off_path, m_pSongBtn_over_path);
	m_pSongBtn->move( 80, 5 );
	QToolTip::add( m_pSongBtn, trUtf8("Song operations") );
	connect( m_pSongBtn, SIGNAL( clicked( Button* ) ), this, SLOT( songBtnClicked( Button* ) ) );

	// Song loop button
	string songLoop_on_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_loop_on.png" );
	string songLoop_off_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_loop_off.png" );
	string songLoop_over_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_loop_over.png" );
	m_pSongLoopBtn = new ToggleButton(this, QSize(20, 20), songLoop_on_path, songLoop_off_path, songLoop_over_path);
	m_pSongLoopBtn->move( 105, 5 );
	m_pSongLoopBtn->setPressed( false );
	QToolTip::add( m_pSongLoopBtn, trUtf8("Song loop") );
	connect( m_pSongLoopBtn, SIGNAL( clicked(Button*) ), this, SLOT( songLoopBtnClicked(Button*) ) );


	// PATTERN LIST
	m_pSv2 = new QScrollView( this );
	m_pSv2->setFrameShape( QFrame::NoFrame );
	m_pSv2->move( 5, 30 );
	m_pSv2->resize( m_nPatternListWidth, m_nInitialHeight - 45 - 5 );
	m_pSv2->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pSv2->setHScrollBarMode( QScrollView::AlwaysOn );

	m_pSongEditorPatternList = new SongEditorPatternList( m_pSv2->viewport() );
	m_pSongEditorPatternList->move( 0, 0 );
	m_pSongEditorPatternList->show();
	m_pSv2->addChild( m_pSongEditorPatternList );


	// EDITOR
	m_pSv1 = new QScrollView( this );
	m_pSv1->setFrameShape( QFrame::NoFrame );
	m_pSv1->move( m_nPatternListWidth + 5, 30 );
	m_pSv1->resize( m_nInitialWidth - 110, m_nInitialHeight - 45 - 5 );

	m_pSongEditor = new SongEditor( m_pSv1->viewport() );
	m_pSongEditor->move( 0, 0 );
	m_pSongEditor->show();
	m_pSv1->addChild( m_pSongEditor );
	connect( m_pSv1, SIGNAL( contentsMoving ( int , int ) ), this, SLOT( contentsMove( int, int ) ) );


	// POSITION
	m_pSv3 = new QScrollView( this );
	m_pSv3->setFrameShape( QFrame::NoFrame );
	m_pSv3->move( m_nPatternListWidth + 5, 5 );
	m_pSv3->resize( m_nInitialWidth - 110, 25 );
	m_pSv3->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pSv3->setHScrollBarMode( QScrollView::AlwaysOff );

	m_pSongEditorPositionRuler = new SongEditorPositionRuler( m_pSv3->viewport() );
	m_pSongEditorPositionRuler->move( 0, 0 );
	m_pSv3->addChild( m_pSongEditorPositionRuler );



	QPixmap patternIcon;
	string patternIcon_path = string( IMG_PATH ) + string( "/img/m_pSongEditor/patternIcon.png" );
	patternIcon.load( patternIcon_path.c_str() );

	m_pSongPopup = new QPopupMenu( this, "patternPopupMenu" );
	m_pSongPopup->insertItem( patternIcon, trUtf8("Song properties"),  this, SLOT( songPopup_songProperties() ) );
	m_pSongPopup->insertItem( patternIcon, trUtf8("Clear pattern sequence"),  this, SLOT( songPopup_clearSequence() ) );
	m_pSongPopup->insertItem( patternIcon, trUtf8("Delete all patterns"),  this, SLOT( songPopup_deleteAllPatterns() ) );


/*
	// Background image
	string background_path = string( IMG_PATH ) + string( "/img/patternEditor/patternEditor_background.png" );
	QPixmap m_backgroundPixmap;
	bool ok = m_backgroundPixmap.load( background_path.c_str() );
	if( ok == false ){
		errorLog( "Error loading pixmap " + background_path );
	}
	setBackgroundPixmap( m_backgroundPixmap );
*/
	// Background solid color
	UIStyle *pStyle = PreferencesMng::getInstance()->getDefaultUIStyle();
	QColor backgroundColor( pStyle->m_patternEditorPanel_backgroundColor.getRed(), pStyle->m_patternEditorPanel_backgroundColor.getGreen(), pStyle->m_patternEditorPanel_backgroundColor.getBlue() );
	this->setEraseColor( backgroundColor );

	updateAll();

	(Hydrogen::getInstance())->addEngineListener(this);
}



SongEditorPanel::~SongEditorPanel() {
//	cout << "SongEditorPanel destroy" << endl;
}



///
/// Synchronize the patternlist with the patternsequence
///
void SongEditorPanel::contentsMove( int x, int y) {
	m_pSv2->verticalScrollBar()->setValue( m_pSv1->verticalScrollBar()->value() );
	m_pSv3->horizontalScrollBar()->setValue( m_pSv1->horizontalScrollBar()->value() );
}



///
/// Update and redraw all...
///
void SongEditorPanel::updateAll() {
	Song *song = (Hydrogen::getInstance())->getSong();
	if ( song->isLoopEnabled() ) {
		m_pSongLoopBtn->setPressed( true );
	}
	else {
		m_pSongLoopBtn->setPressed( false );
	}

	m_pSongEditorPatternList->createBackground();
	m_pSongEditorPatternList->update();

	m_pSongEditor->createBackground();
	m_pSongEditor->update();
}



///
/// Create a new pattern
///
void SongEditorPanel::newPatBtnClicked( Button* btn) {
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();

	Pattern *emptyPattern = Pattern::getEmptyPattern();

	PatternPropertiesDialog *dialog = new PatternPropertiesDialog( this, emptyPattern );
	if ( dialog->exec() == QDialog::Accepted ) {
		patternList->add( emptyPattern );
		song->setModified( true );
		updateAll();
	}
	else {
		patternList->del( emptyPattern );
		delete emptyPattern;
		emptyPattern = NULL;
	}
	delete dialog;
	dialog = NULL;



}



///
/// Show the song popup
///
void SongEditorPanel::songBtnClicked( Button* btn) {
	m_pSongPopup->popup( m_pSongBtn->mapToGlobal( QPoint( 10, 10 ) ) );
}



///
/// Song loop button clicked
///
void SongEditorPanel::songLoopBtnClicked( Button* ref ) {
	bool value = m_pSongLoopBtn->isPressed();

	Hydrogen *pEngine = Hydrogen::getInstance();
	Song *song = pEngine->getSong();

	song->setLoopEnabled( value );
	song->setModified( true );
}



///
/// Move up a pattern in the patternList
///
void SongEditorPanel::upBtnClicked( Button* btn )
{

	Hydrogen *engine = Hydrogen::getInstance();
	int selectedPatternPos = engine->getSelectedPatternNumber();

	engine->lockEngine("SongEditorPanel::m_pUpBtnClicked");
	Song *song = engine->getSong();
	PatternList *list = song->getPatternList();

	if ( ( selectedPatternPos - 1 ) >= 0 ) {
		Pattern *m_tempPixmap = list->get( selectedPatternPos - 1 );
		list->replace( list->get( selectedPatternPos ), selectedPatternPos - 1 );
		list->replace( m_tempPixmap, selectedPatternPos );
		engine->unlockEngine();
		engine->setSelectedPatternNumber( selectedPatternPos - 1 );

		updateAll();
		song->setModified(true);
	}
	else {
		engine->unlockEngine();
	}
}



///
/// Move down a pattern in the patternList
///
void SongEditorPanel::downBtnClicked( Button* btn )
{

	Hydrogen *engine = Hydrogen::getInstance();
	int selectedPatternPos = engine->getSelectedPatternNumber();

	engine->lockEngine("SongEditorPanel::m_pDownBtnClicked");
	Song *song = engine->getSong();
	PatternList *list = song->getPatternList();

	if ( ( selectedPatternPos + 1 ) < (int)list->getSize() ) {
		Pattern *m_tempPixmap = list->get( selectedPatternPos + 1 );
		list->replace( list->get( selectedPatternPos ), selectedPatternPos + 1 );
		list->replace( m_tempPixmap, selectedPatternPos );
		engine->unlockEngine();
		engine->setSelectedPatternNumber( selectedPatternPos + 1 );

		updateAll();
		song->setModified(true);
	}
	else {
		engine->unlockEngine();
	}

}




///
/// Show the song properties dialog
///
void SongEditorPanel::songPopup_songProperties() {
	SongPropertiesDialog *dialog = new SongPropertiesDialog( this );
	if ( dialog->exec() == QDialog::Accepted ) {
		updateAll();
		Hydrogen *engine = Hydrogen::getInstance();
		Song *song = engine->getSong();
		song->setModified( true );
	}
	delete dialog;
	dialog = NULL;
}




void SongEditorPanel::songPopup_clearSequence() {
	Hydrogen *engine = Hydrogen::getInstance();

	engine->lockEngine("SongEditorPanel::m_pSongPopup_clearSequence");

	Song *song = engine->getSong();
	vector<PatternList*> *pPatternGroupsVect = song->getPatternGroupVector();
	for (uint i = 0; i < pPatternGroupsVect->size(); i++) {
		PatternList *pPatternList = (*pPatternGroupsVect)[i];
		pPatternList->clear();
		delete pPatternList;
	}
	pPatternGroupsVect->clear();

	engine->unlockEngine();

	updateAll();
	song->setModified(true);

}



void SongEditorPanel::songPopup_deleteAllPatterns() {
	Hydrogen *engine = Hydrogen::getInstance();
	engine->lockEngine("SongEditorPanel::m_pSongPopup_deleteAllPatterns");

	engine->getCurrentPatternList()->clear();

	Song *song = engine->getSong();
	vector<PatternList*> *pPatternGroupsVect = song->getPatternGroupVector();
	for (uint i = 0; i < pPatternGroupsVect->size(); i++) {
		PatternList *pPatternList = (*pPatternGroupsVect)[i];
		pPatternList->clear();
		delete pPatternList;
	}
	pPatternGroupsVect->clear();

	PatternList *patList = song->getPatternList();
	for (uint i = 0; i < patList->getSize(); i++) {
		Pattern *pat = patList->get(i);
		delete pat;
	}
	patList->clear();

	engine->unlockEngine();

	updateAll();
	song->setModified(true);
}



///
/// This method is called from another thread (audio engine)
///
void SongEditorPanel::patternChanged() {
	H2TextEvent *ev = new H2TextEvent( "patternChanged" );
	QApplication::postEvent( this, ev );
}



void SongEditorPanel::customEvent( QCustomEvent *ev ) {
	if ( ev->type() != H2_TEXT_EVENT ) {	// Must be a H2TextEvent
		return;
	}
	QString message = ( (H2TextEvent*)ev )->getText();

	if ( message == QString( "patternChanged" ) ) {
		m_pSongEditorPatternList->createBackground();	// update the pattern highlight
		m_pSongEditorPatternList->update();

		m_pSongEditorPositionRuler->updateRuler();
	}

}



///
///
///
void SongEditorPanel::resizeEvent ( QResizeEvent *ev )
{
	// scrollview 1
	m_pSv1->resize( width() - (m_nPatternListWidth + 10), height() - 25 - 5 );

	// scrollview 2
	m_pSv2->resize( m_nPatternListWidth, height() - 25 - 5 );

	// scrollview 3
	m_pSv3->resize( width() - (m_nPatternListWidth + 10), 25 );

}




// ::::::::::::::::::::::::::::



SongEditor::SongEditor( QWidget *parent )
  : QWidget( parent , "", WNoAutoErase )
 , Object( "SongEditor" )
 , m_bChanged( true )
 , m_bSequenceChanged( true )
 , m_bDragCopy( false )
{
//	cout << "song editor INIT" << endl;

	m_nGridWidth = 16;
	m_nGridHeight = (PreferencesMng::getInstance())->getPatternEditorGridHeight();
	m_nInitialWidth = m_nMaxPatternSequence * m_nGridWidth;
	m_nInitialHeight = 10;

	this->resize( QSize(m_nInitialWidth, m_nInitialHeight) );

	createBackground();	// create m_backgroundPixmap pixmap

	update();

	setFocusPolicy (StrongFocus);

	m_bLasso = m_bDrag = false;
	m_bLassoRepaint = m_bDragRepaint = false;
	m_bSelect = false;
}



SongEditor::~SongEditor()
{
//	cout << "song editor destroy" << endl;
}



void SongEditor::pos2Grid(QRect area, int &top, int &left, int &bottom, int &right)
{
#define SWAP(a,b) { int swap = a; a = b; b = swap; }

	top = (signed)area.top() / (signed)m_nGridHeight;
	left = (signed)area.left() / (signed)m_nGridWidth;
	bottom = (signed)area.bottom() / (signed)m_nGridHeight;
	right = (signed)area.right() / (signed)m_nGridWidth;

	if (top > bottom) SWAP(top, bottom);
	if (left > right) SWAP(left, right);
	if (left == right) right = left + 1;
	if (top == bottom) bottom = top + 1;
}



// returns true if the current pos is inside the rectangle
bool SongEditor::inSelect( int nColumn, int nRow, QRect area )
{
	int top, left, bottom, right;
	m_bLasso = false;
	pos2Grid(area, top, left, bottom, right);

	if ( nRow >= top && nRow < bottom && nColumn >= left && nColumn < right && m_bSelect)
		return true;
	else
		return false;
}



// swap, add or remove a cell
// the default behaviour (type=SWAP) of this method is just the same as
// when the user clicks on a cell with the left mouse button
void SongEditor::manipulateCell( int column, int row, CellManipulationModes type = SWAP )
{
	Hydrogen *engine = Hydrogen::getInstance();

	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();

	if ( row >= (int)patternList->getSize() || row < 0 || column < 0) {
		return;
	}
	Pattern *pPattern = patternList->get( row );

	// check the patternSequence size
	vector<PatternList*> *pPatternVect = song->getPatternGroupVector();
	if ( column >= (int)pPatternVect->size() ) {
		int distance = column - pPatternVect->size();
		if (distance == 0) {
			PatternList* pList = new PatternList();
			pList->add(pPattern);
			pPatternVect->push_back(pList);
			song->setModified(true);
		}
	}
	else {
		PatternList *pList = (*pPatternVect)[column];

		bool bFound = false;
		for (uint i = 0; i < pList->getSize(); i++) {
			if ( pList->get(i) == pPattern ) { // il pattern e' gia presente
				bFound = true;

				if ( type == SWAP || type == REMOVE) { //we're removing or swapping
					if ( pList->getSize() == 1) {	// e' l'unico pattern
						bool bEndOfSong = ( (int)( pPatternVect->size() - 1 ) == (int)column);
						if ( bEndOfSong ) {
							// posso rimuoverlo solo se e' l'ultimo
							pList->del(i);
							if (pList->getSize() == 0) {
								pPatternVect->erase( pPatternVect->begin() + column );
								delete pList;
							}
							break;
						}
						else if (type == REMOVE) {
							//add a cell on first row
							manipulateCell(column, 0, ADD);
							if (row > 0) {
								//if you comment out the two lines directly above you can delete
								//columns entirely but it will render the application fairly unstable on some occasions
								pList->del(i);
							}
						}
					}
					else {
						pList->del(i);
						break;
					}
				}
			}
		}
		if (bFound == false && (type == SWAP || type == ADD) ) { //we're adding or swapping
			pList->add(pPattern);
		}
		song->setModified(true);
	}
	engine->unlockEngine();
}



// manipulate region, ie delete or copy a set of cells
// it copies from m_srcRegion to m_dstRegion
// or deletes in m_srcRegion
void SongEditor::manipulateRegion(RegionManipulationModes type)	//type: COPY, COPY_OVERWRITE, DELETE
{
	if (!m_bSelect)
		return; //if we have no valid regions yet

	int top, left, bottom, right;
	int dTop, dLeft, dBottom, dRight;


	pos2Grid(m_srcRegion, top, left, bottom, right); //get the the selected region's position
	pos2Grid(m_dstRegion, dTop, dLeft, dBottom, dRight); //and the destination region's

	int songWidth  = (Hydrogen::getInstance())->getSong()->getPatternGroupVector()->size();

	if (right > songWidth) {
		right = songWidth;
	}
	if (left < 0) {
		left = 0;
	}

	// dTop and dLeft are horizontal and vertical distance between
	// the source region and the destination region (i.e. copy destination)
	// they're measured in grids, not pixels

	dTop -= top;
	dLeft -= left;


	switch (type) {
		case DELETE: // delete cells in selected region (from right to left)
			for (int x = right - 1; x >= left; x --) {
				for (int y = top; y < bottom; y++) {
					if (std::find(m_cellList.begin(),m_cellList.end(),QPoint(x,y)) != m_cellList.end()) {
						manipulateCell(x, y, REMOVE);
					}
				}
			}
			break;

		case COPY: // copy cells to destination region (from left to right)
/*		case COPY_OVERWRITE:*/
			for (int x = left; x < right; x ++) {
				for (int y = top; y < bottom; y++) {
					if (std::find(m_cellList.begin(),m_cellList.end(),QPoint(x,y)) != m_cellList.end()) {
						manipulateCell(x + dLeft, y + dTop, ADD);
					}
				}
			}
			break;

		case MOVE:
			for (int x = left; x < right; x ++) {
				for (int y = top; y < bottom; y++) {
					if ( std::find( m_cellList.begin(), m_cellList.end(), QPoint(x,y) ) != m_cellList.end() ) {
						manipulateCell( x + dLeft, y + dTop, ADD );
						manipulateCell( x, y, REMOVE);
					}
				}
			}
			break;
	}

	m_bChanged = true;
	m_bSequenceChanged = true;

	// reset the booleans
	m_bDrag = m_bLasso = m_bSelect = false;
	update();
}



// shift columns left/right from getPatternPos() and forward
// last column will become the first if shifting right and first will be last if shifting left
void SongEditor::shiftColumns(ShiftModes dir) //SHIFT_LEFT, SHIFT_RIGHT
{
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	int currentPos = engine->getPatternPos();
	vector<PatternList*> *pPatternVect = song->getPatternGroupVector();
	int songWidth = pPatternVect->size();

	if (currentPos < 0) {
		currentPos = 0;
	}

	if (currentPos >= songWidth - 1) {
		return;
	}

	if (dir == SHIFT_LEFT) {
		PatternList *pList  = (*pPatternVect)[currentPos]; //temporary save first column

		//rearrange the pointers so each pointer except the last point at the next pointer
		for (int i = currentPos; i < songWidth - 1; i++) {
			(*pPatternVect)[i] = (*pPatternVect)[i + 1];
		}
		(*pPatternVect)[songWidth - 1] = pList; //point the last at the first
	}
	else if (dir == SHIFT_RIGHT) {
		PatternList *pList = (*pPatternVect)[songWidth - 1]; //temporary save last column

		//rearrange the pointers so each pointer except the first point at the previous pointer
		for (int i = songWidth - 1; i > currentPos; i --) {
			(*pPatternVect)[i] = (*pPatternVect)[i - 1];
		}

		(*pPatternVect)[currentPos] = pList; //point the first at the last
	}

	m_bChanged = true;
	m_bSequenceChanged = true;
}



// mark active cells in selected region
void SongEditor::markupCells()
{
	Song* song = (Hydrogen::getInstance())->getSong();
	int top, left, bottom, right;
	int songWidth  = song->getPatternGroupVector()->size();
	vector<PatternList*>* pPatternGroupVect = song->getPatternGroupVector();
	PatternList *patList = song->getPatternList();
	uint listLength = patList->getSize();

	pos2Grid(m_srcRegion, top, left, bottom, right);

	if (right > songWidth) {
		right = songWidth;
	}
	if (left < 0) {
		left = 0;
	}

	m_markedCells.clear();
	m_cellList.clear();

	for (int i = left; i < right; i++) {
		PatternList* pPatternList = (*pPatternGroupVect)[i];

		for (int nPat = 0; (unsigned)nPat < pPatternList->getSize(); nPat++) {
			Pattern *pat = pPatternList->get( nPat );

			int position = -1;
			// find the position in pattern list
			for (uint j = 0; j < listLength; j++) {
				Pattern *pat2 = patList->get( j );

				if (pat == pat2) {
					position = j;
					break;
				}
			}

			if (position != -1) {
				if (i >= left && i < right && position >= top && position < bottom)
				{
				//used in paintEvent() to draw this cell when dragging
				m_markedCells.push_back( QRect( (i-left)*m_nGridWidth + 3, (position-top)*m_nGridHeight + 3, m_nGridWidth - 5, m_nGridHeight - 5) );

				//save this cell's position, used in manipulateRegion()
				m_cellList.push_back(QPoint(i, position));
				}
			}
		}
	}
}



void SongEditor::keyPressEvent ( QKeyEvent * ev )
{
	if (ev->key() == Key_Delete && m_bSelect) { 	//if we press Delete and have a selected region
		// delete selected cells
		manipulateRegion(DELETE);

		update();

		m_oldRegion.setRect(0,0,0,0);
	}
	else if (ev->key() == Key_Delete || ev->key() == Key_Insert) {	// delete/insert columns
		Hydrogen *engine = Hydrogen::getInstance();
		Song *song = engine->getSong();
		int currentPos = engine->getPatternPos();
		vector<PatternList*> *pPatternVect = song->getPatternGroupVector();
		int songWidth = pPatternVect->size();
		int songHeight = song->getPatternList()->getSize();
		m_bSelect = false;

		if (ev->key() == Key_Delete && currentPos < songWidth - 1) {
			shiftColumns(SHIFT_LEFT);

			//delete the last column in the song
			m_srcRegion.setRect( (songWidth - 1) * m_nGridWidth, 0, 1, (songHeight + 1) * m_nGridHeight);
			m_bSelect = true;
			markupCells(); //make sure we have the list of cells
			manipulateRegion(DELETE);
		}
		else if (ev->key() == Key_Insert) {
			//insert a new column at the end of the song, it will become the first column when we shift
			manipulateCell(songWidth, 0, ADD);

			shiftColumns(SHIFT_RIGHT);
		}

		update();
	}
}



void SongEditor::mousePressEvent( QMouseEvent *ev )
{
	int row = (ev->y() / m_nGridHeight);
	int column = (ev->x() / m_nGridWidth);

	if ( ev->button() == MidButton ) {
		m_bLasso = true;
		m_srcRegion.setRect( column * m_nGridWidth, row * m_nGridHeight, 0, 0 );
		m_mousePos = ev->pos();
		return;
	}
	else if ( ev->button() == LeftButton && m_bSelect && inSelect( column, row, m_srcRegion ) ) {
		m_bDragCopy = ev->state() & ControlButton; //use copy overwrite if ctrl-clicking
		m_bDrag = m_bDragRepaint = true;
		m_dstRegion = m_srcRegion;
		m_mousePos = ev->pos();

		update();
		return;
	}

	m_bSelect = false;

	manipulateCell( column, row );

	// update
	m_bChanged = true;
	m_bSequenceChanged = true;
	update();
}



void SongEditor::mouseMoveEvent(QMouseEvent *ev)
{
	int xDiff = (ev->x() / (signed)m_nGridWidth)  - (m_mousePos.x() / (signed)m_nGridWidth);
	int yDiff = (ev->y() / (signed)m_nGridHeight) - (m_mousePos.y() / (signed)m_nGridHeight);

	if (m_bLasso) {	// user is resizing lasso (ie marking cells)
		//setCursor( QCursor( Qt::ArrowCursor ) );

		int row = (ev->y() / (signed)m_nGridHeight);
		int column = (ev->x() / (signed)m_nGridWidth);

		if (xDiff != 0 || yDiff != 0) {
			m_dstRegion.moveBy( xDiff * m_nGridWidth, yDiff * m_nGridHeight );
			QPoint newPos( xDiff * m_nGridWidth, yDiff * m_nGridHeight );

			m_mousePos += newPos;
		m_srcRegion.setBottomRight(QPoint( column * m_nGridWidth, row * m_nGridHeight) );

		m_bLassoRepaint = true;
		update();
	}
	}
	else if (m_bDrag) {	// user is dragging previously created lasso to a new position (ie copying)
		setCursor( QCursor( Qt::SizeAllCursor ) );

		if (xDiff != 0 || yDiff != 0) {
			m_dstRegion.moveBy( xDiff * m_nGridWidth, yDiff * m_nGridHeight );
			QPoint newPos( xDiff * m_nGridWidth, yDiff * m_nGridHeight );

			m_mousePos += newPos;

			m_bDragRepaint = true;
			update();
		}
	}
}



void SongEditor::mouseReleaseEvent( QMouseEvent *ev ) {
	setCursor( QCursor( Qt::ArrowCursor ) );

	if (m_bLasso) { // user is done with marking cells
		m_bChanged = true;
		m_bSequenceChanged = true;
		m_bSelect = true;
		update();

		m_oldRegion.setRect( 0, 0, 0, 0 );

		markupCells();
	}
	else if (m_bDrag) { // user is done with the draggin, copy the region
		if (m_bDragCopy) {
			manipulateRegion( COPY );
		}
		else {
			manipulateRegion( MOVE );
		}
		m_bDragCopy = false;
		update();
		m_oldRegion.setRect( 0, 0, 0, 0 );
	}
}



void SongEditor::paintEvent( QPaintEvent *ev )
{
	if (!isVisible()) {
		return;
	}

	if (m_bChanged) {
		m_bChanged = false;
		// ridisegno tutto solo se sono cambiate le note
		if (m_bSequenceChanged) {
			bitBlt( &m_sequencePixmap, 0, 0, &m_backgroundPixmap, 0, 0, width(), height(), CopyROP );
			drawSequence();
			m_bSequenceChanged = false;
		}

		//bitBlt( &m_tempPixmap, 0, 0, &m_sequencePixmap, 0, 0, width(), height(), CopyROP );
		//setErasePixmap( m_tempPixmap );
		bitBlt( this, 0, 0, &m_sequencePixmap, 0, 0, width(), height(), CopyROP, true);
		setErasePixmap( m_sequencePixmap );
	}


	// repaint lasso and/or the dragged cells
	if (m_bLassoRepaint || m_bDragRepaint) {
		QRect currentRegion;

		if (m_bLassoRepaint)
			currentRegion = m_srcRegion.normalize();
		else
			currentRegion = m_dstRegion.normalize();

		QPainter p( this );
		p.setRasterOp( XorROP );

		QPen pen( gray );
		pen.setStyle( Qt::DotLine );
		p.setPen( pen );

		if ( m_bLassoRepaint ) {
			p.drawRect( m_oldRegion );	// remove old rectangle, xor:ing will take care of that
			p.drawRect( currentRegion );	// add the new one
		}

		// if dragging, also draw each cell
		if (m_bDragRepaint) {
			QRect cell;
			QRect tempRegion;

			// loop twice, to erase old cells and draw the new ones
			for (int j = 0; j < 2; j++) {
				if (j == 0)
					tempRegion = m_oldRegion;
				else
					tempRegion = currentRegion;

				if (j > 0 || !tempRegion.isNull()) {
					for (uint i = 0; i < m_markedCells.size(); i ++) {
						cell = m_markedCells[i];
						cell.moveBy(tempRegion.left(), tempRegion.top());
						p.fillRect(cell, QColor(white));
					}
				}
			}
		}
		//setErasePixmap( m_tempPixmap );
		m_bLassoRepaint = m_bDragRepaint = false;
		m_oldRegion = currentRegion;
	}
}



void SongEditor::updateEditor()
{
	if(!isVisible()) {
		return;
	}

//	createBackground();
	update();
}



void SongEditor::createBackground()
{
	UIStyle *pStyle = PreferencesMng::getInstance()->getDefaultUIStyle();
	QColor backgroundColor( pStyle->m_songEditor_backgroundColor.getRed(), pStyle->m_songEditor_backgroundColor.getGreen(), pStyle->m_songEditor_backgroundColor.getBlue() );
	QColor alternateRowColor( pStyle->m_songEditor_alternateRowColor.getRed(), pStyle->m_songEditor_alternateRowColor.getGreen(), pStyle->m_songEditor_alternateRowColor.getBlue() );
	QColor linesColor( pStyle->m_songEditor_lineColor.getRed(), pStyle->m_songEditor_lineColor.getGreen(), pStyle->m_songEditor_lineColor.getBlue() );

	Hydrogen *pEngine = Hydrogen::getInstance();
	Song *pSong = pEngine->getSong();

	uint nPatterns = pSong->getPatternList()->getSize();

	static int nOldHeight = -1;
	int nNewHeight = m_nGridHeight * nPatterns;

	if (nOldHeight != nNewHeight) {
		// cambiamento di dimensioni...
		if (nNewHeight == 0) {
			nNewHeight = 1;	// the pixmap should not be empty
		}

		m_tempPixmap.resize( width(), nNewHeight );	// initialize the pixmap
		m_backgroundPixmap.resize( width(), nNewHeight );	// initialize the pixmap
		m_sequencePixmap.resize( width(), nNewHeight );	// initialize the pixmap
		m_tempPixmap.fill();
		m_sequencePixmap.fill();
		this->resize( QSize( width(), nNewHeight ) );
	}

	m_backgroundPixmap.fill( backgroundColor );		// fill the pixmap with white color

	QPainter p( &m_backgroundPixmap );
	p.setPen( linesColor );

	// sfondo per celle scure (alternato)
	for (uint i = 0; i < nPatterns; i++) {
		if ( ( i % 2) == 0) {
			uint y = m_nGridHeight * i;
			p.fillRect ( 0, y, m_nMaxPatternSequence * m_nGridWidth, m_nGridHeight, alternateRowColor );
		}
	}

	// celle...
	p.setPen( linesColor );

	// vertical lines
	for (uint i = 0; i < m_nMaxPatternSequence + 1; i++) {
		uint x = i * m_nGridWidth;

		int x1 = x + 2;
		int x2 = x + m_nGridWidth - 2;

		p.drawLine( x1, 0, x1, m_nGridHeight * nPatterns );
		p.drawLine( x2, 0, x2, m_nGridHeight * nPatterns );
	}

	// horizontal lines
	for (uint i = 0; i < nPatterns + 1; i++) {
		uint y = m_nGridHeight * i;

		int y1 = y + 2;
		int y2 = y + m_nGridHeight - 2;

		p.drawLine( 0, y1, (m_nMaxPatternSequence * m_nGridWidth), y1 );
		p.drawLine( 0, y2, (m_nMaxPatternSequence * m_nGridWidth), y2 );
	}


	// vertical lines (erase..)
	for (uint i = 0; i < m_nMaxPatternSequence + 1; i++) {
		uint x = i * m_nGridWidth;

		p.fillRect( x, 0, 2, m_nGridHeight * nPatterns, backgroundColor );
		p.fillRect( x + m_nGridWidth, 0, -2, m_nGridHeight * nPatterns, backgroundColor );
	}

	// horizontal lines (erase..)
	for (uint i = 0; i < nPatterns + 1; i++) {
		uint y = m_nGridHeight * i;

		p.fillRect( 0, y, m_nMaxPatternSequence * m_nGridWidth, 2, backgroundColor );
		p.fillRect( 0, y + m_nGridHeight, m_nMaxPatternSequence * m_nGridWidth, -2, backgroundColor );
	}


	//~ celle


	m_bChanged = true;
	m_bSequenceChanged = true;
}



void SongEditor::drawSequence()
{
	Song* song = (Hydrogen::getInstance())->getSong();

	PatternList *patList = song->getPatternList();
	vector<PatternList*>* pPatternGroupVect = song->getPatternGroupVector();

	uint listLength = patList->getSize();

	for (uint i = 0; i < pPatternGroupVect->size(); i++) {
		PatternList* pPatternList = (*pPatternGroupVect)[i];

		for (uint nPat = 0; nPat < pPatternList->getSize(); nPat++) {
			Pattern *pat = pPatternList->get( nPat );

			int position = -1;
			// find the position in pattern list
			for (uint j = 0; j < listLength; j++) {
				Pattern *pat2 = patList->get( j );
				if (pat == pat2) {
					position = j;
					break;
				}
			}
			if (position == -1) {
				warningLog( "[drawSequence] position == -1, group = " + toString( i ) );
			}
			drawPattern( i, position );
		}
	}
}



void SongEditor::drawPattern( int pos, int number )
{
	QPainter p( &m_sequencePixmap );
	QColor patternColor;

	if ( inSelect( pos, number, m_srcRegion ) ) {
		patternColor.setRgb( 140, 170, 210); // this color is used if the current cell's selected
	}
	else {
		patternColor.setRgb( 110, 140, 180);
	}

	int x = m_nGridWidth * pos;
	int y = m_nGridHeight * number;

	p.fillRect( x + 3, y + 3, m_nGridWidth - 5, m_nGridHeight - 5, patternColor );
}





// :::::::::::::::::::





SongEditorPatternList::SongEditorPatternList( QWidget *parent )
 : QWidget( parent )
 , Object( "SongEditPatList" )
{
	m_nWidth = 150;
	m_nGridHeight = (PreferencesMng::getInstance())->getPatternEditorGridHeight();

	m_bChanged = true;
	this->resize( m_nWidth, m_nInitialHeight );

	QPixmap patternIcon;
	string patternIcon_path = string( IMG_PATH ) + string( "/img/m_pSongEditor/patternIcon.png" );
	patternIcon.load( patternIcon_path.c_str() );

	m_pPatternPopup = new QPopupMenu( this, "patternPopupMenu" );
	m_pPatternPopup->insertItem( patternIcon, trUtf8("Edit"),  this, SLOT( patternPopup_edit() ) );
	m_pPatternPopup->insertItem( patternIcon, trUtf8("Copy"),  this, SLOT( patternPopup_copy() ) );
	m_pPatternPopup->insertItem( patternIcon, trUtf8("Delete"),  this, SLOT( patternPopup_delete() ) );
	m_pPatternPopup->insertItem( patternIcon, trUtf8("Properties"),  this, SLOT( patternPopup_properties() ) );

	createBackground();
	update();
}



SongEditorPatternList::~SongEditorPatternList() {
//	cout << "song editor PatternList destroy" << endl;
}



/// Single click, select the next pattern
void SongEditorPatternList::mousePressEvent( QMouseEvent *ev ) {
	int row = (ev->y() / m_nGridHeight);

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();

	if ( row >= (int)patternList->getSize() ) {
		return;
	}
	engine->setSelectedPatternNumber(row);

	warningLog( "[mousePressEvent] not implemented yet");

	if ( song->getMode() == Song::PATTERN_MODE ) {
		engine->setNextPattern( row );
	}

	if (ev->button() == RightButton) {
		m_pPatternPopup->popup( QPoint( ev->globalX(), ev->globalY() ) );
	}
	createBackground();
	update();
}



void SongEditorPatternList::mouseDoubleClickEvent( QMouseEvent *ev ) {
	int row = (ev->y() / m_nGridHeight);

	Hydrogen *engine = Hydrogen::getInstance();

	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();
	if ( row >= (int)patternList->getSize() ) {
		return;
	}

	engine->setSelectedPatternNumber(row);

	patternPopup_edit();
}



void SongEditorPatternList::paintEvent( QPaintEvent *ev ) {
	if (!isVisible()) {
		return;
	}

	if (m_bChanged) {
		m_bChanged = false;
		bitBlt( &m_tempPixmap, 0, 0, &m_backgroundPixmap, 0, 0, m_nWidth, height(), CopyROP );
		setErasePixmap( m_tempPixmap );
	}
}



void SongEditorPatternList::updateEditor()
{
	if(!isVisible()) {
		return;
	}

	update();
}



void SongEditorPatternList::createBackground()
{
	PreferencesMng *pref = PreferencesMng::getInstance();
	UIStyle *pStyle = pref->getDefaultUIStyle();
	QColor backgroundColor( pStyle->m_songEditor_backgroundColor.getRed(), pStyle->m_songEditor_backgroundColor.getGreen(), pStyle->m_songEditor_backgroundColor.getBlue() );
	QColor alternateRowColor( pStyle->m_songEditor_alternateRowColor.getRed(), pStyle->m_songEditor_alternateRowColor.getGreen(), pStyle->m_songEditor_alternateRowColor.getBlue() );
	QColor linesColor( pStyle->m_songEditor_lineColor.getRed(), pStyle->m_songEditor_lineColor.getGreen(), pStyle->m_songEditor_lineColor.getBlue() );
	QColor selectedRowColor( pStyle->m_songEditor_selectedRowColor.getRed(), pStyle->m_songEditor_selectedRowColor.getGreen(), pStyle->m_songEditor_selectedRowColor.getBlue() );
	QColor textColor( pStyle->m_songEditor_textColor.getRed(), pStyle->m_songEditor_textColor.getGreen(), pStyle->m_songEditor_textColor.getBlue() );

	QString family = pref->getApplicationFontFamily().c_str();
	int size = pref->getApplicationFontPointSize();
	QFont textFont( family, size );

	QFont boldTextFont( textFont);
	boldTextFont.setBold( true );


	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	uint nPatterns = song->getPatternList()->getSize();

	static int oldHeight = -1;
	int newHeight = m_nGridHeight * nPatterns;

	if (oldHeight != newHeight) {
		if (newHeight == 0) {
			newHeight = 1;	// the pixmap should not be empty
		}
		m_backgroundPixmap.resize( m_nWidth, newHeight );	// initialize the pixmap
		m_tempPixmap.resize( m_nWidth, newHeight );	// initialize the pixmap
		m_tempPixmap.fill();
		this->resize( m_nWidth, newHeight );
	}

	m_backgroundPixmap.fill( backgroundColor );

	QPainter p( &m_backgroundPixmap );
	p.setPen( QColor(0, 0, 0) );

	int nSelectedPattern = engine->getSelectedPatternNumber();

	for (uint i = 0; i < nPatterns; i++) {
		uint y = m_nGridHeight * i;
		if ( (int)i == nSelectedPattern ) {
			p.fillRect( 0, y, m_nWidth, m_nGridHeight, selectedRowColor );
		}
		else {
			if ( ( i % 2) == 0 ) {
				p.fillRect( 0, y, m_nWidth, m_nGridHeight, alternateRowColor );
			}
		}
	}

	PatternList *pCurrentPatternList = engine->getCurrentPatternList();

	// horizontal lines
	for (uint i = 0; i < nPatterns; i++) {
		Pattern *pPattern = song->getPatternList()->get(i);
		uint y = m_nGridHeight * i;
		p.setPen( linesColor );
		p.drawLine( 0, y, m_nWidth, y);

		// Text
		p.setPen( textColor );
		bool bActive = false;
		for (uint j = 0; j < pCurrentPatternList->getSize(); j++) {
			if ( pPattern == pCurrentPatternList->get(j) ) {
				bActive = true;
				break;
			}
		}
		if (bActive)
			p.setFont( boldTextFont );
		else
			p.setFont( textFont );

		uint text_y = i * m_nGridHeight + m_nGridHeight;
		string patternName = pPattern->getName();
		p.drawText( 5, text_y - 1 - m_nGridHeight, m_nWidth - 5, m_nGridHeight + 2, Qt::AlignVCenter, patternName.c_str() );
	}
	p.drawLine( 0, m_nGridHeight * nPatterns, m_nWidth, m_nGridHeight * nPatterns);

	p.drawLine( 0, 0, 0, height() );
	p.drawLine( m_nWidth - 1, 0, m_nWidth - 1, height() );

	m_bChanged = true;
}



void SongEditorPatternList::patternPopup_edit()
{
	HydrogenApp::getInstance()->getPatternEditorPanel()->show();
}



void SongEditorPatternList::patternPopup_properties()
{
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();

	int nSelectedPattern = engine->getSelectedPatternNumber();
	Pattern *pattern = patternList->get( nSelectedPattern );

	PatternPropertiesDialog *dialog = new PatternPropertiesDialog(this, pattern);
	if (dialog->exec() == QDialog::Accepted) {
		Hydrogen *engine = Hydrogen::getInstance();
		Song *song = engine->getSong();
		song->setModified( true );
		createBackground();
		update();
	}
	delete dialog;
	dialog = NULL;
}



void SongEditorPatternList::patternPopup_delete()
{
	Hydrogen *engine = Hydrogen::getInstance();
	engine->lockEngine( "SongEditorPatternList::patternPopup_delete" );

	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();

	Pattern *pattern = patternList->get( engine->getSelectedPatternNumber() );
	patternList->del(pattern);

	vector<PatternList*> *patternGroupVect = song->getPatternGroupVector();

	uint i = 0;
	while (i < patternGroupVect->size() ) {
		PatternList *list = (*patternGroupVect)[i];
		for (uint j = 0; j < list->getSize(); j++) {
			Pattern *pOldPattern = list->get( j );
			if (pOldPattern == pattern ) {
				list->del( j );
			}
		}
		if (list->getSize() == 0 ) {
			patternGroupVect->erase( patternGroupVect->begin() + i );
			delete list;
		}
		else {
			i++;
		}
	}

	engine->setSelectedPatternNumber( -1 );	// cosi' il pattern editor viene costretto ad aggiornarsi

	PatternList *list = engine->getCurrentPatternList();
	list->del( pattern );

	delete pattern;

	song->setModified(true);

	engine->unlockEngine();

	( HydrogenApp::getInstance() )->getSongEditorPanel()->updateAll();

}



void SongEditorPatternList::patternPopup_copy()
{
	Hydrogen *pEngine = Hydrogen::getInstance();
	Song *pSong = pEngine->getSong();
	PatternList *pPatternList = pSong->getPatternList();
	int nSelectedPattern = pEngine->getSelectedPatternNumber();
	Pattern *pPattern = pPatternList->get( nSelectedPattern );

	Pattern *pNewPattern = pPattern->copy();
	pPatternList->add( pNewPattern );

	// rename the copied pattern
	PatternPropertiesDialog *dialog = new PatternPropertiesDialog( this, pNewPattern );
	if ( dialog->exec() == QDialog::Accepted ) {
		pSong->setModified( true );
		pEngine->setSelectedPatternNumber(pPatternList->getSize() - 1);	// select the last pattern (the copied one)
		if (pSong->getMode() == Song::PATTERN_MODE) {
			pEngine->setNextPattern( pPatternList->getSize() - 1 );	// select the last pattern (the new copied pattern)
		}
	}
	else {
		pPatternList->del( pNewPattern );
		delete pNewPattern;
	}
	delete dialog;

	HydrogenApp::getInstance()->getSongEditorPanel()->updateAll();
}



// ::::::::::::::::::::::::::



SongEditorPositionRuler::SongEditorPositionRuler( QWidget *parent ) : QWidget( parent ), Object( "SongEditPosRuler" )
{
	m_bChanged = true;
	this->resize( m_nInitialWidth, m_nHeight );

	m_tempPixmap.resize( m_nInitialWidth, m_nHeight );	// initialize the pixmap
	m_tempPixmap.fill();

	m_backgroundPixmap.resize( m_nInitialWidth, m_nHeight );	// initialize the pixmap

	createBackground();	// create m_backgroundPixmap pixmap

	// create tick position pixmap
	string tickPosition_path = string( IMG_PATH ) + string( "/img/patternEditor/tickPosition.png" );
	bool ok = m_tickPositionPixmap.load(tickPosition_path.c_str());
	if( ok == false ){
		errorLog( "Error loading pixmap " + tickPosition_path );
	}

	update();

	m_pTimer = new QTimer(this);
	connect(m_pTimer, SIGNAL(timeout()), this, SLOT(updatePosition()));
	m_pTimer->start(500);

}



SongEditorPositionRuler::~SongEditorPositionRuler() {
	m_pTimer->stop();
}



void SongEditorPositionRuler::updateRuler() {
	update();
}



void SongEditorPositionRuler::createBackground()
{
	UIStyle *pStyle = PreferencesMng::getInstance()->getDefaultUIStyle();
	QColor backgroundColor( pStyle->m_songEditor_backgroundColor.getRed(), pStyle->m_songEditor_backgroundColor.getGreen(), pStyle->m_songEditor_backgroundColor.getBlue() );
	QColor linesColor( pStyle->m_songEditor_lineColor.getRed(), pStyle->m_songEditor_lineColor.getGreen(), pStyle->m_songEditor_lineColor.getBlue() );
	QColor textColor( pStyle->m_songEditor_textColor.getRed(), pStyle->m_songEditor_textColor.getGreen(), pStyle->m_songEditor_textColor.getBlue() );

	m_backgroundPixmap.fill( backgroundColor );

	PreferencesMng *pref = PreferencesMng::getInstance();
	QString family = pref->getApplicationFontFamily().c_str();
	int size = pref->getApplicationFontPointSize();
	QFont font( family, size );

	QPainter p( &m_backgroundPixmap );
	p.setFont( font );

	char tmp[10];
	for (uint i = 0; i < m_nMaxPatternSequence + 1; i++) {
		uint x = i * m_nGridWidth;
		if ( (i % 4) == 0 ) {
			p.setPen( textColor );
			sprintf( tmp, "%d", i );
			p.drawText( x - m_nGridWidth, 0, m_nGridWidth * 2, height(), Qt::AlignCenter, tmp );
		}
		else {
			p.setPen( linesColor );
			p.drawLine( x, 10, x, m_nHeight - 10 );
		}
	}

	p.setPen( linesColor );
	p.drawLine( 0, 0, width(), 0 );
	p.drawLine( 0, height() - 1, width(), height() - 1);

}



void SongEditorPositionRuler::mousePressEvent( QMouseEvent *ev ) {
	int column = (ev->x() / m_nGridWidth);

	( Hydrogen::getInstance() )->setPatternPos( column );
	update();
}



void SongEditorPositionRuler::paintEvent( QPaintEvent *ev )
{
	if (!isVisible()) {
		return;
	}

	static int oldPosition = -1000;
	int pos = ( Hydrogen::getInstance() )->getPatternPos();
	if ( oldPosition != pos ) {
		m_bChanged = true;
		oldPosition = pos;
	}

	if (m_bChanged) {
		m_bChanged = false;
		bitBlt( &m_tempPixmap, 0, 0, &m_backgroundPixmap, 0, 0, m_nInitialWidth, m_nHeight, CopyROP );

		if (pos != -1) {
			uint x = pos * m_nGridWidth;
			bitBlt( &m_tempPixmap, x + 3, height() / 2, &m_tickPositionPixmap, 0, 0, 11, 8, CopyROP );
		}

		setErasePixmap( m_tempPixmap );
	}
}



void SongEditorPositionRuler::updatePosition()
{
	update();
}



