/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: Hydrogen.cpp,v 1.71 2004/07/19 09:25:04 comix Exp $
 *
 */
#include "config.h"

#include <pthread.h>
#include <deque>
#include <sys/time.h>
#include <iostream>
#include <time.h>
#include <math.h>
using std::cout;
using std::cerr;
using std::endl;

#include "Sample.h"
#include "LocalFileMng.h"
#include "Hydrogen.h"

#include "drivers/JackDriver.h"
#include "drivers/OssDriver.h"
#include "drivers/NullDriver.h"
#include "drivers/DiskWriterDriver.h"
#include "drivers/AlsaMidiDriver.h"


// workaround for gcc 2.96
#if __GNUC__ < 3
inline int round(double x) { return x > 0 ? (int) (x+0.5) :
-(int)(-x+0.5); }
#endif


// GLOBALS

pthread_mutex_t m_engineLock_mutex;	/// Mutex for syncronized access to the song object
string m_sLocker = "";

GenericDriver *m_pAudioDriver = NULL;		/// Audio Driver

#ifdef USE_ALSA_SEQ	// check if ALSA_SEQ support is enabled
AlsaMidiDriver *midiDriver = NULL;	/// MIDI driver
#endif

uint m_nMaxNotes;
uint m_nPlayingNotes = 0;		/// number of playing notes

std::deque<Note*> songNoteQueue;		/// Song Note FIFO
std::deque<Note*> midiNoteQueue;		/// Midi Note FIFO
std::vector<Note*> playingNotesQueue;	/// Playing note FIFO


Song *m_pSong;		/// Current song
Pattern* m_pNextPattern;	/// Next pattern (used only in Pattern mode)

PatternList* m_pCurrentPatternList;
int m_nSongPos;		// E' la posizione all'interno della canzone
int m_nSelectedPatternNumber;
int m_nSelectedInstrumentNumber;

/** Metronome Instrument */
Instrument *metronomeInstrument = NULL;
Instrument *m_pPreviewInstrument = NULL;

float masterPeak_L = 0.0f;		/// Master peak (left channel)
float masterPeak_R = 0.0f;		/// Master peak (right channel)

// Buffers used in the process function
uint m_nBufferSize = 0;
float *m_pMainBuffer_L = NULL;
float *m_pMainBuffer_R = NULL;
float *m_pTrackBuffers_L[MAX_INSTRUMENTS];
float *m_pTrackBuffers_R[MAX_INSTRUMENTS];
bool m_bUseTrackOuts = false;
bool m_bUseDefaultOuts = false;


Hydrogen* Hydrogen::instance = NULL;		/// static reference of Hydrogen class (Singleton)
Hydrogen* hydrogenInstance = NULL;		/// Hydrogen class instance (used for log)


int  m_audioEngineState = UNINITIALIZED;		/// Audio engine state

std::vector<EngineListener*> audioEngine_listeners;

float fProcessTime = 0.0f;		/// time used in process function
float fMaxProcessTime = 0.0f;		/// max ms usable in process with no xrun


// LADSPA
uint m_nFXBufferSize = 0;
float* m_pFXBuffer_L[MAX_FX];
float* m_pFXBuffer_R[MAX_FX];
float m_fFXPeak_L[MAX_FX];
float m_fFXPeak_R[MAX_FX];


int m_nPatternStartTick = -1;
int m_nPatternTickPosition = 0;

// used in findPatternInTick
int m_nSongSizeInTicks = 0;

struct timeval m_currentTickTime;

unsigned long m_nRealtimeFrames = 0;



// PROTOTYPES
void	audioEngine_init();
void	audioEngine_destroy();
int	audioEngine_start(bool bLockEngine = false, uint nTotalFrames = 0);
void	audioEngine_stop(bool bLockEngine = false);
void	audioEngine_setSong(Song *newSong);
void	audioEngine_removeSong();
void	audioEngine_noteOn(Note *note);
void	audioEngine_noteOff(Note *note);
int	audioEngine_process(uint32_t nframes, void *arg);
inline void audioEngine_clearNoteQueue();
inline void audioEngine_process_checkBPMChanged();
inline void audioEngine_process_playNotes(unsigned long nframes);
inline void audioEngine_process_transport();

inline uint audioEngine_renderNote(Note* note, int nFrames, uint bufferSize);
inline int audioEngine_updateNoteQueue(uint nFrames);

inline int findPatternInTick( int tick, bool loopMode, int *patternStartTick );

void audioEngine_seek( unsigned long nFrames, bool bLoopMode = false);

void audioEngine_restartAudioDrivers();
void audioEngine_startAudioDrivers();
void audioEngine_stopAudioDrivers();


inline unsigned long currentTime() {
	struct timeval now;
	gettimeofday(&now, NULL);
	return now.tv_sec * 1000 + now.tv_usec / 1000;
}



inline timeval currentTime2() {
	struct timeval now;
	gettimeofday(&now, NULL);
	return now;
}



inline int randomValue( int max ) {
	return rand() % max;
}


inline void audioEngine_lock(string sLocker) {
	pthread_mutex_lock(&m_engineLock_mutex);
	//hydrogenInstance->infoLog( "\n>>>lock.   " + sLocker + "\n" );
	m_sLocker = sLocker;
}


inline void audioEngine_unlock() {
	//if (m_sLocker != "audioEngine_process" ) {
	//	hydrogenInstance->infoLog( "\n<<< unlock. " + m_sLocker + "\n" );
	//}
	pthread_mutex_unlock(&m_engineLock_mutex);
}



void audioEngine_raiseError( unsigned nErrorCode ) {
	// raise error event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->engineError( nErrorCode );
	}
}



void updateTickSize() {
//	hydrogenInstance->infoLog("UpdateTickSize");
	float bpm = m_pSong->getBpm();
	float resolution = m_pSong->getResolution();
	float sampleRate = (float)m_pAudioDriver->getSampleRate();
	m_pAudioDriver->m_transport.m_nTickSize = ( sampleRate * 60.0 /  bpm / resolution );
}



void audioEngine_init()
{
	hydrogenInstance->infoLog("Hydrogen audio engine init");

	// check current state
	if (m_audioEngineState != UNINITIALIZED) {
		hydrogenInstance->errorLog("[audioEngine_init]: Error the audio engine is not in UNINITIALIZED state");
		audioEngine_unlock();
		return;
	}

	m_pSong = NULL;
	m_pCurrentPatternList = new PatternList();
	m_pNextPattern = NULL;
	m_nSongPos = -1;
	m_nSelectedPatternNumber = -1;
	m_nSelectedInstrumentNumber = 0;
	m_nPatternTickPosition = 0;
	metronomeInstrument = NULL;
	m_pAudioDriver = NULL;

	m_pMainBuffer_L = NULL;
	m_pMainBuffer_R = NULL;
	for (uint i = 0; i < MAX_FX; i++) {
		m_pFXBuffer_L[i] = NULL;
		m_pFXBuffer_R[i] = NULL;
	}
	for (uint i=0; i < MAX_INSTRUMENTS; i++) {
		m_pTrackBuffers_L[i] = NULL;
		m_pTrackBuffers_R[i] = NULL;
	}

	PreferencesMng *preferences = PreferencesMng::getInstance();
	m_nMaxNotes = preferences->getMaxNotes();
	m_bUseTrackOuts = preferences->isJackTrackOuts();
	m_bUseDefaultOuts = preferences->isJackConnectDefaults();

	pthread_mutex_init(&m_engineLock_mutex, NULL);

	srand(time(NULL));

	// Create metronome instrument
	string sMetronomeFilename = string(DATA_DIR) + "/click.wav";
	metronomeInstrument = new Instrument( sMetronomeFilename, "metronome", "", 0.0 );
	metronomeInstrument->setLayer( new InstrumentLayer( Sample::load( sMetronomeFilename ) ), 0 );

	// instrument used in file preview
	string sEmptySampleFilename = string(DATA_DIR) + "/emptySample.wav";
	m_pPreviewInstrument = new Instrument( sEmptySampleFilename, "preview", "", 0.8 );
	m_pPreviewInstrument->setLayer( new InstrumentLayer( Sample::load( sEmptySampleFilename ) ), 0 );

	// set initial metronome volume
	if ( preferences->isMetronomeEnabled() ) {
		metronomeInstrument->setVolume( preferences->getMetronomeVolume() );
	}
	else {
		metronomeInstrument->setVolume( 0.0 );
	}

	// Change the current audio engine state
	m_audioEngineState = INITIALIZED;
	hydrogenInstance->infoLog( "[audioEngine_init] INITIALIZED" );

	// raise stateChanged event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->stateChanged( INITIALIZED );
	}
}



void audioEngine_destroy()
{
	audioEngine_lock("audioEngine_destroy");
	hydrogenInstance->infoLog("Hydrogen audio engine destroy");

	// check current state
	if (m_audioEngineState != INITIALIZED) {
		hydrogenInstance->errorLog("[audioEngine_destroy] Error the audio engine is not in INITIALIZED state");
		return;
	}

	// change the current audio engine state
	m_audioEngineState = UNINITIALIZED;
	hydrogenInstance->infoLog( "[audioEngine_destroy] UNINITIALIZED" );

	// raise stateChanged event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->stateChanged(UNINITIALIZED);
	}

	delete m_pCurrentPatternList;

	// delete all copied notes in the song notes queue
	for (uint i = 0; i < songNoteQueue.size(); i++) {
		Note *note = songNoteQueue[i];
		delete note;
		note = NULL;
	}
	songNoteQueue.clear();

	// delete all copied notes in the playing notes queue
	for (uint i = 0; i < playingNotesQueue.size(); i++) {
		Note *note = playingNotesQueue[i];
		delete note;
		note = NULL;
	}
	playingNotesQueue.clear();

	// delete all copied notes in the midi notes queue
	for (uint i = 0; i < midiNoteQueue.size(); i++) {
		Note *note = midiNoteQueue[i];
		delete note;
		note = NULL;
	}
	midiNoteQueue.clear();

	hydrogenInstance->infoLog( "[audioEngine_destroy] all notes buffers cleared" );

	delete metronomeInstrument;
	metronomeInstrument = NULL;

	delete m_pPreviewInstrument;
	m_pPreviewInstrument = NULL;

	for (uint i = 0; i < MAX_FX; i++) {
		delete m_pFXBuffer_L[i];
		m_pFXBuffer_L[i] = NULL;
	}

	audioEngine_unlock();
}





/// Start playing
/// return 0 = OK
/// return -1 = NULL Audio Driver
/// return -2 = Driver connect() error
int audioEngine_start(bool bLockEngine, uint nTotalFrames) {
	if (bLockEngine) {
		audioEngine_lock("audioEngine_start");
	}

	hydrogenInstance->infoLog("[audioEngine_start]");

	// check current state
	if (m_audioEngineState != READY) {
		hydrogenInstance->errorLog("[audioEngine_start]: Error the audio engine is not in READY state");
		if (bLockEngine) {
			audioEngine_unlock();
		}
		return 0;	// FIXME!!
	}


	PreferencesMng *preferencesMng = PreferencesMng::getInstance();
	masterPeak_L = 0.0f;
	masterPeak_R = 0.0f;
	m_pAudioDriver->m_transport.m_nFrames = nTotalFrames;	// reset total frames
	m_nSongPos = -1;
	m_nPatternStartTick = -1;
	m_nPatternTickPosition = 0;

	// prepare the tickSize for this song
	updateTickSize();

	m_nMaxNotes = preferencesMng->getMaxNotes();

	// set initial metronome volume
	if ( preferencesMng->isMetronomeEnabled() ) {
		metronomeInstrument->setVolume( preferencesMng->getMetronomeVolume() );
	}
	else {
		metronomeInstrument->setVolume( 0.0 );
	}


	// change the current audio engine state
	m_audioEngineState = PLAYING;
	hydrogenInstance->infoLog("[audioEngine_start] PLAYING");
	// raise stateChanged event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->stateChanged(PLAYING);
	}

	if (bLockEngine) {
		audioEngine_unlock();
	}
	return 0; // per ora restituisco sempre OK
}



/// Stop the audio engine
void audioEngine_stop(bool bLockEngine) {
	if (bLockEngine) {
		audioEngine_lock( "audioEngine_stop" );
	}
	hydrogenInstance->infoLog( "[audioEngine_stop]" );

	// check current state
	if (m_audioEngineState != PLAYING) {
		hydrogenInstance->errorLog("[audioEngine_stop]: Error the audio engine is not in PLAYING state");
		cout << "state = " << m_audioEngineState << endl;
		if (bLockEngine) {
			audioEngine_unlock();
		}
		return;
	}

	// change the current audio engine state
	m_audioEngineState = READY;
	hydrogenInstance->infoLog("[audioEngine_stop] READY");
	// raise stateChanged event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->stateChanged(READY);
	}

	masterPeak_L = 0.0f;
	masterPeak_R = 0.0f;
	m_nPatternTickPosition = 0;
	m_nPatternStartTick = -1;

	// delete all copied notes in the song notes queue
	for (uint i = 0; i < songNoteQueue.size(); i++) {
		Note *note = songNoteQueue[i];
		delete note;
	}
	songNoteQueue.clear();

/*	// delete all copied notes in the playing notes queue
	for (uint i = 0; i < playingNotesQueue.size(); i++) {
		Note *note = playingNotesQueue[i];
		delete note;
	}
	playingNotesQueue.clear();
*/

	// delete all copied notes in the midi notes queue
	for (uint i = 0; i < midiNoteQueue.size(); i++) {
		Note *note = midiNoteQueue[i];
		delete note;
	}
	midiNoteQueue.clear();


	if (bLockEngine) {
		audioEngine_unlock();
	}
}



inline void audioEngine_process_checkBPMChanged()
{
	if ( ( m_audioEngineState == READY) || (m_audioEngineState == PLAYING ) ) {

		float fSampleRate = m_pAudioDriver->getSampleRate();
		float fBPM = m_pSong->getBpm();

		float newTickSize = ( fSampleRate * 60.0 /  fBPM / m_pSong->getResolution() );

		if (newTickSize != m_pAudioDriver->m_transport.m_nTickSize) {


			// update tickSize
			m_pAudioDriver->m_transport.m_nTickSize = newTickSize;
			hydrogenInstance->infoLog("[audioEngine_process_checkBPMChanged] TickSize=" + toString(newTickSize) + ", transport bpm=" + toString(m_pAudioDriver->m_transport.m_nBPM) + ", song bpm=" + toString(m_pSong->getBpm()) );

			// delete all copied notes in the song notes queue
			for (uint i = 0; i < songNoteQueue.size(); i++) {
				delete songNoteQueue[i];
			}
			songNoteQueue.clear();

		/*	// delete all copied notes in the playing notes queue
			for (uint i = 0; i < playingNotesQueue.size(); i++) {
				Note *note = playingNotesQueue[i];
				delete note;
			}
			playingNotesQueue.clear();
		*/

			// delete all copied notes in the midi notes queue
			for (uint i = 0; i < midiNoteQueue.size(); i++) {
				delete midiNoteQueue[i];
			}
			midiNoteQueue.clear();
		}
	}
}



inline void audioEngine_process_playNotes ( unsigned long nframes )
{
	unsigned int framepos;

	if (  m_audioEngineState == PLAYING ) {
		framepos = m_pAudioDriver->m_transport.m_nFrames;
	}
	else {
		// use this to support realtime events when not playing
		framepos = m_nRealtimeFrames;
	}

	// leggo da songNoteQueue
	while (songNoteQueue.size() > 0) {
		Note *note = songNoteQueue[0];

		// verifico se la nota rientra in questo ciclo
		uint noteStartInFrames = (uint) (note->getPosition() * m_pAudioDriver->m_transport.m_nTickSize);

		// m_nTotalFrames <= NotePos < m_nTotalFrames + bufferSize
		bool isNoteStart = ( ( noteStartInFrames >= framepos ) && ( noteStartInFrames < ( framepos + nframes ) ) );
		bool isOldNote = noteStartInFrames < framepos;
		if ( isNoteStart || isOldNote ) {
			playingNotesQueue.push_back( note );		// aggiungo la nota alla lista di note da eseguire
			songNoteQueue.pop_front();			// rimuovo la nota dalla lista di note

			// exclude notes
			Instrument *instr = note->getInstrument();
			if ( instr->m_excludeVect.size() != 0 ) {
				for (uint i = 0; i < instr->m_excludeVect.size(); i++) {
					Instrument *pExcluded = instr->m_excludeVect[ i ];
					// remove all notes listed in excludeVect
					for ( uint j = 0; j < playingNotesQueue.size(); j++ ) {	// delete older note
						Note *oldNote = playingNotesQueue[ j ];

						if ( oldNote->getInstrument() == pExcluded ) {
							//hydrogenInstance->infoLog("Delete excluded note");
							playingNotesQueue.erase( playingNotesQueue.begin() + j );
							delete oldNote;
						}
					}
				}
			}

			// raise noteOn event
			for ( uint nList = 0; nList < audioEngine_listeners.size(); nList++ ) {
				audioEngine_listeners[nList]->noteOn( note->copy() );	// the note must be deleted!
			}
			continue;
		}
		else {
			// la nota non andra' in esecuzione
			break;
		}
	}

}



void audioEngine_seek( unsigned long nFrames, bool bLoopMode)
{
	if (m_pAudioDriver->m_transport.m_nFrames == nFrames) {
		return;
	}

	char tmp[200];
	sprintf(tmp, "[audioEngine_seek()] seek in %d (old pos = %d)", (int)nFrames, (int)m_pAudioDriver->m_transport.m_nFrames);
	hydrogenInstance->warningLog(tmp);

	m_pAudioDriver->m_transport.m_nFrames = nFrames;

	int tickNumber_start = (uint)( m_pAudioDriver->m_transport.m_nFrames / m_pAudioDriver->m_transport.m_nTickSize );
//	sprintf(tmp, "[audioEngine_seek()] tickNumber_start = %d", tickNumber_start);
//	hydrogenInstance->infoLog(tmp);

	bool loop = m_pSong->isLoopEnabled();

	if (bLoopMode) {
		loop = true;
	}

	m_nSongPos = findPatternInTick( tickNumber_start, loop, &m_nPatternStartTick );
//	sprintf(tmp, "[audioEngine_seek()] m_nSongPos = %d", m_nSongPos);
//	hydrogenInstance->infoLog(tmp);

	audioEngine_clearNoteQueue();
}



inline void audioEngine_process_transport()
{
	if ( (m_audioEngineState == READY ) || (m_audioEngineState == PLAYING) ) {

		m_pAudioDriver->updateTransportInfo();
		unsigned long nNewFrames = m_pAudioDriver->m_transport.m_nFrames;

		audioEngine_seek( nNewFrames, true );

		switch ( m_pAudioDriver->m_transport.m_status ) {
		case TransportInfo::ROLLING:

			if ( m_audioEngineState == READY ) {
				//hydrogenInstance->infoLog( "[audioEngine_process_transport] first start frames is " + toString(m_pAudioDriver->m_transport.m_nFrames) );
				audioEngine_start( false, m_pAudioDriver->m_transport.m_nFrames );	// no engine lock
			}

			if (m_pSong->getBpm() != m_pAudioDriver->m_transport.m_nBPM ) {
				m_pSong->setBpm( m_pAudioDriver->m_transport.m_nBPM );
			}

			m_nRealtimeFrames = m_pAudioDriver->m_transport.m_nFrames;
			break;


		case TransportInfo::STOPPED:
			if ( m_audioEngineState == PLAYING ) {
				audioEngine_stop(false);	// no engine lock
			}

			// go ahead and increment the realtimeframes by buffersize
			// to support our realtime keyboard and midi event timing
			m_nRealtimeFrames += m_nBufferSize;

			break;
		}
	}
}



void audioEngine_clearNoteQueue() {
	hydrogenInstance->infoLog("[audioEngine_clearNoteQueue()]");

	for (uint i = 0; i < songNoteQueue.size(); i++) {	// delete all copied notes in the song notes queue
		delete songNoteQueue[i];
	}
	songNoteQueue.clear();

	for (uint i = 0; i < playingNotesQueue.size(); i++) {	// delete all copied notes in the playing notes queue
		delete playingNotesQueue[i];
	}
	playingNotesQueue.clear();

	for (uint i = 0; i < midiNoteQueue.size(); i++) {	// delete all copied notes in the midi notes queue
		delete midiNoteQueue[i];
	}
	midiNoteQueue.clear();

}



/// Clear all audio buffers
inline void audioEngine_process_clearAudioBuffers(uint32_t nFrames)
{
	memset( m_pMainBuffer_L, 0, nFrames * sizeof( float ) );	// clear main out Left
	memset( m_pMainBuffer_R, 0, nFrames * sizeof( float ) );	// clear main out Right
	if ( ( m_audioEngineState == READY ) || ( m_audioEngineState == PLAYING) ) {
		for (uint i = 0; i < MAX_FX; i++) {	// clear FX buffers
			memset( m_pFXBuffer_L[i], 0, nFrames * sizeof( float ) );
			memset( m_pFXBuffer_R[i], 0, nFrames * sizeof( float ) );
		}
	}
	if (m_bUseTrackOuts) {
	 	for (uint n=0; n < MAX_INSTRUMENTS; ++n) {	// clear track buffers
 			memset( m_pTrackBuffers_L[n], 0, nFrames * sizeof( float ) );
 			memset( m_pTrackBuffers_R[n], 0, nFrames * sizeof( float ) );
	 	}
	}
}



/// Main audio processing function. Called by audio drivers.
int audioEngine_process(uint32_t nframes, void *arg)
{
	// try to lock the engine mutex. If fails returns without sound processing
	int res = pthread_mutex_trylock(&m_engineLock_mutex);
	if (res != 0) {
		hydrogenInstance->warningLog( "[audioEngine_process] trylock != 0. Frames = " + toString(m_pAudioDriver->m_transport.m_nFrames ) + ". Lock in " + m_sLocker );
		return 0;
	}
	m_sLocker = "audioEngine_process";

	timeval startTimeval = currentTime2();


	if ( m_nBufferSize != nframes ) {
		hydrogenInstance->infoLog( "[audioEngine_process] Buffer size changed. Old size = " + toString(m_nBufferSize) +", new size = " + toString(nframes) );
		if ( m_nFXBufferSize != nframes ) {
			hydrogenInstance->errorLog( "[audioEngine_process] Ladspa FX buffersize != nframes. FXBuffersize = " + toString( m_nFXBufferSize ) );
		}
		m_nBufferSize = nframes;
	}

	audioEngine_process_transport();

	audioEngine_process_clearAudioBuffers(nframes);

	audioEngine_process_checkBPMChanged();

	// check ticksize
	if ( m_pAudioDriver->m_transport.m_nTickSize == 0 ) {
		hydrogenInstance->warningLog( "[audioEngine_process] m_transport.m_nTickSize == 0" );
		if ( m_audioEngineState != PREPARED ) {
			hydrogenInstance->errorLog( "m_transport.m_nTickSize == 0 m_audioEngineState != PREPARED!" );
		}
		return 0;	// return without sound processing
	}

	bool sendPatternChange = false;
	// always update note queue.. could come from pattern or realtime input (midi, keyboard)
	int res2 = audioEngine_updateNoteQueue( nframes );	// update the notes queue
	if ( res2 == -1 ) {	// end of song
		hydrogenInstance->infoLog( "[audioEngine_process] End of song received, calling engine_stop()" );
		audioEngine_unlock();
		m_pAudioDriver->stop();
		m_pAudioDriver->locate( 0 );	// locate 0, reposition from start of the song

		if ( m_pAudioDriver->getClassName() == "DiskWriterDriver" ) {
			hydrogenInstance->infoLog( "[audioEngine_process] Exiting DiskWriterDriver thread!!" );
			return 1;	// kill the audio AudioDriver thread
		}

		return 0;
	}
	else if ( res2 == 2 ) {	// send pattern change
		sendPatternChange = true;
	}


	// play all notes
	audioEngine_process_playNotes( nframes );

	// Max notes
	while ( playingNotesQueue.size() > m_nMaxNotes ) {
		Note *oldNote = playingNotesQueue[ 0 ];
		playingNotesQueue.erase( playingNotesQueue.begin() );
		delete oldNote;	// FIXME: send note-off instead of removing the note from the list?
	}
	m_nPlayingNotes = playingNotesQueue.size();	// update the number of playing notes


	timeval renderTime_start = currentTime2();
	// eseguo tutte le note nella lista di note in esecuzione
	uint i = 0;
	Note* note = NULL;
	while ( i < playingNotesQueue.size() ) {
		note = playingNotesQueue[ i ];		// recupero una nuova nota
		uint res = audioEngine_renderNote( note, nframes, nframes );
		if ( res == 1 ) {	// la nota e' finita
			playingNotesQueue.erase( playingNotesQueue.begin() + i );
			if ( note != NULL ) {
				delete note;
				note = NULL;
			}
		}
		else {
			i++; // carico la prox nota
		}
	}
	timeval renderTime_end = currentTime2();


	// Process LADSPA FX
	timeval ladspaTime_start = currentTime2();
	if ( ( m_audioEngineState == READY ) || ( m_audioEngineState == PLAYING ) ) {
		for (uint nFX = 0; nFX < MAX_FX; nFX++) {
			LadspaFX *pFX = m_pSong->getLadspaFX( nFX );
			if ( ( pFX ) && (pFX->isEnabled()) ) {
				pFX->processFX( nframes );
				float *buf_L = NULL;
				float *buf_R = NULL;
				if (pFX->getPluginType() == LadspaFX::STEREO_FX) {	// STEREO FX
					buf_L = m_pFXBuffer_L[nFX];
					buf_R = m_pFXBuffer_R[nFX];
				}
				else { // MONO FX
					buf_L = m_pFXBuffer_L[nFX];
					buf_R = buf_L;
				}
				for ( uint i = 0; i < nframes; i++) {
					m_pMainBuffer_L[ i ] += buf_L[ i ];
					m_pMainBuffer_R[ i ] += buf_R[ i ];
					if (buf_L[ i ] > m_fFXPeak_L[nFX] )	m_fFXPeak_L[nFX] = buf_L[ i ];
					if (buf_R[ i ] > m_fFXPeak_R[nFX] )	m_fFXPeak_R[nFX] = buf_R[ i ];
				}
			}
		}
	}
	timeval ladspaTime_end = currentTime2();

	// update master peaks
	float val_L;
	float val_R;
	for ( uint i = 0; i < nframes; i++) {
		val_L = m_pMainBuffer_L[i];
		val_R = m_pMainBuffer_R[i];
		if (val_L > masterPeak_L) {
			masterPeak_L = val_L;
		}
		if (val_R > masterPeak_R) {
			masterPeak_R = val_R;
		}
	}

	// update total frames number
	if ( m_audioEngineState == PLAYING ) {
		m_pAudioDriver->m_transport.m_nFrames += nframes;
	}

	float fRenderTime = (renderTime_end.tv_sec - renderTime_start.tv_sec) * 1000.0 + (renderTime_end.tv_usec - renderTime_start.tv_usec) / 1000.0;
	float fLadspaTime = (ladspaTime_end.tv_sec - ladspaTime_start.tv_sec) * 1000.0 + (ladspaTime_end.tv_usec - ladspaTime_start.tv_usec) / 1000.0;


	timeval finishTimeval = currentTime2();
	fProcessTime = (finishTimeval.tv_sec - startTimeval.tv_sec) * 1000.0 + (finishTimeval.tv_usec - startTimeval.tv_usec) / 1000.0;

	float sampleRate = (float)m_pAudioDriver->getSampleRate();
	fMaxProcessTime = 1000.0 / (sampleRate / nframes);


	//DEBUG
	if ( fProcessTime > fMaxProcessTime ) {
		hydrogenInstance->warningLog( "" );
		hydrogenInstance->warningLog( "----XRUN----" );
		hydrogenInstance->warningLog( "XRUN of " + toString((fProcessTime - fMaxProcessTime)) + string(" msec (") + toString(fProcessTime) + string(" > ") + toString(fMaxProcessTime) + string(")") );
		hydrogenInstance->warningLog( "Playing notes = " + toString( m_nPlayingNotes ) + string( ", render time = ") + toString( fRenderTime ) );
		hydrogenInstance->warningLog( "Ladspa process time = " + toString( fLadspaTime ) );
		hydrogenInstance->warningLog( "------------" );
		hydrogenInstance->warningLog( "" );
		// raise xRun event
		for (uint i = 0; i < audioEngine_listeners.size(); ++i) {
			audioEngine_listeners[i]->xRunEvent( fProcessTime - fMaxProcessTime, fMaxProcessTime );
		}
	}

	audioEngine_unlock();

	if ( sendPatternChange ) {
//		hydrogenInstance->infoLog( "[audioEngine_process] pattern change" );
		// raise patternChanged event
		for (uint i = 0; i < audioEngine_listeners.size(); ++i) {
			audioEngine_listeners[i]->patternChanged();
		}
	}

	return 0;
}



/// Render a note
/// Return 0: the note is not ended
/// Return 1: the note is ended
inline uint audioEngine_renderNote(Note* note, int nFrames, uint bufferSize)
{
	// nFrames = frame disponibili da copiare nei buffer
	/*if (note->samplePosition == 0) {
		hydrogenInstance->infoLog( "note on" );
	}*/

	unsigned int nFramepos;
	if (  m_audioEngineState == PLAYING ) {
		nFramepos = m_pAudioDriver->m_transport.m_nFrames;
	}
	else {
		// use this to support realtime events when not playing
		nFramepos = m_nRealtimeFrames;
	}


	Instrument *instr = note->getInstrument();
	if (instr == NULL) {
		hydrogenInstance->errorLog( "[audioEngine_renderNote] NULL instrument" );
		return 1;
	}

	int track_n = m_pSong->getInstrumentList()->getPos(instr);

	float fLayerGain = 1.0;
	float fLayerPitch = 0.0;

	// scelgo il sample da usare in base alla velocity
	Sample *pSample = NULL;
	for (uint nLayer = 0; nLayer < MAX_LAYERS; nLayer++) {
		InstrumentLayer *pLayer = instr->getLayer( nLayer );
		if ( pLayer == NULL ) continue;

		if ( ( note->getVelocity() >= pLayer->getStartVelocity() ) && ( note->getVelocity() <= pLayer->getEndVelocity() ) ) {
			pSample = pLayer->getSample();
			fLayerGain = pLayer->getGain();
			fLayerPitch = pLayer->getPitch();
			break;
		}
	}
	if (pSample == NULL) {
		hydrogenInstance->warningLog("[audioEngine_renderNote] NULL sample");
		return 1;
	}

	float *sample_data_L = pSample->getData_L();
	float *sample_data_R = pSample->getData_R();
	float instrPeak_L = instr->getPeak_L(); // this value will be reset to 0 by the mixer..
	float instrPeak_R = instr->getPeak_R(); // this value will be reset to 0 by the mixer..

	int noteStartInFrames = (int) ( note->getPosition() * m_pAudioDriver->m_transport.m_nTickSize ) + note->getHumanizeDelay();

	int nInitialSilence = 0;
	if (noteStartInFrames > (int) nFramepos) {	// scrivo silenzio prima dell'inizio della nota
		nInitialSilence = noteStartInFrames - nFramepos;
		nFrames = nFrames - nInitialSilence;
		if (nFrames < 0) {	// delay note execution
			return 0;
		}
	}

	float cost_L = 1;
	float cost_R = 1;
	float cost_track = 1;

	if (instr->isMuted()) {	// is instrument muted?
		cost_L = 0.0;
		cost_R = 0.0;
	}
	else {	// Precalcolo i valori per le nota
		cost_L = cost_L * note->getVelocity();		// note velocity
		cost_L = cost_L * note->getPan_L();		// note pan
		cost_L = cost_L * instr->getPan_L();		// instrument pan
		cost_L = cost_L * instr->getVolume();		// instrument volume
		cost_L = cost_L * fLayerGain;			// layer gain
		cost_L = cost_L * m_pSong->getVolume();	// song volume

		cost_R = cost_R * note->getVelocity();	// note velocity
		cost_R = cost_R * note->getPan_R();		// note pan
		cost_R = cost_R * instr->getPan_R();		// instrument pan
		cost_R = cost_R * instr->getVolume();		// instrument volume
		cost_R = cost_R * fLayerGain;			// layer gain
		cost_R = cost_R * m_pSong->getVolume();	// song pan
	}

	// direct track outputs only use velocity
	cost_track = note->getVelocity();

	int nNoteLength = -1;
	if (note->getLength() != -1) {
		nNoteLength = (int)( note->getLength() * m_pAudioDriver->m_transport.m_nTickSize );
	}
	float fNotePitch = note->getPitch() + fLayerPitch;

	// Se non devo fare resample (drumkit) posso evitare di utilizzare i float e gestire il tutto in
	// maniera ottimizzata
	//	constant^12 = 2, so constant = 2^(1/12) = 1.059463.
	//	float nStep = 1.0;1.0594630943593

	float fStep = 1.0;
	if ((int)fNotePitch == 0.0) {	// NO RESAMPLE
		int avail_bytes = pSample->getNFrames() - (int)note->m_fSamplePosition;	// verifico il numero di frame disponibili ancora da eseguire
		if ( ( nNoteLength != -1 ) && ( nNoteLength < (avail_bytes + note->m_fSamplePosition)  ) ) {
			avail_bytes = (int)( nNoteLength - note->m_fSamplePosition );	// taglia la nota in base alla durata
		}


		int retValue = 1; // the note is ended
		if (avail_bytes > (int)(bufferSize - nInitialSilence)) {	// il sample e' piu' grande del buffersize
			// imposto il numero dei bytes disponibili uguale al buffersize
			avail_bytes = bufferSize - nInitialSilence;
			retValue = 0; // the note is not ended yet
		}

		int nInitialBufferPos = nInitialSilence;
		int nInitialSamplePos = (int)note->m_fSamplePosition;

		// copio il sample nel buffer
		float val_L;
		float val_R;
		int bufferPos = nInitialBufferPos;
		int samplePos = nInitialSamplePos;
		int i;
		for (i = 0; i < avail_bytes; ++i) {
			val_L = sample_data_L[samplePos] * cost_L;
			val_R = sample_data_R[samplePos] * cost_R;

			m_pMainBuffer_L[bufferPos] += val_L;
			m_pMainBuffer_R[bufferPos] += val_R;

			if (m_bUseTrackOuts && track_n >= 0) {	// for the track output
				m_pTrackBuffers_L[track_n][bufferPos] = sample_data_L[samplePos] * cost_track;
				m_pTrackBuffers_R[track_n][bufferPos] = sample_data_R[samplePos] * cost_track;
			}

			// update instr peak
			if (val_L > instrPeak_L) {	instrPeak_L = val_L;	}
			if (val_R > instrPeak_R) {	instrPeak_R = val_R;	}

			++samplePos; //samplePos += 1;
			++bufferPos;
		}
		note->m_fSamplePosition += i;
		instr->setPeak_L( instrPeak_L );
		instr->setPeak_R( instrPeak_R );

		// LADSPA
		for (uint nFX = 0; nFX < MAX_FX; nFX++) {
			LadspaFX *pFX = m_pSong->getLadspaFX( nFX );
			float fLevel = instr->getFXLevel(nFX);
			if ( ( fLevel != 0.0 ) && ( pFX ) ) {
				fLevel = fLevel * pFX->getVolume();
				float *buf_L = m_pFXBuffer_L[nFX];
				float *buf_R = m_pFXBuffer_R[nFX];
				int bufferPos = nInitialBufferPos;
				int samplePos = nInitialSamplePos;
				for (int i = 0; i < avail_bytes; ++i) {
					val_L = sample_data_L[samplePos] * cost_L;
					val_R = sample_data_R[samplePos] * cost_R;
					buf_L[ bufferPos ] += val_L * fLevel;
					buf_R[ bufferPos ] += val_R * fLevel;
					++samplePos; //samplePos += 1;
					++bufferPos;
				}
			}
		}
		// ~LADSPA

		return retValue;
	}
	else {	// RESAMPLE
		//hydrogenInstance->infoLog( "resample: " + toString( fNotePitch ) );
		// posso precalcolare questi valori, conviene?
		fStep = pow( 1.0594630943593, (int)fNotePitch );	/// \todo precalcolare o cercare un modo migliore per questo		//fStep = 2;	// test only
		int avail_bytes = (int)( (float)(pSample->getNFrames() - note->m_fSamplePosition) / fStep );	// verifico il numero di frame disponibili ancora da eseguire
		if ( ( nNoteLength != -1 ) && ( nNoteLength < (avail_bytes + note->m_fSamplePosition)  ) ) {
			avail_bytes = (int)( nNoteLength - note->m_fSamplePosition );
		}

		int retValue = 1; // the note is ended
		if (avail_bytes > (int)(bufferSize - nInitialSilence)) {	// il sample e' piu' grande del buffersize
			// imposto il numero dei bytes disponibili uguale al buffersize
			avail_bytes = bufferSize - nInitialSilence;
			retValue = 0; // the note is not ended yet
		}

		int nInitialBufferPos = nInitialSilence;
		float nInitialSamplePos = note->m_fSamplePosition;

		// copio il sample nel buffer
		float fVal_L;
		float fVal_R;
		float fSample_L;
		float fSample_R;
		int nBufferPos = nInitialSilence;
		float fSamplePos = note->m_fSamplePosition;
		int i;
		float fVal1, fVal2;	// used in linear interpolation
		for (i = 0; i < avail_bytes; ++i) {
			// linear interpolation
			// new = old(int)+(old(int+1)-old(int))*fract
			//8-11-2003	samplePos += nStep;
			/// \todo: error! samplepos+1 can exceed the array bounds
			int nSamplePos = (int)fSamplePos;


/*			sample_L = sample_data_L[nSamplePos] + ( sample_data_L[nSamplePos + 1] - sample_data_L[nSamplePos] ) * (fSamplePos - nSamplePos);
			sample_R = sample_data_R[nSamplePos] + ( sample_data_R[nSamplePos + 1] - sample_data_R[nSamplePos] ) * (fSamplePos - nSamplePos);*/

			if ( ( nSamplePos + 1 ) >= (int)pSample->getNFrames() ) {
				// out of bounds..
				//hydrogenInstance->errorLog( "out of bounds..." );
				continue;
			}


			fVal1 = sample_data_L[nSamplePos];
			fVal2 = sample_data_L[nSamplePos + 1];
			fSample_L = fVal1 + ( fVal2 - fVal1 ) * (fSamplePos - nSamplePos);

			fVal1 = sample_data_R[nSamplePos];
			fVal2 = sample_data_R[nSamplePos + 1];
			fSample_R = fVal1 + ( fVal2 - fVal1 ) * (fSamplePos - nSamplePos);

			fVal_L = fSample_L * cost_L;
			fVal_R = fSample_R * cost_R;

			m_pMainBuffer_L[nBufferPos] += fVal_L;
			m_pMainBuffer_R[nBufferPos] += fVal_R;

			if (m_bUseTrackOuts && track_n >= 0) {	// for the track output
				m_pTrackBuffers_L[track_n][nBufferPos] = fSample_L * cost_track;
				m_pTrackBuffers_R[track_n][nBufferPos] = fSample_R * cost_track;
			}

			// update instr peak
			if (fVal_L > instrPeak_L) {	instrPeak_L = fVal_L;	}
			if (fVal_R > instrPeak_R) {	instrPeak_R = fVal_R;	}

			fSamplePos += fStep;
			++nBufferPos;
		}
		note->m_fSamplePosition += i * fStep;
		instr->setPeak_L( instrPeak_L );
		instr->setPeak_R( instrPeak_R );

		// LADSPA
		for (uint nFX = 0; nFX < MAX_FX; nFX++) {
			LadspaFX *pFX = m_pSong->getLadspaFX( nFX );
			float fLevel = instr->getFXLevel(nFX);
			if ( ( fLevel != 0.0 ) && ( pFX ) ) {
				fLevel = fLevel * pFX->getVolume();
				float *buf_L = m_pFXBuffer_L[nFX];
				float *buf_R = m_pFXBuffer_R[nFX];
				int bufferPos = nInitialBufferPos;
				float samplePos = nInitialSamplePos;
				for (int i = 0; i < avail_bytes; ++i) {
					/// \todo: #warning "FIXME: error! samplepos+1 can exceed the array bounds"
					fVal_L = sample_data_L[(int)samplePos] + ( sample_data_L[(int)(samplePos + 1)] - sample_data_L[(int)samplePos] ) * (samplePos - (int)samplePos);
					fVal_R = sample_data_R[(int)samplePos] + ( sample_data_R[(int)(samplePos + 1)] - sample_data_R[(int)samplePos] ) * (samplePos - (int)samplePos);
					fVal_L = fVal_L * cost_L;
					fVal_R = fVal_R * cost_R;
					buf_L[ bufferPos ] += fVal_L * fLevel;
					buf_R[ bufferPos ] += fVal_R * fLevel;
					samplePos += fStep;
					++bufferPos;
				}
			}
		}



		return retValue;
	}
}



void audioEngine_setupLadspaFX( uint nBufferSize ) {
	hydrogenInstance->infoLog( "[audioEngine_setupLadspaFX] buffersize=" + toString(nBufferSize) );

	if (m_pSong == NULL) {
		hydrogenInstance->warningLog( "[audioEngine_setupLadspaFX] m_pSong=NULL" );
		return;
	}
	if (nBufferSize == 0) {
		hydrogenInstance->errorLog( "[audioEngine_setupLadspaFX] nBufferSize=0" );
		return;
	}

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		if ( m_pSong->getLadspaFX(nFX) )	m_pSong->getLadspaFX(nFX)->deactivate();
		delete[] m_pFXBuffer_L[nFX];
		delete[] m_pFXBuffer_R[nFX];
		if (nBufferSize != 0) {
			m_nFXBufferSize = nBufferSize;
			m_pFXBuffer_L[nFX] = new float[ nBufferSize ];
			m_pFXBuffer_R[nFX] = new float[ nBufferSize ];

			//touch all the memory (is this really necessary?)
			for (uint i = 0; i < nBufferSize; i++) {
				m_pFXBuffer_L[nFX][i] = 0;
				m_pFXBuffer_R[nFX][i] = 0;
			}
		}
		if ( m_pSong->getLadspaFX(nFX) ) {
			m_pSong->getLadspaFX(nFX)->connectAudioPorts( m_pFXBuffer_L[nFX], m_pFXBuffer_R[nFX], m_pFXBuffer_L[nFX], m_pFXBuffer_R[nFX] );
			m_pSong->getLadspaFX(nFX)->activate();
		}
	}
}



void audioEngine_setSong(Song *newSong) {
	hydrogenInstance->infoLog( "[audioEngine_setSong] " + newSong->getName()  );

	audioEngine_lock( "audioEngine_setSong" );

	if ( m_audioEngineState == PLAYING ) {
		m_pAudioDriver->stop();
		audioEngine_stop( false );
	}

	// check current state
	if (m_audioEngineState != PREPARED) {
		hydrogenInstance->errorLog("[audioEngine_setSong] Error the audio engine is not in PREPARED state");
	}

	m_pCurrentPatternList->clear();
	m_pNextPattern = NULL;

	audioEngine_clearNoteQueue();

	m_pSong = newSong;

	// update ticksize
	audioEngine_process_checkBPMChanged();

	// find the first pattern and set as current
	if ( m_pSong->getPatternList()->getSize() > 0 ) {
		m_pCurrentPatternList->add( m_pSong->getPatternList()->get(0) );
	}

	// setup LADSPA FX
	audioEngine_setupLadspaFX( m_pAudioDriver->getBufferSize() );

	m_pAudioDriver->setBpm( m_pSong->getBpm() );


	// change the current audio engine state
	m_audioEngineState = READY;
	hydrogenInstance->infoLog("READY");


	m_pAudioDriver->locate( 0 );

	audioEngine_unlock();

	// raise selectedInstrumentChanged event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->selectedInstrumentChanged();
	}

	// raise patternChanged event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->patternChanged();
	}

	// raise stateChanged event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->stateChanged(READY);
	}
}



void audioEngine_removeSong() {
	audioEngine_lock( "audioEngine_removeSong" );

	if ( m_audioEngineState == PLAYING ) {
		m_pAudioDriver->stop();
		audioEngine_stop( false );
	}

	// check current state
	if (m_audioEngineState != READY) {
		hydrogenInstance->errorLog( "audioEngine_removeSong: Error the audio engine is not in READY state" );
		audioEngine_unlock();
		return;
	}
//	hydrogenInstance->warningLog( "audioEngine_removeSong()" );

	m_pSong = NULL;
	m_pCurrentPatternList->clear();
	m_pNextPattern = NULL;

	audioEngine_clearNoteQueue();

	// change the current audio engine state
	m_audioEngineState = PREPARED;
	audioEngine_unlock();

	// raise stateChanged event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->stateChanged(PREPARED);
	}

}



// return -1 = end of song
// return 2 = send pattern changed event!!
inline int audioEngine_updateNoteQueue(uint nFrames)
{
	static int lastTick = -1;
	bool sendPatternChange = false;

	unsigned int framepos;
	if (  m_audioEngineState == PLAYING ) {
		framepos = m_pAudioDriver->m_transport.m_nFrames;
	}
	else {
		// use this to support realtime events when not playing
		framepos = m_nRealtimeFrames;
	}

 	int tickNumber_start = (int)(  framepos / m_pAudioDriver->m_transport.m_nTickSize );
 	int tickNumber_end = (int)( ( framepos + nFrames ) / m_pAudioDriver->m_transport.m_nTickSize );


	int tick = tickNumber_start;

	// get initial timestamp for first tick
	gettimeofday (&m_currentTickTime, NULL);

	while ( tick <= tickNumber_end ) {
		if (tick == lastTick) {
			tick++;
			continue;
		}
		else {
			lastTick = tick;
		}


		// midi events now get put into the songNoteQueue as well, based on their timestamp
		while ( midiNoteQueue.size() > 0 ) {
			Note *note = midiNoteQueue[0];

			if ((int)note->getPosition() <= tick) {
				// printf ("tick=%d  pos=%d\n", tick, note->getPosition());
				midiNoteQueue.pop_front();
				songNoteQueue.push_back( note );
			}
			else {
				break;
			}
		}

		if (  m_audioEngineState != PLAYING ) {
			// only keep going if we're playing
			continue;
		}


		// metronome
		if (( tick % 48 ) == 0) {
			if (metronomeInstrument->getVolume() != 0.0) {
				Note *metronomeNote = new Note( tick, 1.0, 1.0, 1.0, -1, 1.0 );
				metronomeNote->setInstrument( metronomeInstrument );
				songNoteQueue.push_back (metronomeNote);
			}
		}

		// SONG MODE
		if ( m_pSong->getMode() == Song::SONG_MODE ) {

			m_nSongPos = findPatternInTick( tick, m_pSong->isLoopEnabled(), &m_nPatternStartTick );
			if (m_nSongSizeInTicks != 0) {
				m_nPatternTickPosition = (tick - m_nPatternStartTick) % m_nSongSizeInTicks;
			}
			else {
				m_nPatternTickPosition = tick - m_nPatternStartTick;
			}

			if (m_nPatternTickPosition == 0) {
				sendPatternChange = true;
			}

			PatternList *pPatternList = (*(m_pSong->getPatternGroupVector()))[m_nSongPos];
			if ( m_nSongPos == -1 ) {
				hydrogenInstance->infoLog("[audioEngine_updateNoteQueue()] song pos = -1");
				if ( m_pSong->isLoopEnabled() == true ) {	// check if the song loop is enabled
					m_nSongPos = findPatternInTick( 0, true, &m_nPatternStartTick );

/*					Pattern *pPat = pPatternList->get(0);	// prendo solo il primo
					if ( pPat->getSize() != 0 ) {
						hydrogenInstance->infoLog("[audioEngine_updateNoteQueue()] Loop enabled");
						cout << "nSongSizeInTicks " << tick << endl;
						m_nSongPos = findPatternInTick( 0, true, &m_nPatternStartTick );
					}
*/
				}
				else {
					hydrogenInstance->infoLog( "[audioEngine_updateNoteQueue()] End of Song" );
					return -1;
				}
			}
			// copio tutti i pattern
			m_pCurrentPatternList->clear();
			if ( pPatternList ) {
				for (uint i = 0; i < pPatternList->getSize(); i++) {
					m_pCurrentPatternList->add( pPatternList->get(i) );
				}
			}

			if (m_nPatternTickPosition > MAX_NOTES) {
				char tmp[200];
				sprintf(tmp, "[audioEngine_updateNoteQueue()]1 Error m_nPatternTickPosition > MAX_NOTES. (%d > %d)", m_nPatternTickPosition, MAX_NOTES);
				hydrogenInstance->errorLog(tmp);

				sprintf(tmp, "[audioEngine_updateNoteQueue()] tick =%d, m_nPatternStartTick = %d, m_nSongPos = %d", tick, m_nPatternStartTick, m_nSongPos);
				hydrogenInstance->errorLog(tmp);

				// PANIC!!!
				hydrogenInstance->errorLog("[audioEngine_updateNoteQueue()] Panic! Stopping audio engine");
				audioEngine_stop(false);	// stop the audio engine
				return 0;
			}
		}

		// PATTERN MODE
		else if ( m_pSong->getMode() == Song::PATTERN_MODE )	{
			//hydrogenInstance->warningLog( "pattern mode not implemented yet" );

			// per ora considero solo il primo pattern, se ce ne saranno piu' di uno
			// bisognera' prendere quello piu' piccolo
			Pattern *pFirstPattern = m_pCurrentPatternList->get( 0 );
			int nPatternSize = pFirstPattern->getSize();

			if ( ( tick == (int)(m_nPatternStartTick + nPatternSize) ) || ( m_nPatternStartTick == -1 ) ) {
				if ( m_pNextPattern != NULL ) {
					m_pCurrentPatternList->clear();
					m_pCurrentPatternList->add( m_pNextPattern );
					m_pNextPattern = NULL;
					sendPatternChange = true;
				}
				m_nPatternStartTick = tick;
			}
			//m_nPatternTickPosition = tick % m_pCurrentPattern->getSize();
			m_nPatternTickPosition = tick % nPatternSize;
		}

		// update notes queue
		if ( m_pCurrentPatternList->getSize() != 0 ) {

			for (uint nPat = 0; nPat < m_pCurrentPatternList->getSize(); nPat++) {
				Pattern *pPat = m_pCurrentPatternList->get( nPat );

				uint nSequences = ( pPat->getSequenceList() )->getSize();
				for ( uint i = 0; i < nSequences; i++ ) {
					Sequence* sequence = ( (pPat->getSequenceList() )->get(i) );
//					Note* note = sequence->noteList[ m_nPatternTickPosition ];
					Note* note = sequence->m_noteList[ m_nPatternTickPosition ];
					if ( note ) {
						uint offset = 0;
						float velocity = note->getVelocity();

						// Swing
						if ( m_pSong->isSwingEnabled() ) {
							float swingFactor = m_pSong->getSwingFactor();
							if ( ( (m_nPatternTickPosition % 12) == 0 ) && ( (m_nPatternTickPosition % 24) != 0) ) {	// da l'accento al tick 4, 12, 20, 36...
								offset += (int)( ( 6.0 * m_pAudioDriver->m_transport.m_nTickSize ) * swingFactor );
							}
						}

						// Humanize - Time parameter
						if ( m_pSong->isHumanizeTimeEnabled() ) {
							int maxTimeHumanize = 2000;
							if (m_pSong->getHumanizeTimeValue() != 0) {
								float random = randomValue(100) / 100.0;
								offset += (int)(random * m_pSong->getHumanizeTimeValue() * maxTimeHumanize);
							}
						}

						// Humanize - Velocity parameter
						if ( m_pSong->isHumanizeVelocityEnabled() ) {
							if (m_pSong->getHumanizeVelocityValue() != 0) {
								float random = m_pSong->getHumanizeVelocityValue() * (randomValue(100) / 100.0);
								velocity += (random - (m_pSong->getHumanizeVelocityValue() / 2.0));
								if (velocity > 1.0) {
									velocity = 1.0;
								}
								else if (velocity < 0.0) {
									velocity = 0.0;
								}
							}
						}

						Note *copiedNote = new Note( note->getPosition(), velocity, note->getPan_L(), note->getPan_R(), note->getLength(), note->getPitch() );
						copiedNote->setInstrument( note->getInstrument() );
						copiedNote->setPosition( tick );
						copiedNote->m_fSamplePosition = 0;
						copiedNote->setHumanizeDelay( offset );	// humanize time
						songNoteQueue.push_back( copiedNote );
					}
				}
			}
		}
		tick++;
	}

	// audioEngine_process must send the pattern change event after mutex unlock
	if (sendPatternChange) {
		return 2;
	}
	return 0;
}



/// restituisce l'indice relativo al patternGroup in base al tick
inline int findPatternInTick( int tick, bool loopMode,int *pPatternStartTick  )
{
	if (m_pSong == NULL) {
		return -1;
	}

//	cout << "find pattern in tick " << tick << " [lastTick=" << findPatternInTick_lastTick << "] lastTotal[" << findPatternInTick_lastTotal << "] "	 << endl;
//	int pos = -1;

	int totalTick = 0;
	int nPatternGroups = (m_pSong->getPatternGroupVector())->size();
	int nPatternSize = 0;

	m_nSongSizeInTicks = 0;

	vector<PatternList*> *patternGroupVector = m_pSong->getPatternGroupVector();

	// vecchia ricerca
	for ( int i = 0; i < nPatternGroups; i++ ) {
		PatternList *pPatternGroup = (*patternGroupVector)[i];
		Pattern *pPattern = pPatternGroup->get(0);	// prendo solo il primo. I pattern nel gruppo devono avere la stessa lunghezza
		nPatternSize = pPattern->getSize();
		if ( (tick >= totalTick) && ( tick < totalTick + nPatternSize ) ) {
			(*pPatternStartTick) = totalTick;
			return i;
		}
		totalTick += nPatternSize;
	}

	if (loopMode){
		m_nSongSizeInTicks = totalTick;
		int loopTick = 0;
		if (m_nSongSizeInTicks != 0) {
			loopTick = tick % m_nSongSizeInTicks;
		}
		totalTick = 0;
		for ( int i = 0; i < nPatternGroups; i++ ) {
			PatternList *pPatternGroup = (*patternGroupVector)[i];
			Pattern *pPattern = pPatternGroup->get(0);	// prendo solo il primo. I pattern nel gruppo devono avere la stessa lunghezza
			nPatternSize = pPattern->getSize();
			if ( ( loopTick >= totalTick ) && ( loopTick < totalTick + nPatternSize ) ) {
				(*pPatternStartTick) = totalTick;
				return i;
			}
			totalTick += nPatternSize;
		}
	}

	char tmp[200];
	sprintf( tmp, "[findPatternInTick] tick = %d. No pattern found", tick );
	hydrogenInstance->infoLog( tmp );
	return -1;



/*	totalTick = findPatternInTick_lastTotal;
	// nuova ricerca semi-ottimizzata
	Pattern *pattern = NULL;
	for ( int i = findPatternInTick_lastTick; i < nPatterns; i++ ) {
//		cout << i << endl;
		pattern = m_pSong->getPatternSequence()->get( i );
		if (pattern == NULL) {
			hydrogenInstance->errorLog( "[Hydrogen::findPatternInTick] Pattern == null");
			return -1;
		}
		patternSize = pattern->getSize();

		if ( (tick >= totalTick) && ( tick < totalTick + patternSize ) ) {
			findPatternInTick_lastTick = i;
			findPatternInTick_lastTotal = totalTick;
			(*pPatternStartTick) = totalTick;
			return i;
		}
		totalTick += patternSize;
	}


	if ( loopMode ) {
		hydrogenInstance->infoLog( "[Hydrogen::findPatternInTick] loop mode" );
		// se arriva qua e' attivo il loop mode
		int songTick = totalTick;

		int loopTick = 0;
		if (songTick != 0) {
			loopTick = tick % songTick;
		}

		totalTick = 0;
		for ( int i = 0; i < nPatterns; i++ ) {
			patternSize = m_pSong->getPatternSequence()->get( i )->getSize();
			if ( ( loopTick >= totalTick ) && ( loopTick < totalTick + patternSize ) ) {
				findPatternInTick_lastTick = i;
				findPatternInTick_lastTotal = totalTick;
				(*pPatternStartTick) = totalTick;
				return i;
			}
			totalTick += patternSize;
		}
	}

	findPatternInTick_lastTick = 0;
	totalTick = 0;
	(*pPatternStartTick) = totalTick;
	return pos;
*/
}



void audioEngine_noteOn(Note *note) {
	// check current state
	if ( (m_audioEngineState != READY) && (m_audioEngineState != PLAYING) ) {
		hydrogenInstance->errorLog("audioEngine_noteOn: Error the audio engine is not in READY state");
		delete note;
		return;
	}

	midiNoteQueue.push_back(note);
}



void audioEngine_noteOff( Note *note )
{
	if (note == NULL)	{
		hydrogenInstance->errorLog("audioEngine_noteOff: Error, note == NULL");
	}

	audioEngine_lock( "audioEngine_noteOff" );

	// check current state
	if ( (m_audioEngineState != READY) && (m_audioEngineState != PLAYING) ) {
		hydrogenInstance->errorLog("audioEngine_noteOff: Error the audio engine is not in READY state");
		delete note;
		audioEngine_unlock();
		return;
	}

	for ( uint i = 0; i < playingNotesQueue.size(); i++ ) {	// delete old note
		Note *oldNote = playingNotesQueue[ i ];

		if ( oldNote->getInstrument() == note->getInstrument() ) {
			playingNotesQueue.erase( playingNotesQueue.begin() + i );
			delete oldNote;
			break;
		}
	}
	audioEngine_unlock();

	delete note;
}



unsigned long audioEngine_getTickPosition() {
	return m_nPatternTickPosition;
}



/// Start all audio drivers
void audioEngine_startAudioDrivers() {
	PreferencesMng *preferencesMng = PreferencesMng::getInstance();
	m_bUseTrackOuts = false;


	audioEngine_lock( "audioEngine_startAudioDrivers" );

	hydrogenInstance->infoLog("[audioEngine_startAudioDrivers] Starting audio drivers");

	// check current state
	if (m_audioEngineState != INITIALIZED) {
		hydrogenInstance->errorLog("[audioEngine_startAudioDrivers] Error the audio engine is not in INITIALIZED state. state=" + toString(m_audioEngineState) );
		audioEngine_unlock();
		return;
	}

	// check if the audio m_pAudioDriver is still alive
	if (m_pAudioDriver) {
		hydrogenInstance->errorLog( "[audioEngine_startAudioDrivers] The audio driver is still alive" );
	}

	// Create Audio Driver
	string audioDriver = preferencesMng->getAudioDriver();
	if (audioDriver == "Oss") {
#ifdef OSS_SUPPORT
		m_pAudioDriver = new OssDriver( audioEngine_process );
#else
		audioEngine_raiseError( Hydrogen::NO_OSS_SUPPORT );
		m_pAudioDriver = new NullDriver( audioEngine_process );
#endif
	}
	else if (audioDriver == "Jack") {
#ifdef JACK_SUPPORT
		m_pAudioDriver = new JackDriver(audioEngine_process);

		// set use track outputs flag
		m_bUseTrackOuts = preferencesMng->isJackTrackOuts();
		((JackDriver *) m_pAudioDriver)->setTrackOuts( m_bUseTrackOuts );

		m_bUseDefaultOuts = preferencesMng->isJackConnectDefaults();
		((JackDriver *) m_pAudioDriver)->setConnectDefaults( m_bUseDefaultOuts );
#else
		audioEngine_raiseError( Hydrogen::NO_JACK_SUPPORT );
		m_pAudioDriver = new NullDriver( audioEngine_process );
#endif
	}
	else {
		audioEngine_raiseError( Hydrogen::UNKNOWN_DRIVER );
	}

	// initialize the audio driver
	int res = m_pAudioDriver->init( preferencesMng->getBufferSize() );
	if (res != 0) {
		audioEngine_raiseError( Hydrogen::ERROR_STARTING_DRIVER );
		hydrogenInstance->errorLog("[audioEngine_startAudioDrivers] Error starting audio driver [audioDriver::init()]");
		delete m_pAudioDriver;
		m_bUseTrackOuts = false;

		m_pAudioDriver = new NullDriver( audioEngine_process );
		m_pAudioDriver->init(0);
	}

	m_pMainBuffer_L = m_pAudioDriver->getOut_L();
	m_pMainBuffer_R = m_pAudioDriver->getOut_R();

#ifdef JACK_SUPPORT
	if (m_bUseTrackOuts) {
		for (uint i = 0; i < MAX_INSTRUMENTS; ++i) {
			m_pTrackBuffers_L[i] = ((JackDriver*)m_pAudioDriver)->getTrackOut_L(i);
			m_pTrackBuffers_R[i] = ((JackDriver*)m_pAudioDriver)->getTrackOut_R(i);
		}
	}
#endif


// check if ALSA_SEQ support is enabled
#ifdef USE_ALSA_SEQ

	// check if midi driver is still alive
	if (midiDriver) {
		hydrogenInstance->errorLog( "[audioEngine_startAudioDrivers] The MIDI driver is still alive");
	}

	// Create MIDI driver
	midiDriver = new AlsaMidiDriver();
	midiDriver->open();
	midiDriver->setActive(true);
#endif

	// change the current audio engine state
	if (m_pSong == NULL) {
		m_audioEngineState = PREPARED;
		hydrogenInstance->infoLog("[audioEngine_startAudioDrivers] PREPARED");
	}
	else {
		m_audioEngineState = READY;
		hydrogenInstance->infoLog("[audioEngine_startAudioDrivers] READY");
	}


	audioEngine_setupLadspaFX( m_pAudioDriver->getBufferSize() );

	if (m_pSong) {
		m_pAudioDriver->setBpm( m_pSong->getBpm() );
	}
	hydrogenInstance->infoLog( "[audioEngine_startAudioDrivers] unlock engine" );

	res = m_pAudioDriver->connect();
	if (res != 0) {
		audioEngine_raiseError( Hydrogen::ERROR_STARTING_DRIVER );
		hydrogenInstance->errorLog("[audioEngine_startAudioDrivers] Error starting audio driver [audioDriver::connect()]");

		delete m_pAudioDriver;
		m_bUseTrackOuts = false;
		m_pAudioDriver = new NullDriver( audioEngine_process );
		m_pAudioDriver->init(0);
		m_pAudioDriver->connect();
	}
	audioEngine_unlock();


	// raise stateChanged event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->stateChanged(m_audioEngineState);
	}
}



/// Stop all audio drivers
void audioEngine_stopAudioDrivers() {
	hydrogenInstance->infoLog("Stopping audio drivers");

	// check current state
	if (m_audioEngineState == PLAYING) {
		audioEngine_stop();
	}

	if ( (m_audioEngineState != PREPARED) && (m_audioEngineState != READY) ) {
		hydrogenInstance->errorLog("audioEngine_stopAudioDrivers: Error the audio engine is not in PREPARED or READY state. state=" + toString(m_audioEngineState) );
		return;
	}

// check if ALSA_SEQ support is enabled
#ifdef USE_ALSA_SEQ
	hydrogenInstance->infoLog("Stopping MIDI driver");
	// delete MIDI driver
	if (midiDriver) {
		midiDriver->close();
		delete midiDriver;
		midiDriver = NULL;
	}
#endif

	// delete audio driver
	if (m_pAudioDriver) {
		m_pAudioDriver->disconnect();
		delete m_pAudioDriver;
		m_pAudioDriver = NULL;
	}


	audioEngine_lock( "audioEngine_stopAudioDrivers" );
	// change the current audio engine state
	m_audioEngineState = INITIALIZED;
	hydrogenInstance->infoLog("INITIALIZED");
	// raise stateChanged event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->stateChanged(INITIALIZED);
	}
	audioEngine_unlock();
}



/// Restart all audio and midi drivers
void audioEngine_restartAudioDrivers() {
	audioEngine_stopAudioDrivers();
	audioEngine_startAudioDrivers();
	//audioEngine_setupLadspaFX( m_pAudioDriver->getBufferSize() );
}






//----------------------------------------------------------------------------
//
// Implementation of Hydrogen class
//
//----------------------------------------------------------------------------




Hydrogen::Hydrogen(EngineListener *mainListener) : Object( "Hydrogen   " ) {
	if (instance) {
		errorLog( "Hydrogen audio engine is already running" );
	}

	hydrogenInstance = this;
	instance = this;
	audioEngine_init();
	addEngineListener( mainListener );
	audioEngine_startAudioDrivers();
}



Hydrogen::~Hydrogen() {
	infoLog( "[~Hydrogen]" );
	if ( m_audioEngineState == PLAYING ) {
		audioEngine_stop();
	}
	removeSong();
	audioEngine_stopAudioDrivers();
	audioEngine_destroy();
	instance = NULL;
}



/// Return the Hydrogen instance
Hydrogen* Hydrogen::getInstance(EngineListener *listener) {
	if (instance == NULL) {
		instance = new Hydrogen( listener );
	}
	return instance;
}



/// Start the internal sequencer
void Hydrogen::start() {
	// play from start if pattern mode is enabled
	if (m_pSong->getMode() == Song::PATTERN_MODE) {
		setPatternPos( 0 );
	}
	m_pAudioDriver->play();
}



/// Stop the internal sequencer
void Hydrogen::stop() {
	m_pAudioDriver->stop();
}



void Hydrogen::setSong(Song *newSong) {
	audioEngine_setSong(newSong);
}



void Hydrogen::removeSong() {
	audioEngine_removeSong();
}



Song* Hydrogen::getSong() {
	return m_pSong;
}



void Hydrogen::noteOn(Note *note) {
	audioEngine_noteOn(note);
}



void Hydrogen::noteOff(Note *note) {
	audioEngine_noteOff(note);
}



void Hydrogen::addRealtimeNote(int instrument, float velocity, float pan_L, float pan_R, float pitch, bool forcePlay)
{
	lockEngine( "Hydrogen::addRealtimeNote" );	// lock the audio engine

	unsigned int column = getTickPosition();
	unsigned int realcolumn = 0;

	realcolumn = getRealtimeTickPosition();

	//printf ("orig tickpos=%u  real=%u\n", column, realcolumn);

	PreferencesMng *pref = PreferencesMng::getInstance();

	// quantize it to scale
	uint res = pref->getPatternEditorGridResolution();
	int nBase = pref->isPatternEditorUsingTriplets() ? 3 : 4;
	int scalar = (4 * MAX_NOTES) / (res*nBase);
	int qcolumn = (int)::round(column / (double)scalar) * scalar;
	if (qcolumn == MAX_NOTES) qcolumn = 0;

	//printf ("column=%d  qcol=%d\n", column, qcolumn);

	if (pref->getQuantizeEvents()) {
		column = qcolumn;
	}

	uint position = column;
	bool hearnote = forcePlay;

//	Pattern* currentPattern = getCurrentPattern();
	Pattern* currentPattern = NULL;
	PatternList *pPatternList = m_pSong->getPatternList();
	if ( (m_nSelectedPatternNumber != -1) && (m_nSelectedPatternNumber < (int)pPatternList->getSize() ) ) {
		currentPattern = pPatternList->get( m_nSelectedPatternNumber );
	}

	Song *song = getSong();
	Instrument *instrRef=0;
	if (song) {
		instrRef = (song->getInstrumentList())->get(instrument);
	}

	if (currentPattern) {
		SequenceList *sequenceList = currentPattern->getSequenceList();
		Sequence *seq = sequenceList->get(instrument);

		if (seq->m_noteList[column] != NULL) {
			// in this case, we'll leave the note alone
			// hear note only if not playing too
			if ( pref->getHearNewNotes() && getState() == READY) {
				hearnote = true;
			}
		}
		else if (!pref->getRecordEvents())
		{
			if ( pref->getHearNewNotes() && (getState() == READY || getState() == PLAYING )) {
				hearnote = true;
			}
		}
		else {
			// create the new note
			Note *note = new Note(position, velocity, pan_L, pan_R, -1);
			note->setInstrument(instrRef);
			seq->m_noteList[column] = note;

			// hear note if its not in the future
			if ( pref->getHearNewNotes() && position <= getTickPosition()) {
				hearnote = true;
			}

			song->setModified(true);

			/// \todo spedire fuori dal lock..
			// raise patternModified event
			for (uint i = 0; i < audioEngine_listeners.size(); i++) {
				audioEngine_listeners[i]->patternModified();
			}
		}
	}
	else if (pref->getHearNewNotes()) {
		hearnote = true;
	}

	if (hearnote && instrRef) {
		Note *note2 = new Note(realcolumn, velocity, pan_L, pan_R, -1);
		note2->setInstrument(instrRef);
		noteOn( note2 );
	}

	unlockEngine(); // unlock the audio engine
}



float Hydrogen::getMasterPeak_L() {
	return masterPeak_L;
}



float Hydrogen::getMasterPeak_R() {
	return masterPeak_R;
}



unsigned long Hydrogen::getTickPosition() {
	return audioEngine_getTickPosition();
}



unsigned long Hydrogen::getRealtimeTickPosition()
{
	//unsigned long initTick = audioEngine_getTickPosition();
	unsigned int initTick = (unsigned int)( m_nRealtimeFrames / m_pAudioDriver->m_transport.m_nTickSize );
	unsigned long retTick;

	struct timeval currtime, deltatime;

	gettimeofday (&currtime, NULL);

	timersub (&currtime, &m_currentTickTime, &deltatime);

	double sampleRate = (double) m_pAudioDriver->getSampleRate();
	// add a buffers worth for jitter resistance
	double deltaSec = (double) deltatime.tv_sec + (deltatime.tv_usec / 1000000.0) +  (m_pAudioDriver->getBufferSize() / (double)sampleRate);
	retTick = (unsigned long) ((sampleRate / (double) m_pAudioDriver->m_transport.m_nTickSize) * deltaSec);

	retTick = initTick + retTick;


	return retTick;
}



PatternList* Hydrogen::getCurrentPatternList() {
	return m_pCurrentPatternList;
}



/// Set the next pattern (Pattern mode only)
void Hydrogen::setNextPattern( int pos )
{
	audioEngine_lock( "Hydrogen::setNextPattern" );
	//infoLog( "[setNextPattern] " + toString(pos) );

	if ( m_pSong && m_pSong->getMode() == Song::PATTERN_MODE ) {
		PatternList *patternList = m_pSong->getPatternList();
		if ( (pos >= 0) && ( pos < (int)patternList->getSize() ) ) {
			m_pNextPattern = patternList->get(pos);
		}
		else {
			errorLog("[setNextPattern] pos not in patternList range. pos=" + toString(pos) + " patternListSize=" + toString(patternList->getSize()));
		}
	}
	else {
		errorLog("[setNextPattern] can't set next pattern in song mode");
	}

	audioEngine_unlock();
}



int Hydrogen::getPatternPos()
{
	return m_nSongPos;
}



uint Hydrogen::getPlayingNotes()
{
	return m_nPlayingNotes;
}



void Hydrogen::restartDrivers()
{
	audioEngine_restartAudioDrivers();
}



/// Export a song to a wav file, returns the elapsed time in mSec
uint Hydrogen::exportSong(string filename)
{
	if ( getState() == PLAYING ) {
		stop();
	}

	Song::SongMode oldEngineMode = m_pSong->getMode();
	m_pSong->setMode( Song::SONG_MODE );

	bool oldLoopEnabled = m_pSong->isLoopEnabled();
	m_pSong->setLoopEnabled( false );

	PreferencesMng *preferencesMng = PreferencesMng::getInstance();

	// stop all audio drivers
	audioEngine_stopAudioDrivers();

	/*
		FIXME: Questo codice fa davvero schifo....
	*/

	m_pAudioDriver = new DiskWriterDriver(audioEngine_process);

	// reset
	m_pAudioDriver->m_transport.m_nFrames = 0;	// reset total frames
	m_pAudioDriver->setBpm( m_pSong->getBpm() );
	m_nSongPos = -1;
	m_nPatternTickPosition = 0;
	m_audioEngineState = PLAYING;
	m_nPatternStartTick = -1;


	( (DiskWriterDriver*)m_pAudioDriver )->setFilename(filename);

	int res = m_pAudioDriver->init(preferencesMng->getBufferSize());
	if (res != 0) {
		errorLog("[exportSong] Error starting disk writer driver [DiskWriterDriver::init()]");
	}

	m_pMainBuffer_L = m_pAudioDriver->getOut_L();
	m_pMainBuffer_R = m_pAudioDriver->getOut_R();

	audioEngine_setupLadspaFX(m_pAudioDriver->getBufferSize());

	uint start = currentTime();


	res = m_pAudioDriver->connect();
	if (res != 0) {
		errorLog("[exportSong] Error starting disk writer driver [DiskWriterDriver::connect()]");
	}

	m_pAudioDriver->disconnect();

	m_audioEngineState = INITIALIZED;
	delete m_pAudioDriver;
	m_pAudioDriver = NULL;

	m_pMainBuffer_L = NULL;
	m_pMainBuffer_R = NULL;

	uint finish = currentTime();
	uint elapsedTime = finish - start;

	char tmpbuf[200];
	sprintf(tmpbuf, "[exportSong] Exported in %d msec", elapsedTime);
	hydrogenInstance->infoLog( tmpbuf );

	m_pSong->setMode( oldEngineMode );
	m_pSong->setLoopEnabled( oldLoopEnabled );

	m_nSongPos = -1;
	m_nPatternTickPosition = 0;
	audioEngine_startAudioDrivers();

	if (m_pAudioDriver) {
		m_pAudioDriver->setBpm( m_pSong->getBpm() );
	}
	else {
		errorLog( "[exportSong] m_pAudioDriver = NULL" );
	}

	return elapsedTime;
}



uint Hydrogen::getSongNotesQueue() {
	return songNoteQueue.size();
}



/// Used to display audio driver info
GenericDriver* Hydrogen::getAudioDriver() {
	return m_pAudioDriver;
}



/// Used to display midi driver info
AlsaMidiDriver* Hydrogen::getMidiDriver() {
// check if ALSA_SEQ support is enabled
#ifdef USE_ALSA_SEQ
	return midiDriver;
#else
	return NULL;
#endif
}



void Hydrogen::setMasterPeak_L(float value) {
	masterPeak_L = value;
}



void Hydrogen::setMasterPeak_R(float value) {
	masterPeak_R = value;
}



int Hydrogen::getState() {
	return m_audioEngineState;
}



void Hydrogen::setCurrentPatternList(PatternList *pPatternList) {
	audioEngine_lock( "Hydrogen::setCurrentPatternList" );

	m_pCurrentPatternList = pPatternList;
	// raise patternChanged event
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->patternChanged();
	}

	audioEngine_unlock();
}



void Hydrogen::addEngineListener(EngineListener *newListener) {
	audioEngine_lock( "Hydrogen::addEngineListener" );

	if (newListener) {
		audioEngine_listeners.push_back(newListener);
	}

	audioEngine_unlock();
}



void Hydrogen::removeEngineListener(EngineListener *pListener)
{
	audioEngine_lock( "Hydrogen::removeEngineListener" );

	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		EngineListener *pOldListener = audioEngine_listeners[i];
		if ( pOldListener == pListener) {
			audioEngine_listeners.erase( audioEngine_listeners.begin() + i );
			break;
		}
	}

	audioEngine_unlock();
}



/// Lock the audio engine
void Hydrogen::lockEngine(string sLocker) {
	audioEngine_lock(sLocker);
}



/// Unlock the audio engine
void Hydrogen::unlockEngine() {
	audioEngine_unlock();
}



float Hydrogen::getProcessTime() {
	return fProcessTime;
}



float Hydrogen::getMaxProcessTime() {
	return fMaxProcessTime;
}



int Hydrogen::loadDrumkit( DrumkitInfo *drumkitInfo )
{
	infoLog( "[loadDrumkit] " + drumkitInfo->getName() );
	LocalFileMng fileMng;
	string sDrumkitPath = fileMng.getDrumkitDirectory( drumkitInfo->getName() );

	InstrumentList *songInstrList = m_pSong->getInstrumentList();
	InstrumentList *pDrumkitInstrList = drumkitInfo->getInstrumentList();
	for (uint i = 0; i < pDrumkitInstrList->getSize(); i++ ) {
		Instrument *pInstr = songInstrList->get( i );
		Instrument *pNewInstr = pDrumkitInstrList->get( i );
		infoLog( "[loadDrumkit] Loading instrument (" + toString( i ) + " of " + toString( pDrumkitInstrList->getSize() ) + ") [ " + pNewInstr->getName() + " ]" );

		// creo i nuovi layer in base al nuovo strumento
		for (uint nLayer = 0; nLayer < MAX_LAYERS; nLayer++) {
			InstrumentLayer *pNewLayer = pNewInstr->getLayer( nLayer );
			if (pNewLayer != NULL) {
				Sample *pNewSample = (pNewLayer)->getSample();
				string sSampleFilename = sDrumkitPath + drumkitInfo->getName() + "/" + pNewSample->getFilename();
				infoLog( "[loadDrumkit]    |-> Loading layer [ " + sSampleFilename + " ]" );

				// carico il nuovo sample e creo il nuovo layer
				Sample *pSample = Sample::load( sSampleFilename );
//				pSample->setFilename( pNewSample->getFilename() );	// riuso il path del nuovo sample (perche' e' gia relativo al path del drumkit)
				if (pSample == NULL) {
					errorLog( "[loadDrumkit] Error Loading drumkit: NULL sample, now using /emptySample.wav" );
					pSample = Sample::load( string(DATA_DIR) + "/emptySample.wav" );
					pSample->setFilename( string(DATA_DIR) + "/emptySample.wav" );
				}
				InstrumentLayer *pLayer = new InstrumentLayer( pSample );
				pLayer->setStartVelocity( pNewLayer->getStartVelocity() );
				pLayer->setEndVelocity( pNewLayer->getEndVelocity() );
				pLayer->setGain( pNewLayer->getGain() );

				InstrumentLayer *pOldLayer = pInstr->getLayer(nLayer);
				audioEngine_lock( "Hydrogen::loadDrumkit" );
				pInstr->setLayer( pLayer, nLayer );	// set the new layer
				audioEngine_unlock();
				delete pOldLayer;		// delete the old layer

			}
			else {
				InstrumentLayer *pOldLayer = pInstr->getLayer(nLayer);
				audioEngine_lock( "Hydrogen::loadDrumkit" );
				pInstr->setLayer( NULL, nLayer );
				audioEngine_unlock();
				delete pOldLayer;		// delete the old layer
			}

		}

		audioEngine_lock( "Hydrogen::loadDrumkit" );
		// update instrument properties
		pInstr->setName( pNewInstr->getName() );
		pInstr->setPan_L( pNewInstr->getPan_L() );
		pInstr->setPan_R( pNewInstr->getPan_R() );
		pInstr->setVolume( pNewInstr->getVolume() );
		pInstr->setDrumkitName( pNewInstr->getDrumkitName() );
		pInstr->setMuted( pNewInstr->isMuted() );
		pInstr->m_excludeVectId.clear();
		for (uint i = 0; i < pNewInstr->m_excludeVectId.size(); i++) {
			pInstr->m_excludeVectId.push_back( pNewInstr->m_excludeVectId[ i ] );
		}
		pInstr->m_excludeVect.clear();

		audioEngine_unlock();
	}

	audioEngine_lock("Hydrogen::loadDrumkit");
	// rebuild the exclude vector
	for (uint nInstr = 0; nInstr < songInstrList->getSize(); nInstr++) {
		Instrument* pInstr = songInstrList->get( nInstr );
		for (uint i = 0; i < pInstr->m_excludeVectId.size(); i++) {
			int id = pInstr->m_excludeVectId[ i ];
			Instrument* pExcluded = songInstrList->get( id );
			pInstr->m_excludeVect.push_back( pExcluded );
		}
	}
	audioEngine_unlock();

	return 0;	//ok
}



void Hydrogen::raiseError( uint nErrorCode )
{
	audioEngine_raiseError( nErrorCode );
}



/// Used by AlsaMidiDriver
void Hydrogen::raiseMidiActivityEvent()
{
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->midiActivityEvent();
	}
}



unsigned long Hydrogen::getTotalFrames()
{
	return m_pAudioDriver->m_transport.m_nFrames;
}



/// Set the position in the song
void Hydrogen::setPatternPos( int pos )
{
	audioEngine_lock( "Hydrogen::setPatternPos" );

	int nPatternGroups = (m_pSong->getPatternGroupVector())->size();
	if ( pos >= nPatternGroups ) {
		hydrogenInstance->warningLog( "[Hydrogen::setPatternPos()] patternPos > nPatternGroups" );
		audioEngine_unlock();
		return;
	}

	vector<PatternList*> *patternGroupVector = m_pSong->getPatternGroupVector();
	int totalTick = 0;
	int nPatternSize;
	Pattern *pPattern = NULL;
	for ( int i = 0; i < pos; i++ ) {
		PatternList *pPatternGroup = (*patternGroupVector)[i];
		pPattern = pPatternGroup->get(0);	// prendo solo il primo. I pattern nel gruppo devono avere la stessa lunghezza
		nPatternSize = pPattern->getSize();
		totalTick += nPatternSize;
	}

	if (getState() != PLAYING) {
		// find pattern immediately when not playing
		int dummy;
		m_nSongPos = findPatternInTick( totalTick, m_pSong->isLoopEnabled(), &dummy);
	}

	m_pAudioDriver->locate( (int) (totalTick * m_pAudioDriver->m_transport.m_nTickSize) );


	audioEngine_unlock();
}



/// Preview, usa solo il primo layer
void Hydrogen::previewSample( Sample *pSample )
{
	audioEngine_lock( "Hydrogen::previewSample" );

	InstrumentLayer *pLayer = m_pPreviewInstrument->getLayer(0);

	Sample *pOldSample = pLayer->getSample();
	pLayer->setSample( pSample );
	delete pOldSample;

	Note *previewNote = new Note( 0, 1.0, 1.0, 1.0, MAX_NOTES, 0 );
	previewNote->setInstrument( m_pPreviewInstrument );
	audioEngine_noteOn( previewNote );

	audioEngine_unlock();
}



void Hydrogen::getLadspaFXPeak( int nFX, float *fL, float *fR )
{
	(*fL) = m_fFXPeak_L[nFX];
	(*fR) = m_fFXPeak_R[nFX];
}



void Hydrogen::setLadspaFXPeak( int nFX, float fL, float fR )
{
	m_fFXPeak_L[nFX] = fL;
	m_fFXPeak_R[nFX] = fR;
}



void Hydrogen::setTapTempo( float fInterval )
{
//	infoLog( "set tap tempo" );
	static float fOldBpm1 = -1;
	static float fOldBpm2 = -1;
	static float fOldBpm3 = -1;
	static float fOldBpm4 = -1;
	static float fOldBpm5 = -1;
	static float fOldBpm6 = -1;
	static float fOldBpm7 = -1;
	static float fOldBpm8 = -1;

	float fBPM = 60000.0 / fInterval;

	if ( fabs(fOldBpm1 - fBPM) > 20 ) {	// troppa differenza, niente media
		fOldBpm1 = fBPM;
		fOldBpm2 = fBPM;
		fOldBpm3 = fBPM;
		fOldBpm4 = fBPM;
		fOldBpm5 = fBPM;
		fOldBpm6 = fBPM;
		fOldBpm7 = fBPM;
		fOldBpm8 = fBPM;
	}

	if ( fOldBpm1 == -1 ) {
		fOldBpm1 = fBPM;
		fOldBpm2 = fBPM;
		fOldBpm3 = fBPM;
		fOldBpm4 = fBPM;
		fOldBpm5 = fBPM;
		fOldBpm6 = fBPM;
		fOldBpm7 = fBPM;
		fOldBpm8 = fBPM;
	}

	fBPM = (fBPM + fOldBpm1 + fOldBpm2 + fOldBpm3 + fOldBpm4 + fOldBpm5 + fOldBpm6 + fOldBpm7 + fOldBpm8) / 9.0;


	infoLog( "[setTapTempo] avg BPM = " + toString(fBPM) );
	fOldBpm8 = fOldBpm7;
	fOldBpm7 = fOldBpm6;
	fOldBpm6 = fOldBpm5;
	fOldBpm5 = fOldBpm4;
	fOldBpm4 = fOldBpm3;
	fOldBpm3 = fOldBpm2;
	fOldBpm2 = fOldBpm1;
	fOldBpm1 = fBPM;

	lockEngine("Hydrogen::setTapTempo");

// 	m_pAudioDriver->setBpm( fBPM );
// 	m_pSong->setBpm( fBPM );

	setBPM( fBPM );

	unlockEngine();
}



void Hydrogen::setBPM( float fBPM )
{
	if (m_pAudioDriver && m_pSong) {
		m_pAudioDriver->setBpm( fBPM );
		m_pSong->setBpm( fBPM );
		audioEngine_process_checkBPMChanged();
	}
}



void Hydrogen::restartLadspaFX() {
	if (m_pAudioDriver) {
		lockEngine("Hydrogen::restartLadspaFX");
		audioEngine_setupLadspaFX( m_pAudioDriver->getBufferSize() );
		unlockEngine();
	}
	else {
		errorLog( "[restartLadspaFX] m_pAudioDriver = NULL" );
	}
}



int Hydrogen::getSelectedPatternNumber()
{
	return m_nSelectedPatternNumber;
}



void Hydrogen::setSelectedPatternNumber(int nPat)
{
	// FIXME: controllare se e' valido..
	if (nPat == m_nSelectedPatternNumber)	return;

	m_nSelectedPatternNumber = nPat;

	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->selectedPatternChanged();
	}
}



int Hydrogen::getSelectedInstrumentNumber()
{
	return m_nSelectedInstrumentNumber;
}



void Hydrogen::setSelectedInstrumentNumber( int nInstrument )
{
	if (m_nSelectedInstrumentNumber == nInstrument)	return;

	m_nSelectedInstrumentNumber = nInstrument;
	for (uint i = 0; i < audioEngine_listeners.size(); i++) {
		audioEngine_listeners[i]->selectedInstrumentChanged();
	}
}
