/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: PreferencesMng.cpp,v 1.36 2004/07/19 09:25:04 comix Exp $
 *
 */

#include <stdlib.h>
#include "PreferencesMng.h"

#include "xml/tinyxml.h"
#include "config.h"
#include "LocalFileMng.h"

PreferencesMng* PreferencesMng::instance = NULL;



/// Return an instance of PreferencesMng
PreferencesMng* PreferencesMng::getInstance() {
	if (instance == NULL) {
		instance = new PreferencesMng();
	}

	return instance;
}




PreferencesMng::PreferencesMng()
 : Object( "PreferencesMng" )
 // general properties
 , demoPath( string(DATA_DIR) + "/demo_songs/" )
 , restoreLastSong( true )
 , lastSongFilename( "" )
 , hearNewNotes( true )
{
	infoLog( "INIT" );


	char * ladpath = getenv("LADSPA_PATH");	// read the Environment variable LADSPA_PATH
	if (ladpath) {
		infoLog( "Found LADSPA_PATH enviroment variable" );
		string sLadspaPath = string(ladpath);
		int pos;
		while ( (pos = sLadspaPath.find(":")) != -1) {
			string sPath = sLadspaPath.substr(0, pos);
			m_ladspaPathVect.push_back( sPath );
			sLadspaPath = sLadspaPath.substr( pos + 1, sLadspaPath.length() );
		}
		m_ladspaPathVect.push_back(sLadspaPath);
	}
	else {
		m_ladspaPathVect.push_back("/usr/lib/ladspa");
		m_ladspaPathVect.push_back("/usr/local/lib/ladspa");
	}
	
	recordEvents = false;
	quantizeEvents = true;

	// Audio engine properties
	setAudioDriver( "Oss" );
	useMetronome = false;
	metronomeVolume = 0.5f;
	maxNotes = 32;

	// OSS Driver properties
	bufferSize = 1024;
	sampleRate = 44100;
	ossDevice = "/dev/dsp";

	// Jack driver properties
	jackPortName1 = "alsa_pcm:playback_1";
	jackPortName2 = "alsa_pcm:playback_2";
	m_jackTransportMode = NO_JACK_TRANSPORT;
	jackTrackOuts = false;
	jackConnectDefaults = true;

	// Alsa MIDI driver properties
	midiPortChannel = -1;
	midiDest_name = "-";
	midiDest_client = -1;
	midiDest_port = -1;
	m_bIgnoreMidiNoteOff = true;

	// GUI Properties
	m_sQTStyle = "";
	m_interfaceMode = PreferencesMng::MDI;
	applicationFontFamily = "Helvetica";
	applicationFontPointSize = 10;
	mixerFontFamily = "Helvetica";
	mixerFontPointSize = 8;
	m_nPatternEditorGridHeight = 16;
	m_nPatternEditorGridWidth = 3;
	mixerFalloffSpeed = FALLOFF_NORMAL;
	m_nPatternEditorGridResolution = 8;
	m_bPatternEditorUsingTriplets = false;
	m_bShowInstrumentPeaks = true;

	// window properties
	mainFormProperties.x = 0;
	mainFormProperties.y = 0;
	mainFormProperties.width = 723;
	mainFormProperties.height = 645;

	mixerProperties.x = 10;
	mixerProperties.y = 350;
	mixerProperties.width = 829;
	mixerProperties.height = 220;

	patternEditorProperties.x = 280;
	patternEditorProperties.y = 100;
	patternEditorProperties.width = 706;
	patternEditorProperties.height = 302;

	songEditorProperties.x = 10;
	songEditorProperties.y = 10;
	songEditorProperties.width = 600;
	songEditorProperties.height = 250;

	drumkitManagerProperties.x = 500;
	drumkitManagerProperties.y = 20;
	drumkitManagerProperties.visible = false;

	audioEngineInfoProperties.x = 720;
	audioEngineInfoProperties.y = 120;
	audioEngineInfoProperties.visible = false;

	m_instrumentEditorProperties.x = 10;
	m_instrumentEditorProperties.y = 100;
	m_instrumentEditorProperties.visible = false;
	
	m_ladspaProperties[0].x = 100;
	m_ladspaProperties[0].y = 100;
	m_ladspaProperties[0].visible = false;

	m_ladspaProperties[1].x = 110;
	m_ladspaProperties[1].y = 110;
	m_ladspaProperties[1].visible = false;

	m_ladspaProperties[2].x = 120;
	m_ladspaProperties[2].y = 120;
	m_ladspaProperties[2].visible = false;

	m_ladspaProperties[3].x = 130;
	m_ladspaProperties[3].y = 130;
	m_ladspaProperties[3].visible = false;

	
	// default UI Style
	
	m_pDefaultUIStyle = new UIStyle();
	
	m_pDefaultUIStyle->m_songEditor_backgroundColor = RGBColor( 100, 100, 100 );
	m_pDefaultUIStyle->m_songEditor_selectedRowColor = RGBColor( 100, 100, 150 );
	m_pDefaultUIStyle->m_songEditor_alternateRowColor = RGBColor( 92, 92, 92 );
	m_pDefaultUIStyle->m_songEditor_lineColor = RGBColor( 130, 130, 130 );
	m_pDefaultUIStyle->m_songEditor_textColor = RGBColor( 200, 200, 200 );

	m_pDefaultUIStyle->m_patternEditorPanel_backgroundColor = RGBColor( 131, 149, 180 );
	
	m_pDefaultUIStyle->m_patternEditor_backgroundColor = RGBColor( 230, 230, 230 );
	m_pDefaultUIStyle->m_patternEditor_alternateRowColor = RGBColor( 240, 240, 255 );
	m_pDefaultUIStyle->m_patternEditor_selectedRowColor = RGBColor( 209, 233, 245 );
	m_pDefaultUIStyle->m_patternEditor_selectedMuteRowColor = RGBColor( 169, 183, 195 );
	m_pDefaultUIStyle->m_patternEditor_mutedRowColor= RGBColor( 200, 200, 200 );
	m_pDefaultUIStyle->m_patternEditor_textColor = RGBColor( 40, 40, 40 );
	m_pDefaultUIStyle->m_patternEditor_noteColor = RGBColor( 40, 40, 40 );
	m_pDefaultUIStyle->m_patternEditor_lineColor = RGBColor( 150, 150, 150 );
	m_pDefaultUIStyle->m_patternEditor_line1Color = RGBColor( 100, 100, 100 );
	m_pDefaultUIStyle->m_patternEditor_line2Color = RGBColor( 130, 130, 130 );
	m_pDefaultUIStyle->m_patternEditor_line3Color = RGBColor( 160, 160, 160 );
	m_pDefaultUIStyle->m_patternEditor_line4Color = RGBColor( 190, 190, 190 );
	m_pDefaultUIStyle->m_patternEditor_line5Color = RGBColor( 190, 190, 190 );
	
	loadPreferences();
}



PreferencesMng::~PreferencesMng() {
	savePreferences();
	infoLog( "DESTROY" );
	instance = NULL;
	delete m_pDefaultUIStyle;
}




///
/// Get user's home directory.
///
string PreferencesMng::getUserHome() {
	string home;
	struct passwd *pwdbuf;

	pwdbuf = getpwuid(getuid());
	home = pwdbuf->pw_dir;

	return home;
}




///
/// Load the preferences file
///
void PreferencesMng::loadPreferences() {
	bool recreate = false;	// configuration file must be recreated?
	string prefDir = getUserHome() + "/.hydrogen";
	string dataDir = getUserHome() + "/.hydrogen/data";
	string filename = prefDir + "/hydrogen.conf";

	infoLog("Loading preferences file in " + filename);

	// pref directory exists?
	DIR* dirp = opendir( prefDir.c_str() );
	if ( dirp == NULL ) {
		warningLog( "[loadPreferences] Configuration directory not found." );
		createPreferencesDirectory();
	}

	// data directory exists?
	DIR* dirp2 = opendir( dataDir.c_str() );
	if ( dirp2 == NULL ) {
		warningLog( "[loadPreferences] Data directory not found." );
		createDataDirectory();
	}


	// pref file exists?
	std::ifstream input(filename.c_str() , std::ios::in | std::ios::binary);
	if (input){
		// read preferences file
		TiXmlDocument doc(filename.c_str());
		doc.LoadFile();

		TiXmlNode* rootNode;
		if ( (rootNode = doc.FirstChild("hydrogen_preferences")) ) {

			// version
			string version = LocalFileMng::readXmlString( this, rootNode, "version", "" );
			if ( version == "" ) {
				recreate = true;
			}

			//////// GENERAL ///////////
			//m_sLadspaPath = LocalFileMng::readXmlString( this, rootNode, "ladspaPath", m_sLadspaPath );
			restoreLastSong = LocalFileMng::readXmlBool( this, rootNode, "restoreLastSong", restoreLastSong );
			hearNewNotes = LocalFileMng::readXmlBool( this, rootNode, "hearNewNotes", hearNewNotes );
			recordEvents = LocalFileMng::readXmlBool( this, rootNode, "recordEvents", recordEvents );
			quantizeEvents = LocalFileMng::readXmlBool( this, rootNode, "quantizeEvents", quantizeEvents );

			TiXmlNode* pRecentUsedSongsNode = rootNode->FirstChild( "recentUsedSongs" );
			if ( pRecentUsedSongsNode ) {
				TiXmlNode* pSongNode = 0;
				for( pSongNode = pRecentUsedSongsNode->FirstChild("song"); pSongNode; pSongNode = pSongNode->NextSibling( "song" ) ) {
					string sFilename = pSongNode->FirstChild()->Value();
					m_recentFiles.push_back( sFilename );
				}
			}
			else {
				warningLog( "[loadPreferences] recentUsedSongs node not found" );
			}


			/////////////// AUDIO ENGINE //////////////
			TiXmlNode* audioEngineNode;
			if ( !(audioEngineNode = rootNode->FirstChild( "audio_engine" ) ) ) {
				warningLog( "[loadPreferences] audio_engine node not found" );
				recreate = true;
			}
			else {
				// Audio driver
				audioDriver = LocalFileMng::readXmlString( this, audioEngineNode, "audio_driver", audioDriver );

				// Use metronome
				useMetronome = LocalFileMng::readXmlBool( this, audioEngineNode, "use_metronome", useMetronome );

				// Metronome volume
				metronomeVolume = LocalFileMng::readXmlFloat( this, audioEngineNode, "metronome_volume", metronomeVolume );

				// Max notes
				maxNotes = LocalFileMng::readXmlInt( this, audioEngineNode, "maxNotes", maxNotes );

				//// OSS DRIVER ////
				TiXmlNode* ossDriverNode;
				if ( !(ossDriverNode = audioEngineNode->FirstChild( "oss_driver" ) ) ) {
					warningLog("[loadPreferences] oss_driver node not found");
					recreate = true;
				}
				else {
					// Buffer size
					bufferSize = LocalFileMng::readXmlInt( this, ossDriverNode, "buffer_size", bufferSize );

					// Sample rate
					sampleRate = LocalFileMng::readXmlInt( this, ossDriverNode, "samplerate", sampleRate );

					ossDevice = LocalFileMng::readXmlString( this, ossDriverNode, "ossDevice", ossDevice );
				}

				//// JACK DRIVER ////
				TiXmlNode* jackDriverNode;
				if ( !(jackDriverNode = audioEngineNode->FirstChild( "jack_driver" ) ) ) {
					warningLog("[loadPreferences] jack_driver node not found");
					recreate = true;
				}
				else {
					// Jack port name 1
					jackPortName1 = LocalFileMng::readXmlString( this, jackDriverNode, "jack_port_name_1", jackPortName1 );

					// Jack port name 2
					jackPortName2 = LocalFileMng::readXmlString( this, jackDriverNode, "jack_port_name_2", jackPortName2 );

					// Jack transport mode
					string sMode = LocalFileMng::readXmlString( this, jackDriverNode, "jack_transport_mode", "NO_JACK_TRANSPORT" );
					if (sMode == "NO_JACK_TRANSPORT") {
						m_jackTransportMode = NO_JACK_TRANSPORT;
					}
					else if (sMode == "USE_JACK_TRANSPORT") {
						m_jackTransportMode = USE_JACK_TRANSPORT;
					}


					// Jack track outs?
					jackTrackOuts = LocalFileMng::readXmlBool( this, jackDriverNode, "jack_track_outs", jackTrackOuts );

					// Connect to default jack outputs?
					jackConnectDefaults = LocalFileMng::readXmlBool( this, jackDriverNode, "jack_connect_defaults", jackConnectDefaults );
				}

				//// ALSA MIDI DRIVER ////
				TiXmlNode* alsaMidiDriverNode;
				if ( !(alsaMidiDriverNode = audioEngineNode->FirstChild( "alsa_midi_driver" ) ) ) {
					warningLog( "[loadPreferences] alsa_midi_driver node not found" );
					recreate = true;
				}
				else {
					// midi port channel
					midiPortChannel = LocalFileMng::readXmlInt( this, alsaMidiDriverNode, "midi_port_channel", midiPortChannel );

					// midi destination name
					midiDest_name = LocalFileMng::readXmlString( this, alsaMidiDriverNode, "midi_dest_name", midiDest_name );

					// midi destination client
					midiDest_client = LocalFileMng::readXmlInt( this, alsaMidiDriverNode, "midi_dest_client", midiDest_client );

					// midi destination port
					midiDest_port = LocalFileMng::readXmlInt( this, alsaMidiDriverNode, "midi_dest_port", midiDest_port );

					// ignore note off
					m_bIgnoreMidiNoteOff = LocalFileMng::readXmlBool( this, alsaMidiDriverNode, "ignoreMidiNoteOff", m_bIgnoreMidiNoteOff );
				}
			}

			/////////////// GUI //////////////
			TiXmlNode* guiNode;
			if ( !(guiNode = rootNode->FirstChild( "gui" ) ) ) {
				warningLog("[loadPreferences] gui node not found");
				recreate = true;
			}
			else {
				// QT Style
				m_sQTStyle = LocalFileMng::readXmlString( this, guiNode, "QTStyle", m_sQTStyle );

				// Interface mode
				
				string sMode = LocalFileMng::readXmlString( this, guiNode, "interface_mode", "" );
				if (sMode == "Child frame") {
					m_interfaceMode = MDI;
				}
				else if ( sMode == "Top level" ) {
					m_interfaceMode = TOP_LEVEL;
				}
				else if ( sMode == "Single paned" ) {
					m_interfaceMode = SINGLE_PANED;
				}
				else {
					m_interfaceMode = SINGLE_PANED;
				}

				// Application font family
				applicationFontFamily = LocalFileMng::readXmlString( this, guiNode, "application_font_family", applicationFontFamily );

				// Application font pointSize
				applicationFontPointSize = LocalFileMng::readXmlInt( this, guiNode, "application_font_pointsize", applicationFontPointSize );

				// mixer font family
				mixerFontFamily = LocalFileMng::readXmlString( this, guiNode, "mixer_font_family", mixerFontFamily );

				// mixer font pointSize
				mixerFontPointSize = LocalFileMng::readXmlInt( this, guiNode, "mixer_font_pointsize", mixerFontPointSize );

				// Mixer falloff speed
				mixerFalloffSpeed = LocalFileMng::readXmlFloat( this, guiNode, "mixer_falloff_speed", mixerFalloffSpeed );

				// pattern editor grid resolution
				m_nPatternEditorGridResolution = LocalFileMng::readXmlInt( this, guiNode, "patternEditorGridResolution", m_nPatternEditorGridResolution );
				m_bPatternEditorUsingTriplets = LocalFileMng::readXmlBool( this, guiNode, "patternEditorUsingTriplets", m_bPatternEditorUsingTriplets );
				m_bShowInstrumentPeaks = LocalFileMng::readXmlBool( this, guiNode, "showInstrumentPeaks", m_bShowInstrumentPeaks );

				// pattern editor grid height
				m_nPatternEditorGridHeight = LocalFileMng::readXmlInt( this, guiNode, "patternEditorGridHeight", m_nPatternEditorGridHeight );

				// pattern editor grid width
				m_nPatternEditorGridWidth = LocalFileMng::readXmlInt( this, guiNode, "patternEditorGridWidth", m_nPatternEditorGridWidth );

				// mainForm window properties
				setMainFormProperties( readWindowProperties( guiNode, "mainForm_properties", mainFormProperties ) );
				setMixerProperties( readWindowProperties( guiNode, "mixer_properties", mixerProperties ) );
				setPatternEditorProperties( readWindowProperties( guiNode, "patternEditor_properties", patternEditorProperties ) );
				setSongEditorProperties( readWindowProperties( guiNode, "songEditor_properties", songEditorProperties ) );
				setDrumkitManagerProperties( readWindowProperties( guiNode, "drumkitManager_properties", drumkitManagerProperties ) );
				setAudioEngineInfoProperties( readWindowProperties( guiNode, "audioEngineInfo_properties", audioEngineInfoProperties ) );
				setInstrumentEditorProperties( readWindowProperties( guiNode, "instrumentEditor_properties", audioEngineInfoProperties ) );

				
				for (uint nFX = 0; nFX < MAX_FX; nFX++) {
					string sNodeName = "ladspaFX_properties" + toString(nFX);
					setLadspaProperties(nFX, readWindowProperties( guiNode, sNodeName, m_ladspaProperties[nFX] ) );
				}
				
				TiXmlNode *pUIStyle = guiNode->FirstChild( "UI_Style" );
				if ( pUIStyle ) {
					readUIStyle( *pUIStyle );
				}
				else {
					warningLog( "[loadPreferences] UI_Style node not found" );
					recreate = true;
				}
			}

			/////////////// FILES //////////////
			TiXmlNode* filesNode;
			if ( !(filesNode = rootNode->FirstChild( "files" ) ) ) {
				warningLog( "[loadPreferences] files node not found");
				recreate = true;
			}
			else {
				// last used song
				lastSongFilename = LocalFileMng::readXmlString( this, filesNode, "lastSongFilename", lastSongFilename );
			}

			
		} // rootNode
		else {
			warningLog("[loadPreferences] hydrogen_preferences node not found");
			recreate = true;
		}
	}
	else {
		warningLog("Configuration file not found.");
		recreate = true;
	}


	// The preferences file should be recreated?
	if (recreate == true) {
		infoLog("Recreating configuration file");
		savePreferences();
	}

}



///
/// Save the preferences file
///
void PreferencesMng::savePreferences() {
	infoLog("Saving preferences file");
	string prefDir = getUserHome() + "/.hydrogen";
	string filename = prefDir + "/hydrogen.conf";

	TiXmlDocument doc(filename.c_str());

	TiXmlElement rootNode("hydrogen_preferences");

	// hydrogen version
	LocalFileMng::writeXmlString( &rootNode, "version", string(VERSION) );

	////// GENERAL ///////
	string restoreLastSongStr = "false";
	if ( restoreLastSong ) {
		restoreLastSongStr = "true";
	}
	LocalFileMng::writeXmlString( &rootNode, "restoreLastSong", restoreLastSongStr );

	// hear new notes in the pattern editor
	string hearNewNotesStr = "false";
	if ( hearNewNotes ) {
		hearNewNotesStr = "true";
	}
	LocalFileMng::writeXmlString( &rootNode, "hearNewNotes", hearNewNotesStr );

	// key/midi event prefs
	LocalFileMng::writeXmlString( &rootNode, "recordEvents", recordEvents ? "true": "false" );
	LocalFileMng::writeXmlString( &rootNode, "quantizeEvents", quantizeEvents ? "true": "false" );


	// Recent used songs
	TiXmlElement recentUsedSongsNode( "recentUsedSongs" );
	{
		uint nSongs = 5;
		if ( m_recentFiles.size() < 5 ) {
			nSongs = m_recentFiles.size();
		}
		for ( uint i = 0; i < nSongs; i++ ) {
			LocalFileMng::writeXmlString( &recentUsedSongsNode, "song", m_recentFiles[ i ] );
		}
	}
	rootNode.InsertEndChild( recentUsedSongsNode );


	//---- AUDIO ENGINE ----
	TiXmlElement audioEngineNode( "audio_engine" );
	{
		// audio driver
		LocalFileMng::writeXmlString( &audioEngineNode, "audio_driver", audioDriver );

		// use metronome
		string useMetronomeStr = "false";
		if ( useMetronome ) {
			useMetronomeStr = "true";
		}
		LocalFileMng::writeXmlString( &audioEngineNode, "use_metronome", useMetronomeStr );


		// Metronome volume
		LocalFileMng::writeXmlString( &audioEngineNode, "metronome_volume", toString( metronomeVolume ) );

		// Max notes
		LocalFileMng::writeXmlString( &audioEngineNode, "maxNotes", toString( maxNotes ) );


		//// OSS DRIVER ////
		TiXmlElement ossDriverNode("oss_driver");
		{
			// buffer size
			LocalFileMng::writeXmlString( &ossDriverNode, "buffer_size", toString( bufferSize ) );

			// sample rate
			LocalFileMng::writeXmlString( &ossDriverNode, "samplerate", toString( sampleRate ) );

			LocalFileMng::writeXmlString( &ossDriverNode, "ossDevice", ossDevice );
		}
		audioEngineNode.InsertEndChild( ossDriverNode );

		//// JACK DRIVER ////
		TiXmlElement jackDriverNode( "jack_driver" );
		{
			LocalFileMng::writeXmlString( &jackDriverNode, "jack_port_name_1", jackPortName1 );	// jack port name 1
			LocalFileMng::writeXmlString( &jackDriverNode, "jack_port_name_2", jackPortName2 );	// jack port name 2

			// jack transport slave
			string sMode;
			if ( m_jackTransportMode == NO_JACK_TRANSPORT) {
				sMode = "NO_JACK_TRANSPORT";
			}
			else if ( m_jackTransportMode == USE_JACK_TRANSPORT) {
				sMode = "USE_JACK_TRANSPORT";
			}
			LocalFileMng::writeXmlString( &jackDriverNode, "jack_transport_mode", sMode );

			// jack default connection
			string jackConnectDefaultsString = "false";
			if (jackConnectDefaults) {
				jackConnectDefaultsString = "true";
			}
			LocalFileMng::writeXmlString( &jackDriverNode, "jack_connect_defaults", jackConnectDefaultsString );

			// jack track outs
			string jackTrackOutsString = "false";
			if (jackTrackOuts) {
				jackTrackOutsString = "true";
			}
			LocalFileMng::writeXmlString( &jackDriverNode, "jack_track_outs", jackTrackOutsString );

		}
		audioEngineNode.InsertEndChild( jackDriverNode );

		//// ALSA MIDI DRIVER ////
		TiXmlElement alsaMidiDriverNode( "alsa_midi_driver" );
		{
			LocalFileMng::writeXmlString( &alsaMidiDriverNode, "midi_port_channel", toString( midiPortChannel ) );	// Midi port channel
			LocalFileMng::writeXmlString( &alsaMidiDriverNode, "midi_dest_name", midiDest_name );		// Midi destination name
			LocalFileMng::writeXmlString( &alsaMidiDriverNode, "midi_dest_client", toString( midiDest_client ) );		// Midi destination client
			LocalFileMng::writeXmlString( &alsaMidiDriverNode, "midi_dest_port", toString( midiDest_port ) );		// Midi destination port

			// Ignore midi note off
			string sIgnore = "false";
			if (m_bIgnoreMidiNoteOff) {
				sIgnore = "true";
			}
			LocalFileMng::writeXmlString( &alsaMidiDriverNode, "ignoreMidiNoteOff", sIgnore );
		}
		audioEngineNode.InsertEndChild( alsaMidiDriverNode );

	}
	rootNode.InsertEndChild( audioEngineNode );

	//---- GUI ----
	TiXmlElement guiNode("gui");
	{
		LocalFileMng::writeXmlString( &guiNode, "QTStyle", m_sQTStyle );

		// Interface mode
		switch ( m_interfaceMode ) {
			case TOP_LEVEL:
				LocalFileMng::writeXmlString( &guiNode, "interface_mode", "Top level" );
				break;
				
			case MDI:
				LocalFileMng::writeXmlString( &guiNode, "interface_mode", "Child frame" );
				break;
			
			case SINGLE_PANED:
			default:
				LocalFileMng::writeXmlString( &guiNode, "interface_mode", "Single paned" );
				break;
		}

		LocalFileMng::writeXmlString( &guiNode, "application_font_family", applicationFontFamily );
		LocalFileMng::writeXmlString( &guiNode, "application_font_pointsize", toString( applicationFontPointSize ) );
		LocalFileMng::writeXmlString( &guiNode, "mixer_font_family", mixerFontFamily );
		LocalFileMng::writeXmlString( &guiNode, "mixer_font_pointsize", toString( mixerFontPointSize ) );
		LocalFileMng::writeXmlString( &guiNode, "mixer_falloff_speed", toString( mixerFalloffSpeed ) );
		LocalFileMng::writeXmlString( &guiNode, "patternEditorGridResolution", toString( m_nPatternEditorGridResolution ) );
		LocalFileMng::writeXmlString( &guiNode, "patternEditorGridHeight", toString( m_nPatternEditorGridHeight ) );
		LocalFileMng::writeXmlString( &guiNode, "patternEditorGridWidth", toString( m_nPatternEditorGridWidth ) );
		LocalFileMng::writeXmlBool( &guiNode, "patternEditorUsingTriplets", m_bPatternEditorUsingTriplets );
		LocalFileMng::writeXmlBool( &guiNode, "showInstrumentPeaks", m_bShowInstrumentPeaks );

		// MainForm window properties
		writeWindowProperties( guiNode, "mainForm_properties", mainFormProperties );
		writeWindowProperties( guiNode, "mixer_properties", mixerProperties );
		writeWindowProperties( guiNode, "patternEditor_properties", patternEditorProperties );
		writeWindowProperties( guiNode, "songEditor_properties", songEditorProperties );
		writeWindowProperties( guiNode, "drumkitManager_properties", drumkitManagerProperties );
		writeWindowProperties( guiNode, "audioEngineInfo_properties", audioEngineInfoProperties );
		writeWindowProperties( guiNode, "instrumentEditor_properties", m_instrumentEditorProperties );
		for (uint nFX = 0; nFX < MAX_FX; nFX++) {
			string sNode = "ladspaFX_properties" + toString(nFX);
			writeWindowProperties( guiNode, sNode, m_ladspaProperties[nFX] );
		}
		
		// User interface style
		writeUIStyle( guiNode );
	}
	rootNode.InsertEndChild( guiNode );

	//---- FILES ----
	TiXmlElement filesNode( "files" );
	{
		// last used song
		LocalFileMng::writeXmlString( &filesNode, "lastSongFilename", lastSongFilename );
	}
	rootNode.InsertEndChild( filesNode );

	doc.InsertEndChild(rootNode);
	doc.SaveFile();
}



///
/// Create preferences directory
///
void PreferencesMng::createPreferencesDirectory() {
	string prefDir = getUserHome() + "/.hydrogen";

	warningLog("Creating preference file directory in " + prefDir);

	mkdir(prefDir.c_str(),S_IRWXU);

}



///
/// Create data directory
///
void PreferencesMng::createDataDirectory() {
	string dir = getUserHome() + "/.hydrogen/data";

	warningLog("Creating data directory in " + dir);

	mkdir(dir.c_str(),S_IRWXU);
}



void PreferencesMng::setRecentFiles( vector<string> recentFiles )
{
	// find single filenames. (skip duplicates)
	vector<string> temp;
	for (uint i = 0; i < recentFiles.size(); i++) {
		string sFilename = recentFiles[ i ];

		bool bExists = false;
		for (uint j = 0; j < temp.size(); j++) {
			if ( sFilename == temp[ j ] ) {
				bExists = true;
				break;
			}
		}
		if ( !bExists ) {
			temp.push_back( sFilename );
		}
	}

	m_recentFiles = temp;
}



/// Read the xml nodes related to window properties
WindowProperties PreferencesMng::readWindowProperties( TiXmlNode *parent, string windowName, WindowProperties defaultProp ) {
	WindowProperties prop = defaultProp;

	TiXmlNode* windowPropNode;
	if ( !(windowPropNode = parent->FirstChild( windowName.c_str() ) ) ) {
		warningLog( "Error reading configuration file: " + windowName + " node not found" );
	}
	else {
		prop.visible = LocalFileMng::readXmlBool( this, windowPropNode, "visible", true );
		prop.x = LocalFileMng::readXmlInt( this, windowPropNode, "x", prop.x );
		prop.y = LocalFileMng::readXmlInt( this, windowPropNode, "y", prop.y );
		prop.width = LocalFileMng::readXmlInt( this, windowPropNode, "width", prop.width );
		prop.height = LocalFileMng::readXmlInt( this, windowPropNode, "height", prop.height );
	}

	return prop;
}



/// Write the xml nodes related to window properties
void PreferencesMng::writeWindowProperties( TiXmlNode& parent, const string& windowName, const WindowProperties& prop ) 
{
	TiXmlElement windowPropNode( windowName.c_str() );
		if (prop.visible) {
			LocalFileMng::writeXmlString( &windowPropNode, "visible", "true" );
		}
		else {
			LocalFileMng::writeXmlString( &windowPropNode, "visible", "false" );
		}
		
		LocalFileMng::writeXmlString( &windowPropNode, "x", toString( prop.x ) );
		LocalFileMng::writeXmlString( &windowPropNode, "y", toString( prop.y ) );
		LocalFileMng::writeXmlString( &windowPropNode, "width", toString( prop.width ) );
		LocalFileMng::writeXmlString( &windowPropNode, "height", toString( prop.height ) );
	parent.InsertEndChild( windowPropNode );
}



void PreferencesMng::writeUIStyle( TiXmlNode& parent )
{
	infoLog( "[writeUIStyle]" );
	TiXmlElement node( "UI_Style" );
	
	// SONG EDITOR
	TiXmlElement songEditorNode( "songEditor" );
	LocalFileMng::writeXmlString( &songEditorNode, "backgroundColor", m_pDefaultUIStyle->m_songEditor_backgroundColor.toString() );
	LocalFileMng::writeXmlString( &songEditorNode, "alternateRowColor", m_pDefaultUIStyle->m_songEditor_alternateRowColor.toString() );
	LocalFileMng::writeXmlString( &songEditorNode, "selectedRowColor", m_pDefaultUIStyle->m_songEditor_selectedRowColor.toString() );
	LocalFileMng::writeXmlString( &songEditorNode, "lineColor", m_pDefaultUIStyle->m_songEditor_lineColor.toString() );
	LocalFileMng::writeXmlString( &songEditorNode, "textColor", m_pDefaultUIStyle->m_songEditor_textColor.toString() );
	node.InsertEndChild( songEditorNode );
	
	// PATTERN EDITOR PANEL
	TiXmlElement patternEditorPanelNode( "patternEditorPanel" );
	LocalFileMng::writeXmlString( &patternEditorPanelNode, "backgroundColor", m_pDefaultUIStyle->m_patternEditorPanel_backgroundColor.toString() );
	node.InsertEndChild( patternEditorPanelNode );
	
	// PATTERN EDITOR
	TiXmlElement patternEditorNode( "patternEditor" );
	LocalFileMng::writeXmlString( &patternEditorNode, "backgroundColor", m_pDefaultUIStyle->m_patternEditor_backgroundColor.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "alternateRowColor", m_pDefaultUIStyle->m_patternEditor_alternateRowColor.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "selectedMuteRowColor", m_pDefaultUIStyle->m_patternEditor_selectedMuteRowColor.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "selectedRowColor", m_pDefaultUIStyle->m_patternEditor_selectedRowColor.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "mutedRowColor", m_pDefaultUIStyle->m_patternEditor_mutedRowColor.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "textColor", m_pDefaultUIStyle->m_patternEditor_textColor.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "noteColor", m_pDefaultUIStyle->m_patternEditor_noteColor.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "lineColor", m_pDefaultUIStyle->m_patternEditor_lineColor.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "line1Color", m_pDefaultUIStyle->m_patternEditor_line1Color.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "line2Color", m_pDefaultUIStyle->m_patternEditor_line2Color.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "line3Color", m_pDefaultUIStyle->m_patternEditor_line3Color.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "line4Color", m_pDefaultUIStyle->m_patternEditor_line4Color.toString() );
	LocalFileMng::writeXmlString( &patternEditorNode, "line5Color", m_pDefaultUIStyle->m_patternEditor_line5Color.toString() );
	node.InsertEndChild( patternEditorNode );
	
	parent.InsertEndChild( node );
}



void PreferencesMng::readUIStyle( TiXmlNode& parent )
{
	// SONG EDITOR
	TiXmlNode* pSongEditorNode = parent.FirstChild( "songEditor" );
	if ( pSongEditorNode ) {
		m_pDefaultUIStyle->m_songEditor_backgroundColor = RGBColor( LocalFileMng::readXmlString( this, pSongEditorNode, "backgroundColor", m_pDefaultUIStyle->m_songEditor_backgroundColor.toString() ) );
		m_pDefaultUIStyle->m_songEditor_alternateRowColor = RGBColor( LocalFileMng::readXmlString( this, pSongEditorNode, "alternateRowColor", m_pDefaultUIStyle->m_songEditor_alternateRowColor.toString() ) );
		m_pDefaultUIStyle->m_songEditor_selectedRowColor = RGBColor( LocalFileMng::readXmlString( this, pSongEditorNode, "selectedRowColor", m_pDefaultUIStyle->m_songEditor_selectedRowColor.toString() ) );
		m_pDefaultUIStyle->m_songEditor_lineColor = RGBColor( LocalFileMng::readXmlString( this, pSongEditorNode, "lineColor", m_pDefaultUIStyle->m_songEditor_lineColor.toString() ) );
		m_pDefaultUIStyle->m_songEditor_textColor = RGBColor( LocalFileMng::readXmlString( this, pSongEditorNode, "textColor", m_pDefaultUIStyle->m_songEditor_textColor.toString() ) );
	}
	else {
		warningLog( "[readUIStyle] songEditor node not found" );
	}
	
	// PATTERN EDITOR PANEL
	TiXmlNode* pPatternEditorPanelNode = parent.FirstChild( "patternEditorPanel" );
	if ( pPatternEditorPanelNode ) {
		m_pDefaultUIStyle->m_patternEditorPanel_backgroundColor = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorPanelNode, "backgroundColor", m_pDefaultUIStyle->m_patternEditorPanel_backgroundColor.toString() ) );
	}
	else {
		warningLog( "[readUIStyle] songEditorPanel node not found" );
	}
	
	// PATTERN EDITOR
	TiXmlNode* pPatternEditorNode = parent.FirstChild( "patternEditor" );
	if ( pPatternEditorNode ) {
		m_pDefaultUIStyle->m_patternEditor_backgroundColor = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "backgroundColor", m_pDefaultUIStyle->m_patternEditor_backgroundColor.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_alternateRowColor = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "alternateRowColor", m_pDefaultUIStyle->m_patternEditor_alternateRowColor.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_selectedRowColor = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "selectedRowColor", m_pDefaultUIStyle->m_patternEditor_selectedRowColor.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_selectedMuteRowColor = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "selectedMuteRowColor", m_pDefaultUIStyle->m_patternEditor_selectedMuteRowColor.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_mutedRowColor = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "mutedRowColor", m_pDefaultUIStyle->m_patternEditor_mutedRowColor.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_textColor = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "textColor", m_pDefaultUIStyle->m_patternEditor_textColor.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_noteColor = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "noteColor", m_pDefaultUIStyle->m_patternEditor_noteColor.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_lineColor = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "lineColor", m_pDefaultUIStyle->m_patternEditor_lineColor.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_line1Color = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "line1Color", m_pDefaultUIStyle->m_patternEditor_line1Color.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_line2Color = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "line2Color", m_pDefaultUIStyle->m_patternEditor_line2Color.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_line3Color = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "line3Color", m_pDefaultUIStyle->m_patternEditor_line3Color.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_line4Color = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "line4Color", m_pDefaultUIStyle->m_patternEditor_line4Color.toString() ) );
		m_pDefaultUIStyle->m_patternEditor_line5Color = RGBColor( LocalFileMng::readXmlString( this, pPatternEditorNode, "line5Color", m_pDefaultUIStyle->m_patternEditor_line5Color.toString() ) );
	}
	else {
		warningLog( "[readUIStyle] patternEditor node not found" );
	}
}


// -----------------------




WindowProperties::WindowProperties()
 : Object( "WindowProperties" )
{
//	infoLog( "INIT" );
	x = 0;
	y = 0;
	width = 0;
	height = 0;
	visible = true;
}



WindowProperties::~WindowProperties() {
//	infoLog( "DESTROY" );
}




// :::::::::::::::::::::::::::::::



UIStyle::UIStyle()
 : Object( "UIStyle" )
{
	infoLog( "INIT" );
}



// ::::::::::::::::::::::::::::::::::::::



RGBColor::RGBColor(int r, int g, int b)
 : Object( "RGBColor" ) 
 , m_red( r )
 , m_green( g )
 , m_blue( b ) 
{
//	infoLog( "INIT" );
}



RGBColor::~RGBColor() 
{
//	infoLog( "DESTROY" );
}



RGBColor::RGBColor( const string& sColor )
 : Object ( "RGBColor" )
{
//	infoLog( "INIT " + sColor );
	string temp = sColor;

	int nPos = temp.find(',');
	string sRed = temp.substr(0, nPos);
	temp.erase(0, nPos + 1);
	
	nPos = temp.find(',');
	string sGreen = temp.substr(0, nPos);
	temp.erase(0, nPos + 1);
	
	nPos = temp.find(',');
	string sBlue = temp.substr(0, nPos);
	
	m_red = atoi( sRed.c_str() );
	m_green = atoi( sGreen.c_str() );
	m_blue = atoi( sBlue.c_str() );

}



string RGBColor::toString()
{
	string sRes = ::toString( m_red ) + "," + ::toString( m_green ) + "," + ::toString( m_blue ); 
	
//	infoLog( "[toString] " + sRes );
	
	return sRes;
}


