/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: PreferencesMng.h,v 1.27 2004/07/07 14:44:46 comix Exp $
 *
 */

#ifndef PREFERENCES_MNG
#define PREFERENCES_MNG

#include <string>
using std::string;
#include <vector>
using std::vector;

#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <fstream>
#include <iostream>
#include <dirent.h>
#include <stdio.h>

#include "Object.h"
#include "Globals.h"

const float FALLOFF_SLOW = 	1.08;
const float FALLOFF_NORMAL=	1.1;
const float FALLOFF_FAST =		1.5;

// forward declaration
class TiXmlNode;

class WindowProperties : public Object{
	public:
		int x;
		int y;
		int width;
		int height;
		bool visible;
		
		WindowProperties();
		~WindowProperties();
};



class RGBColor : public Object
{
	public:
		RGBColor(int r = -1, int g = -1, int b = -1);
		RGBColor( const string& sColor );
		~RGBColor();
		
		string toString();

		int getRed() {	return m_red;	}
		int getGreen() {	return m_green;	}
		int getBlue() {	return m_blue;	}

	private:
		int m_red;
		int m_green;
		int m_blue;

};


///
/// Colors for hydogen
///
class UIStyle : public Object
{
	public:
		UIStyle();

		
		RGBColor m_songEditor_backgroundColor;
		RGBColor m_songEditor_alternateRowColor;
		RGBColor m_songEditor_selectedRowColor;
		RGBColor m_songEditor_lineColor;
		RGBColor m_songEditor_textColor;
		
		RGBColor m_patternEditorPanel_backgroundColor;
		
		RGBColor m_patternEditor_backgroundColor;
		RGBColor m_patternEditor_alternateRowColor;
		RGBColor m_patternEditor_selectedRowColor;
		RGBColor m_patternEditor_selectedMuteRowColor;
		RGBColor m_patternEditor_mutedRowColor;
		RGBColor m_patternEditor_textColor;
		RGBColor m_patternEditor_noteColor;
		RGBColor m_patternEditor_lineColor;
		RGBColor m_patternEditor_line1Color;
		RGBColor m_patternEditor_line2Color;
		RGBColor m_patternEditor_line3Color;
		RGBColor m_patternEditor_line4Color;
		RGBColor m_patternEditor_line5Color;


};



/// Manager for User Preferences File (singleton)
class PreferencesMng : public Object {
	public:
		enum {
			USE_JACK_TRANSPORT,
			NO_JACK_TRANSPORT
		};

		enum InterfaceMode {
			MDI,
			TOP_LEVEL,
			SINGLE_PANED
		};
		
		/// Returns an instance of PreferencesMng class
		static PreferencesMng* getInstance();

		~PreferencesMng();

		/// Load the preferences file
		void loadPreferences();

		/// Save the preferences file
		void savePreferences();

		string getDemoPath() {	return demoPath;	}

		/// Get user's home directory
		string getUserHome();


		// General
		void setRestoreLastSongEnabled( bool restore ) {	restoreLastSong = restore;	}
		bool isRestoreLastSongEnabled() {	return restoreLastSong;	}

		void setLastSongFilename( string filename ) {	lastSongFilename = filename;	}
		string getLastSongFilename() {	return lastSongFilename;	}

		void setHearNewNotes( bool value ) {	hearNewNotes = value;	}
		bool getHearNewNotes() {	return hearNewNotes;	}

		void setRecordEvents( bool value ) {	recordEvents = value;	}
		bool getRecordEvents() {	return recordEvents;	}

		void setQuantizeEvents( bool value ) {	quantizeEvents = value;	}
		bool getQuantizeEvents() {	return quantizeEvents;	}

		vector<string> getRecentFiles() {	return m_recentFiles;	}
		void setRecentFiles( vector<string> recentFiles );

		vector<string> getLadspaPath() {	return m_ladspaPathVect;	}
		void setLadspaPath( vector<string> pathVect ) {	m_ladspaPathVect = pathVect;	}


		// Audio Engine properties
		string getAudioDriver() {	return audioDriver;	}
		void setAudioDriver(string audioDriver) {	this->audioDriver = audioDriver;	}

		bool isMetronomeEnabled() {	return useMetronome;	}
		void setUseMetronome(bool useMetronome) {	this->useMetronome = useMetronome;	}

		float getMetronomeVolume() {	return metronomeVolume;	}
		void setMetronomeVolume(float metronomeVolume) {	this->metronomeVolume = metronomeVolume;	}

		void setMaxNotes(uint notes) {	maxNotes = notes;	}
		uint getMaxNotes() {	return maxNotes;	}


		// OSS Driver properties
		uint getBufferSize() {	return bufferSize;	}
		void setBufferSize(uint bufferSize) {	this->bufferSize = bufferSize;	}

		uint getSampleRate() {	return sampleRate;	}
		void setSampleRate(uint sampleRate) {	this->sampleRate = sampleRate;	}

		void setOssDevice(string device) {	ossDevice = device;	}
		string getOssDevice() {	return ossDevice;	}


		// JACK Driver properties
		string getJackPortName1() {	return jackPortName1;	}
		void setJackPortName1( string port ) {	jackPortName1 = port;	}

		string getJackPortName2() {	return jackPortName2;	}
		void setJackPortName2( string port ) {	jackPortName2 = port;	}

		bool getJackTransportMode() {	return m_jackTransportMode; }
		void setJackTransportMode( bool mode ) {	m_jackTransportMode = mode;	}

		bool isJackTrackOuts() {	return jackTrackOuts; }
		void setJackTrackOuts( bool slave ) {	jackTrackOuts = slave;	}

		bool isJackConnectDefaults() {	return jackConnectDefaults; }
		void setJackConnectDefaults( bool slave ) {	jackConnectDefaults = slave;	}


		// ALSA MIDI Driver
		int getMidiPortChannel() {	return midiPortChannel;	}
		void setMidiPortChannel(int channel) {	midiPortChannel = channel;	}

		void setMidiDest_name( string name ) {	this->midiDest_name = name;	}
		string getMidiDest_name() {	return midiDest_name;	}

		void setMidiDest_client( int client ) {	this->midiDest_client = client;	}
		int getMidiDest_client() {	return midiDest_client;	}

		void setMidiDest_port( int port ) {	this->midiDest_port = port; }
		int getMidiDest_port() {	return midiDest_port;	}

		bool isIgnoringMidiNoteOff() {	return m_bIgnoreMidiNoteOff;	}
		void setIgnoreMidiNoteOff(bool value) {	m_bIgnoreMidiNoteOff = value;	}


		// GUI Properties
		string getQTStyle() {	return m_sQTStyle;	}
		void setQTStyle( string sStyle ) {	m_sQTStyle = sStyle;	}

		InterfaceMode getInterfaceMode() {	return m_interfaceMode;	}
		void setInterfaceMode( InterfaceMode mode ) {	m_interfaceMode = mode;	}

		string getApplicationFontFamily() {	return applicationFontFamily;	}
		void setApplicationFontFamily(string family) {	applicationFontFamily = family;	}

		int getApplicationFontPointSize() {	return applicationFontPointSize;	}
		void setApplicationFontPointSize(int size) {	applicationFontPointSize = size;	}

		string getMixerFontFamily() {	return mixerFontFamily;	}
		void setMixerFontFamily(string family) {	mixerFontFamily = family;	}

		int getMixerFontPointSize() {	return mixerFontPointSize;	}
		void setMixerFontPointSize(int size) {	mixerFontPointSize = size;	}

		float getMixerFalloffSpeed() {	return mixerFalloffSpeed;	}
		void setMixerFalloffSpeed(float value) {	mixerFalloffSpeed = value;	}

		int getPatternEditorGridResolution() {	return m_nPatternEditorGridResolution;	}
		void setPatternEditorGridResolution( int value ) {	m_nPatternEditorGridResolution = value;	}

		bool isPatternEditorUsingTriplets() {	return m_bPatternEditorUsingTriplets;	}
		void setPatternEditorUsingTriplets( bool value ) {	m_bPatternEditorUsingTriplets = value;	}

		bool showInstrumentPeaks() {	return m_bShowInstrumentPeaks;	}
		void setInstrumentPeaks( bool value ) {	m_bShowInstrumentPeaks = value;	}

		uint getPatternEditorGridHeight() {	return m_nPatternEditorGridHeight;	}
		void setPatternEditorGridHeight(uint value) {	m_nPatternEditorGridHeight = value;	}

		uint getPatternEditorGridWidth() {	return m_nPatternEditorGridWidth;	}
		void setPatternEditorGridWidth(uint value) {	m_nPatternEditorGridWidth = value;	}

		WindowProperties getMainFormProperties() {	return mainFormProperties;	}
		void setMainFormProperties( WindowProperties prop ) {	mainFormProperties = prop;	}

		WindowProperties getMixerProperties() {	return mixerProperties;	}
		void setMixerProperties( WindowProperties prop ) {	mixerProperties = prop;	}

		WindowProperties getPatternEditorProperties() {	return patternEditorProperties;	}
		void setPatternEditorProperties( WindowProperties prop ) {	patternEditorProperties = prop;	}

		WindowProperties getSongEditorProperties() {	return songEditorProperties;	}
		void setSongEditorProperties( WindowProperties prop ) {	songEditorProperties = prop;	}

		WindowProperties getDrumkitManagerProperties() {	return drumkitManagerProperties;	}
		void setDrumkitManagerProperties( WindowProperties prop ) {	drumkitManagerProperties = prop;	}

		WindowProperties getAudioEngineInfoProperties() {	return audioEngineInfoProperties;	}
		void setAudioEngineInfoProperties( WindowProperties prop ) {	audioEngineInfoProperties = prop;	}

		WindowProperties getLadspaProperties(uint nFX) {	return m_ladspaProperties[nFX];	}
		void setLadspaProperties( uint nFX, WindowProperties prop ) {	m_ladspaProperties[nFX] = prop;	}
		
		WindowProperties getInstrumentEditorProperties() {	return m_instrumentEditorProperties;	}
		void setInstrumentEditorProperties( WindowProperties prop ) {	m_instrumentEditorProperties = prop;	}

		UIStyle* getDefaultUIStyle(){	return m_pDefaultUIStyle;	}
		
	private:
		static PreferencesMng *instance;

		/** directory of demo songs */
		string demoPath;

		//___ General properties ___
		bool restoreLastSong;		///< Restore last song?
		string lastSongFilename;	///< Last song used
		bool hearNewNotes;
		vector<string> m_recentFiles;
		vector<string> m_ladspaPathVect;
		bool quantizeEvents;
		bool recordEvents;

		//___ audio engine properties ___
		string audioDriver;		///< Audio Driver
		bool useMetronome;		///< Use metronome?
		float metronomeVolume;	///< Metronome volume FIXME: remove this volume!!
		uint maxNotes;			///< max notes

		//___ oss driver properties ___
		uint bufferSize;			///< Audio buffer size
		uint sampleRate;		///< Audio sample rate
		string ossDevice;		///< Device used for output


		//___  jack driver properties ___
		string jackPortName1;
		string jackPortName2;
		int m_jackTransportMode;
		bool jackTrackOuts;
		bool jackConnectDefaults;


		//___ alsa midi driver properties ___
		int midiPortChannel;
		string midiDest_name;
		int midiDest_client;
		int midiDest_port;
		bool m_bIgnoreMidiNoteOff;


		//___ GUI properties ___
		string m_sQTStyle;
		InterfaceMode m_interfaceMode;
		string applicationFontFamily;
		int applicationFontPointSize;
		string mixerFontFamily;
		int mixerFontPointSize;
		float mixerFalloffSpeed;
		int m_nPatternEditorGridResolution;
		bool m_bPatternEditorUsingTriplets;
		bool m_bShowInstrumentPeaks;
		uint m_nPatternEditorGridHeight;
		uint m_nPatternEditorGridWidth;
		WindowProperties mainFormProperties;
		WindowProperties mixerProperties;
		WindowProperties patternEditorProperties;
		WindowProperties songEditorProperties;
		WindowProperties drumkitManagerProperties;
		WindowProperties audioEngineInfoProperties;
		WindowProperties m_ladspaProperties[MAX_FX];
		WindowProperties m_instrumentEditorProperties;

		UIStyle*  m_pDefaultUIStyle;
		

		PreferencesMng();

		/// Create preferences directory
		void createPreferencesDirectory();

		/// Create data directory
		void createDataDirectory();
		
		WindowProperties readWindowProperties( TiXmlNode *parent, string windowName, WindowProperties defaultProp );
		void writeWindowProperties( TiXmlNode& parent, const string& windowName, const WindowProperties& prop );
		
		void writeUIStyle( TiXmlNode& parent );
		void readUIStyle( TiXmlNode& parent );
};

#endif

