package net.i2p.router.transport.udp;

import java.util.*;

import net.i2p.data.Base64;
import net.i2p.data.SessionKey;
import net.i2p.data.RouterInfo;
import net.i2p.data.RouterAddress;
import net.i2p.router.RouterContext;
import net.i2p.util.Log;

/**
 *
 */
public class IntroductionManager {
    private RouterContext _context;
    private Log _log;
    private UDPTransport _transport;
    private PacketBuilder _builder;
    /** map of relay tag to PeerState that should receive the introduction */
    private Map _outbound;
    /** list of peers (PeerState) who have given us introduction tags */
    private List _inbound;

    public IntroductionManager(RouterContext ctx, UDPTransport transport) {
        _context = ctx;
        _log = ctx.logManager().getLog(IntroductionManager.class);
        _transport = transport;
        _builder = new PacketBuilder(ctx, transport);
        _outbound = Collections.synchronizedMap(new HashMap(128));
        _inbound = new ArrayList(128);
        ctx.statManager().createRateStat("udp.receiveRelayIntro", "How often we get a relayed request for us to talk to someone?", "udp", new long[] { 60*1000, 5*60*1000, 10*60*1000 });
        ctx.statManager().createRateStat("udp.receiveRelayRequest", "How often we receive a request to relay to someone else?", "udp", new long[] { 60*1000, 5*60*1000, 10*60*1000 });
    }
    
    public void reset() {
        _inbound.clear();
        _outbound.clear();
    }
    
    public void add(PeerState peer) {
        if (peer == null) return;
        if (_log.shouldLog(Log.DEBUG))
            _log.debug("Adding peer " + peer.getRemoteHostId() + ", weRelayToThemAs " 
                       + peer.getWeRelayToThemAs() + ", theyRelayToUsAs " + peer.getTheyRelayToUsAs());
        if (peer.getWeRelayToThemAs() > 0) 
            _outbound.put(new Long(peer.getWeRelayToThemAs()), peer);
        if (peer.getTheyRelayToUsAs() > 0) {
            synchronized (_inbound) {
                if (!_inbound.contains(peer))
                    _inbound.add(peer);
            }
        }
    }
    
    public void remove(PeerState peer) {
        if (peer == null) return;
        if (_log.shouldLog(Log.DEBUG))
            _log.debug("removing peer " + peer.getRemoteHostId() + ", weRelayToThemAs " 
                       + peer.getWeRelayToThemAs() + ", theyRelayToUsAs " + peer.getTheyRelayToUsAs());
        if (peer.getWeRelayToThemAs() > 0) 
            _outbound.remove(new Long(peer.getWeRelayToThemAs()));
        if (peer.getTheyRelayToUsAs() > 0) {
            synchronized (_inbound) {
                _inbound.remove(peer);
            }
        }
    }
    
    public PeerState get(long id) {
        return (PeerState)_outbound.get(new Long(id));
    }
    
    /**
     * Grab a bunch of peers who are willing to be introducers for us that
     * are locally known (duh) and have published their own SSU address (duh^2).
     * The picked peers have their info tacked on to the ssuOptions parameter for
     * use in the SSU RouterAddress.
     *
     */
    public int pickInbound(Properties ssuOptions, int howMany) {
        List peers = null;
        int start = _context.random().nextInt(Integer.MAX_VALUE);
        synchronized (_inbound) {
            if (_log.shouldLog(Log.DEBUG))
                _log.debug("Picking inbound out of " + _inbound);
            if (_inbound.size() <= 0) return 0;
            peers = new ArrayList(_inbound);
        }
        int sz = peers.size();
        start = start % sz;
        int found = 0;
        for (int i = 0; i < sz && found < howMany; i++) {
            PeerState cur = (PeerState)peers.get((start + i) % sz);
            RouterInfo ri = _context.netDb().lookupRouterInfoLocally(cur.getRemotePeer());
            if (ri == null) {
                if (_log.shouldLog(Log.INFO))
                    _log.info("Picked peer has no local routerInfo: " + cur);
                continue;
            }
            RouterAddress ra = ri.getTargetAddress(UDPTransport.STYLE);
            if (ra == null) {
                if (_log.shouldLog(Log.INFO))
                    _log.info("Picked peer has no SSU address: " + ri);
                continue;
            }
            UDPAddress ura = new UDPAddress(ra);
            ssuOptions.setProperty(UDPAddress.PROP_INTRO_HOST_PREFIX + found, cur.getRemoteHostId().toHostString());
            ssuOptions.setProperty(UDPAddress.PROP_INTRO_PORT_PREFIX + found, String.valueOf(cur.getRemotePort()));
            ssuOptions.setProperty(UDPAddress.PROP_INTRO_KEY_PREFIX + found, Base64.encode(ura.getIntroKey()));
            ssuOptions.setProperty(UDPAddress.PROP_INTRO_TAG_PREFIX + found, String.valueOf(cur.getTheyRelayToUsAs()));
            found++;
        }
        return found;
    }
    
    public void receiveRelayIntro(RemoteHostId bob, UDPPacketReader reader) {
        if (_context.router().isHidden())
            return;
        if (_log.shouldLog(Log.INFO))
            _log.info("Receive relay intro from " + bob);
        _context.statManager().addRateData("udp.receiveRelayIntro", 1, 0);
        _transport.send(_builder.buildHolePunch(reader));
    }
    
    public void receiveRelayRequest(RemoteHostId alice, UDPPacketReader reader) {
        _context.statManager().addRateData("udp.receiveRelayRequest", 1, 0);
        if (_context.router().isHidden())
            return;
        long tag = reader.getRelayRequestReader().readTag();
        PeerState charlie = _transport.getPeerState(tag);
        if (_log.shouldLog(Log.INFO))
            _log.info("Receive relay request from " + alice 
                      + " for tag " + tag
                      + " and relaying with " + charlie);
        if (charlie == null)
            return;
        byte key[] = new byte[SessionKey.KEYSIZE_BYTES];
        reader.getRelayRequestReader().readAliceIntroKey(key, 0);
        SessionKey aliceIntroKey = new SessionKey(key);
        // send that peer an introduction for alice
        _transport.send(_builder.buildRelayIntro(alice, charlie, reader.getRelayRequestReader()));
        // send alice back charlie's info
        _transport.send(_builder.buildRelayResponse(alice, charlie, reader.getRelayRequestReader().readNonce(), aliceIntroKey));
    }
}
