/*
 * Copyright (c) 2000-2001, Sascha Schumann. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.  
 * 
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution. 
 * 
 * Neither the name of Sascha Schumann nor the names of his contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

/* $Id: if_irc.h,v 1.24.2.3 2001/08/01 08:48:17 sas Exp $ */

#ifndef IF_IRC_H
#define IF_IRC_H

#define IRCG_API_VERSION 20010601

#include <st.h>
#include <sys/socket.h>

#include "ircg_config.h"
#include "queue.h"
#include "php_smart_str_public.h"

#ifdef IRCG_DEFINE_SOCKADDR_STORAGE
struct sockaddr_storage {
	char sockaddr_storage_data[256];
};
#endif

extern time_t ircg_now_time_t;

typedef struct irc irconn_t;
typedef struct chan irc_chan_t;

typedef void (*msg_func_t)(irconn_t *, smart_str *chan, smart_str *from, smart_str *msg, void *conn_data, void *chan_data);
typedef void (*quit_func_t)(irconn_t *, void *);
typedef void (*connect_func_t)(irconn_t *, void *);
typedef void (*error_func_t)(irconn_t *, int, int, smart_str *msg, void *);

typedef void (*nick_func_t)(irconn_t *, smart_str *oldnick, smart_str *newnick, void *data);
typedef void (*part_func_t)(irconn_t *, smart_str *channel, void *);
typedef void (*user_add_t)(irconn_t *, smart_str *channel, smart_str *users, int nr, void *);
typedef void (*user_leave_t)(irconn_t *, smart_str *channel, smart_str *users, void *);
typedef void (*user_quit_t)(irconn_t *, smart_str *user, smart_str *reason, void *);
typedef void (*user_kick_t)(irconn_t *, smart_str *channel, smart_str *who, smart_str *kicked_by, smart_str *reason, void *);
typedef void (*new_topic_t)(irconn_t *, smart_str *channel, smart_str *who, smart_str *topic, void *);

typedef void (*whois_user_t)(irconn_t *, smart_str *nick, smart_str *user,
		smart_str *host, smart_str *real_name, void *);
typedef void (*whois_server_t)(irconn_t *, smart_str *nick, smart_str *server,
		smart_str *server_info, void *);
typedef void (*whois_idle_t)(irconn_t *, smart_str *nick, smart_str *idletime,
		void *);
typedef void (*whois_channels_t)(irconn_t *, smart_str *nick,
		smart_str *channel_names, int nr, void *);
typedef void (*end_of_whois_t)(irconn_t *, smart_str *nick, void *);

typedef void (*banlist_t)(irconn_t *, smart_str *channel, smart_str *mask, void *);
typedef void (*end_of_banlist_t)(irconn_t *, smart_str *channel, void *);

enum {
	IRCG_MODE_VOICE = (1 << 0),
	IRCG_MODE_OP    = (1 << 1)
};

typedef void (*mode_channel_t)(irconn_t *, smart_str *nick, smart_str *channel,
		smart_str *who, int mode, int give, void *data, void *chan_data);

typedef void (*idle_recv_queue_t)(irconn_t *, void *data);

enum {
	IRCG_PART,
	IRCG_USER_ADD,
	IRCG_USER_LEAVE,
	IRCG_USER_KICK,
	IRCG_TOPIC,
	IRCG_MSG,
	IRCG_QUIT,
	IRCG_ERROR,
	IRCG_CONNECT,
	IRCG_NICK,
	IRCG_USER_QUIT,
	IRCG_WHOISUSER,
	IRCG_WHOISSERVER,
	IRCG_WHOISIDLE,
	IRCG_WHOISCHANNELS,
	IRCG_ENDOFWHOIS,
	IRCG_MODE_CHANNEL,
	IRCG_IDLE_RECV_QUEUE,
	IRCG_BANLIST,
	IRCG_ENDOFBANLIST,
	IRCG_NR_HOOKS
};

#define MAX_USERNAME_LEN 32
#define MAX_IDENT_LEN 9
#define MAX_PASSWORD_LEN 32
#define MAX_REALNAME_LEN 64

typedef void (*irc_hook)();

struct irc_ignore {
	smart_str who;
	SLIST_ENTRY(irc_ignore) next;
};

struct irc {
	const char username[MAX_USERNAME_LEN];
	char username_len;
	
	st_netfd_t c;

	irc_hook hooks[IRCG_NR_HOOKS];
	
	struct sockaddr_storage server;
	size_t server_len;
	int sockpf;

	char status;

	void *data;
	st_thread_t dispatcher;

	SLIST_HEAD(chan_head, chan) channels;
	SLIST_HEAD(, irc_ignore) ignore;

	smart_str msg_buffer;
	
	char ident[MAX_IDENT_LEN];
	char password[MAX_PASSWORD_LEN];
	char realname[MAX_REALNAME_LEN];
};

#define MAX_CHANNEL_LEN 32

struct chan {
	irconn_t *user;

	const char channel[MAX_CHANNEL_LEN];
	unsigned char channel_len;
	
	void *data;

	SLIST_ENTRY(chan) next;
};

typedef void (*register_hooks_t)(irconn_t *, void *);

int irc_connect(const char *username,
		register_hooks_t hooks,
		void *data,
		const char *server,
		int port,
		irconn_t *conn
		);

int irc_join(irconn_t *, 
		const char *channel, 
		const char *key,
		void *data
		);

int irc_nick(irconn_t *, const char *newnick);

/* not C99 compatible -- yet */
int irc_register_hook(irconn_t *, int type, irc_hook);

int irc_set_password(irconn_t *, smart_str *);

int irc_set_ident(irconn_t *, smart_str *);

int irc_set_realname(irconn_t *, smart_str *);

int irc_ignore_check(irconn_t *, smart_str *);
int irc_ignore_add(irconn_t *, smart_str *, int count);
int irc_ignore_del(irconn_t *, smart_str *);
int irc_ignore_destroy(irconn_t *);

int irc_sizeof_irconn(void);
int irc_sizeof_write_buf(void);

int irc_part(irconn_t *, const char *channel);

int irc_disconnect(irconn_t *, const char *reason);

int irc_msg(irconn_t *, const char *recipient, const char *msg);

#define irc_handle_command irc_send_message

/* do NOT use this function directly.  Always use irc_handle_command() */
void irc_send_message(irconn_t *, const char *cmd, int nr, ...);

#endif
