/*
 * Copyright (c) 2000-2001, Sascha Schumann. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.  
 * 
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution. 
 * 
 * Neither the name of Sascha Schumann nor the names of his contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

/* $Id: irc_dispatcher.c,v 1.38.2.2 2001/09/14 19:29:51 sas Exp $ */


#include <string.h>
#include <st.h>
#include <stdlib.h>
#include <stdio.h>

#include "if_irc.h"
#include "irc_private.h"
#include "php_smart_str.h"
#include "irc_codes.h"

static void ircmsg_new(ircmsg_t *msg)
{
	memset(msg, 0, sizeof(*msg));
}

static void ircmsg_free(ircmsg_t *msg)
{
	msg->nr_para = 0;
}

#define IRC_CMD_ENTRYX(symbol, func) { symbol, func }
#define IRC_CMD_ENTRY(n) IRC_CMD_ENTRYX( #n, irc_cmd_##n)
#define IRC_CMD_ALIAS(m,n) IRC_CMD_ENTRYX( #m, irc_cmd_##n )
#define IRC_CMD_ENTRN(n) IRC_CMD_ENTRYX( n, irc_cmd_##n)
#define IRC_CMD_FUNC(n) static int irc_cmd_##n(irconn_t *conn, ircmsg_t *msg)

typedef int (*cmd_handler_func)(irconn_t *, ircmsg_t *);

int ircg_log_opt;

IRC_CMD_FUNC(PING)
{
	if (msg->nr_para == 1) {
		irc_send_message_ex(conn, 0, "PONG", 1, msg->para[0].c);
	}
	return 0;
}

IRC_CMD_FUNC(NOTICE)
{

	return 0;
}

IRC_CMD_FUNC(TOPIC)
{
	irc_chan_t *ep;

	if (msg->nr_para < 2) return -1;

	ep = irc_locate_channel(conn, msg->para[0].c);

	if (ep && HOOK_FUNC(new_topic_t, IRCG_TOPIC)) {
		smart_str tmp;
		smart_str_setl(&tmp, ep->channel, ep->channel_len);
		HOOK_FUNC(new_topic_t, IRCG_TOPIC)(conn, &tmp, &msg->nickname, &msg->para[1], ep->data);
	}

	return 0;
}

IRC_CMD_FUNC(RPL_TOPIC)
{
	irc_chan_t *ep;
	smart_str who = {0};

	if (msg->nr_para < 3) return -1;

	ep = irc_locate_channel(conn, msg->para[1].c);

	if (ep && HOOK_FUNC(new_topic_t, IRCG_TOPIC)) {
		smart_str tmp;
		smart_str_setl(&tmp, ep->channel, ep->channel_len);
		smart_str_setl(&who, "Server", 6);
		HOOK_FUNC(new_topic_t, IRCG_TOPIC)(conn, &tmp, &who, &msg->para[2], ep->data);
	}

	return 0;
}

IRC_CMD_FUNC(RPL_WHOISUSER)
{
	if (msg->nr_para < 6) return -1;

	CALL_HOOK_FUNC(whois_user_t, IRCG_WHOISUSER)(conn, &msg->para[1],
			&msg->para[2], &msg->para[3], &msg->para[5], conn->data);
	return 0;
}

IRC_CMD_FUNC(RPL_WHOISSERVER)
{
	if (msg->nr_para < 4) return -1;
	
	CALL_HOOK_FUNC(whois_server_t, IRCG_WHOISSERVER)(conn, &msg->para[1],
			&msg->para[2], &msg->para[3], conn->data);
	return 0;
}

IRC_CMD_FUNC(RPL_WHOISIDLE)
{
	if (msg->nr_para < 3) return -1;

	CALL_HOOK_FUNC(whois_idle_t, IRCG_WHOISIDLE)(conn, &msg->para[1],
			&msg->para[2], conn->data);
	return 0;
}

#define MAX_CHANNELS 50

IRC_CMD_FUNC(RPL_WHOISCHANNELS)
{
	smart_str channels[MAX_CHANNELS];
	int i;
	char *p;
	
	if (msg->nr_para < 3) return -1;

	/* Split the channel list and store each element in the array */
	for (i = 0, p = strtok(msg->para[2].c, " "); 
			p && i < MAX_CHANNELS; 
			p = strtok(NULL, " "))
		smart_str_sets(&channels[i++], p);

	if (i)
		CALL_HOOK_FUNC(whois_channels_t, IRCG_WHOISCHANNELS)(conn,
				&msg->para[1], channels, i, conn->data);
	
	return 0;
}

IRC_CMD_FUNC(RPL_ENDOFWHOIS)
{
	if (msg->nr_para < 2) return -1;

	CALL_HOOK_FUNC(end_of_whois_t, IRCG_ENDOFWHOIS)(conn, &msg->para[1],
			conn->data);
	return 0;
}

#define MAX_NICKS 300

IRC_CMD_FUNC(RPL_NAMREPLY)
{
	smart_str users[MAX_NICKS];
	int i;
	char *p;
	irc_chan_t *ep;
	
	if (msg->nr_para < 4) return -1;

	ep = irc_locate_channel(conn, msg->para[2].c);

	if (!ep || !HOOK_FUNC(user_add_t, IRCG_USER_ADD)) return -1;
	
	/* Split the nick list and store each element in the array */
	for (i = 0, p = strtok(msg->para[3].c, " "); 
			p && i < MAX_NICKS; 
			p = strtok(NULL, " "))
		smart_str_sets(&users[i++], p);

	if (i) {
		smart_str tmp;
		smart_str_setl(&tmp, ep->channel, ep->channel_len);

		HOOK_FUNC(user_add_t, IRCG_USER_ADD)(conn, &tmp, users, i, ep->data);
	}

	return 0;
}

static void irc_flush_msg_buffer(irconn_t *conn)
{
	if (conn->msg_buffer.c) {
		if (st_write(conn->c, conn->msg_buffer.c, conn->msg_buffer.len, 0) == -1) {
			irc_disconnect(conn, "st_write failed");
		}
		smart_str_free(&conn->msg_buffer);
	}
}

IRC_CMD_FUNC(RPL_WELCOME)
{
	/* The IRC server might limit the username to a certain length. */
	if (msg->nr_para > 0 && conn->username_len != msg->para[0].len) {
		conn->username_len = msg->para[0].len;
		memcpy((char *) conn->username, msg->para[0].c, conn->username_len + 1);
	}
	conn->status = 1;	
	irc_flush_msg_buffer(conn);	
	if (HOOK_FUNC(connect_func_t, IRCG_CONNECT)) HOOK_FUNC(connect_func_t, IRCG_CONNECT)(conn, conn->data);
	return 0;
}

IRC_CMD_FUNC(RPL_BANLIST)
{
	if (msg->nr_para < 3) return 1;
	
	CALL_HOOK_FUNC(banlist_t, IRCG_BANLIST)(conn, &msg->para[1], &msg->para[2], conn->data);
	return 0;
}

IRC_CMD_FUNC(RPL_ENDOFBANLIST)
{
	if (msg->nr_para < 2) return 1;
	
	CALL_HOOK_FUNC(end_of_banlist_t, IRCG_ENDOFBANLIST)(conn, &msg->para[1], conn->data);
	return 0;
}

IRC_CMD_FUNC(JOIN)
{
	irc_chan_t *ep;

	if (msg->nr_para < 1) return -1;

	ep = irc_locate_channel(conn, msg->para[0].c);

	if (conn->username_len == msg->nickname.len 
			&& !memcmp(msg->nickname.c, conn->username, conn->username_len))
		return 0;

	if (ep) {
		smart_str tmp;
		smart_str_setl(&tmp, ep->channel, ep->channel_len);
		if (HOOK_FUNC(user_add_t, IRCG_USER_ADD)) HOOK_FUNC(user_add_t, IRCG_USER_ADD)(conn, &tmp, &msg->nickname, 1, ep->data);
	}

	return 0;
}

IRC_CMD_FUNC(QUIT)
{
	if (HOOK_FUNC(user_quit_t, IRCG_USER_QUIT))
		HOOK_FUNC(user_quit_t, IRCG_USER_QUIT)(conn, &msg->nickname, msg->nr_para > 0 ? &msg->para[0] : NULL, conn->data);

	return 0;
}

IRC_CMD_FUNC(KICK)
{
	irc_chan_t *ep;

	if (msg->nr_para < 2) return -1;

	ep = irc_locate_channel(conn, msg->para[0].c);

	if (ep && HOOK_FUNC(user_kick_t, IRCG_USER_KICK)) {
		smart_str tmp;
		smart_str_setl(&tmp, ep->channel, ep->channel_len);

		HOOK_FUNC(user_kick_t, IRCG_USER_KICK)(conn, &tmp, &msg->para[1], &msg->nickname, msg->nr_para > 2 ? &msg->para[2] : NULL, ep->data);
	}

	return 0;
}

IRC_CMD_FUNC(PART)
{
	irc_chan_t *ep;

	if (msg->nr_para < 1) return -1;

	ep = irc_locate_channel(conn, msg->para[0].c);

	if (ep && HOOK_FUNC(user_leave_t, IRCG_USER_LEAVE)) {
		smart_str tmp;
		smart_str_setl(&tmp, ep->channel, ep->channel_len);

		HOOK_FUNC(user_leave_t, IRCG_USER_LEAVE)(conn, &tmp, &msg->nickname, ep->data);
	}

	return 0;
}

IRC_CMD_FUNC(MODE)
{
	irc_chan_t *ep;
	int give = 0;
	char *p;
	int argc = 1;
	int type;
	char c;

	if (msg->nr_para < 2) return -1;

	/* We only handle channel modes right now */
	ep = irc_locate_channel(conn, msg->para[0].c);

	if (ep && HOOK_FUNC(mode_channel_t, IRCG_MODE_CHANNEL)) {
		argc = 1;

again:
		if (msg->nr_para <= argc) return 0;
		p = msg->para[argc++].c;

split_it:
		while (*p) {
			c = *p;
			p++;
			switch (c) {
			case '+': give = 1; break;
			case '-': give = 0; break;
			case 'v': type = IRCG_MODE_VOICE; goto need_arg;
			case 'o': type = IRCG_MODE_OP; goto need_arg;
			/* Ignore Channel Creator Status */
			case 'O': argc++; break;
			}
		}
		goto again;

need_arg:
		if (msg->nr_para <= argc) return -1;

		HOOK_FUNC(mode_channel_t, IRCG_MODE_CHANNEL)(conn, &msg->para[argc],
				&msg->para[0], &msg->nickname, type, give, conn->data, ep->data);

		argc++;
		goto split_it;
	}

	return 0;
}

IRC_CMD_FUNC(PRIVMSG)
{
	irc_chan_t *ep;

	if (msg->nr_para < 2) return -1;

	if (irc_ignore_check(conn, &msg->nickname))
		return 0;

	ep = irc_locate_channel(conn, msg->para[0].c);
	
	if (HOOK_FUNC(msg_func_t, IRCG_MSG)) {
		smart_str tmp;
		if (ep) smart_str_setl(&tmp, ep->channel, ep->channel_len);

		HOOK_FUNC(msg_func_t, IRCG_MSG)(conn, ep?&tmp:NULL, &msg->nickname, 
			&msg->para[1], conn->data, ep?ep->data:NULL);
	}

	return 0;
}

IRC_CMD_FUNC(NICK)
{
	if (msg->nr_para < 1) return -1;

	/* Are we changing our own nick? */
	if (conn->username_len == msg->nickname.len 
			&& !memcmp(msg->nickname.c, conn->username, conn->username_len)) {
		if (msg->para[0].len >= MAX_USERNAME_LEN - 1)
			return -1;
		conn->username_len = msg->para[0].len;
		memcpy((char *) conn->username, msg->para[0].c, conn->username_len + 1);
	}
	
	if (HOOK_FUNC(nick_func_t, IRCG_NICK)) 
		HOOK_FUNC(nick_func_t, IRCG_NICK)(conn, &msg->nickname, &msg->para[0], conn->data);

	return 0;
}

IRC_CMD_FUNC(ERR_BANNEDFROMCHAN)
{
	CALL_HOOK_FUNC(error_func_t, IRCG_ERROR)(conn, 474, 0, &msg->para[1], conn->data);
	return 0;
}

IRC_CMD_FUNC(ERR_BADCHANNELKEY)
{
	CALL_HOOK_FUNC(error_func_t, IRCG_ERROR)(conn, 475, 0, &msg->para[1], conn->data);
	return 0;
}

IRC_CMD_FUNC(ERR_ERRONEUSNICKNAME)
{
	if (HOOK_FUNC(error_func_t, IRCG_ERROR)) HOOK_FUNC(error_func_t, IRCG_ERROR)(conn, 432, 1, &msg->para[1], conn->data);
	irc_send_message_ex(conn, 0, "QUIT :Errorneus Nickname", 0);
	return 0;
}

IRC_CMD_FUNC(ERR_NICKCOLLISION)
{
	if (HOOK_FUNC(error_func_t, IRCG_ERROR)) HOOK_FUNC(error_func_t, IRCG_ERROR)(conn, 436, 1, &msg->para[1], conn->data);
	irc_send_message_ex(conn, 0, "QUIT :Nick collision", 0);
	return 0;
}

IRC_CMD_FUNC(ERR_NICKNAMEINUSE)
{
	if (HOOK_FUNC(error_func_t, IRCG_ERROR)) HOOK_FUNC(error_func_t, IRCG_ERROR)(conn, 433, conn->status == 0 ? 1 : 0, &msg->para[2], conn->data);
	if (conn->status == 0)
		irc_send_message_ex(conn, 0, "QUIT :Nickname in use", 0);
	return 0;
}

IRC_CMD_FUNC(ERR_UNAVAILRESOURCE)
{
	if (HOOK_FUNC(error_func_t, IRCG_ERROR)) HOOK_FUNC(error_func_t, IRCG_ERROR)(conn, 437, 1, &msg->para[1], conn->data);
	irc_send_message_ex(conn, 0, "QUIT :Unavail resource", 0);
	return 0;
}


IRC_CMD_FUNC(ERR_NOSUCHNICK)
{
	CALL_HOOK_FUNC(error_func_t, IRCG_ERROR)(conn, 401, 0, &msg->para[1], conn->data);
	return 0;
}

IRC_CMD_FUNC(ERR_NOSUCHCHANNEL)
{
	CALL_HOOK_FUNC(error_func_t, IRCG_ERROR)(conn, 406, 0, &msg->para[1], conn->data);
	return 0;
}

typedef struct {
	const char *cmd;
	cmd_handler_func hnd;
} cmd_table_t;

static cmd_table_t irc_commands[] = {
	IRC_CMD_ENTRY(PRIVMSG),
	IRC_CMD_ENTRN(RPL_WELCOME),
	IRC_CMD_ENTRY(NOTICE),
	IRC_CMD_ALIAS(NOTICE, PRIVMSG),
	IRC_CMD_ENTRN(RPL_NAMREPLY),
	IRC_CMD_ENTRN(RPL_TOPIC),
	IRC_CMD_ENTRN(RPL_WHOISUSER),
	IRC_CMD_ENTRN(RPL_WHOISSERVER),
	IRC_CMD_ENTRN(RPL_WHOISIDLE),
	IRC_CMD_ENTRN(RPL_ENDOFWHOIS),
	IRC_CMD_ENTRN(RPL_WHOISCHANNELS),
	IRC_CMD_ENTRY(PING),
	IRC_CMD_ENTRY(JOIN),
	IRC_CMD_ENTRY(PART),
	IRC_CMD_ENTRY(MODE),
	IRC_CMD_ENTRY(NICK),
	IRC_CMD_ENTRY(QUIT),
	IRC_CMD_ENTRN(RPL_BANLIST),
	IRC_CMD_ENTRN(RPL_ENDOFBANLIST),
	IRC_CMD_ENTRN(ERR_ERRONEUSNICKNAME),
	IRC_CMD_ENTRN(ERR_BADCHANNELKEY),
	IRC_CMD_ENTRN(ERR_BANNEDFROMCHAN),
	IRC_CMD_ENTRN(ERR_NICKCOLLISION),
	IRC_CMD_ENTRN(ERR_NICKNAMEINUSE),
	IRC_CMD_ENTRN(ERR_UNAVAILRESOURCE),
	IRC_CMD_ENTRN(ERR_NOSUCHNICK),
	IRC_CMD_ENTRN(ERR_NOSUCHCHANNEL),
	IRC_CMD_ENTRY(KICK),
	IRC_CMD_ENTRY(TOPIC),
	{0}
};

static void dispatch_message(irconn_t *conn, ircmsg_t *msg)
{
	cmd_table_t *table = irc_commands;
	int i;

	for (i = 0; table[i].cmd; i++) {
		if (!strcasecmp(table[i].cmd, msg->cmd.c)) {
			table[i].hnd(conn, msg);
			return;
		}
	}
}

extern struct chan_head ircg_free_chan_list; 

static void cleanup_conn(irconn_t *conn)
{
	irc_chan_t *ep, *next;

	for (ep = SLIST_FIRST(&conn->channels);
			ep;
			ep = next) {
		next = SLIST_NEXT(ep, next);
		if (HOOK_FUNC(part_func_t, IRCG_PART)) {
			smart_str tmp;
			smart_str_setl(&tmp, ep->channel, ep->channel_len);
		
			HOOK_FUNC(part_func_t, IRCG_PART)(conn, &tmp, ep->data);
		}
		SLIST_INSERT_HEAD(&ircg_free_chan_list, ep, next);
	}

	smart_str_free(&conn->msg_buffer);
	
	irc_ignore_destroy(conn);
	
	if (conn->c) st_netfd_close(conn->c);
}

static int handle_login(irconn_t *conn)
{
	int s;

	if ((s = socket(conn->sockpf, SOCK_STREAM, 0)) == -1)
		return -1;

	if ((conn->c = st_netfd_open_socket(s)) == NULL) {
		close(s);
		return -1;
	}	

	if (st_connect(conn->c, (struct sockaddr *) &conn->server, conn->server_len, -1)) {
		st_netfd_close(conn->c);
		return -1;
	}

	if (conn->password[0])
		irc_send_message_ex(conn, 0, "PASS", 1, conn->password);
	irc_send_message_ex(conn, 0, "NICK", 1, conn->username);
	irc_send_message_ex(conn, 0, "USER", 4, conn->ident, "0", "*", conn->realname);

	return s;
}

#define BUF_SIZE sizeof(buf)

void *irc_dispatcher(void *dummy)
{
	irconn_t *conn = (irconn_t *) dummy;
	char buf[2048];
	ssize_t n;
	int off = 0;
	int s;
	ircmsg_t msg;
	char *p;
	size_t len, nlen;

	if (ircg_log_opt == 0) {
		char *env = getenv("IRCG_OPTIONS");

		if (env) {
			while (1) {
				switch (*env++) {
					case 'D': ircg_log_opt |= 2; break;
					case 'E': ircg_log_opt |= 4; break;
					case 'F': ircg_log_opt |= 8; break;
					case 'G': ircg_log_opt |= 16; break;
					case 0: goto done;
				}
			}
		}
done:
		ircg_log_opt |= 1;
	}
	
	if ((s = handle_login(conn)) < 0) {
		conn->c = NULL;
		goto cleanup;
	}

#define IDLE_RECV_TIME 30

	ircmsg_new(&msg);
	while ((n = st_read(conn->c, buf + off, BUF_SIZE - off - 1, IDLE_RECV_TIME * 1000000)) || 1) {
//	while ((n = st_read(conn->c, buf + off, BUF_SIZE - off - 1, 1000000000 )) || 1) {
		if (conn->status == 101) break;

		if (n == 0) break;
		if (n == -1) {
			if (errno == ETIME) {
				CALL_HOOK_FUNC(idle_recv_queue_t, IRCG_IDLE_RECV_QUEUE)(conn,
						conn->data);
				continue;
			}
			break;
		}

		if (ircg_log_opt & 2) {
			printf("RCVD(%d):%s\n", n, buf + off);
		}
		off += n;
		buf[off] = '\0';

		while (off > 0 && (p = memchr(buf, '\r', off)) && p[1] == '\n') {
			len = p - buf;
			if (!irc_scan_message(buf, len, &msg)) {
				if (ircg_log_opt & 4) {
					int i;
					
					printf("nick(%s),cmd(%s)", msg.nickname.len?msg.nickname.c:"", msg.cmd.c);
					for (i = 0; i < msg.nr_para; i++)
						printf(",para%d(%s)", i, msg.para[i].c);
					printf("\n");
				}
				dispatch_message(conn, &msg);
			}
			nlen = off - len - 2;
			if (nlen > 0) {
				memmove(buf, p + 2, off - len + 1); /* Incl. NUL */
			}
			off = nlen;
		}
	
		if (BUF_SIZE == off + 1) {
			/* Try to sync the buffer */
			off = 0;
		}

		if (off == 1 && buf[0] == '\0') {
			off = 0;
		}
	}
	
	if (conn->status < 101) {
		irc_disconnect(conn, "EOF from server");
	}
	if (ircg_log_opt & 8 && off > 0) {
		printf("WHOOPS. %d(%d) unparsed:--\n%s\n--\n", off, buf[0], buf);
	}
	
	ircmsg_free(&msg);
cleanup:	
	cleanup_conn(conn);
	if (HOOK_FUNC(quit_func_t, IRCG_QUIT)) HOOK_FUNC(quit_func_t, IRCG_QUIT)(conn, conn->data);
	st_thread_exit(NULL);
	return NULL;
}
