/*
 * Copyright (c) 2000-2001, Sascha Schumann. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.  
 * 
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution. 
 * 
 * Neither the name of Sascha Schumann nor the names of his contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

/* $Id: irc_write_buffer.c,v 1.21.2.7 2001/08/02 00:59:41 sas Exp $ */

unsigned long long smart_str_allocated_bytes;
#include "irc_write_buffer.h"
#include "php_smart_str.h"

#include <st.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

time_t ircg_now_time_t;

static st_thread_t irc_write_buf_th;
static LIST_HEAD(write_buf_head, write_buf) irc_write_bufs;
static SLIST_HEAD(free_irc_bucket_head, irc_bucket) free_irc_buckets;
static int recompute;
static unsigned long long bytes_sent;
static unsigned long nr_buckets;
static int sleeping;

#define BUF_PUNISH 1
#define MAX_VEC 96
#define MAX_BUF 12000

static long qsize;

static irc_write_buf *fd_to_write_buf[MAX_BUF];
static struct pollfd fd_array[MAX_BUF];

#ifndef MAX_QUEUE_SIZE
#define MAX_QUEUE_SIZE 3000000
#endif

static int log_opt;

static unsigned long lbuf;

static void destroy_brigade(struct write_buf *buf)
{
	struct irc_bucket *next, *b;

	b = STAILQ_FIRST(&buf->brigade);
	
	while (b) {
		next = STAILQ_NEXT(b, next);
		
		smart_str_free(&b->buf);
		SLIST_INSERT_HEAD(&free_irc_buckets, b, fnext);
		b = next;
	}
	buf->bsize = 0;
	STAILQ_INIT(&buf->brigade);
}

static void remove_from_queue(struct write_buf *m)
{
	if (m->queued) {
		qsize -= m->bsize;
		LIST_REMOVE(m, next);
		m->queued = 0;
		recompute = 1;
	}
	destroy_brigade(m);
}

static void inline dead_buffer(irc_write_buf *m)
{
	m->dead = 1;
	remove_from_queue(m);
	fd_to_write_buf[m->fd] = NULL;
}

static int insert_into_queue(irc_write_buf *buf)
{
	if (!buf->queued && buf->bsize) {
		qsize += buf->bsize;
		LIST_INSERT_HEAD(&irc_write_bufs, buf, next);
		buf->queued = 1;
		recompute = 1;
		if (sleeping) {
			sleeping = 0;
			st_thread_interrupt(irc_write_buf_th);
		}
		return -1;
	}
	return 0;
}

static void try_send(struct write_buf *m, int last_try)
{
	int n;
	struct iovec vec[MAX_VEC];
	int v = 0;
	struct irc_bucket *b;
	
	if (m->dead) return;

	STAILQ_FOREACH(b, &m->brigade, next) {
		vec[v].iov_base = b->buf.c;
		vec[v].iov_len = b->buf.len;
		v++;

		if (v >= MAX_VEC) break;
	}

	n = writev(m->fd, vec, v);

	if (n > 0) {
		m->blocked = 0;
		bytes_sent += n;
	}
	
	if (n == m->bsize) {
		remove_from_queue(m);
		return;
	} else if (n > 0) {
		struct irc_bucket *remove_head_until = NULL;

		if (m->queued) qsize -= n;
		m->bsize -= n;
		STAILQ_FOREACH(b, &m->brigade, next) {
			/* Was only a part of this bucket sent? */
			if (n < b->buf.len) {
				/* Truncate the buffer */
				memmove(b->buf.c, b->buf.c + n, b->buf.len - n);
				b->buf.len -= n;
				break;
			}
			/* We are going to remove the whole bucket */
			n -= b->buf.len;
			smart_str_free(&b->buf);
			remove_head_until = b;
			/* Insert the bucket into the free list */
			SLIST_INSERT_HEAD(&free_irc_buckets, b, fnext);
		}
		/* Remove all sent buckets in one rush */
		if (remove_head_until) {
			STAILQ_REMOVE_HEAD_UNTIL(&m->brigade, remove_head_until, next);
		}
	} else if (n == 0 || (n == -1 && (errno == EAGAIN || errno == EWOULDBLOCK))) {
		/* Punish the socket and try sending through it later */
		m->blocked = BUF_PUNISH;
		if (m->bsize > 50000) {
			shutdown(m->fd, 2);
			dead_buffer(m);
			return;
		}
	} else {
		dead_buffer(m);
		return;
	}

	if (!last_try) insert_into_queue(m);
}

static inline int rebuild_poll_array(struct pollfd *fds)
{
	int nfds = 0;
	irc_write_buf *m;
	
	LIST_FOREACH(m, &irc_write_bufs, next) {
		if (m->bsize) {
			fds[nfds].fd      = m->fd;
			fds[nfds].events  = POLLOUT;
			fds[nfds].revents = 0;
			nfds++;
		}
	}

	return nfds;
}


static void *irc_write_buf_thread(void *dummy)
{
	struct pollfd *fds = fd_array;
	int nfds;
	int n = 0, i;
	irc_write_buf *m;
	int nothing_to_do = 0;

	while (1) {
		nfds = rebuild_poll_array(fds);
		
		if (log_opt & 1) printf("%ld,nfds %d,bytes %llu,%d,lbuf %lu,nrbuckets %lu\n", qsize, nfds, bytes_sent, n, lbuf, nr_buckets);

		if (nfds == 0) {
			if (nothing_to_do++ > 40) {
				sleeping = 1;
				st_sleep(-1);
			} else
				st_usleep(200000);
			n = -1911;
			continue;
		} else
			nothing_to_do = 0;
		
		n = st_poll(fds, nfds, 200000);

		time(&ircg_now_time_t);
		
		if (n <= 0) continue;

		for (i = 0; i < nfds; i++) {
			m = fd_to_write_buf[fds[i].fd];
			if (!m) continue;
			
			if (fds[i].revents) {
				
				if (fds[i].revents & POLLOUT) {
					try_send(m, 0);
				} else if (fds[i].revents & (POLLNVAL | POLLERR | POLLHUP)) {
					dead_buffer(m);
				} else {
					printf("unknown event mask: %d\n", fds[i].revents);
				}
			} else {
				m->blocked = BUF_PUNISH;
			}
		}
	}
	/* We never reach this point */

	return (NULL);
}

void irc_write_buf_add(irc_write_buf *buf, int fd)
{
	/* In this case, we might have hit a race which is caused by
	   the asynchronous nature of this API */
	if (fd_to_write_buf[fd]) {
		dead_buffer(fd_to_write_buf[fd]);
	}
	fd_to_write_buf[fd] = buf;
	buf->fd = fd;
	buf->queued = buf->dead = 0;
	buf->blocked = 0;
	buf->bsize = 0;
	STAILQ_INIT(&buf->brigade);
	
	lbuf++;
	if (!irc_write_buf_th) {
		char *env;

		env = getenv("IRCG_OPTIONS");
		if (env) {
			while (1) {
				switch (*env++) {
					case 'A': log_opt |= 1; break;
					case 'B': log_opt |= 2; break;
					case 'C': log_opt |= 4; break;
					case 0: goto done;
				}
			}
		}
done:
		irc_write_buf_th = st_thread_create(irc_write_buf_thread, NULL, 0, 0);
	}
}

void irc_write_buf_del(irc_write_buf *buf)
{
	if (!buf->dead && buf->bsize > 0) {
		try_send(buf, 1);
	}
	/* Do we "own" the map entry? -- race protection */
	if (fd_to_write_buf[buf->fd] == buf)
		fd_to_write_buf[buf->fd] = NULL;
	lbuf--;
	remove_from_queue(buf);
}

int irc_write_buf_append_ex(irc_write_buf *buf, smart_str *data, int copy)
{
	struct irc_bucket *b;

	if (!buf->dead) {
		b = SLIST_FIRST(&free_irc_buckets);
		if (!b) {
			b = malloc(sizeof(struct irc_bucket));
			b->buf.c = NULL;
			b->buf.len = b->buf.a = 0;
			nr_buckets++;
		} else {
			SLIST_REMOVE_HEAD(&free_irc_buckets, fnext);
		}
		
		if (copy) {
			smart_str_appendl(&b->buf, data->c, data->len);
		} else {
			smart_str_setl(&b->buf, data->c, data->len);
		}
		
		STAILQ_INSERT_TAIL(&buf->brigade, b, next);
	
		buf->bsize += data->len;
		
		if (buf->queued) { 
			qsize += data->len;
		}
		return 0;
	} else if (!copy) {
		smart_str_free(data);
	}
	return -1;
}

void irc_write_buf_flush(irc_write_buf *buf)
{
	if (buf->dead) return;

	if (!buf->blocked && buf->bsize > 1400)
		try_send(buf, 0);
	else
		insert_into_queue(buf);
}
