/*-
 * Copyright (c) 1999,2000
 *	Konstantin Chuguev.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Konstantin Chuguev
 *	and its contributors.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	iconv (Charset Conversion Library) v2.0
 */

#include <ctype.h>	/* isspace */
#include <limits.h>	/* PATH_MAX */
#include <stdlib.h>	/* free */
#include <string.h>	/* strlen, strsep, strstr */

#define ICONV_INTERNAL
#include <iconv.h>

static const char builtin_aliases[] = 
#include "aliases.h"
	"";

static const char *
map_aliases(const char *path, size_t *len)
{
	const char *result;
	char file[PATH_MAX];
	off_t size = iconv_filesize(path, "charset.aliases", file);

	if (size <= 0)
		return NULL;
	result = (const char *)iconv_mmap(file, ++ size);
	if (result != NULL)
		*len = size;
	return result;
}

static int
normalise_alias(char **name)
{
	char *alias;
	if (!name || !(alias = *name))
		return 0;
	if (tolower(alias[0]) == 'x' && alias[1] == '-')
		*name += 2;
	alias = strdup(alias);
	if (!alias)
		return 1;
	*name = alias;
	for (*name = alias; *alias; alias ++)
			*alias = tolower(*alias);
	return 0;
}

static char *
find_alias(const char *charset, const char *aliases)
{
	const char *ptr = aliases - 1;
	size_t len = 0;

	while ((ptr = strstr(ptr + 1, charset))) {
		const char *end;
		char *result;
		if (ptr > aliases && !isspace(ptr[-1]))
			continue;
		if (len == 0)
			len = strlen(charset);
		if (ptr[len] != '\0' && !isspace(ptr[len]))
			continue;
		while (ptr-- > aliases)
			if (*ptr == '\n' && ptr[1] != '#' && !isspace(ptr[1]))
				break;
		for (end = ++ptr; *end != '\0' && !isspace(*end); end++);
		if (iconv_malloc(end+1 - ptr, (void **)&result))
			break;
		strncpy(result, ptr, end - ptr);
		return result;
	}
	return NULL;
}

int
iconv_get_aliases(char **name1, char **name2)
{
	char *alias1, *alias2;

	if (normalise_alias(name1))
		return 1;
	if (normalise_alias(name2)) {
		free(*name1);
		return 1;
	}
	alias1 = find_alias(*name1, builtin_aliases),
	alias2 = *name2 ? find_alias(*name2, builtin_aliases) : NULL;
	if (!alias1 || (*name2 && !alias2)) {
		char *path, *ptr, *plist = getenv("ICONV_TABLE_PATH");

		ptr = plist = strdup(plist ? plist : ICONV_TABLE_PATH);
		while ((path = strsep(&plist, ":")) != NULL) {
			size_t len;
			const char *file_aliases = map_aliases(path, &len);

			if (file_aliases) {
				if (!alias1)
					alias1 = find_alias(*name1, file_aliases);
				if (*name2 && !alias2)
					alias2 = find_alias(*name2, file_aliases);
				iconv_munmap(file_aliases, len);
			}
			if (alias1 && (!*name2 || alias2))
				break;
		}
		free(ptr);
	}
	if(alias1) {
		free(*name1);
		*name1 = alias1;
	}
	if(alias2) {
		free(*name2);
		*name2 = alias2;
	}
	return 0;
}
