/*-
 * Copyright (c) 1999, 2000
 *	Konstantin Chuguev.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Konstantin Chuguev
 *	and its contributors.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	iconv (Charset Conversion Library) v2.0
 */

#include <errno.h>		/* errno */
#include <limits.h>		/* PATH_MAX */
#include <stdio.h>		/* snprintf */
#include <stdlib.h>		/* free, malloc */
#include <string.h>		/* strncmp */
#include <machine/endian.h>	/* ntohl, ntohs */

#define ICONV_INTERNAL
#include <iconv.h>	/* iconv_ccs */


/* Generic coded character set conversion table */

typedef struct {
	unsigned char	label[8];  /* CSconvT<N>; N=[0-3] */
	uint32_t	tables[2]; /* offsets to 2 unidirectional tables */
} iconv_ccs_convtable;

#define ICONV_TBL_LABEL		"\003CSCT"
#define ICONV_TBL_LABEL_SIZE	5
#define ICONV_TBL_BYTE_ORDER(table)	(((table)->label[5]) & 1)
#define ICONV_TBL_NBITS(table)		((table)->label[6])
#define ICONV_TBL_VERSION(table)	((table)->label[7])

/* Indices for unidirectional conversion tables */
enum { iconv_ccs_to_ucs = 0, iconv_ccs_from_ucs = 1 };


/* Unidirectional conversion table types */

/* one-level tables */
typedef struct {
	ucs2_t data[128];
} iconv_ccs_table_7bit;	/* 7-bit charset to Unicode */

typedef struct {
	ucs2_t data[256];
} iconv_ccs_table_8bit;	/* 8-bit charset to Unicode */

/* two-level tables */
typedef struct {
	uint32_t  data[128];
} iconv_ccs_table_14bit;	/* 14-bit charset to Unicode */

typedef struct {
	uint32_t  data[256];
} iconv_ccs_table_16bit;	/* 16-bit charset to Unicode;
				 * Unicode to any charset */
/* abstract table */
typedef union {
	iconv_ccs_table_7bit	_7bit;
	iconv_ccs_table_8bit	_8bit;
	iconv_ccs_table_14bit	_14bit;
	iconv_ccs_table_16bit	_16bit;
} iconv_ccs_table_abstract;


/* host and network byte order array element macros */

#define iconv_table_elt(base, i, type)	\
	((type *)(((char *)(base)) + ((uint32_t *)(base))[(i)]))

#define iconv_table_elt_n(base, i, type)	\
	((type *)(((char *)(base)) + ntohl(((uint32_t *)(base))[(i)])))


#define abstable	((const iconv_ccs_table_abstract *)table)

/* Functions for host byte order tables */

static ucs2_t
cvt_7bit_h(const void *table, ucs2_t ch)
{
	return ch & 0x80 ? UCS_CHAR_INVALID : abstable->_7bit.data[ch];
}

static ucs2_t
cvt_8bit_h(const void *table, ucs2_t ch)
{
	return abstable->_8bit.data[ch];
}

static ucs2_t
cvt_14bit_h(const void *table, ucs2_t ch)
{
	const iconv_ccs_table_7bit *sub_table;

	if (ch & 0x8080)
		return UCS_CHAR_INVALID;
	sub_table =  iconv_table_elt(table, ch >> 8, iconv_ccs_table_7bit);
	return sub_table == &(abstable->_7bit) ? UCS_CHAR_INVALID
	                                       : sub_table->data[ch & 0x7F];
}

static ucs2_t
cvt_16bit_h(const void *table, ucs2_t ch)
{
	const iconv_ccs_table_8bit *sub_table =
		iconv_table_elt(table, ch >> 8, iconv_ccs_table_8bit);
	return sub_table == &(abstable->_8bit) ? UCS_CHAR_INVALID
	                                       : sub_table->data[ch & 0xFF];
}

static iconv_ccs_convert_t * const converters_h[] = {
	cvt_7bit_h, cvt_8bit_h, cvt_14bit_h, cvt_16bit_h
};


/* Functions for network byte order tables */

static ucs2_t
cvt_7bit_n(const void *table, ucs2_t ch)
{
	return ch & 0x80 ? UCS_CHAR_INVALID : ntohs(abstable->_7bit.data[ch]);
}

static ucs2_t
cvt_8bit_n(const void *table, ucs2_t ch)
{
	return ntohs(abstable->_8bit.data[ch]);
}

static ucs2_t
cvt_14bit_n(const void *table, ucs2_t ch)
{
	const iconv_ccs_table_7bit *sub_table;

	if (ch & 0x8080)
		return UCS_CHAR_INVALID;
	sub_table =  iconv_table_elt_n(table, ch >> 8, iconv_ccs_table_7bit);
	return sub_table == &(abstable->_7bit) ? UCS_CHAR_INVALID
	                                       : ntohs(sub_table->data[ch & 0x7F]);
}

static ucs2_t
cvt_16bit_n(const void *table, ucs2_t ch)
{
	const iconv_ccs_table_8bit *sub_table =
		iconv_table_elt_n(table, ch >> 8, iconv_ccs_table_8bit);
	return sub_table == &(abstable->_8bit) ? UCS_CHAR_INVALID
	                                       : ntohs(sub_table->data[ch & 0xFF]);
}

static iconv_ccs_convert_t * const converters_n[] = {
	cvt_7bit_n, cvt_8bit_n, cvt_14bit_n, cvt_16bit_n
};

#undef abstable


/* Generic coded character set initialisation function */

static int
ccs_init(struct iconv_ccs *ccs, const iconv_ccs_convtable *table)
{
	if (strncmp(table->label, ICONV_TBL_LABEL, ICONV_TBL_LABEL_SIZE))
		return EINVAL;
	if (ICONV_TBL_VERSION(table) > 3)
		return EINVAL;
	ccs->nbits = ICONV_TBL_NBITS(table);
	if (ICONV_TBL_BYTE_ORDER(table)) {
		ccs->from_ucs = iconv_table_elt(table->tables,
		                                   iconv_ccs_from_ucs,
		                                   const iconv_ccs_convtable);
		ccs->to_ucs = iconv_table_elt(table->tables,
		                                 iconv_ccs_to_ucs,
		                                 const iconv_ccs_convtable);
		ccs->convert_from_ucs = cvt_16bit_h;
		ccs->convert_to_ucs = converters_h[ICONV_TBL_VERSION(table)];
	} else {
		ccs->from_ucs = iconv_table_elt_n(table->tables,
		                                     iconv_ccs_from_ucs,
		                                     const iconv_ccs_convtable);
		ccs->to_ucs = iconv_table_elt_n(table->tables,
		                                   iconv_ccs_to_ucs,
		                                   const iconv_ccs_convtable);
		ccs->convert_from_ucs = cvt_16bit_n;
		ccs->convert_to_ucs = converters_n[ICONV_TBL_VERSION(table)];
	}
	return 0;
}


static int
close_builtin(struct iconv_ccs *desc)
{
	return 0;
}

static int
iconv_ccs_init_builtin(struct iconv_ccs *ccs, const char *name)
{
	const iconv_builtin_table *ccs_ptr;
	for (ccs_ptr = iconv_builtin_ccs; ccs_ptr->key != NULL; ccs_ptr ++) {
		if (strcmp(ccs_ptr->key, name) == 0) {
			int res = ccs_init(ccs, (const iconv_ccs_convtable *)
			                        (ccs_ptr->value));
			if (res == 0)
				ccs->close = close_builtin;
			return res;
		}
	}
	return EINVAL;
}


/* External CCS initialisation */

struct external_extra {
	const iconv_ccs_convtable *ptr;
	off_t size;
};

static int
close_external(struct iconv_ccs *desc)
{
	int res = 0;
	struct external_extra *e = desc->extra;

	if (e != NULL) {
		res = iconv_munmap(e->ptr, e->size);
		free(e);
	}
	return res;
}

static int
iconv_ccs_init_external(struct iconv_ccs *ccs, const char *name)
{
	char file[PATH_MAX], buffer[PATH_MAX];
	off_t size;
	const iconv_ccs_convtable *ccs_ptr;
	struct external_extra *extra;
	char *pathlist = getenv("ICONV_TABLE_PATH");

	if (pathlist == NULL)
		pathlist = ICONV_TABLE_PATH;
	snprintf(file, sizeof(file), "%s.cct", name);
	size = iconv_filesize(pathlist, file, buffer);
	if (size <= 0)
		return EINVAL;
	ccs_ptr = (const iconv_ccs_convtable *)iconv_mmap(buffer, size);
	if (ccs_ptr == NULL)
		return errno;
	extra = malloc(sizeof(struct external_extra *));
	if (extra != NULL) {
		int result = ccs_init(ccs, ccs_ptr);
		if (result == 0) {
			extra->ptr = ccs_ptr;
			extra->size = size;
			ccs->extra = extra;
			ccs->close = close_external;
			return 0;
		}
		free(extra);
	}
	iconv_munmap(ccs_ptr, size);
	return errno;
}

int
iconv_ccs_init(struct iconv_ccs *ccs, const char *name)
{
	int res = iconv_ccs_init_builtin(ccs, name);
	if (res)
		res = iconv_ccs_init_external(ccs, name);
	if (res)
		errno = res;
	return res;
}
