/*-
 * Copyright (c) 1999, 2000
 *	Konstantin Chuguev.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Konstantin Chuguev
 *	and its contributors.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	iconv (Charset Conversion Library) v2.0
 */

#include <dlfcn.h>	/* dlclose, dlopen, RTLD_NOW */
#include <err.h>	/* warnx */
#include <errno.h>	/* errno */
#include <limits.h>	/* PATH_MAX */
#include <stdio.h>	/* snprintf */
#include <stdlib.h>	/* free, malloc */
#include <unistd.h>	/* issetugid */

#define ICONV_INTERNAL
#include <iconv.h>	/* iconv_ccs_desc, iconv_ccs */

static int close_builtin(void *ces)
{
	return ((struct iconv_ces *)ces)->desc->close(
	        ((struct iconv_ces *)ces)->data);
}

static int
ces_instance_init(struct iconv_ces *ces, const char *name,
                  const struct iconv_ces_desc *desc)
{
	int res = desc->init(&(ces->data), name, desc->data);

	if (res)
		return errno = res;
	ces->desc = desc;
	ces->close = close_builtin;
	return 0;
}

static int
ces_init_builtin(struct iconv_ces *ces, const char *name)
{
	const iconv_builtin_table *ces_ptr;
	for (ces_ptr = iconv_builtin_ces; ces_ptr->key != NULL; ces_ptr ++)
		if (strcmp(ces_ptr->key, name) == 0)
			return ces_instance_init(ces, name,
			       (const struct iconv_ces_desc *)(ces_ptr->value));
	return errno = EINVAL;
}

#ifdef PIC

static int close_external(void *ces)
{
	void *handle = ((struct iconv_ces *)ces)->handle;
	int res = close_builtin(ces);
	return dlclose(handle) || res;
}

static const struct iconv_ces_desc *
iconv_load_module(const char *path, const char *name, void **handle)
{
	const struct iconv_ces_desc *res;
	char symbol[64], *ptr;
	void *h = dlopen(path, RTLD_NOW);

	if (handle == NULL) {
		warnx("cannot dlopen file %s: %s", path, dlerror());
		errno = EINVAL;
		return NULL;
	}
	snprintf(symbol, sizeof(symbol), "iconv_ces_%s", name);
	for (ptr = symbol; *ptr; ptr++)
		if (*ptr == '-')
			*ptr = '_';
	res = (struct iconv_ces_desc *)dlsym(h, symbol);
	if (res)
		*handle = h;
	else {
		warnx("invalid file %s: no external symbol %s", path, symbol);
		errno = EINVAL;
		dlclose(h);
	}
	return res;
}

static int
ces_init_external(struct iconv_ces *ces, const char *name)
{
	char file[PATH_MAX], buffer[PATH_MAX];
	void *handle;
	const struct iconv_ces_desc *desc;
	int res;
	char *pathlist;

	if (issetugid() || !(pathlist = getenv("ICONV_MODULE_PATH")))
		pathlist = ICONV_MODULE_PATH;
	snprintf(file, sizeof(file), "%s.so", name);
	if (iconv_filesize(pathlist, file, buffer) <= 0)
		return EINVAL;
	desc = iconv_load_module(buffer, name, &handle);
	if (desc == NULL)
		return errno;
	res = ces_instance_init(ces, name, desc);
	if (res)
		dlclose(handle);
	else {
		ces->handle = handle;
		ces->close = close_external;
	}
	return res;
}

#endif

int
iconv_ces_init(struct iconv_ces *ces, const char *name)
{
	return ces_init_builtin(ces, name)
	       && ces_instance_init(ces, name, &iconv_ces_table_driven)
#ifdef PIC
	       && ces_init_external(ces, name)
#endif
	       ;
}

/*ARGSUSED*/
int
iconv_ces_init_state(void **res, const char *name, const void *data)
{
	return iconv_malloc(sizeof(int), res);
}

int
iconv_ces_close_state(void *data)
{
	if (data != NULL)
		free(data);
	return 0;
}

void
iconv_ces_reset_state(void *data)
{
	*(int *)data = 0;
}

int
iconv_ces_init_null(void **res, const char *name, const void *data)
{
	return 0;
}

int
iconv_ces_close_null(void *data)
{
	return 0;
}

void
iconv_ces_reset_null(void *data)
{
}
