/*-
 * Copyright (c) 1999,2000
 *	Konstantin Chuguev.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Konstantin Chuguev
 *	and its contributors.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	iconv (Charset Conversion Library) v2.0
 */

#include <errno.h>	/* errno */
#include <stdlib.h>	/* free, malloc */
#include <string.h>	/* memcmp, memcpy */

#define ICONV_INTERNAL
#include <iconv.h>


typedef struct {
	int nccs;
	struct iconv_ccs ccs[1];
} iconv_ces_euc_state;

int
iconv_euc_init(void **data, const void *desc_data, size_t num)
{
	size_t stsz = sizeof(iconv_ces_euc_state) +
	              sizeof(struct iconv_ccs) * (num - 1);
	int i;
	iconv_ces_euc_state *state = (iconv_ces_euc_state *)malloc(stsz);

	if (state == NULL)
		return errno;
	for (i = 0; i < num; i++) {
		int res = iconv_ccs_init(&(state->ccs[i]),
		                         ((const iconv_ces_euc_ccs *)
					  desc_data)[i].name);
		if (res) {
			while (--i >= 0)
				state->ccs[i].close(&(state->ccs[i]));
			free(state);
			return res;
		}
	}
	state->nccs = num;
	(iconv_ces_euc_state *)*data = state;
	return 0;
}

int
iconv_euc_close(void *data)
{
#define state ((iconv_ces_euc_state *)data)
	int i, res = 0;

	for (i = 0; i < state->nccs; i++)
		res = state->ccs[i].close(&(state->ccs[i])) || res;
	free(data);
	return res;
#undef state
}

#define is_7_14bit(data) ((data)->nbits & 7)
#define is_7bit(data) ((data)->nbits & 1)

ssize_t
iconv_euc_convert_from_ucs(struct iconv_ces *ces, ucs_t in,
                           unsigned char **outbuf, size_t *outbytesleft)
{
	iconv_ces_euc_state *euc_state;
	size_t bytes;
	int i;

	if (in == UCS_CHAR_NONE)
		return 1;	/* No state reinitialization for table charsets */
	if (iconv_char32bit(in))
		return -1;
	euc_state = (iconv_ces_euc_state *)(ces->data);
	for (i = 0; i < euc_state->nccs; i++) {
		const iconv_ces_euc_ccs *ccsattr;
		const struct iconv_ccs *ccs = &(euc_state->ccs[i]);
		ucs_t res = ICONV_CCS_CONVERT_FROM_UCS(ccs, in);

		if (res == UCS_CHAR_INVALID)
			continue;
		ccsattr = &(((const iconv_ces_euc_ccs *)(ces->desc->data))[i]);
		if (i) {
			if (is_7_14bit(ccs))
				res |= is_7bit(ccs) ? 0x80 : 0x8080;
			else if (!(res & 0x8080))
				continue;
		} else if (res & 0x8080)
			continue;
		bytes = (res & 0xFF00 ? 2 : 1) + ccsattr->prefixlen;
		if (*outbytesleft < bytes)
			return 0;	/* No space in the output buffer */
		if (ccsattr->prefixlen) {
			memcpy(*outbuf, ccsattr->prefix, ccsattr->prefixlen);
			(*outbuf) += ccsattr->prefixlen;
		}
		if (res & 0xFF00)
			*(*outbuf)++ = (unsigned char)(res >> 8);
		*(*outbuf)++ = (unsigned char)res;
		*outbytesleft -= bytes;
		return 1;
	}
	return -1;	/* No character in output charset */
}

static ucs_t
cvt2ucs(struct iconv_ccs *ccs, const unsigned char *inbuf,
	size_t inbytesleft, int hi_plane, const unsigned char **bufptr)
{
	size_t bytes = ccs->nbits > 8 ? 2 : 1;
	ucs_t ch = *(const unsigned char *)inbuf++;

	if (inbytesleft < bytes)
		return UCS_CHAR_NONE;	/* Not enough bytes in the input buffer */
	if (bytes == 2)
		ch = (ch << 8) | *(const unsigned char *)inbuf++;
	*bufptr = inbuf;
	if (hi_plane) {
		if (!(ch & 0x8080))
			return UCS_CHAR_INVALID;
		if (is_7_14bit(ccs))
			ch &= 0x7F7F;
	} else if (ch & 0x8080)
		return UCS_CHAR_INVALID;
	return ICONV_CCS_CONVERT_TO_UCS(ccs, ch);
}

ucs_t
iconv_euc_convert_to_ucs(struct iconv_ces *ces, const unsigned char **inbuf,
                         size_t *inbytesleft)
{
	iconv_ces_euc_state *euc_state =
		(iconv_ces_euc_state *)(ces->data);
	ucs_t res = UCS_CHAR_INVALID;
	const unsigned char *ptr;
        int i;

	if (**inbuf & 0x80) {
		for (i = 1; i < euc_state->nccs; i++) {
			const iconv_ces_euc_ccs *ccsattr =
				 &(((const iconv_ces_euc_ccs *)
			           (ces->desc->data))[i]);
			if (ccsattr->prefixlen + 1 > *inbytesleft)
				return UCS_CHAR_NONE;
			if (ccsattr->prefixlen &&
			    memcmp(*inbuf, ccsattr->prefix, ccsattr->prefixlen))
				continue;
			res = cvt2ucs(&(euc_state->ccs[i]),
			              *inbuf + ccsattr->prefixlen,
			              *inbytesleft - ccsattr->prefixlen,
				      1, &ptr);
			if (res != UCS_CHAR_INVALID)
				break;
		}
		if (res == UCS_CHAR_INVALID)
			ptr = *inbuf + 1;
	} else
		res = cvt2ucs(euc_state->ccs, *inbuf, *inbytesleft, 0, &ptr);
	if (res == UCS_CHAR_NONE)
		return res;	/* Not enough bytes in the input buffer */
	*inbytesleft -= ptr - *inbuf;
	*inbuf = ptr;
	return res;
}
