/*-
 * Copyright (c) 1999,2000
 *	Konstantin Chuguev.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Konstantin Chuguev
 *	and its contributors.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	iconv (Charset Conversion Library) v2.0
 */

#include <errno.h>	/* errno */
#include <stdlib.h>	/* free, malloc */
#include <string.h>	/* bcopy */

#define ICONV_INTERNAL
#include <iconv.h>

static iconv_converter *
converter_init(iconv_conv_t conv_func, iconv_close_t close_func, size_t extra)
{
	iconv_converter *res = malloc(sizeof(iconv_converter) + extra);
	if (res) {
		res->convert = conv_func;
		res->close = close_func;
	}
	return res;
}

typedef struct {
	struct iconv_ces from;
	struct iconv_ces to;
	ucs_t  missing;
} unicode_converter;

static int
unicode_close(void *data)
{
	int res;
	unicode_converter *uc = (unicode_converter *)data;

	res = ICONV_CES_CLOSE(&(uc->from));
	res = ICONV_CES_CLOSE(&(uc->to)) || res;
	return res;
}

static size_t
unicode_conv(void *data, const unsigned char **inbuf, size_t *inbytesleft,
             unsigned char **outbuf, size_t *outbytesleft)
{
	size_t res = 0;
	unicode_converter *uc = (unicode_converter *)data;

	if (inbuf == NULL || *inbuf == NULL) {
		if (ICONV_CES_CONVERT_FROM_UCS(&(uc->to), UCS_CHAR_NONE,
		                               outbuf, outbytesleft) <= 0) {
			errno = E2BIG;
			return (size_t)(-1);
		}
		ICONV_CES_RESET(&(uc->from));
		ICONV_CES_RESET(&(uc->to));
		return res;
	}
	if (inbytesleft == NULL || *inbytesleft == 0)
		return 0;
	while (*inbytesleft > 0 && *outbytesleft > 0) {
		ssize_t size;
		const unsigned char *ptr = *inbuf;
		ucs_t ch = ICONV_CES_CONVERT_TO_UCS(&(uc->from), inbuf,
		                                    inbytesleft);
		if (ch == UCS_CHAR_NONE) {
			/* Incomplete character in input buffer */
			errno = EINVAL;
			return res;
		}
		if (ch == UCS_CHAR_INVALID) {
			/* Invalid character in source buffer */
			*inbytesleft += *inbuf - ptr;
			*inbuf = ptr;
			errno = EILSEQ;
			return res;
		}
		size = ICONV_CES_CONVERT_FROM_UCS(&(uc->to), ch,
		                                  outbuf, outbytesleft);
		if (size < 0) {
			/* No equivalent in destination charset */
			size = ICONV_CES_CONVERT_FROM_UCS(&(uc->to),
			                                  uc->missing,
			                                  outbuf, outbytesleft);
			if (size)
				res ++;
		}
		if (!size) {
			/* Not enough space in output buffer */
			*inbytesleft += *inbuf - ptr;
			*inbuf = ptr; 
			errno = E2BIG;
			return res;
		}
	}
	return res;
}

iconv_converter *
iconv_unicode_conv_init(const char *to, const char *from)
{
	unicode_converter *uc;
	iconv_converter *ic = converter_init(unicode_conv, unicode_close,
	                                     sizeof(unicode_converter));

	if (ic == NULL)
		return NULL;
	uc = (unicode_converter *)(ic + 1);
	if (!iconv_ces_init(&(uc->from), from)) {
		if(!iconv_ces_init(&(uc->to), to))
		{
			uc->missing = '_';
			return ic;
		}
		ICONV_CES_CLOSE(&(uc->from));
	}
	free(ic);
	return NULL;
}

static int
null_close(void *data)
{
	return 0;
}

static size_t
null_conv(void *data, const unsigned char **inbuf, size_t *inbytesleft,
          unsigned char **outbuf, size_t *outbytesleft)
{
	if (inbuf && *inbuf && inbytesleft && *inbytesleft > 0 && outbuf
			&& *outbuf && outbytesleft && *outbytesleft > 0) {
		size_t len = *inbytesleft < *outbytesleft ? *inbytesleft
							  : *outbytesleft;
		bcopy(*inbuf, *outbuf, len);
		*inbuf += len;
		*inbytesleft -= len;
		*outbuf += len;
		*outbytesleft -= len;
	}
	return 0;
}

iconv_converter *
iconv_null_conv_init(const char *to, const char *from)
{
	return converter_init(null_conv, null_close, 0);
}
