/*-
 * Copyright (c) 1999,2000
 *	Konstantin Chuguev.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Konstantin Chuguev
 *	and its contributors.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	iconv (Charset Conversion Library) v2.0
 */

#include <errno.h>	/* errno */
#include <fcntl.h>	/* O_RDONLY */
#include <limits.h>	/* PATH_MAX */
#include <stdio.h>	/* snprintf */
#include <stdlib.h>	/* free, malloc */
#include <string.h>	/* bzero, strdup, strsep */
#include <unistd.h>	/* close, open */
#include <sys/mman.h>	/* MAP_FAILED, PROT_READ, mmap, munmap */
#include <sys/stat.h>	/* stat */

#define ICONV_INTERNAL
#include <iconv.h>

int
iconv_malloc(size_t size, void **pp)
{
	void *p = malloc(size);

	if (p == NULL)
		return errno;
	bzero(p, size);
	*pp = p;
	return 0;
}

off_t
iconv_filesize(const char *pathlist, const char *filename, char *result)
{
	off_t size = -1;
	char *path, *plist = strdup(pathlist);

	if (plist == NULL)
		return -1;
	while ((path = strsep(&plist, ":")) != NULL ) {
		struct stat st;

		snprintf(result, PATH_MAX, "%s/%s", path, filename);
		if (stat(result, &st) < 0 || !S_ISREG(st.st_mode))
			continue;
		size = st.st_size;
		break;
	}
	free(plist);
	return size;
}

const void *
iconv_mmap(const char *filename, size_t size)
{
	const void *ptr;
	int fd = open(filename, O_RDONLY);

	if (fd < 0)
		return NULL;
	ptr = (const void *)mmap(NULL, size, PROT_READ, 0, fd, 0);
	close(fd);
	if (ptr == MAP_FAILED)
		return NULL;
	return ptr;
}

int
iconv_munmap(const void *addr, size_t size)
{
	return munmap((void *)addr, size);
}

#ifdef ICONV_DEBUG
#include <stdarg.h>	/* va_end, va_list, va_start */

void iconv_debug(const char *file, int line, const char *function,
                 const char *format, ...)
{
    static char buffer[256];
    va_list ap;
    va_start(ap, format);
    vsnprintf(buffer, sizeof(buffer), format, ap);
    fprintf(stderr, "%-14s[%3d]->%-14s: %s\n", file, line, function, buffer);
    fflush(stderr);
    va_end(ap);
}
#endif
