/* Vars data file utility */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <vars.h>

enum {
    O_NONE, O_FREEZE, O_WRITE
};

int
main(int argc, char *argv[])
{
    int output = O_NONE, errs = 0, binary = 0, numargs;
    vscalar *elt;
    vhash *opts;
    vlist *args;
    char *file;
    void *ptr;
    FILE *fp;

    /* Initialise */
    v_debug_env();

    /* Call extra declaration functions */
#ifdef EXTRA_DECLS
    EXTRA_DECLS;
#endif

    /* Set options */
    v_option('b', "binary", V_OPT_FLAG, NULL,
             "Assume input is binary");

    v_option('p', "print", V_OPT_FLAG, NULL,
             "Print ASCII data");

    v_option('w', "write", V_OPT_FLAG, NULL,
             "Write binary data");

    v_option('o', "output", V_OPT_ARG, "file",
             "Write output to specified file");

    v_option('h', "help", V_OPT_FLAG, NULL,
             "This help message");

    /* Parse arguments */
    if ((opts = vh_getopt(argc, argv)) == NULL)
        v_die("Type `%s -help' for help", argv[0]);

    if (vh_exists(opts, "help")) {
        v_usage("Usage: %s [options] [files]", argv[0]);
        return 0;
    }

    if (vh_exists(opts, "output")) {
        file = vh_sgetref(opts, "output");
        if ((fp = freopen(file, "w", stdout)) == NULL)
            v_die("%s: can't open %s for writing: %s",
                  argv[0], file, strerror(errno));
    }

    if (vh_exists(opts, "binary"))
        binary = 1;

    if (vh_exists(opts, "write"))
        output = O_WRITE;

    if (vh_exists(opts, "print"))
        output = O_FREEZE;

    /* Decompress if required */
    v_open_with("\\.gz$", "r", "gzip -qcd %s");
    v_open_with("\\.Z$",  "r", "uncompress -c %s");

    /* Process arguments */
    args = vh_pget(opts, "ARGS");
    if ((numargs = vl_length(args)) == 0) {
        vl_spush(args, "-");
        numargs++;
    }

    vl_foreach(elt, args) {
        file = vs_sgetref(elt);

        /* Open file */
        if (V_STREQ(file, "-")) {
            fp = stdin;
            file = "stdin";
        } else if ((fp = v_open(file, "r")) == NULL) {
            v_die("%s: can't open %s: %s",
                  argv[0], file, strerror(errno));
        }

        /* Read contents */
        if (binary)
            ptr = v_fread(fp);
        else
            ptr = v_thaw(fp);

        /* Close it */
        if (fp != stdin)
            v_close(fp);

        /* Check for errors */
        if (ptr == NULL) {
            if (binary)
                v_warn("%s: not Vars binary data", file);
            else
                v_warn("%s, %s", file, v_thaw_error());
            errs++;
            continue;
        }

        /* Write output */
        switch (output) {
        case O_FREEZE:
            if (numargs > 1)
                printf("%s: ", file);
            v_freeze(ptr, stdout);
            break;
        case O_WRITE:
            v_fwrite(ptr, stdout);
            break;
        default:
            printf("%s: %s OK\n", file, binary ? "data" : "syntax");
            break;
        }

        /* Clean up */
        v_destroy(ptr);
    }

    return errs != 0;
}
