/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @defgroup array Arrays
  @ingroup types

  An array is an object which contains an ordered list of value-scalar
  pairs.  The value is a real number.
*/

/*!
  @defgroup array_create Creating and destroying arrays
  @ingroup array

  These functions create and destroy arrays.
*/

/*!
  @defgroup array_modify Modifying arrays
  @ingroup array

  These functions are available for adding and removing value-scalar pairs
  from an array.  Note that entries can be added in any order -- when
  required, the pairs are sorted by increasing value.

  When a pair is added to an array, any existing pair with the same value
  is replaced by it.  The definition of 'same value' is any pair with a
  value which differs by less than the resolution of the array.  The
  default resolution is 1.0e-8, but can be changed via va_resolution().

  If you're storing pointers to other structures in array entries, you may
  want to deal with them when their array entries get overwritten (e.g.
  reclaim memory). The function va_remove_func() can do that.
*/

/*!
  @defgroup array_interp Array value interpolation
  @ingroup array

  These functions do linear interpolation to find entries.
*/

/*!
  @defgroup array_access Accessing array information
  @ingroup array

  These functions are available for getting miscellaneous information from
  arrays.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include "vars-config.h"
#include "vars-array.h"
#include "vars-macros.h"
#include "vars-memory.h"

#ifndef ARRAY_DEFAULT_SIZE
#define ARRAY_DEFAULT_SIZE 20
#endif

#define FLOAT_FORMAT "%.7g"

/* Various shortcut macros */
#define NP(a)       ((a)->numpoints)
#define INITP(a)    ((a)->initpoints)
#define MAXP(a)     ((a)->maxpoints)
#define MOD(a)      ((a)->modflag)
#define RES(a)      ((a)->resolution)

#define XVAL(a, n)  ((a)->points[n].x)
#define SVAL(a, n)  ((a)->points[n].s)
#define CODE(a, n)  ((a)->points[n].code)

#define XMIN(a)     XVAL(a, 0)
#define XMAX(a)     XVAL(a, NP(a) - 1)
#define EMPTY(a)    (NP(a) == 0)

#define ARRAY_END    -1

#define SORT(a)     qsort((a)->points, NP(a), sizeof(vpoint), va_sortpoints)

#define VA_UPDATE(A)	if (MOD(a)) va_update(a)

#define ADDP(a, x, s)                                                   \
        do {                                                            \
            if (NP(a) == MAXP(a) - 1) {                                 \
                MAXP(a) += INITP(a);                                    \
                a->points = V_REALLOC(a->points, vpoint, MAXP(a));      \
            }                                                           \
            XVAL(a, NP(a)) = x;                                         \
            SVAL(a, NP(a)) = s;                                         \
            CODE(a, NP(a)) = ++sortcode;                                \
            MOD(a) = 1;                                                 \
            NP(a)++;                                                    \
        } while (0)

/* A single point */
struct v_point {
    float x;            /* X value */
    vscalar *s;		/* Table value */
    int code;           /* Sort code */
};

typedef struct v_point vpoint;

/* Type definition */
struct v_array {
    struct v_header id;         /* Type marker */

    struct v_point *points;     /* Function points */
    unsigned int initpoints;    /* Initial point allocation */
    unsigned int maxpoints;     /* Max. points allocated */
    unsigned int numpoints;     /* No. of points used */
    float resolution;           /* Resolution of X values */
    int modflag;                /* Function modified? */

    int ipos;                   /* Iteration location */

    void (*remove)(vscalar *s); /* Scalar removal function */
};

/* Type variable */
vtype *varray_type = NULL;

/* Sort code values */
static int sortcode = 0;

/* X value resolution */
static float resolution;

/* Duplicate-point flag */
static int duplicate;

/* Internal functions */
static int va_search(varray *a, float x, int *entry, float *fraction);
static int va_sortpoints(const void *v1, const void *v2);
static void va_update(varray *a);

/*!
  @brief   Add a list of value-scalar pairs to an array.
  @ingroup array_modify
  @param   a Array.
  @param   x List of X values to add.
  @param   s List of scalars to add.
*/
void
va_add_list(varray *a, vlist *x, vlist *s)
{
    int i, num;

    VA_CHECK(a);
    VL_CHECK(x);
    VL_CHECK(s);

    num = V_MIN(vl_length(x), vl_length(s));

    for (i = 0; i < num; i++)
	ADDP(a, vl_dget(x, i), vs_copy(vl_get(s, i)));
}

/*!
  @brief   Signal break out of an each-loop.
  @ingroup array_access
  @param   a Array.
  @see     va_each, va_foreach

  Stop iterating over an array.
*/
void
va_break(varray *a)
{
    VA_CHECK(a);
    a->ipos = ARRAY_END;
}

/*!
  @brief   Return a copy of an array.
  @ingroup array_create
  @param   a Array.
  @return  A copy of the array.
*/
varray *
va_copy(varray *a)
{
    varray *copy;
    int i;

    VA_CHECK(a);

    copy = va_create_size(INITP(a));

    for (i = 0; i < NP(a); i++)
	ADDP(copy, XVAL(a, i), vs_copy(SVAL(a, i)));

    RES(copy) = RES(a);

    if (V_DEBUG(V_DBG_INFO))
	v_info("Copied %s to 0x%p", v_vinfo(a), copy);

    return copy;
}

/*!
  @brief   Create and return a new (empty) array.
  @ingroup array_create
  @return  New array.
*/
varray *
va_create(void)
{
    return va_create_size(ARRAY_DEFAULT_SIZE);
}

/*!
  @brief   Create a new (empty) array with the given initial size.
  @ingroup array_create
  @param   size Initial size of the array.
  @return  New array.
*/
varray *
va_create_size(unsigned int size)
{
    static vheader *id = NULL;
    varray *a;

    if (id == NULL) {
	va_declare();
	id = vt_header(varray_type);
    }

    a = V_ALLOC(varray, 1);
    a->id = *id;

    a->points = V_ALLOC(vpoint, size);
    a->initpoints = V_MAX(size, 1);
    a->maxpoints = V_MAX(size, 1);
    a->numpoints = 0;
    a->modflag = 0;
    a->resolution = 1.0e-4;
    a->remove = NULL;

    a->ipos = ARRAY_END;

    if (V_DEBUG(V_DBG_INFO))
	v_info("Created %s", v_vinfo(a));

    return a;
}

/* Declare array type */
vtype *
va_declare(void)
{
    if (varray_type == NULL) {
	varray_type = vt_create("ARRAY", "A");
	vt_copy_with(varray_type, (void *(*)()) va_copy);
	vt_read_with(varray_type, (void *(*)()) va_fread);
	vt_write_with(varray_type, va_fwrite);
	vt_freeze_with(varray_type, va_freeze);
	vt_thaw_with(varray_type, (void *(*)()) va_thaw);
	vt_print_with(varray_type, va_print);
	vt_destroy_with(varray_type, va_destroy);
	vt_traverse_with(varray_type, va_traverse);
    }

    return varray_type;
}

/*!
  @brief   Return whether an interpolation value is defined.
  @ingroup array_access
  @param   a Array.
  @param   x Value to check.
  @return  Yes or no.

  Return whether an array is defined at a given value (i.e. whether the
  value is between the minimum and maximum values). This is always false
  for an empty array.
*/
int
va_defined(varray *a, float x)
{
    VA_CHECK(a);
    VA_UPDATE(a);

    if (EMPTY(a))
	return 0;

    return (x >= XMIN(a) && x <= XMAX(a));
}

/*!
  @brief   Delete an array entry.
  @ingroup array_modify
  @param   a Array to modify.
  @param   num The index of the entry to delete.
  @return  Whether anything was deleted.
*/
int
va_delete(varray *a, int num)
{
    VA_CHECK(a);

    if (num < 0 || num >= NP(a))
	return 0;

    XVAL(a, num) = XVAL(a, NP(a) - 1);
    SVAL(a, num) = SVAL(a, NP(a) - 1);
    CODE(a, num) = CODE(a, NP(a) - 1);
    MOD(a) = 1;
    NP(a)--;

    return 1;
}

/*!
  @brief   Deallocate an array.
  @ingroup array_create
  @param   a Array to destroy.
*/
void
va_destroy(varray *a)
{
    int i;

    VA_CHECK(a);

    if (V_DEBUG(V_DBG_INFO))
	v_info("Destroyed %s", v_vinfo(a));

    for (i = 0; i < NP(a); i++)
        vs_destroy(SVAL(a, i));

    V_DEALLOC(a->points);
    V_DEALLOC(a);
}

/*!
  @brief   Iterate over an array, returning values.
  @ingroup array_access
  @param   a Array.
  @param   x Value (returned).
  @param   s Scalar (returned).
  @return  Whether there are more values.
  @see     va_break(), va_foreach()

  Iterate over the specified array, returning each value-scalar pair in
  order. Returns zero when at the end of the array (and the following call
  starts again). Note that if you finish iterating without getting to the
  end of the function (via break, for instance) then you must call
  va_break() to indicate that you've finished.
*/
int
va_each(varray *a, float *x, vscalar **s)
{
    VA_CHECK(a);
    VA_UPDATE(a);

    if (a->ipos == ARRAY_END) {
	if (NP(a) > 0)
	    a->ipos = 0;
    } else if (a->ipos == NP(a) - 1) {
	a->ipos = ARRAY_END;
    } else {
	a->ipos++;
    }

    if (a->ipos != ARRAY_END) {
	*x = XVAL(a, a->ipos);
	*s = SVAL(a, a->ipos);
	return 1;
    }

    return 0;
}

/* Check start of foreach loop */
void
va_each_start(varray *a)
{
    VA_CHECK(a);
    if (a->ipos != ARRAY_END)
        v_fatal("va_each_start(): iteration error");
}

/*!
  @brief   Empty an array of all points.
  @ingroup array_modify
  @param   a Array to empty.
*/
void
va_empty(varray *a)
{
    VA_CHECK(a);
    NP(a) = 0;
}

/*!
  @brief   Find a place in an array.
  @ingroup array_access
  @param   a Array.
  @param   x Value to find.
  @return  Entry index.
  @retval  -1 if not present (within array's resolution).
*/
int
va_find(varray *a, float x)
{
    int entry, extflag;
    float fraction;

    VA_CHECK(a);
    VA_UPDATE(a);

    if (EMPTY(a))
	return -1;

    /* Find nearest point */
    extflag = va_search(a, x, &entry, &fraction);
    if (!extflag && fraction > 0.5)
	entry++;

    /* Is it close enough? */
    x -= XVAL(a, entry);
    if (fabs(x) > RES(a))
	return -1;

    /* Found it */
    return entry;
}

/* Read array from a stream */
varray *
va_fread(FILE *fp)
{
    int num, size;
    vscalar *s;
    varray *a;
    float x;

    /* Read initial array size */
    if (!v_read_long(&size, fp))
	return NULL;

    /* Read no. of points */
    if (!v_read_long(&num, fp))
	return NULL;

    /* Read points */
    a = va_create_size(size);

    while (num-- > 0) {
	/* Read key */
	if (!v_read_float(&x, fp))
	    return NULL;
	/* Read value */
	if ((s = vs_fread(fp)) == NULL)
	    return NULL;

	ADDP(a, x, s);
    }

    if (V_DEBUG(V_DBG_IO))
	v_info("Read %s", v_vinfo(a));

    return a;
}

/* Freeze an array to a stream */
int
va_freeze(varray *a, FILE *fp)
{
    int i, first = 1;

    VA_CHECK(a);
    VA_UPDATE(a);

    v_freeze_start(fp);

    fputs("{\n", fp);
    v_push_indent();

    for (i = 0; i < NP(a); i++) {
	if (first)
	    first = 0;
	else
	    fputs(",\n", fp);

	v_indent(fp);
	fprintf(fp, FLOAT_FORMAT, XVAL(a, i));
	fputs(" = ", fp);
	if (!vs_freeze(SVAL(a, i), fp))
	    return 0;
    }

    fputc('\n', fp);
    v_pop_indent();
    v_indent(fp);
    fputc('}', fp);

    v_freeze_finish(fp);

    return 1;
}

/* Write array to a stream */
int
va_fwrite(varray *a, FILE *fp)
{
    int i;

    VA_CHECK(a);

    if (V_DEBUG(V_DBG_IO))
	v_info("Writing %s", v_vinfo(a));

    if (!v_write_long(a->initpoints, fp))
	return 0;

    if (!v_write_long(a->numpoints, fp))
	return 0;

    for (i = 0; i < a->numpoints; i++) {
	if (!v_write_float(XVAL(a, i), fp))
	    return 0;
	if (!vs_fwrite(SVAL(a, i), fp))
	    return 0;
    }

    return 1;
}

/*!
  @brief   Find nearest scalar to a value.
  @ingroup array_interp
  @param   a Array.
  @param   x Value to find.
  @return  Nearest scalar.
  @retval  NULL if array is empty.
*/
vscalar *
va_get(varray *a, float x)
{
    int extflag, entry;
    float fraction;

    VA_CHECK(a);
    VA_UPDATE(a);

    if (EMPTY(a))
	return NULL;

    extflag = va_search(a, x, &entry, &fraction);
    if (!extflag && fraction > 0.5)
        entry++;

    return SVAL(a, entry);
}

/*!
  @brief   Find point exactly.
  @ingroup array_interp
  @param   a Array.
  @param   x Value to find.
  @return  Scalar.
  @retval  NULL if no exact match.

  Like va_get(), but only an "exact" match (i.e. within the resolution of
  the array).
*/
vscalar *
va_get_exact(varray *a, float x)
{
    int extflag, entry;
    float fraction;

    VA_CHECK(a);
    VA_UPDATE(a);

    if (EMPTY(a))
	return NULL;

    extflag = va_search(a, x, &entry, &fraction);

    if (!extflag && fraction > 0.5)
	entry++;

    if (fabs(x - XVAL(a, entry)) > RES(a))
	return NULL;

    return SVAL(a, entry);
}

/*!
  @brief   Return a point in an array.
  @ingroup array_access
  @param   a Array.
  @param   n Entry number.
  @param   x Value (returned).
  @param   s Scalar (returned).
  @return  Whether \c n is valid.

  Get the Nth value-scalar pair in an array, counting from zero.
*/
int
va_get_point(varray *a, int n, float *x, vscalar **s)
{
    VA_CHECK(a);

    if (n < 0 || n >= NP(a))
	return 0;

    VA_UPDATE(a);
    *x = XVAL(a, n);
    *s = SVAL(a, n);

    return 1;
}

/*!
  @brief   Find the two nearest values for interpolation purposes.
  @ingroup array_interp
  @param   a Array.
  @param   x Value to find.
  @param   before Scalar before the value (returned).
  @param   after Scalar after the value (returned).
  @param   fraction Interpolation fraction (returned).
  @return  Number of scalars returned.
  @retval  0 if array is empty.

  Find the two nearest scalars to the specified value. If the value
  is outside the range of the array, only \c before is set.
*/
int
va_interp(varray *a, float x, vscalar **before,
	  vscalar **after, float *fraction)
{
    int extflag, entry;

    VA_CHECK(a);
    VA_UPDATE(a);

    if (EMPTY(a)) {
	*before = *after = NULL;
	return 0;
    }

    extflag = va_search(a, x, &entry, fraction);
    *before = SVAL(a, entry);

    if (extflag) {
	/* Extrapolating, only one point */
	*after = NULL;
	return 1;
    } else {
	/* Interpolating, two points */
	*after = SVAL(a, entry + 1);
	return 2;
    }
}


/*!
  @brief   Return maximum value of an array.
  @ingroup array_access
  @param   a Array.
  @return  Maximum value.
  @retval  0 if array is empty.
*/
float
va_maximum(varray *a)
{
    VA_CHECK(a);
    return (EMPTY(a) ? 0.0 : XMAX(a));
}

/*!
  @brief   Return minimum value of an array.
  @ingroup array_access
  @param   a Array.
  @return  Minimum value.
  @retval  0 if array is empty.
*/
float
va_minimum(varray *a)
{
    VA_CHECK(a);
    return (EMPTY(a) ? 0.0 : XMIN(a));
}

/*!
  @brief   Return no. of points in an array.
  @ingroup array_access
  @param   a Array.
  @return  No. of points.
*/
int
va_point_count(varray *a)
{
    VA_CHECK(a);
    VA_UPDATE(a);

    return NP(a);
}

/*!
  @brief   Return a list of array scalars.
  @ingroup array_access
  @param   a Array.
  @return  List of scalars.
*/
vlist *
va_points_s(varray *a)
{
    vlist *list;
    int i;

    VA_CHECK(a);
    VA_UPDATE(a);

    list = vl_create();

    for (i = 0; i < NP(a); i++)
	vl_push(list, vs_copy(SVAL(a, i)));

    return list;
}

/*!
  @brief   Return a list of array values.
  @ingroup array_access
  @param   a Array.
  @return  List of values.
*/
vlist *
va_points_x(varray *a)
{
    vlist *list;
    int i;

    VA_CHECK(a);
    VA_UPDATE(a);

    list = vl_create();

    for (i = 0; i < NP(a); i++)
	vl_dpush(list, XVAL(a, i));

    return list;
}

/* Print contents of an array */
void
va_print(varray *a, FILE *fp)
{
    int i;

    VA_CHECK(a);
    VA_UPDATE(a);

    v_print_start();
    v_push_indent();

    v_print_type(varray_type, a, fp);

    for (i = 0; i < NP(a); i++) {
	v_indent(fp);
	fprintf(fp, FLOAT_FORMAT " => ", XVAL(a, i));
	v_print(SVAL(a, i), fp);
    }

    v_pop_indent();
    v_print_finish();
}

/*!
  @brief   Set an array scalar removal function.
  @ingroup array_modify
  @param   a Array.
  @param   remove Removal function (or \c NULL ).

  Define a scalar removal function for the given array. The function is
  called whenever a scalar is about to be removed from the array, giving
  you the chance to do whatever you want with it. If a removal function is
  not defined, the default action is just to destroy the scalar. Note that
  destruction doesn't happen if there \e is a removal function defined.
*/
void
va_remove_func(varray *a, void (*remove)(vscalar *s))
{
    VA_CHECK(a);
    a->remove = remove;
}

/*!
  @brief   Set resolution of array values.
  @ingroup array_modify
  @param   a Array to set.
  @param   res The resolution.

  Set the resolution of an array to the given (non-negative) value.
*/
void
va_resolution(varray *a, float res)
{
    VA_CHECK(a);
    RES(a) = V_MAX(res, 0.0);
}

/* Search array for interpolation point */
static int
va_search(varray *a, float x, int *entry, float *fraction)
{
    int low = 0, mid, high = NP(a) - 1;
    float x_low, x_mid, x_high;

    if (high <= 0) {
	/* Not enough points */
	*entry = low;
	*fraction = 0.0;
	return 1;
    } else if (x <= (x_low = XVAL(a,low)) ) {
	/* Before first point */
	*entry = low;
	*fraction = 0.0;
	return 1;
    } else if (x >= (x_high = XVAL(a, high)) ) {
	/* After last point */
	*entry = high;
	*fraction = 0.0;
	return 1;
    }

    /* Find point with a binary search */
    while (high > low + 1) {
	mid = (low + high) / 2;
	x_mid = XVAL(a,mid);
	if (x >= x_mid) {
	    low = mid;
	    x_low = x_mid;
	} else {
	    high = mid;
	    x_high = x_mid;
	}
    }

    *entry = low;
    *fraction = (x - x_low) / (x_high - x_low);

    return 0;
}

/* Point sorting array */
static int
va_sortpoints(const void *v1, const void *v2)
{
    vpoint *p1 = (vpoint *) v1;
    vpoint *p2 = (vpoint *) v2;

    /* If point has been removed, move it to last */
    if (p1->code == 0)
	return 1;
    else if (p2->code == 0)
	return -1;

    /* Sort on X values */
    if (p1->x < p2->x - resolution)
	return -1;
    else if (p1->x > p2->x + resolution)
	return 1;

    /* If the same, use sort code */
    duplicate = 1;
    if (p1->code > p2->code)
	return -1;
    else if (p1->code < p2->code)
	return 1;

    return 0;
}

/*!
  @brief   Add a value-scalar pair to an array.
  @ingroup array_modify
  @param   a Array to add to.
  @param   x Value to add.
  @param   s Scalar to add (not copied).
*/
void
va_store(varray *a, float x, vscalar *s)
{
    VA_CHECK(a);
    VS_CHECK(s);
    ADDP(a, x, s);
}

/* Thaw an array from a stream */
varray *
va_thaw(FILE *fp)
{
    double x;
    int token;
    varray *a;
    vscalar *s;

    v_thaw_start();
    a = va_create();

    if (!v_thaw_follow(fp, '{', "open-brace"))
	goto fail;

    while (1) {
	if (v_thaw_peek(fp) == '}') {
	    v_thaw_token(fp);
	    break;
	}

	if (!v_thaw_double(fp, &x))
	    goto fail;

	if (!v_thaw_follow(fp, '=', "'='"))
	    goto fail;

	if ((s = vs_thaw(fp)) == NULL)
	    goto fail;

	va_store(a, (float) x, s);

	token = v_thaw_token(fp);
	if (token == '}') {
	    break;
	} else if (token != ',') {
	    v_thaw_expected("comma or close-brace");
	    goto fail;
	}
    }

    if (V_DEBUG(V_DBG_IO))
	v_info("Thawed %s", v_vinfo(a));

    v_thaw_finish();
    return a;

  fail:
    v_thaw_finish();
    v_destroy(a);
    return NULL;
}

/* Traverse an array */
int
va_traverse(varray *a, int (*func) (void *ptr))
{
    int i, val;
    vscalar *s;

    VA_CHECK(a);

    if ((val = func(a)) != 0)
	return val;

    if (v_traverse_seen(a))
	return 0;

    v_push_traverse(a);

    for (i = 0; i < NP(a); i++) {
	s = SVAL(a, i);
	if (vs_type(s) == V_POINTER) {
	    val = v_traverse(vs_pget(s), func);
	    if (val != 0) {
		v_pop_traverse();
		return val;
	    }
	}
    }

    v_pop_traverse();

    return 0;
}

/* Update array internals */
static void
va_update(varray *a)
{
    int i, ndup = 0;
    float xlast;

    /* Flag unmodified */
    MOD(a) = 0;

    /* Do nothing if empty, or not modified */
    if (EMPTY(a))
	return;

    /* Sort points into ascending X values */
    duplicate = 0;
    resolution = RES(a);
    SORT(a);

    /* Remove duplicates, if any */
    if (duplicate) {
	xlast = XVAL(a, 0);
	for (i = 1; i < NP(a); i++) {
	    if (XVAL(a, i) - xlast <= resolution) {
		CODE(a, i) = 0;

                if (a->remove != NULL)
                    a->remove(SVAL(a, i));
                else
                    vs_destroy(SVAL(a, i));

		ndup++;
	    }

	    xlast = XVAL(a, i);
	}

	SORT(a);
	NP(a) -= ndup;
    }
}
