/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @file
  @brief Array functions and macros.
  @ingroup array
*/

#ifndef VARS_ARRAY_H
#define VARS_ARRAY_H

#include <vars-list.h>

/*! @brief Abort if a pointer is not an array. */
#define VA_CHECK(ptr)           V_CHECK(ptr, varray_type, "ARRAY")

/*! @brief Check if a pointer is an array type. */
#define v_isarray(ptr)          (ptr != NULL && vt_type(ptr) == varray_type) 

/*! @brief Iterate over an array. */
#define va_foreach(x, s, a)     va_each_start(a); while (va_each(a, &x, &s))

/*! @brief Get array string value (pointer to internal buffer). */
#define va_sget(a, key)		vs_sget(va_get(a, key))

/*! @brief Get array string value by allocating a copy. */
#define va_sgetcopy(a, key)     vs_sgetcopy(va_get(a, key))

/*! @brief Get array string value by reference. */
#define va_sgetref(a, key)	vs_sgetref(va_get(a, key))

/*! @brief Set array string value. */
#define va_sstore(a, key, n)	va_store(a, key, vs_screate(n))

/*! @brief Get array float value. */
#define va_fget(a, key)		vs_fget(va_get(a, key))

/*! @brief Set array float value. */
#define va_fstore(a, key, n)	va_store(a, key, vs_fcreate(n))

/*! @brief Get array double value. */
#define va_dget(a, key)		vs_dget(va_get(a, key))

/*! @brief Set array double value. */
#define va_dstore(a, key, n)	va_store(a, key, vs_dcreate(n))

/*! @brief Get array integer value. */
#define va_iget(a, key)         vs_iget(va_get(a, key))

/*! @brief Set array integer value. */
#define va_istore(a, key, n)    va_store(a, key, vs_icreate(n))

/*! @brief Get array pointer value. */
#define va_pget(a, key)		vs_pget(va_get(a, key))

/*! @brief Set array pointer value. */
#define va_pstore(a, key, p)	va_store(a, key, vs_pcreate((void *) p))

/* Fix for va_copy() macros */
#ifdef va_copy
#undef va_copy
#endif

/*! @brief Array type. */
typedef struct v_array varray;

/*! @brief Array type variable. */
extern vtype *varray_type;

#ifdef __cplusplus
extern "C" {
#endif

extern void va_add_list(varray *a, vlist *x, vlist *s);
extern void va_break(varray *a);
extern varray *va_copy(varray *a);
extern varray *va_create(void);
extern varray *va_create_size(unsigned int size);
extern vtype *va_declare(void);
extern int va_defined(varray *a, float x);
extern int va_delete(varray *a, int num);
extern void va_destroy(varray *a);
extern int va_each(varray *a, float *x, vscalar **s);
extern void va_each_start(varray *a);
extern void va_empty(varray *a);
extern int va_find(varray *a, float x);
extern varray *va_fread(FILE *fp);
extern int va_freeze(varray *a, FILE *fp);
extern int va_fwrite(varray *a, FILE *fp);
extern vscalar *va_get(varray *a, float x);
extern vscalar *va_get_exact(varray *a, float x);
extern int va_get_point(varray *a, int n, float *x, vscalar **s);
extern int va_interp(varray *a, float x, vscalar **before, vscalar **after,
                     float *fraction);
extern float va_maximum(varray *a);
extern float va_minimum(varray *a);
extern int va_point_count(varray *a);
extern vlist *va_points_s(varray *a);
extern vlist *va_points_x(varray *a);
extern void va_print(varray *a, FILE *fp);
extern void va_remove_func(varray *a, void (*remove)(vscalar *s));
extern void va_resolution(varray *a, float res);
extern void va_store(varray *a, float x, vscalar *s);
extern varray *va_thaw(FILE *fp);
extern int va_traverse(varray *a, int (*func) (void *ptr));

#ifdef __cplusplus
}
#endif

#endif
