/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @defgroup copy Copying functions
  @ingroup methods
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "vars-config.h"
#include "vars-copy.h"
#include "vars-hash.h"

/* Deep-copy flag */
int v_deepcopy = 0;

/*!
  @brief   Deep-copy an object.
  @ingroup copy
  @param   ptr Vars object.
  @return  Copy of the object (or \c NULL if the pointer is \c NULL).

  A "deep copy" of an object is done by recursively copying all referenced
  pointers inside the object.  If any referenced pointer is not recognized
  as a Vars object, a fatal error occurs.  See \ref extend for more
  details.
*/
void *
v_copy(void *ptr)
{
    static vhash *seen = NULL;
    char *key, keybuf[20];
    static int depth = 0;
    void *cptr;
    vtype *t;

    /* Do nothing if NULL */
    if (ptr == NULL)
        return NULL;

    /* Get pointer type */
    if ((t = vt_type(ptr)) == NULL)
        v_fatal("v_copy(): unknown pointer type");

    if (t->copy == NULL)
        v_fatal("v_copy(): type %s has no copy function", t->name);

    /* Initialise if required */
    if (depth++ == 0) {
        v_deepcopy = 1;
        if (seen == NULL)
            seen = vh_create_table(1000, 0);
    }

    key = vh_pkey_buf(ptr, keybuf);
    if ((cptr = vh_pget(seen, key)) == NULL) {
        /* Copy the pointer */
        ptr = t->copy(ptr);

        /* Flag it as copied */
        vh_pstore(seen, key, ptr);
    } else {
        /* Seen before -- point to the copy */
        ptr = cptr;
    }

    if (ptr == NULL)
        v_fatal("v_copy(): copy function failed");

    /* Clean up if required */
    if (--depth == 0) {
        vh_empty(seen);
        v_deepcopy = 0;
    }

    return ptr;
}
