/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @defgroup destroy Destruction functions
  @ingroup methods
*/

/*!
  @defgroup destroy_rec Recursive destruction
  @ingroup destroy

  In all cases, whenever a function returns a newly-created Vars object to
  you, you are responsible for deallocating it.  There are several ways to
  do this.  Firstly, the standard destruction functions (vs_destroy(),
  vl_destroy(), etc.) can be used to destroy individual variables and their
  contents.

  However, if a variable contains a pointer to another variable, this other
  variable will not be touched.  For those times when you have created a
  complicated data structure (e.g. a hash containing pointers to other
  hashes or lists, and so on) and you want to destroy all of it, these
  functions are available.
*/

/*!
  @defgroup destroy_gc Garbage collection
  @ingroup destroy

  These functions are also available, which implement 'garbage collection'.
  These can be used to mark pointers for garbage collection at certain
  points, so that you don't have to explicitly destroy them when you've
  finished with them.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "vars-config.h"
#include "vars-hash.h"

/* Current garbage-collection hash */
static vhash *gc_hash = NULL; 

/* List of garbage-collection pointer hashes */
static vlist *gc_stack = NULL; 

/* List of pointers to destroy */
static vlist *pointers = NULL; 

/* Pointer destruction functions */
/*@-exportheadervar@*/ 
vhash *free_funcs = NULL; 
/*@=exportheadervar@*/ 

/* Default destruction function */
static void (*freefunc)(void *p) = NULL;

/* Internal functions */
static int collect_pointers(void *ptr);

/* Scribble buffer */
static char buf[BUFSIZ];

/* Pointer collection function */
static int
collect_pointers(void *ptr)
{
    if (!v_traverse_seen(ptr))
        vl_ppush(pointers, ptr);

    return 0;
}

/*!
  @brief   Recursively destroy a data structure.
  @ingroup destroy_rec
  @param   ptr Pointer to Vars structure.

  Given an anonymous pointer to data, recursively destroy it and all its
  pointer references.  This is done by firstly traversing the pointer tree
  using v_traverse() and collecting a list of pointers which require
  destruction.  Then each pointer is destroyed depending on its type.  If
  the pointer type is unknown, because you have explicitly added it
  yourself, then by default nothing is done.
*/
void
v_destroy(void *ptr)
{
    static int destroying = 0;
    void (*func)(void *ptr);
    vscalar *elt;
    vtype *type;
    char *key;
    void *p;

    /* Check for recursive calling */
    if (destroying++)
        return;

    /* Traverse data and collect pointers */
    pointers = vl_create();
    v_traverse(ptr, collect_pointers);

    /* Destroy each one */
    vl_foreach(elt, pointers) {
        p = vs_pget(elt);
        type = vt_type(p);

        if (type != NULL && type->destroy != NULL) {
            (*type->destroy)(p);
        } else {
            key = vh_pkey_buf(p, buf);

            /*@-type@*/ 
            if (free_funcs != NULL &&
                (func = vh_pget(free_funcs, key)) != NULL) {
                /* Use explicit destruction function */
                func(p);
                vh_delete(free_funcs, key);
            } else if (freefunc != NULL) {
                /* Use default function */
                freefunc(p);
            }
            /*@=type@*/ 

            break;
        }
    }

    /* Clean up */
    vl_destroy(pointers);
    destroying = 0;
}

/*!
  @brief   Set a default destruction function.
  @ingroup destroy_rec
  @param   func Function (or \c NULL to unset it).

  Set a default destruction function for any anonymous pointers that don't
  have an explicit destruction function associated.  Using a \c NULL
  argument will unset this default.  Note that if this is set, then no
  unknown pointer will be left untouched by v_destroy().
*/
void
v_destroy_default(void (*func)(void *ptr))
{
    freefunc = func;
}

/*!
  @brief   Set a destruction function for a pointer.
  @ingroup destroy_rec
  @param   ptr Pointer to Vars structure.
  @param   func Function (or \c NULL to unset it).

  Given an anonymous pointer to data, associate with it a function that can
  be used to destroy it.  Then, when v_destroy() sees it, it will use that
  function.  Note that when the pointer is destroyed via this function, the
  association is lost.  Also note that this function will have no effect on
  pointers to standard Vars variables.
*/
void
v_destroy_with(void *ptr, void (*func)(void *ptr))
{
    if (free_funcs == NULL) {
        free_funcs = vh_create_noreuse();
        v_cleanup(free_funcs);
    }

    vh_pstore(free_funcs, vh_pkey_buf(ptr, buf), func);
}

/*!
  @brief   Flag a pointer for garbage collection.
  @ingroup destroy_gc
  @param   ptr Pointer to Vars structure.
  @return  The argument.

  Mark the given pointer to be garbage collected by the current
  garbage collector.
*/
void *
v_gc(void *ptr)
{
    if (gc_stack == NULL || gc_hash == NULL)
        v_fatal("v_gc(): no gc on stack");

    vh_pstore(gc_hash, vh_pkey_buf(ptr, buf), ptr);
    return ptr;
}

/*!
  @brief   Do garbage collection.
  @ingroup destroy_gc

  Pop the current garbage collector off the stack, deallocating all the
  pointers it contains using v_destroy().
*/
void
v_pop_gc(void)
{
    if (gc_stack == NULL || gc_hash == NULL)
        v_fatal("v_pop_gc(): unmatched pop");

    v_destroy(gc_hash);
    gc_hash = vl_ppop(gc_stack);
}

/*!
  @brief   Push a new garbage-collection hash onto the stack.
  @ingroup destroy_gc

  Push a new garbage collector onto the stack and make it the current one.
*/
void
v_push_gc(void)
{
    if (gc_stack == NULL) {
        gc_stack = vl_create();
        v_cleanup(gc_stack);
    }

    if (gc_hash != NULL)
        vl_ppush(gc_stack, gc_hash);

    gc_hash = vh_create();
}
