/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @defgroup freeze Freeze/thaw functions
  @ingroup methods

  These functions perform text I/O on objects.  See @ref write for more
  details.
*/

/*!
  @defgroup freeze_high High-level functions
  @ingroup freeze

  These functions are for high-level use.
*/

/*!
  @defgroup freeze_low Low-level functions
  @ingroup freeze

  These functions are for use when defining new object types (see @ref
  extend).
*/

%{
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <errno.h>
#include <math.h>

#include "vars-config.h"
#include "vars-buffer.h"
#include "vars-hash.h"
#include "vars-macros.h"
#include "vars-system.h"
#include "vars-utils.h"

#define varswrap() 1

/* Current token values */
int v_thaw_ivalue = 0;
double v_thaw_dvalue = 0.0;
char *v_thaw_svalue = "";
void *v_thaw_pvalue = NULL;

static char buf[BUFSIZ];        /* Scribble buffer */
static char *filename = NULL;   /* Current thaw filename */
static int ft_tag = 0;          /* Freeze tag count */
static int ft_depth = 0;        /* Freeze/thaw recursion depth */
static int last_line_number;    /* Last line number */
static int line_number;         /* Current line number */
static int start_line = 1;      /* Starting line number */
static vbuffer *ebuf = NULL;    /* Error buffer */
static vbuffer *sbuf = NULL;    /* String value buffer */
static vhash *multiref = NULL;  /* Multiple reference flags */
static vhash *seen = NULL;      /* Data already processed */

static enum v_tokentype lookahead = 0; /* Lookahead token */

/* Internal functions */
static int flag_refs(void *ptr);

#ifndef DOXYGEN_IGNORE 
%}

SIGN            [-+]
DIGIT		[0-9]
DECIMAL         \.{DIGIT}*
EXP		[Ee]{SIGN}?{DIGIT}*
LETTER		[A-Za-z_]
ALNUM		{LETTER}|{DIGIT}

%%

\n {
    /* Newline */
    line_number++;
}

\#.* {
    /* Comment */
}

[ \t\f\r]+ {
    /* Whitespace */
}

{SIGN}?{DIGIT}*{DECIMAL}?{EXP} {
    /* Real number with exponent */
    if (sscanf(yytext, "%lg", &v_thaw_dvalue) == 1) {
        v_thaw_ivalue = (int) v_thaw_dvalue;
        return V_TOKEN_REAL;
    } else {
        return V_TOKEN_ERROR;
    }
}

{SIGN}?{DIGIT}*{DECIMAL} {
    /* Real number with decimal point and no exponent */
    if (sscanf(yytext, "%lg", &v_thaw_dvalue) == 1) {
        v_thaw_ivalue = (int) v_thaw_dvalue;
        return V_TOKEN_REAL;
    } else {
        return V_TOKEN_ERROR;
    }
}

{SIGN}?{DIGIT}+ {
    /* Integer */
    if (sscanf(yytext, "%d", &v_thaw_ivalue) == 1) {
        v_thaw_dvalue = (double) v_thaw_ivalue;
        return V_TOKEN_INT;
    } else {
        return V_TOKEN_ERROR;
    }
}

\" {
    /* String */
    int c;

    vb_init(sbuf);
    while ((c = input()) != EOF) {
        switch (c) {
        case '"':
            v_thaw_svalue = vb_get(sbuf);
            return V_TOKEN_STRING;
        case '\\':
            switch (c = input()) {
            case 'n':
                vb_putc(sbuf, '\n');
                break;
            case EOF:
                return V_TOKEN_ERROR;
            default:
                vb_putc(sbuf, c);
                break;
            }
            break;
        case '\n':
            line_number++;
            vb_putc(sbuf, c);
            break;
        default:
            vb_putc(sbuf, c);
            break;
        }
    }

    return V_TOKEN_ERROR;
}

{LETTER}{ALNUM}* {
    /* Identifier */
    vtype *type;

    v_thaw_svalue = yytext;

    if ((type = vt_find_name(yytext)) != NULL) {
        v_thaw_pvalue = type;
        return V_TOKEN_VTYPE;
    }

    if      (V_STREQ(yytext, V_TOKEN_REF_STRING))
        return V_TOKEN_REF;
    else if (V_STREQ(yytext, V_TOKEN_UNDEF_STRING))
        return V_TOKEN_UNDEF;
    else if (V_STREQ(yytext, V_TOKEN_NULL_STRING))
        return V_TOKEN_NULL;

    return V_TOKEN_ID;
}

. {
    /* Single character */
    return yytext[0];
}

%%

#endif /* DOXYGEN_IGNORE */

/* Flag multiple references to pointers */
static int
flag_refs(void *ptr)
{
    char *key;

    if (v_traverse_seen(ptr)) {
        key = vh_pkey_buf(ptr, buf);
        vh_istore(multiref, key, 1);
    }

    return 0;
}

/*!
  @brief   Freeze contents of a pointer.
  @ingroup freeze_high
  @param   ptr Pointer to Vars structure.
  @param   fp Stream to write.
  @return  Whether successful.
*/
int
v_freeze(void *ptr, FILE *fp)
{
    static vhash *stack = NULL;
    static int depth = 0;
    int id, code = 1;
    char *key;
    vtype *t;

    if (seen == NULL) {
        seen = vh_create_table(1000, 0);
        v_cleanup(seen);
    }

    if (stack == NULL) {
        stack = vh_create_noreuse();
        v_cleanup(stack);
    }

    /* Check special case */
    if (ptr == NULL) {
        v_freeze_start(fp);
        fprintf(fp, "%s", V_TOKEN_NULL_STRING);
        v_freeze_finish(fp);
        return code;
    }

    /* Get pointer type */
    if ((t = vt_type(ptr)) == NULL)
        v_fatal("v_freeze(): unknown pointer type");
    if (t->freeze == NULL)
        v_fatal("v_freeze(): type %s has no freeze function", t->name);

    /* Initialise if required */
    v_freeze_start(fp);

    if (depth++ == 0) {
        vh_init(multiref);
        v_traverse(ptr, flag_refs);
    }        

    key = vh_pkey_buf(ptr, buf);
    id = vh_iget(seen, key);

    if (id == 0) {
        /* Flag it as written */
        if (vh_exists(multiref, key)) {
            id = ++ft_tag;
            vh_istore(seen, key, id);
        }

        vh_istore(stack, key, 1);

        if (V_DEBUG(V_DBG_IO))
            v_info("Freezing %s", v_vinfo(ptr));

        /* Write name */
        fprintf(fp, "%s ", t->name);

        /* Write ID if required */
        if (id)
            fprintf(fp, "%d ", id);

        /* Write the data */
        if (!(*t->freeze)(ptr, fp))
            code = 0;
    } else {
        /* Seen before -- write old ID */
        fprintf(fp, "%s %d", V_TOKEN_REF_STRING, id);

        /* Give error if invalid reference */
        if (vh_iget(stack, key))
            code = 0;
    }

    /* Clean up if required */
    key = vh_pkey_buf(ptr, buf);
    vh_delete(stack, key);

    if (--depth == 0)
        vh_empty(multiref);

    v_freeze_finish(fp);

    return code;
}

/*!
  @brief   Freeze a pointer to a file.
  @ingroup freeze_high
  @param   ptr Pointer to Vars structure.
  @param   file File to write.
  @retval  Whether successful.
*/
int
v_freeze_file(void *ptr, char *file)
{
    int ok = 0;
    FILE *fp;

    if ((fp = v_open(file, "w")) != NULL) {
        ok = v_freeze(ptr, fp);
        v_close(fp);
    }

    if (!ok)
        v_exception("can't freeze file '%s'", file);

    return ok;
}

/*!
  @brief   Finish freezing.
  @ingroup freeze_low
  @param   fp Stream.
*/
void
v_freeze_finish(FILE *fp)
{
    if (--ft_depth == 0) {
        vh_empty(seen);
        fputc('\n', fp);
    }
}

/*!
  @brief   Start freezing.
  @ingroup freeze_low
  @param   fp Stream.
*/
void
v_freeze_start(FILE *fp)
{
    if (ft_depth++ == 0)
        ft_tag = 0;
}

/*!
  @brief   Freeze a string to file.
  @ingroup freeze_low
  @param   string String to freeze.
  @param   fp Stream.
  @return  Whether successful.
*/
int
v_freeze_string(char *string, FILE *fp)
{
    fputc('"', fp);

    while (*string != '\0') {
        switch (*string) {
        case '\n':
            fputc('\\', fp);
            fputc('n', fp);
            break;
        case '\\':
            fputc('\\', fp);
            fputc('\\', fp);
            break;
        case '"':
            fputc('\\', fp);
            fputc('"', fp);
            break;
        default:
            fputc(*string, fp);
            break;
        }

        string++;
    }

    fputc('"', fp);
    return 1;
}

/*!
  @brief   Thaw a pointer from file.
  @ingroup freeze_high
  @param   fp Stream to read.
  @return  Pointer to Vars structure.
  @retval  NULL if it failed.
  @see     v_thaw_error()
*/
void *
v_thaw(FILE *fp)
{
    char idbuf[20];
    vtype *type;
    void *data;
    int id;

    v_thaw_start();

    /* Get data type */
    switch (v_thaw_token(fp)) {
    case V_TOKEN_VTYPE:
        type = v_thaw_pvalue;
        if (type->thaw == NULL) {
            v_thaw_err("type '%s' has no thaw function", type->name);
            v_thaw_finish();
            return NULL;
        }
        break;
    case V_TOKEN_REF:
        type = NULL;
        break;
    default:
        v_thaw_expected("type name or reference");
        v_thaw_finish();
        return NULL;
    }

    /* Get its ID (if set) */
    if (type != NULL) {
        if (v_thaw_peek(fp) == V_TOKEN_INT) {
            v_thaw_token(fp);
            id = v_thaw_ivalue;
        } else {
            id = 0;
        }
    } else if (v_thaw_token(fp) != V_TOKEN_INT) {
        v_thaw_expected("reference tag");
        v_thaw_finish();
        return NULL;
    } else {
        id = v_thaw_ivalue;
    }

    /* See if it's been read before */
    if (id) {
        sprintf(idbuf, "%d", id);
        data = vh_pget(seen, idbuf);
        if (data == NULL && vh_exists(seen, idbuf)) {
            v_thaw_err("reference to parent data unsupported");
            v_thaw_finish();
            return NULL;
        } else if (type != NULL && vh_exists(seen, idbuf)) {
            v_thaw_err("reference tag %d already defined", id);
            v_thaw_finish();
            return NULL;
        }
    } else {
        data = NULL;
    }

    /* Read it if not */
    if (data == NULL) {
        if (type == NULL) {
            v_thaw_err("reference tag %d not yet defined", id);
            v_thaw_finish();
            return NULL;
        }

        if (id)
            vh_pstore(seen, idbuf, NULL);

        data = (*type->thaw)(fp);

        /* Flag it as read */
        if (id)
            vh_pstore(seen, idbuf, data);
    }

    v_thaw_finish();

    return data;
}

/*!
  @brief   Thaw a double value.
  @ingroup freeze_low
  @param   fp Stream.
  @param   value Pointer to value (returned).
  @return  Whether successful.
*/
int
v_thaw_double(FILE *fp, double *value)
{
    switch (v_thaw_token(fp)) {
    case V_TOKEN_INT:
        *value = v_thaw_ivalue;
        break;
    case V_TOKEN_REAL:
        *value = v_thaw_dvalue;
        break;
    default:
        v_thaw_expected("integer or real number");
        return 0;
    }

    return 1;
}

/*!
  @brief   Give a thaw error.
  @ingroup freeze_low
  @param   fmt Format string.
*/
void
v_thaw_err(char *fmt, ...)
{
    V_VPRINT(buf, fmt);

    vb_init(ebuf);

    if (filename != NULL && line_number > 0)
        vb_printf(ebuf, "%s, line %d: ", filename, line_number);
    else if (line_number > 0)
        vb_printf(ebuf, "line %d: ", line_number);
    else if (filename != NULL)
        vb_printf(ebuf, "%s: ", filename);

    vb_puts(ebuf, buf);
    v_exception("%s", vb_get(ebuf));
}

/*!
  @brief   Return the last thaw error.
  @ingroup freeze_high
  @return  Error string.
  @retval  NULL if no errors found.
*/
char *
v_thaw_error(void)
{
    if (ebuf == NULL)
        return NULL;
    return vb_get(ebuf);
}

/*!
  @brief   Give expected-token error.
  @ingroup freeze_low
  @param   desc Description of what was expected.
*/
void
v_thaw_expected(char *desc)
{
    if (strlen(yytext) == 0)
        v_thaw_err("expected %s (last token: EOF)", desc);
    else if (yytext[0] == '"')
        v_thaw_err("expected %s (last token: '%c')", desc, yytext[0]);
    else
        v_thaw_err("expected %s (last token: '%s')", desc, yytext);
}

/*!
  @brief   Thaw a pointer from a file.
  @ingroup freeze_high
  @param   file File to read.
  @return  Pointer to Vars structure.
  @retval  NULL if it failed.
*/
void *
v_thaw_file(char *file)
{
    void *ptr = NULL;
    FILE *fp;

    if ((fp = v_open(file, "r")) != NULL) {
        filename = file;
        ptr = v_thaw(fp);
        v_close(fp);
    } else {
        v_thaw_err("can't open %s: %s", file, strerror(errno));
    }

    return ptr;
}

/*!
  @brief   Set thaw filename.
  @ingroup freeze_low
  @param   file Filename.
*/
void
v_thaw_filename(char *file)
{
    filename = file;
}

/*!
  @brief   Finish thawing.
  @ingroup freeze_low
*/
void
v_thaw_finish(void)
{
    if (--ft_depth == 0) {
        vh_empty(seen);
        last_line_number = line_number;
        line_number = 0;
        start_line = 1;
        filename = NULL;
        yyrestart(yyin);
    }
}

/*!
  @brief   Expect a particular token to follow.
  @ingroup freeze_low
  @param   fp Stream.
  @param   token Token expected.
  @param   desc Description of what's expected.
  @return  Whether it followed.
*/
int
v_thaw_follow(FILE *fp, enum v_tokentype token, char *desc)
{
    if (v_thaw_token(fp) == token)
        return 1;

    v_thaw_expected(desc);
    return 0;
}

/*!
  @brief   Get line number of last thawed line.
  @ingroup freeze_low
  @return  Line number.
*/
int
v_thaw_getline(void)
{
    return last_line_number;
}

/*!
  @brief   Thaw an int value.
  @ingroup freeze_low
  @param   fp Stream.
  @param   value Pointer to value (returned).
  @return  Whether successful.
*/
int
v_thaw_int(FILE *fp, int *value)
{
    if (v_thaw_token(fp) == V_TOKEN_INT) {
        *value = v_thaw_ivalue;
        return 1;
    }

    v_thaw_expected("integer");
    return 0;
}

/*!
  @brief   Look at the next token without reading it.
  @ingroup freeze_low
  @param   fp Stream.
  @return  Next token.
*/
enum v_tokentype
v_thaw_peek(FILE *fp)
{
    if (!lookahead)
        lookahead = v_thaw_token(fp);

    return lookahead;
}

/*!
  @brief   Set current line number.
  @ingroup freeze_low
  @param   num Line number.
*/
void
v_thaw_setline(int num)
{
    start_line = num;
}

/*!
  @brief   Start thawing.
  @ingroup freeze_low
*/
void
v_thaw_start(void)
{
    static int decl = 0;

    /* Declare standard types if required */
    if (!decl) {
        vt_declare();
        decl++;
    }

    /* Initialise if required */
    if (ft_depth++ == 0) {
        line_number = start_line;
        lookahead = 0;
    }

    if (seen == NULL)
        seen = vh_create_table(1000, 0);
}

/*!
  @brief   Thaw a string.
  @ingroup freeze_low
  @param   fp Stream.
  @param   value Pointer to value (returned).
  @return  Whether successful.
*/
int
v_thaw_string(FILE *fp, char **value)
{
    if (v_thaw_token(fp) == V_TOKEN_STRING) {
        *value = v_thaw_svalue;
        return 1;
    }

    v_thaw_expected("string");
    return 0;
}

/*!
  @brief   Get the next token.
  @ingroup freeze_low
  @param   fp Stream.
  @return  Next token.
*/
enum v_tokentype
v_thaw_token(FILE *fp)
{
    enum v_tokentype token;

    if (lookahead) {
        token = lookahead;
        lookahead = 0;
    } else {
        yyin = fp;
        token = yylex();

        if (V_DEBUG(V_DBG_INTERN)) {
            fprintf(stderr, "Read token: ");

            switch (token) {
            case V_TOKEN_INT:
                fprintf(stderr, "integer (%d)", v_thaw_ivalue);
                break;
            case V_TOKEN_REAL:
                fprintf(stderr, "real (%g)", v_thaw_dvalue);
                break;
            case V_TOKEN_STRING:
                fprintf(stderr, "string (%s)", v_thaw_svalue);
                break;
            case V_TOKEN_VTYPE:
                fprintf(stderr, "variable type (%s)", v_thaw_svalue);
                break;
            case V_TOKEN_REF:
                fprintf(stderr, "variable reference");
                break;
            case V_TOKEN_NULL:
                fprintf(stderr, "NULL");
                break;
            case V_TOKEN_UNDEF:
                fprintf(stderr, "UNDEF");
                break;
            case V_TOKEN_ID:
                fprintf(stderr, "ID name (%s)", v_thaw_svalue);
                break;
            case V_TOKEN_ERROR:
                fprintf(stderr, "error");
                break;
            default:
                fprintf(stderr, "literal (%s)", yytext);
                break;
            }

            fprintf(stderr, "\n");
        }
    }

    return token;
}
