/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @file
  @brief Function functions and macros.
  @ingroup function
*/

#ifndef VARS_FUNC_H
#define VARS_FUNC_H

#include <vars-list.h>

/*! @brief Abort if a pointer is not a function. */
#define VF_CHECK(ptr)            V_CHECK(ptr, vfunc_type, "FUNC")

/*! @brief Check if a pointer is a function type. */
#define v_isfunc(ptr)            (ptr != NULL && vt_type(ptr) == vfunc_type) 

/*! @brief Iterate over a function. */
#define vf_foreach(x, fx, f)     vf_each_start(f); while (vf_each(f, &x, &fx))

/*! @brief Function interpolation types. */
enum v_interptype {
    VF_NONE,                    /* Internal use */
    VF_LINEAR,                  /*!< Linear */
    VF_LAGRANGE,                /*!< Lagrange 4-point */
    VF_SPLINE,                  /*!< Spline */
    VF_NEAREST,                 /*!< Nearest point */
    VF_EXTRAPOLATE		/*!< Linear with extrapolation  */
};

/*! @brief Function type. */
typedef struct v_func vfunc;

/*! @brief Function type variable. */
extern vtype *vfunc_type;

#ifdef __cplusplus
extern "C" {
#endif

extern void vf_add_array(vfunc *f, int num, double *x, double *fx);
extern void vf_add_func(vfunc *f, double (*func)(double val),
                        double min, double max, unsigned int num);
extern void vf_add_list(vfunc *f, vlist *x, vlist *fx);
extern void vf_add_point(vfunc *f, double x, double fx);
extern void vf_break(vfunc *f);
extern vfunc *vf_copy(vfunc *f);
extern vfunc *vf_create(void);
extern vfunc *vf_create_size(unsigned int size);
extern vtype *vf_declare(void);
extern int vf_defined(vfunc *f, double x);
extern int vf_delete_point(vfunc *f, int num);
extern void vf_destroy(vfunc *f);
extern vfunc *vf_differential(vfunc *f);
extern int vf_each(vfunc *f, double *x, double *fx);
extern void vf_each_start(vfunc *f);
extern void vf_empty(vfunc *f);
extern vfunc *vf_fread(FILE *fp);
extern int vf_freeze(vfunc *f, FILE *fp);
extern int vf_fwrite(vfunc *f, FILE *fp);
extern int vf_get_point(vfunc *f, int n, double *x, double *fx);
extern int vf_get_range(vfunc *f, double *fmin, double *fmax);
extern vfunc *vf_integral(vfunc *f);
extern void vf_interpolate(vfunc *f, enum v_interptype type);
extern double vf_maximum(vfunc *f);
extern double vf_minimum(vfunc *f);
extern int vf_point_count(vfunc *f);
extern vlist *vf_points_fx(vfunc *f);
extern vlist *vf_points_x(vfunc *f);
extern void vf_print(vfunc *f, FILE *fp);
extern void vf_resolution(vfunc *f, double res);
extern void vf_set_cycle(vfunc *f, double cycle);
extern void vf_spline_deriv(vfunc *f, double first, double last);
extern vfunc *vf_thaw(FILE *fp);
extern int vf_traverse(vfunc *f, int (*func)(void *ptr));
extern double vf_value(vfunc *f, double x);
extern double vf_value_extrapolate(vfunc *f, double x);
extern double vf_value_lagrange(vfunc *f, double x);
extern double vf_value_linear(vfunc *f, double x);
extern double vf_value_nearest(vfunc *f, double x);
extern double vf_value_spline(vfunc *f, double x);

#ifdef __cplusplus
}
#endif

#endif
