/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @defgroup hash Hashes
  @ingroup types

  A hash (or hash table, or associative array) is a set of key-value pairs.
  Each key is a string, and its corresponding value is a scalar.
*/

/*!
  @defgroup hash_create Creating and destroying hashes
  @ingroup hash
*/

/*!
  @defgroup hash_access Accessing hash elements
  @ingroup hash
*/

/*!
  @defgroup hash_modify Modifying hash elements
  @ingroup hash
*/

/*!
  @defgroup hash_convert Converting hashes to lists
  @ingroup hash
*/

/*!
  @defgroup hash_misc Other hash functions
  @ingroup hash
*/

/*!
  @defgroup hash_intern Internal hash functions
  @ingroup hash

  Hash tables are represented by an internal array of hash 'buckets'.  Each
  table entry hashes to one of these buckets.  Multiple entries that hash
  to the same bucket are strung together in a list -- this is known as a
  'collision'.  The number of collisions increases as the hash table fills
  up, and if nothing is done this will slow down access times, so at
  certain points a hash table is 'rehashed' -- all its contents are moved
  to a bigger table to reduce collisions.

  Rehashing occurs in the vh_store() function, when a threshold number of
  collisions is detected while adding a new table entry.  The next biggest
  table size is chosen from an internal list of 'good' table sizes -- these
  are the primes closest to the powers of 2 from 8 to 65536, for reasons of
  efficiency.  Once the maximum table size is reached, no more rehashing is
  done.

  If you know you're going to add a lot of entries to a hash, you may as
  well start with a big table size, using vh_create_size(), and save the
  overhead of rehashing.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>

#include "vars-config.h"
#include "vars-hash.h"
#include "vars-macros.h"
#include "vars-memory.h"

/* Calculate a hash value */
#define VH_HASHVAL(size, key, value)                                    \
        {                                                               \
                char *kp = key;                                         \
                for (value = 0; *kp != '\0'; kp++)                      \
                        value = *kp + 31 * value;                       \
                value %= size;                                          \
        }

/* Calculate a hash value */
#define VH_HASH(hash, key, value)                                       \
        VH_HASHVAL(hash->tablesize, key, value)

/* Find entry in hash table */
#define VH_FIND(hash, key, value, entry)                                \
        for (entry = hash->table[value];                                \
             entry != NULL;                                             \
             entry = entry->next)                                       \
                if (V_STREQ(key, entry->key))                           \
                        break;

/* Find an entry with collision counting */
#define VH_FIND_COUNT(hash, key, value, entry, count)                   \
        for (entry = hash->table[value], count = 0;                     \
             entry != NULL;                                             \
             entry = entry->next, count++)                              \
                if (V_STREQ(key, entry->key))                           \
                        break;

/* Whether to print a hash key */
#define PRINTKEY(key) (noprint == NULL || !vh_iget(noprint, key))

/* Non-printing string */
#define NOPRINT_STRING "<not printed>"

/* Type definition */
struct v_hash {
    struct v_header id;         /* Type marker */
    unsigned tablesize;         /* Hash table size */
    int hpos;                   /* Iteration position */
    struct v_table *tpos;       /* Iteration position */
    struct v_table **table;     /* Hash table */
    int reuse;                  /* Whether to reuse hash keys */
};

/* Hash table elements */
struct v_table {
    char *key;                  /* Key value */
    struct v_scalar *val;       /* Scalar value */
    struct v_table *next;       /* Link */
};

/* Type variable */
vtype *vhash_type = NULL;

/* Internal type abbreviations */
typedef struct v_table vtable;

/* List of primes closest to powers of 2 */
static int primes[] = {
    7, 17, 31, 61, 127, 257, 509, 1021, 2053, 4099, 8191, 16381, 32771,
    65537, 0
};

/* Scribble buffer */
static char buf[BUFSIZ];

/* Rehashing collision threshold */
static unsigned vh_collision = 10;

/* Hash key sorting function */
static int (*keysort)(vscalar **s1, vscalar **s2) = NULL;

/* Non-printing hash keys */
static vhash *noprint = NULL; 

/* Hash key table */
static vhash *hashkeys = NULL;

/* Buffer for hash key conversions */
char vh_keybuf[20];

/* Internal functions */
static char *vh_keystring(char *key);
static int vh_nextsize(int num);

/*!
  @brief   Create abbreviation table from word list.
  @ingroup hash_misc
  @param   words List of words.
  @param   nocase Whether to ignore case.
  @return  Hash of abbreviations.
  @see     vh_abbrev(), vh_abbrev_nocase()

  Given a list of words, return a hash of the unique abbreviations of those
  words.  If a character string is an abbreviation of a word in the list,
  it appears as a key in the hash.  If it a unique abbreviation, the
  corresponding value is the word itself.  If not, the corresponding value
  is undefined.  Words in the original list always map to themselves, even
  if they are an abbreviation of a longer word.

  If \c nocase is set, make the hash keys lower-case.  Hash values are
  still words of the original list.  Ambiguities will arise if two words in
  the list map to the same lower-case word.
*/
vhash *
vh_abbrev_table(vlist *words, int nocase)
{
    char *word, *cp;
    vhash *table;
    vscalar *elt;
    int len;

    VL_CHECK(words);
    table = vh_create();

    /* Add abbreviations */
    vl_foreach(elt, words) {
        word = vs_sgetref(elt);
        if ((len = strlen(word)) == 0)
            continue;

        strcpy(buf, word);
        if (nocase)
            for (cp = buf; *cp != '\0'; cp++)
                *cp = tolower(*cp);

        while (len > 0) {
            buf[len--] = '\0';
            if (vh_exists(table, buf))
                vh_undef(table, buf);
            else
                vh_sstore(table, buf, word);
        }
    }

    /* Add the words themselves */
    vl_foreach(elt, words) {
        word = vs_sgetref(elt);
        strcpy(buf, word);

        if (nocase)
            for (cp = buf; *cp != '\0'; cp++)
                *cp = tolower(*cp);

        vh_sstore(table, buf, word);
    }

    return table;
}

/*!
  @brief   Signal break out of an each-loop.
  @ingroup hash_access
  @param   h Hash.
  @see     vh_each()
*/
void
vh_break(vhash *h)
{
    VH_CHECK(h);
    h->hpos = 0;
    h->tpos = NULL;
}

/*!
  @brief   Return a copy of a hash table.
  @ingroup hash_create
  @param   h Hash.
  @return  Copy of the hash.
*/
vhash *
vh_copy(vhash *h)
{
    vhash *hc;
    vtable *hp;
    int i;

    VH_CHECK(h);

    hc = vh_create_size(h->tablesize);

    for (i = 0; i < h->tablesize; i++)
	for (hp = h->table[i]; hp != NULL; hp = hp->next)
	    vh_store(hc, hp->key, vs_copy(hp->val));

    if (V_DEBUG(V_DBG_INFO))
        v_info("Copied %s", v_vinfo(h));

    return hc;
}

/*!
  @brief   Return a new hash.
  @ingroup hash_create
  @param   size Initial size.
  @param   reuse Whether to reuse hash keys.
  @return  New hash.
  @see     vh_create(), vh_create_noreuse()
*/
vhash *
vh_create_table(unsigned size, int reuse)
{
    static vheader *id = NULL;
    vhash *h;
    int i;

    if (id == NULL) {
        vh_declare();
        id = vt_header(vhash_type);
    }

    size = vh_nextsize(size);

    h = V_ALLOC(vhash, 1);

    h->id = *id;
    h->tablesize = size;
    h->table = V_ALLOC(vtable *, size);
    for (i = 0; i < size; i++)
	h->table[i] = NULL;
    h->hpos = 0;
    h->tpos = NULL;
    h->reuse = reuse;

    if (V_DEBUG(V_DBG_INFO))
        v_info("Created %s", v_vinfo(h));

    return h;
}

/* Declare hash type */
vtype *
vh_declare(void)
{
    if (vhash_type == NULL) {
        vhash_type = vt_create("HASH", "H");
        vt_copy_with(vhash_type, (void *(*)()) vh_copy);
        vt_read_with(vhash_type, (void *(*)()) vh_fread);
        vt_write_with(vhash_type, vh_fwrite);
        vt_freeze_with(vhash_type, vh_freeze);
        vt_thaw_with(vhash_type, (void *(*)()) vh_thaw);
        vt_print_with(vhash_type, vh_print);
        vt_destroy_with(vhash_type, vh_destroy);
        vt_traverse_with(vhash_type, vh_traverse);
    }

    return vhash_type;
}

/*!
  @brief   Return whether a hash entry is defined.
  @ingroup hash_access
  @param   h Hash.
  @param   key Entry to check.
  @return  Yes or no.
*/
int
vh_defined(vhash *h, char *key)
{
    unsigned hashval;
    vtable *hp;

    VH_CHECK(h);
    VH_HASH(h, key, hashval);
    VH_FIND(h, key, hashval, hp);

    if (hp == NULL)
        return 0;
    if (hp->val == NULL)
        return 0;

    return vs_defined(hp->val);
}

/*!
  @brief   Delete a key-value pair from a hash.
  @ingroup hash_modify
  @param   h Hash.
  @param   key Entry to delete.
  @return  Whether anything was deleted.

  Differs from vh_undef() in that the key is also deleted (i.e. vh_exists()
  will return false).
*/
int
vh_delete(vhash *h, char *key)
{
    vtable *hp, *hpprev = NULL;
    unsigned hval;

    VH_CHECK(h);
    VH_HASH(h, key, hval);

    for (hp = h->table[hval]; hp != NULL; hp = hp->next) {
	if (V_STREQ(key, hp->key)) {
	    if (hpprev == NULL)
		h->table[hval] = hp->next;
	    else
		hpprev->next = hp->next;

            if (V_DEBUG(V_DBG_INFO))
                v_info("Deleted %s from %s", v_vinfo(hp->val), v_vinfo(h));

            if (!h->reuse)
                V_DEALLOC(hp->key);
	    if (hp->val != NULL)
		vs_destroy(hp->val);
	    V_DEALLOC(hp);
	    return 1;
	} else {
	    hpprev = hp;
	}
    }

    return 0;
}

/*!
  @brief   Deallocate a hash table and its contents.
  @ingroup hash_create
  @param   h Hash.
*/
void
vh_destroy(vhash *h)
{
    VH_CHECK(h);
    vh_empty(h);

    if (V_DEBUG(V_DBG_INFO))
        v_info("Destroyed %s", v_vinfo(h));

    V_DEALLOC(h->table);
    V_DEALLOC(h);
}

/*!
  @brief   Iterate over a hash, returning key-value pairs.
  @ingroup hash_access
  @param   h Hash.
  @param   key Pointer to key value (returned).
  @param   val Pointer to scalar value (returned).
  @return  Whether there are more entries.
  @see     vh_foreach(), vh_break()

  Iterate over the key-value pairs in a hash, in an apparently random
  order.  Returns whether there are no more entries (if not, the next call
  starts again).  Note that if you finish iterating without getting to the
  end of the hash (via \c break, for instance) then you must call
  vh_break() to indicate that you've finished.  The returned scalar is a
  pointer to the actual hash element, so modifying it will change the hash
  -- but do \e not destroy it!  And treat the key as read-only, too.
*/
int
vh_each(vhash *h, char **key, vscalar **val)
{
    VH_CHECK(h);

    while (h->hpos < h->tablesize) {
	if (h->tpos == NULL) {
	    h->tpos = h->table[h->hpos];
	    if (h->tpos == NULL)
		h->hpos++;
	} else {
	    *key = h->tpos->key;
	    *val = h->tpos->val;
	    h->tpos = h->tpos->next;
	    if (h->tpos == NULL)
		h->hpos++;
	    return 1;
	}
    }

    h->hpos = 0;
    h->tpos = NULL;

    return 0;
}

/* Check start of foreach loop */
void
vh_each_start(vhash *h)
{
    VH_CHECK(h);
    if (h->hpos != 0 || h->tpos != NULL)
        v_fatal("vh_each_start(): iteration error");
}

/*!
  @brief   Empty a hash.
  @ingroup hash_modify
  @param   h Hash.
*/
void
vh_empty(vhash *h)
{
    vtable *hp, *hpnext;
    int i;

    VH_CHECK(h);

    if (V_DEBUG(V_DBG_INFO))
        v_info("Emptying %s", v_vinfo(h));

    for (i = 0; i < h->tablesize; i++) {
	for (hp = h->table[i]; hp != NULL; hp = hpnext) {
	    hpnext = hp->next;
            if (!h->reuse)
                V_DEALLOC(hp->key);
	    if (hp->val != NULL)
		vs_destroy(hp->val);
	    V_DEALLOC(hp);
	}

        h->table[i] = NULL;
    }

    h->hpos = 0;
    h->tpos = NULL;
}

/*!
  @brief   Return the no. of entries in a hash.
  @ingroup hash_misc
  @param   h Hash.
  @return  No. of entries.
*/
int
vh_entry_count(vhash *h)
{
    int i, size = 0;
    vtable *hp;

    VH_CHECK(h);

    for (i = 0; i < h->tablesize; i++)
	for (hp = h->table[i]; hp != NULL; hp = hp->next)
            size++;

    return size;
}

/*!
  @brief   Return whether a hash entry exists.
  @ingroup hash_access
  @param   h Hash.
  @param   key Entry to check.
  @return  Yes or no.
*/
int
vh_exists(vhash *h, char *key)
{
    unsigned hashval;
    vtable *hp;

    VH_CHECK(h);
    VH_HASH(h, key, hashval);
    VH_FIND(h, key, hashval, hp);

    return (hp != NULL);
}

/* Read hash from a stream */
vhash *
vh_fread(FILE *fp)
{
    char keybuf[BUFSIZ];
    vscalar *s;
    int num, i;
    char *key;
    vhash *h;

    /* Read no. of entries */
    if (!v_read_long(&num, fp))
        return NULL;

    h = vh_create_size(num);

    /* Read key-value pairs */
    for (i = 0; i < num; i++) {
        /* Read key */
        if ((key = v_read_string(fp)) == NULL)
            return NULL;
        strcpy(keybuf, key);

        /* Read value */
        if ((s = vs_fread(fp)) == NULL)
            return NULL;

        /* Add entry */
        vh_store(h, keybuf, s);
    }

    if (V_DEBUG(V_DBG_IO))
        v_info("Read %s", v_vinfo(h));

    return h;
}

/* Freeze contents of a hash */
int
vh_freeze(vhash *h, FILE *fp)
{
    int i, first = 1;
    vtable *hp;

    VH_CHECK(h);

    v_freeze_start(fp);

    fputs("{\n", fp);
    v_push_indent();

    for (i = 0; i < h->tablesize; i++) {
        for (hp = h->table[i]; hp != NULL; hp = hp->next) {
            if (first)
                first = 0;
            else
                fputs(",\n", fp);
            v_indent(fp);
            if (!v_freeze_string(hp->key, fp))
                return 0;
            fputs(" = ", fp);
            if (!vs_freeze(hp->val, fp))
                return 0;
        }
    }

    fputc('\n', fp);
    v_pop_indent();
    v_indent(fp);
    fputc('}', fp);

    v_freeze_finish(fp);

    return 1;
}

/* Write hash to a stream */
int
vh_fwrite(vhash *h, FILE *fp)
{
    int *info, i, num;
    vtable *hp;

    VH_CHECK(h);

    if (V_DEBUG(V_DBG_IO))
        v_info("Writing %s", v_vinfo(h));

    /* Write no. of hash entries */
    info = vh_hashinfo(h);
    num = info[2];
    if (!v_write_long(num, fp))
        return 0;

    for (i = 0; i < h->tablesize; i++) {
        for (hp = h->table[i]; hp != NULL; hp = hp->next) {
            /* Write hash key */
            if (!v_write_string(hp->key, fp))
                return 0;

            /* Write hash value */
            if (!vs_fwrite(hp->val, fp))
                return 0;
        }
    }

    return 1;
}

/*!
  @brief   Return a value from a hash.
  @ingroup hash_access
  @param   h Hash.
  @param   key Entry to get.
  @return  Scalar value.
  @retval  NULL if not defined.
*/
vscalar *
vh_get(vhash *h, char *key)
{
    unsigned hashval;
    vtable *hp;

    VH_CHECK(h);
    VH_HASH(h, key, hashval);
    VH_FIND(h, key, hashval, hp);

    if (V_DEBUG(V_DBG_HASHREFS) && hp == NULL)
        v_warn_internal("access of non-existent hash entry: %s", key);

    return (hp == NULL ? NULL : hp->val);
}

/*!
  @brief   Build a hash from a list of arguments.
  @ingroup hash_create
  @return  New hash.

  Create and return a hash containing a list of given key-value pairs.
  Each hash entry is specified by three parameters: a <tt>char *</tt> key,
  an \c int type, and a value depending on its type.  The type can be one
  of V_INT, V_FLOAT, V_DOUBLE, V_STRING, V_POINTER or V_UNDEF.  The type of
  the following value should correspond to this, except in the case of
  V_UNDEF, where the following value should be omitted.  The list of
  key-value pairs should be terminated by a \c NULL key.  If a value has
  the wrong type, or no terminator is supplied, chaos will surely result.
*/
vhash *
vh_hash(char *key, ...)
{
    va_list ap;
    vhash *h;
    int type;

    h = vh_create();
    va_start(ap, key);

    while (key != NULL) {
        strcpy(buf, key);
        type = va_arg(ap, int);

        switch (type) {
        case V_INT:
            vh_istore(h, buf, va_arg(ap, int));
            break;
        case V_FLOAT:
            /* Read as double */
            vh_fstore(h, buf, va_arg(ap, double));
            break;
        case V_DOUBLE:
            vh_dstore(h, buf, va_arg(ap, double));
            break;
        case V_STRING:
            vh_sstore(h, buf, va_arg(ap, char *));
            break;
        case V_POINTER:
            vh_pstore(h, buf, va_arg(ap, void *));
            break;
        case V_UNDEF:
            vh_store(h, buf, vs_create(V_UNDEF));
            break;
        default:
            v_fatal("vh_hash(): invalid scalar type in arg list");
        }

        key = va_arg(ap, char *);
    }

    return h;
}

/*!
  @brief   Return some stats on a hash.
  @ingroup hash_intern
  @param   h Hash.
  @return  Stats array.

  Entries in the array are:
  @verbatim
  0 Total no. of hash buckets
  1 No. of hash buckets used
  2 Total no. of entries in the hash
  3 Largest no. of entries in a hash bucket
  @endverbatim
*/
int *
vh_hashinfo(vhash *h)
{
    static int sizes[4];
    vtable *hp;
    int i, ncol;

    VH_CHECK(h);

    for (i = 0; i < 4; i++)
	sizes[i] = 0;

    sizes[0] = h->tablesize;

    for (i = 0; i < h->tablesize; i++) {
	if (h->table[i] != NULL)
	    sizes[1]++;
	for (hp = h->table[i], ncol = 0; hp != NULL; hp = hp->next, ncol++)
	    sizes[2]++;
	if (ncol > sizes[3])
	    sizes[3] = ncol;
    }

    return sizes;
}

/*!
  @brief   Return a list of keys of a hash (in unspecified order).
  @ingroup hash_convert
  @param   h Hash.
  @return  List of keys.
*/
vlist *
vh_keys(vhash *h)
{
    vtable *hp;
    vlist *l;
    int i;

    VH_CHECK(h);

    l = vl_create();

    for (i = 0; i < h->tablesize; i++)
	for (hp = h->table[i]; hp != NULL; hp = hp->next)
	    vl_spush(l, hp->key);

    return l;
}

/* Return a key string given a key */
static char *
vh_keystring(char *key)
{
    unsigned hashval;
    vtable *hp;

    if (hashkeys == NULL) {
        hashkeys = vh_create_table(0, 0);
        v_cleanup(hashkeys);
    }

    /* If key allocated already, return it */
    VH_HASH(hashkeys, key, hashval);
    VH_FIND(hashkeys, key, hashval, hp);
    if (hp != NULL)
        return hp->key;

    /* Add key */
    if (V_DEBUG(V_DBG_INFO))
        v_info("Storing hash key '%s'", key);

    vh_store(hashkeys, key, NULL);

    /* Find pointer to key in table */
    VH_HASH(hashkeys, key, hashval);
    VH_FIND(hashkeys, key, hashval, hp);

    return hp->key;
}

/*!
  @brief   Build a hash from a NULL-terminated list of strings.
  @ingroup hash_create
  @param   list List of strings.
  @return  New hash.

  Consecutive strings represent a single key-value pair.  If there are an
  odd number of strings, then the last key-value pair has a value of
  V_UNDEF.  The list of strings should terminate with \c NULL.
*/
vhash *
vh_makehash(char **list)
{
    char *key;
    vhash *h;

    h = vh_create();
    if (list == NULL)
        return h;

    while (*list != NULL) {
	key = *list++;
        if (*list != NULL)
            vh_sstore(h, key, *list++);
        else
            vh_store(h, key, vs_create(V_UNDEF));
    }

    return h;
}

/* Return the next highest hash table size */
static int
vh_nextsize(int num)
{
    int i = 0, next = primes[0];

    while (primes[i + 1] != 0 && num > primes[i++])
        next = primes[i];

    return next;
}

/* Flag a hash key as non-printing */
void
vh_noprint(char *key)
{
    if (key == NULL) {
        if (noprint != NULL) {
            vh_destroy(noprint);
            noprint = NULL;
        }
    } else {
        if (noprint == NULL) {
            noprint = vh_create();
            v_cleanup(noprint);
        }

        vh_istore(noprint, key, 1);
    }
}

/* Print contents of a hash */
void
vh_print(vhash *h, FILE *fp)
{
    char *key, keybuf[BUFSIZ];
    vscalar *val;
    vlist *keys;
    vtable *hp;
    int i;

    VH_CHECK(h);

    v_print_start();
    v_push_indent();

    v_print_type(vhash_type, h, fp);

    if (keysort == NULL) {
        /* Unsorted keys */
        for (i = 0; i < h->tablesize; i++) {
            for (hp = h->table[i]; hp != NULL; hp = hp->next) {
                v_indent(fp);
                fprintf(fp, "%s => ", hp->key);
                if (PRINTKEY(hp->key))
                    v_print(hp->val, fp);
                else
                    fprintf(fp, "%s\n", NOPRINT_STRING); 
            }
        }
    } else {
        /* Sorted keys */
        keys = vh_sortkeys(h, keysort);

        vl_foreach(val, keys) {
            key = vs_sget_buf(val, keybuf);
            v_indent(fp);
            fprintf(fp, "%s => ", key);
            if (PRINTKEY(key))
                v_print(vh_get(h, key), fp);
            else
                fprintf(fp, "%s\n", NOPRINT_STRING); 
        }

        vl_destroy(keys);
    }

    v_pop_indent();
    v_print_finish();
}

/*!
  @brief   Set the rehashing collision threshold.
  @ingroup hash_intern
  @param   count No. of collisions before rehashing.

  Set the rehashing threshold.  Default is 10.  Rehashing will occur when
  this many collisions are detected by vh_store().  A value of zero will
  turn rehashing off completely.
*/
void
vh_rehash(unsigned count)
{
    vh_collision = count;
}

/* Sort hash keys when printing */
void
vh_sortprint(int compare(vscalar **s1, vscalar **s2))
{
    keysort = (compare ? compare : vs_cmp);
}

/*!
  @brief   Store a key-value pair in a hash.
  @ingroup hash_modify
  @param   h Hash.
  @param   key Entry to set.
  @param   val Value to set it to.
*/
void
vh_store(vhash *h, char *key, vscalar *val)
{
    vtable *hp, *hpnext, **table;
    static unsigned maxsize = 0;
    int count, size, i;
    unsigned hashval;

    VH_CHECK(h);

    if (h != hashkeys) {
        VS_CHECK(val);
        if (V_DEBUG(V_DBG_INFO))
            v_info("Adding %s to %s", v_vinfo(val), v_vinfo(h));
    }

    /* Find entry */
    VH_HASH(h, key, hashval);
    VH_FIND_COUNT(h, key, hashval, hp, count);

    /* Add new entry if it doesn't exist */
    if (hp == NULL) {
        /* Get or allocate key string */
        if (!h->reuse)
            key = V_STRDUP(key);
        else
            key = vh_keystring(key);

        /* Add the entry */
	hp = V_ALLOC(vtable, 1);
	hp->key = key;
	hp->val = NULL;
	hp->next = h->table[hashval];
	h->table[hashval] = hp;
    }

    /* Add new value */
    if (hp->val != NULL && hp->val != val)
	vs_destroy(hp->val);
    hp->val = val;

    /* Rehash not required if collision threshold not reached */
    if (vh_collision == 0 || count < vh_collision)
        return;

    /* Can't rehash if at maximum size */
    if (h->tablesize == maxsize)
        return;

    size = vh_nextsize(h->tablesize + 1);
    if (size == h->tablesize) {
        maxsize = size;
        return;
    }

    /* Rehash required and possible -- allocate new table */
    table = V_ALLOC(vtable *, size);
    for (i = 0; i < size; i++)
	table[i] = NULL;

    /* Transfer old entries to new table */
    for (i = 0; i < h->tablesize; i++) {
	for (hp = h->table[i]; hp != NULL; hp = hpnext) {
            hpnext = hp->next;
            VH_HASHVAL(size, hp->key, hashval);
            hp->next = table[hashval];
            table[hashval] = hp;
        }
    }

    /* Dispose of old table */
    V_DEALLOC(h->table);

    /* Install new one */
    h->table = table;
    h->tablesize = size;
    h->hpos = 0;
    h->tpos = NULL;
}

/* Thaw a hash from file */
vhash *
vh_thaw(FILE *fp)
{
    vscalar *s;
    int token;
    char *key;
    vhash *h;

    v_thaw_start();

    h = vh_create();

    if (!v_thaw_follow(fp, '{', "open-brace"))
        goto fail;

    while (1) {
        if (v_thaw_peek(fp) == '}') {
            v_thaw_token(fp);
            break;
        }

        if (!v_thaw_follow(fp, V_TOKEN_STRING, "hash key string"))
            goto fail;

        key = V_STRDUP(v_thaw_svalue);

        if (!v_thaw_follow(fp, '=', "'='"))
            goto fail;

        if ((s = vs_thaw(fp)) == NULL)
            goto fail;

        vh_store(h, key, s);
        v_free(key);

        if ((token = v_thaw_token(fp)) == '}') {
            break;
        } else if (token != ',') {
            v_thaw_expected("comma or close-brace");
            goto fail;
        }
    }

    if (V_DEBUG(V_DBG_IO) && h != NULL)
        v_info("Thawed %s", v_vinfo(h));

    v_thaw_finish();
    return h;

  fail:
    v_thaw_finish();
    v_destroy(h);
    return NULL;
}

/* Traverse a hash */
int
vh_traverse(vhash *h, int (*func)(void *ptr))
{
    vtable *hp;
    int i, val;

    VH_CHECK(h);

    if ((val = func(h)) != 0)
        return val;

    if (v_traverse_seen(h))
        return 0;

    v_push_traverse(h);

    for (i = 0; i < h->tablesize; i++) {
        for (hp = h->table[i]; hp != NULL; hp = hp->next) {
            if (vs_type(hp->val) == V_POINTER &&
                (val = v_traverse(vs_pget(hp->val), func)) != 0) {
                v_pop_traverse();
                return val;
            }
        }
    }

    v_pop_traverse();

    return 0;
}

/*!
  @brief   Undefine a hash entry.
  @ingroup hash_modify
  @param   h Hash.
  @param   key Entry to undefine.

  If the hash entry doesn't exist, it is created.  The scalar value is then
  set to V_UNDEF.
*/
void
vh_undef(vhash *h, char *key)
{
    unsigned hashval;
    vtable *hp;

    VH_CHECK(h);
    VH_HASH(h, key, hashval);
    VH_FIND(h, key, hashval, hp);

    if (hp != NULL && hp->val != NULL)
        vs_undef(hp->val);
    else
        vh_store(h, key, vs_create(V_UNDEF));
}

/*!
  @brief   Return a list of values of a hash (in unspecified order).
  @ingroup hash_convert
  @param   h Hash.
  @return  List of values.
*/
vlist *
vh_values(vhash *h)
{
    vtable *hp;
    vlist *l;
    int i;

    VH_CHECK(h);

    l = vl_create();

    for (i = 0; i < h->tablesize; i++)
	for (hp = h->table[i]; hp != NULL; hp = hp->next)
	    vl_push(l, vs_copy(hp->val));

    return l;
}
