/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @file
  @brief Hash functions and macros.
  @ingroup hash
*/

#ifndef VARS_HASH_H
#define VARS_HASH_H

#include <vars-list.h>
#include <vars-utils.h>

/*! @brief Abort if a pointer is not a hash. */
#define VH_CHECK(ptr)           V_CHECK(ptr, vhash_type, "HASH")

/*! @brief Check if a pointer is a hash type. */
#define v_ishash(ptr)           (ptr != NULL && vt_type(ptr) == vhash_type)

/*! @brief Create a new hash. */
#define vh_create()             vh_create_table(0, 1)

/*! @brief Create a new hash with the given size. */
#define vh_create_size(size)    vh_create_table(size, 1)

/*! @brief Create a new hash that doesn't reuse key strings. */
#define vh_create_noreuse()     vh_create_table(0, 0)

/*! @brief Return type of a hash key. */
#define vh_type(h, key)         vs_type(vh_get(h, key))

/*! @brief Get the string value of a hash key. */
#define vh_sget(h, key)		vs_sget(vh_get(h, key))

/*! @brief Get a copy of the string value of a hash key. */
#define vh_sgetcopy(h, key)     vs_sgetcopy(vh_get(h, key))

/*! @brief Get a reference to the string value of a hash key. */
#define vh_sgetref(h, key)	vs_sgetref(vh_get(h, key))

/*! @brief Set the string value of a hash key. */
#define vh_sstore(h, key, n)	vh_store(h, key, vs_screate(n))

/*! @brief Get the float value of a hash key. */
#define vh_fget(h, key)		vs_fget(vh_get(h, key))

/*! @brief Set the float value of a hash key. */
#define vh_fstore(h, key, n)	vh_store(h, key, vs_fcreate(n))

/*! @brief Get the double value of a hash key. */
#define vh_dget(h, key)		vs_dget(vh_get(h, key))

/*! @brief Set the double value of a hash key. */
#define vh_dstore(h, key, n)	vh_store(h, key, vs_dcreate(n))

/*! @brief Get the integer value of a hash key. */
#define vh_iget(h, key)         vs_iget(vh_get(h, key))

/*! @brief Set the integer value of a hash key. */
#define vh_istore(h, key, n)    vh_store(h, key, vs_icreate(n))

/*! @brief Get the pointer value of a hash key. */
#define vh_pget(h, key)		vs_pget(vh_get(h, key))

/*! @brief Set the pointer value of a hash key. */
#define vh_pstore(h, key, p)	vh_store(h, key, vs_pcreate((void *) p))

/*! @brief Return a sorted list of hash keys. */
#define vh_sortkeys(h, func)    vl_sort_inplace(vh_keys(h), func)

/*! @brief Return a sorted list of hash values. */
#define vh_sortvalues(h, func)  vl_sort_inplace(vh_values(h), func)

/*! @brief Iterate over a hash. */
#define vh_foreach(k, s, h)     vh_each_start(h); while (vh_each(h, &k, &s))

/*! @brief Return hash of unique abbreviations for a list of words. */
#define vh_abbrev(words)        vh_abbrev_table(words, 0)

/*! @brief Return case-insensitive hash of unique abbreviations for a list of words. */
#define vh_abbrev_nocase(words) vh_abbrev_table(words, 1)

/*! @brief Build a hash key from a pointer using a buffer. */
#define vh_pkey_buf(ptr, buf)   v_hexstring(ptr, buf)

/*! @brief Build a hash key from a pointer. */
#define vh_pkey(ptr)            vh_pkey_buf(ptr, vh_keybuf)

/*! @brief Build a hash key from an integer using a buffer. */
#define vh_ikey_buf(num, buf)   (sprintf(buf, "%d", num), buf)

/*! @brief Build a hash key from an integer. */
#define vh_ikey(num)            vh_ikey_buf(num, vh_keybuf)

/*! @brief Build a hash key from a float using a buffer. */
#define vh_fkey_buf(num, buf)   (sprintf(buf, "%g", num), buf)

/*! @brief Build a hash key from a float. */
#define vh_fkey(num)            vh_fkey_buf(num, vh_keybuf)

/*! @brief Build a hash key from a double using a buffer. */
#define vh_dkey_buf(num, buf)   vh_fkey_buf(num, buf)

/*! @brief Build a hash key from a double. */
#define vh_dkey(num)            vh_fkey(num)

/*! @brief Create or empty a hash. */
#define vh_init(h) if (h == NULL) h = vh_create(); else vh_empty(h)

/*! @brief Hash type. */
typedef struct v_hash vhash;

/*! @brief Hash type variable. */
extern vtype *vhash_type;

extern char vh_keybuf[];

#ifdef __cplusplus
extern "C" {
#endif

extern vhash *vh_abbrev_table(vlist *words, int nocase);
extern void vh_break(vhash *h);
extern vhash *vh_copy(vhash *h);
extern vhash *vh_create_table(unsigned size, int reuse);
extern vtype *vh_declare(void);
extern int vh_defined(vhash *h, char *key);
extern int vh_delete(vhash *h, char *key);
extern void vh_destroy(vhash *h);
extern int vh_each(vhash *h, char **key, vscalar **val);
extern void vh_each_start(vhash *h);
extern void vh_empty(vhash *h);
extern int vh_entry_count(vhash *h);
extern int vh_exists(vhash *h, char *key);
extern vhash *vh_fread(FILE *fp);
extern int vh_freeze(vhash *h, FILE *fp);
extern int vh_fwrite(vhash *h, FILE *fp);
extern vscalar *vh_get(vhash *h, char *key);
extern vhash *vh_hash(char *key, ...);
extern int *vh_hashinfo(vhash *h);
extern vlist *vh_keys(vhash *h);
extern vhash *vh_makehash(char **list);
extern void vh_noprint(char *key);
extern void vh_print(vhash *h, FILE *fp);
extern void vh_rehash(unsigned count);
extern void vh_sortprint(int compare(vscalar **s1, vscalar **s2));
extern void vh_store(vhash *h, char *key, vscalar *val);
extern vhash *vh_thaw(FILE *fp);
extern int vh_traverse(vhash *h, int (*func)(void *ptr));
extern void vh_undef(vhash *h, char *key);
extern vlist *vh_values(vhash *h);

#ifdef __cplusplus
}
#endif

#endif
