/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @defgroup memory Memory allocation functions
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <limits.h>

#include "vars-config.h"
#include "vars-debug.h"
#include "vars-destroy.h"
#include "vars-hash.h"
#include "vars-memory.h"
#include "vars-print.h"

/* Memory error function */
static void (*memerr_func)(size_t size);

/*!
  @brief   Allocate some memory.
  @ingroup memory
  @param   size No. of bytes to allocate.
  @return  Allocated memory.
  @see     v_memerr_hook()
*/
void *
v_malloc(size_t size)
{
    void *ptr;

    v_memory_allocated = 1;

    if (size == 0) {
        if (memerr_func == NULL)
            v_fatal("v_malloc(): requested %ld bytes", size);
        else
            memerr_func(size);
    }

    if ((ptr = malloc(size)) == NULL) {
        if (memerr_func == NULL)
            v_fatal("v_malloc(): out of memory (requested %ld bytes)", size);
        else
            memerr_func(size);
    }

    if (v_debug_flags && v_memory_track)
        v_memchange(V_MEM_MALLOC, ptr, NULL, size);

    return ptr;
}

/*!
  @brief   Reallocate memory.
  @ingroup memory
  @param   ptr Pointer to reallocate.
  @param   size Size in bytes.
  @return  New memory.
*/
void *
v_realloc(void *ptr, size_t size)
{
    void *newptr = realloc(ptr, size);

    if (size == 0) {
        if (memerr_func == NULL)
            v_fatal("v_realloc(): requested %ld bytes", size);
        else
            memerr_func(size);
    }

    if (newptr == NULL) {
        if (memerr_func == NULL)
            v_fatal("v_realloc(): out of memory (requested %ld bytes)", size);
        else
            memerr_func(size);
    }

    if (v_debug_flags && v_memory_track)
        v_memchange(V_MEM_REALLOC, newptr, ptr, size);

    return newptr;
}

/*!
  @brief   Free memory.
  @ingroup memory
  @param   ptr Pointer.
*/
void
v_free(void *ptr)
{
    /*@-shadow@*/ 
    extern vhash *print_funcs, *free_funcs;
    /*@=shadow@*/ 
    static char kbuf[30];
    char *key;

    if (v_debug_flags && v_memory_track)
        v_memchange(V_MEM_FREE, ptr, NULL, 0);

    if (free_funcs != NULL) {
        if (ptr != free_funcs) {
            key = vh_pkey_buf(ptr, kbuf);
            vh_delete(free_funcs, key);
        } else {
            free_funcs = NULL;
        }
    }

    if (print_funcs != NULL) {
        if (ptr != print_funcs) {
            key = vh_pkey_buf(ptr, kbuf);
            vh_delete(print_funcs, key);
        } else {
            print_funcs = NULL;
        }
    }

    free(ptr);
}

/*!
  @brief   Set hook function for memory errors.
  @ingroup memory
  @param   func Function.

  This function is called with the number of bytes requested.  It should
  do something which doesn't involve allocating any more memory, and then
  exit.
*/
void
v_memerr_hook(void (*func)(size_t size))
{
    memerr_func = func;
}
