/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @file
  @brief Regexp functions and macros.
  @ingroup regexp
*/

#ifndef VARS_REGEXP_H
#define VARS_REGEXP_H

#include <vars-list.h>

/*! @brief Abort if a pointer is not a regexp. */
#define VR_CHECK(ptr)           V_CHECK(ptr, vregex_type, "REGEX")

/*! @brief Check if a pointer is a regexp type. */
#define v_isregex(ptr)          (ptr != NULL && vt_type(ptr) == vregex_type)

/*! @brief Like vr_matched(), but copy the result. */
#define vr_matchedcopy(num) \
        V_STRDUP(vr_matched(num))

/*! @brief Like vr_subst(), but copy the result. */
#define vr_substcopy(str, pat, sub) \
        V_STRDUP(vr_subst(str, pat, sub))

/*! @brief Create a regexp if not already done. */
#define vr_init(r, regexp) \
        !(r == NULL && (r = vr_create(regexp)) == NULL)

/*! @brief Regexp type. */
typedef struct v_regex vregex;

/*! @brief Regexp type variable. */
extern vtype *vregex_type;

#ifdef __cplusplus
extern "C" {
#endif

extern vregex *vr_copy(vregex *r);
extern vregex *vr_create(char *str);
extern vregex *vr_create_nocase(char *str);
extern vregex *vr_create_translate(char *str, char *table);
extern vtype *vr_declare(void);
extern void vr_destroy(vregex *r);
extern char *vr_error(void);
extern vlist *vr_extract(char *string, vregex *r);
extern vregex *vr_fread(FILE *fp);
extern int vr_freeze(vregex *r, FILE *fp);
extern int vr_fwrite(vregex *r, FILE *fp);
extern vlist *vr_grep(vlist *l, vregex *r);
extern int vr_match(char *str, vregex *r);
extern char *vr_matched(int num);
extern void vr_print(vregex *r, FILE *fp);
extern char *vr_regexp(vregex *r);
extern char *vr_subst(char *str, vregex *r, char *sub);
extern vregex *vr_thaw(FILE *fp);
extern int vr_traverse(vregex *r, int (*func)(void *ptr));

#ifdef __cplusplus
}
#endif

#endif
