/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @defgroup type Type declaration functions

  These functions are used to extend the library; see @ref extend for more
  details.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "vars-config.h"
#include "vars-macros.h"

#include "vars-array.h"
#include "vars-buffer.h"
#include "vars-func.h"
#include "vars-graph.h"
#include "vars-hash.h"
#include "vars-image.h"
#include "vars-list.h"
#include "vars-matrix.h"
#include "vars-parser.h"
#include "vars-queue.h"
#include "vars-regex.h"
#include "vars-scalar.h"
#include "vars-vector.h"

/* Vars magic number */
#define MAGIC 0xff00ee11

/* Grow a list by malloc() or realloc() */
#define GROW_LIST(type, list, size, num, grow, val)                     \
        if (size == 0) {                                                \
                size = grow;                                            \
                list = (type *) malloc(size * sizeof(type));            \
        } else if (num == size - 1) {                                   \
                size += grow;                                           \
                list = (type *) realloc(list, size * sizeof(type));     \
        }                                                               \
        list[num++] = val

/* Unknown type */
vtype *vunknown_type = NULL;

/* List of declared types */
static int num_types = 0;
static int max_types = 0;
static vtype **type_list = NULL;

/* List of type headers */
static int num_hdr = 0;
static int max_hdr = 0;
static vheader **hdr_list = NULL;

/* List of pointers requiring cleanup */
static int num_clean = 0;
static int max_clean = 0;
static void **clean_list = NULL;

/* Mark a pointer as requiring cleanup on exit */
void
v_cleanup(void *ptr)
{
    GROW_LIST(void *, clean_list, max_clean, num_clean, 20, ptr);
}

/* Exit from program, cleaning up internal memory */
void
v_exit(int status)
{
    void *ptr;
    vtype *t;
    int i;

    /* Disable debugging checks */
    v_debug_flags = 0;

    /* Destroy stuff marked with v_cleanup() in reverse order */
    while (num_clean > 0) {
        ptr = clean_list[--num_clean];
        t = vt_type(ptr);

        if (t == vunknown_type)
            free(ptr);
        else if (t->destroy != NULL)
            (t->destroy)(ptr);
    }

    if (clean_list != NULL)
        free(clean_list);

    /* Call specific cleanup functions */
    for (i = 0; i < num_types; i++) {
        t = type_list[i];
        if (t->cleanup != NULL)
            (*t->cleanup)();
    }

    /* Dispose of type information */
    if (num_types > 0) {
        for (i = 0; i < num_types; i++) {
            t = type_list[i];
            free(t->name);
            free(t->code);
            free(t);
        }

        free(type_list);
    }

    if (num_hdr > 0) {
        for (i = 0; i < num_hdr; i++)
            free(hdr_list[i]);
        free(hdr_list);
    }

    /* Bye bye */
    exit(status);
}

/* Set cleanup function for a variable type */
void
vt_cleanup_with(vtype *t, void (*func)())
{
    t->cleanup = func;
}

/*!
  @brief   Set copy function for a variable type.
  @ingroup type
  @param   t Type.
  @param   func Function.
*/
void
vt_copy_with(vtype *t, void *(*func)())
{
    t->copy = func;
}

/*!
  @brief   Create a new variable type.
  @ingroup type
  @param   name Variable name.
  @param   code Short code.
  @return  New type.
*/
vtype *
vt_create(char *name, char *code)
{
    vtype *t;

    /* Check name is valid */
    if (name == NULL)
        v_fatal("vt_create(): no type name specified");
    if (code == NULL)
        v_fatal("vt_create(): no type code specified");
    if (vt_find(code) != NULL)
        v_fatal("vt_create(): type %s already exists with code %s",
                name, code);

    /* Create new type */
    t = (vtype *) malloc(sizeof(vtype));

    t->name = strdup(name);
    t->code = strdup(code);
    t->read = NULL;
    t->write = NULL;
    t->freeze = NULL;
    t->thaw = NULL;
    t->print = NULL;
    t->traverse = NULL;
    t->destroy = NULL;
    t->cleanup = NULL;

    /* Add it to the type list */
    GROW_LIST(vtype *, type_list, max_types, num_types, 10, t);

    if (V_DEBUG(V_DBG_INFO))
        v_info("Created %s type", t->name);

    return t;
}

/*!
  @brief   Declare all the standard types.
  @ingroup type
*/
void
vt_declare(void)
{
    vs_declare();
    vl_declare();
    vh_declare();
    vb_declare();
    vq_declare();
    vg_declare();
    vr_declare();
    vf_declare();
    vp_declare();
    vv_declare();
    va_declare();
    vm_declare();
    vi_declare();
}

/*!
  @brief   Set destruction function for a variable type.
  @ingroup type
  @param   t Type.
  @param   func Function.
*/
void
vt_destroy_with(vtype *t, void (*func)())
{
    t->destroy = func;
}

/* Return a type given its code */
vtype *
vt_find(char *code)
{
    int i;

    for (i = 0; i < num_types; i++)
        if (V_STREQ(code, type_list[i]->code))
            return type_list[i];

    return NULL;
}

/* Return a type given its name */
vtype *
vt_find_name(char *name)
{
    int i;

    for (i = 0; i < num_types; i++)
        if (V_STREQ(name, type_list[i]->name))
            return type_list[i];

    return NULL;
}

/*!
  @brief   Set freeze function for a variable type.
  @ingroup type
  @param   t Type.
  @param   func Function.
*/
void
vt_freeze_with(vtype *t, int (*func)())
{
    t->freeze = func;
}

/*!
  @brief   Return an initialised header for a type.
  @ingroup type
  @param   t Type.
  @return  Header.
*/
vheader *
vt_header(vtype *t)
{
    vheader *h;

    h = (vheader *) malloc(sizeof(vheader));
    GROW_LIST(vheader *, hdr_list, max_hdr, num_hdr, 10, h);

    h->magic = MAGIC;
    h->type = t;

    return h;
}

/* Return the name of a type */
char *
vt_name(vtype *t)
{
    return (t != NULL ? t->name : "NULL");
}


/*!
  @brief   Set print function for a variable type.
  @ingroup type
  @param   t Type.
  @param   func Function.
*/
void
vt_print_with(vtype *t, void (*func)())
{
    t->print = func;
}

/*!
  @brief   Return the name of an unknown pointer.
  @ingroup type
  @param   ptr Pointer.
  @return  Type name.
*/
char *
vt_ptrname(void *ptr)
{
    vheader *id;

    if (ptr == NULL)
	return "NULL";

    id = (vheader *) ptr;
    if (id->magic != MAGIC)
        return "UNKNOWN";

    if (id->type != NULL)
        return id->type->name;

    return "WEIRD";
}

/*!
  @brief   Set read function for a variable type.
  @ingroup type
  @param   t Type.
  @param   func Function.
*/
void
vt_read_with(vtype *t, void *(*func)())
{
    t->read = func;
}

/*!
  @brief   Set thaw function for a variable type.
  @ingroup type
  @param   t Type.
  @param   func Function.
*/
void
vt_thaw_with(vtype *t, void *(*func)())
{
    t->thaw = func;
}


/*!
  @brief   Set traversal function for a variable type.
  @ingroup type
  @param   t Type.
  @param   func Function.
*/
void
vt_traverse_with(vtype *t, int (*func)())
{
    t->traverse = func;
}

/*!
  @brief   Return the type of an unknown pointer.
  @ingroup type
  @param   ptr Pointer.
  @return  Type.
  @retval  NULL if pointer is \c NULL.
*/
vtype *
vt_type(void *ptr)
{
    vheader *id;

    if (vunknown_type == NULL)
        vunknown_type = vt_create("UNKNOWN", "U");

    if (ptr == NULL)
	return NULL;

    id = (vheader *) ptr;
    if (id->magic != MAGIC)
        return vunknown_type;

    return id->type;
}

/*!
  @brief   Set write function for a variable type.
  @ingroup type
  @param   t Type.
  @param   func Function.
*/
void
vt_write_with(vtype *t, int (*func)())
{
    t->write = func;
}
