/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @file
  @brief Type declaration functions and macros.
  @ingroup type
*/

#ifndef VARS_TYPE_H
#define VARS_TYPE_H

/*! @brief Check whether a pointer is unknown. */
#define v_isunknown(ptr)        (ptr != NULL && vt_type(ptr) == vunknown_type)

/* Type definition */
struct v_type {
    char *name;                 /* Type name */
    char *code;                 /* Type code */
    void *(*copy)();            /* Copy function */
    void (*print)();            /* Print function */
    void *(*read)();            /* Read function */
    int (*write)();             /* Write function */
    int (*freeze)();            /* Freeze function */
    void *(*thaw)();            /* Thaw function */
    int (*traverse)();          /* Traversal function */
    void (*destroy)();          /* Destruction function */
    void (*cleanup)();          /* Cleanup function */
};

struct v_header {
    int magic;                  /* Magic number */
    struct v_type *type;        /* Variable type */
};

/*! @brief Variable-type type. */
typedef struct v_type vtype;

/*! @brief Type header type. */
typedef struct v_header vheader;

/*! @brief Unknown type variable. */
extern vtype *vunknown_type;

/* Include auxiliary type stuff */
#include <vars-debug.h>
#include <vars-destroy.h>
#include <vars-freeze.h>
#include <vars-print.h>
#include <vars-traverse.h>
#include <vars-utils.h>
#include <vars-write.h>

#ifdef __cplusplus
extern "C" {
#endif

extern void v_cleanup(void *ptr);
extern void v_exit(int status);
extern void vt_cleanup_with(vtype *t, void (*func)());
extern void vt_copy_with(vtype *t, void *(*func)());
extern vtype *vt_create(char *name, char *code);
extern void vt_declare(void);
extern void vt_destroy_with(vtype *t, void (*func)());
extern vtype *vt_find(char *code);
extern vtype *vt_find_name(char *name);
extern void vt_freeze_with(vtype *t, int (*func)());
extern vheader *vt_header(vtype *t);
extern char *vt_name(vtype *t);
extern void vt_print_with(vtype *t, void (*func)());
extern char *vt_ptrname(void *ptr);
extern void vt_read_with(vtype *t, void *(*func)());
extern void vt_thaw_with(vtype *t, void *(*func)());
extern void vt_traverse_with(vtype *t, int (*func)());
extern vtype *vt_type(void *ptr);
extern void vt_write_with(vtype *t, int (*func)());

#ifdef __cplusplus
}
#endif

#endif
