/*
 * This file is part of the Vars library, copyright (C) Glenn
 * Hutchings 1996-2003.
 *
 * The Vars library comes with ABSOLUTELY NO WARRANTY.  This is free
 * software, and you are welcome to redistribute it under certain
 * conditions; see the file COPYING for details.
 */

/*!
  @defgroup util Utility functions
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

#include "vars-config.h"
#include "vars-debug.h"
#include "vars-macros.h"
#include "vars-memory.h"
#include "vars-utils.h"

#ifdef HAVE_LIBZ
#include <zlib.h>
#endif

/* Hook functions */
static void (*fatal_hook)(char *msg) = NULL;
static void (*exception_hook)(char *msg) = NULL;

/* Buffer for setjmp/longjmp */
jmp_buf catch_buf;

/* Current catch flag (0 means ready for throw) */
int catch_flag = 1;

/* Scribble buffer */
static char buf[BUFSIZ];

/*!
  @brief   Chop the last character from a string and return it.
  @ingroup util
  @param   str String.
  @return  Character removed.
*/
int
v_chop(char *str)
{
    int len = strlen(str);
    int c = EOF;

    if (len > 0) {
        c = str[len - 1];
        str[len - 1] = '\0';
    }

    return c;
}

/*!
  @brief   Compress some data.
  @ingroup util
  @param   data Data to compress.
  @param   size Size of the data.
  @param   csize Pointer to compressed size (returned).
  @return  Compressed data.
  @retval  NULL if it failed.
  @note    This function requires the zlib library.

  Use v_uncompress() to uncompress the data again.  It's up to you to store
  the compressed and uncompressed sizes for passing to that function.
*/
unsigned char *
v_compress(unsigned char *data, unsigned long size, unsigned long *csize)
{
#ifdef HAVE_LIBZ
    unsigned char *cdata;

    *csize = (unsigned long) (size * 1.001 + 12);
    cdata = V_ALLOC(unsigned char, *csize);

    if (compress(cdata, csize, data, size) != Z_OK) {
        V_DEALLOC(cdata);
        return NULL;
    }

    return cdata;
#else
    v_unavailable("v_compress()");
    return NULL;
#endif
}

/*!
  @brief   Give an error message and die.
  @ingroup util
  @ingroup fmt Format string.
*/
void
v_die(char *fmt, ...)
{
    V_VPRINT(buf, fmt);
    fprintf(stderr, "%s\n", buf);
    exit(1);
}

/*!
  @brief   Give an exception.
  @ingroup util
  @param   fmt Format string.

  The default action is to do nothing.  This can be changed using
  v_exception_hook().
*/
void
v_exception(char *fmt, ...)
{
    if (exception_hook != NULL) {
        V_VPRINT(buf, fmt);
        exception_hook(buf);
    }
}

/*!
  @brief   Set hook function for exceptions.
  @ingroup util
  @param   func Function (or \c NULL to unset it).
  @see     v_exception()
*/
void
v_exception_hook(void (*func)(char *msg))
{
    exception_hook = func;
}

/*!
  @brief   Give a fatal error and die.
  @ingroup util
  @param   fmt Format string.

  The default action is to print the message to \c stderr.  This can be
  changed using v_fatal_hook().
*/
void
v_fatal(char *fmt, ...)
{
    V_VPRINT(buf, fmt);

    if (fatal_hook == NULL)
        fprintf(stderr, "Vars fatal: %s\n", buf);
    else
        fatal_hook(buf);

    exit(2);
}

/*!
  @brief   Set hook function for fatal errors.
  @ingroup util
  @param   func Function (or \c NULL to unset it).
  @see     v_fatal()
*/
void
v_fatal_hook(void (*func)(char *msg))
{
    fatal_hook = func;
}

/* Fast conversion of pointer to hex string */
char *
v_hexstring(void *ptr, char *buf)
{
    static char *letters = "0123456789abcdef";
    unsigned long num = (unsigned long) ptr;
    static char hbuf[V_HEXSTRING_SIZE + 1];
    char *cp;

    if (buf == NULL)
        buf = hbuf;

    cp = &buf[V_HEXSTRING_SIZE];
    *cp = '\0';

    while (num) {
        *--cp = letters[num % 16];
        num /= 16;
    }

    return cp;
}

/*!
  @brief   Return a copy of a string.
  @ingroup util
  @param   str String.
  @return  Copy.
*/
char *
v_strdup(char *str)
{
    char *s;

    if (str == NULL)
        return NULL;

    s = V_ALLOC(char, strlen(str) + 1);
    strcpy(s, str);

    return s;
}

/* Throw to the last catch */
void
v_throw(int val)
{
    if (catch_flag) {
        v_fatal("v_throw(): no catch to throw to");
        return;
    }

    if (val == 0) {
        v_fatal("v_throw(): thrown value cannot be zero");
        return;
    }

    longjmp(catch_buf, val);
}

/* Give an unavailable-function error and die */
void
v_unavailable(char *name)
{
    v_fatal("function '%s' is not available on this system", name);
}

/*!
  @brief   Uncompress some data.
  @ingroup util
  @param   cdata Compressed data.
  @param   csize Size of compressed data (as returned by v_compress()).
  @param   size Uncompressed size (as passed to v_compress()).
  @return  Uncompressed data
  @retval  NULL if it failed.
  @note    This function requires the zlib library.

  Uncompresses data that was compressed using v_compress().
*/
unsigned char *
v_uncompress(unsigned char *cdata, unsigned long csize, unsigned long size)
{
#ifdef HAVE_LIBZ
    unsigned char *data;

    data = V_ALLOC(unsigned char, size);

    if (uncompress(data, &size, cdata, csize) != Z_OK) {
        V_DEALLOC(data);
        return NULL;
    }

    return data;
#else
    v_unavailable("v_uncompress()");
    return NULL;
#endif
}

/*!
  @brief   Write a warning message to \c stderr.
  @ingroup util
  @param   fmt Format string.
*/
void
v_warn(char *fmt, ...)
{
    V_VPRINT(buf, fmt);
    fprintf(stderr, "%s\n", buf);
}

/* Give an internal warning message */
void
v_warn_internal(char *fmt, ...)
{
    V_VPRINT(buf, fmt);
    fprintf(stderr, "Vars warning: %s\n", buf);
}
