/* Vars Python interface, in -*- C -*- mode */

%module vars

%include "vars-swig.i"

/* Declare conversion functions */
%define VARS_CONVERT(prefix, name, type)
%inline %{
type *prefix ## _cast(void *ptr) {
    if (!v_is ## name ## (ptr))
        v_exception("%s_cast(): not a %s object", #prefix, #type);
    return ptr;
}

type * ## prefix ## _object(PyObject *obj) {
    return prefix ## _cast(v_from_object(obj));
}
%}
%enddef

/* Declare extension functions */
%define VARS_EXTEND(prefix, name, type)
type *__copy__(void)   {return prefix ## _copy(self);}
PyObject *object(void) {return v_to_object(self);}
%enddef

%inline %{
/* Convert Vars pointer to Python object */
PyObject *
v_to_object(void *ptr)
{
    PyObject *obj, *obj2;
    char name[80], *key;
    double x, fx;
    int i, len;
    vscalar *s;
    float val;
    varray *a;
    vlist *l;
    vhash *h;
    vfunc *f;

    if        (ptr == NULL) {
        /* NULL -> None */
        obj = Py_None;
    } else if (v_isscalar(ptr)) {
        /* Scalar -> single object */
        s = (vscalar *) ptr;
        switch (vs_type(s)) {
        case V_INT:
            obj = PyInt_FromLong(vs_iget(s));
            break;
        case V_FLOAT:
        case V_DOUBLE:
            obj = PyFloat_FromDouble(vs_dget(s));
            break;
        case V_STRING:
            obj = PyString_FromString(vs_sgetref(s));
            break;
        case V_POINTER:
            obj = v_to_object(vs_pget(s));
            break;
        default:
            obj = Py_None;
            break;
        }
    } else if (v_islist(ptr)) {
        /* List -> list */
        l = (vlist *) ptr;
        len = vl_length(l);
        obj = PyList_New(len);

        for (i = 0; i < len; i++) {
            s = vl_get(l, i);
            PyList_SetItem(obj, i, v_to_object(s));
        }
    } else if (v_ishash(ptr)) {
        /* Hash -> dictionary */
        h = (vhash *) ptr;
        obj = PyDict_New();
        vh_foreach(key, s, h)
            PyDict_SetItemString(obj, key, v_to_object(s));
    } else if (v_isfunc(ptr)) {
        /* Function -> tuple of tuples */
        f = (vfunc *) ptr;
        len = vf_point_count(f);
        obj = PyTuple_New(len);

        for (i = 0; i < len; i++) {
            vf_get_point(f, i, &x, &fx);

            obj2 = PyTuple_New(2);
            PyTuple_SetItem(obj2, 0, PyFloat_FromDouble(x));
            PyTuple_SetItem(obj2, 1, PyFloat_FromDouble(fx));

            Py_INCREF(obj2);
            PyTuple_SetItem(obj, i, obj2);
        }
    } else if (v_isarray(ptr)) {
        /* Array -> tuple of float-object mappings */
        a = (varray *) ptr;
        len = va_point_count(a);
        obj = PyTuple_New(len);

        for (i = 0; i < len; i++) {
            va_get_point(a, i, &val, &s);

            obj2 = PyTuple_New(2);
            PyTuple_SetItem(obj2, 0, PyFloat_FromDouble(val));
            PyTuple_SetItem(obj2, 1, v_to_object(s));

            Py_INCREF(obj2);
            PyTuple_SetItem(obj, i, obj2);
        }
    } else {
        /* Everything else -> CObject */
        obj = PyCObject_FromVoidPtr(ptr, NULL);
    }

    Py_INCREF(obj);
    return obj;
}

/* Convert Python object to Vars pointer */ 
void * 
v_from_object(PyObject *obj)
{
    PyObject *repr, *pkey, *pvalue;
    int pos = 0, len, i;
    vscalar *s;
    void *ptr;
    char *key;
    vhash *h;
    vlist *l;

    if        (obj == Py_None) {
        /* None -> UNDEF scalar */
        return vs_create(V_UNDEF);
    } else if (PyInt_Check(obj)) {
        /* Integer -> integer scalar */
        return vs_icreate(PyInt_AsLong(obj));
    } else if (PyFloat_Check(obj)) {
        /* Float -> double scalar */
        return vs_dcreate(PyFloat_AsDouble(obj));
    } else if (PyString_Check(obj)) {
        /* String -> string scalar */
        return vs_screate(PyString_AsString(obj));
    } else if (PySequence_Check(obj)) {
        /* Sequence -> list */
        l = vl_create();
        len = PySequence_Size(obj);

        for (i = 0; i < len; i++) {
            pkey = PySequence_GetItem(obj, i);
            ptr = v_from_object(pkey);

            if (v_isscalar(ptr))
                vl_store(l, i, (vscalar *) ptr);
            else
                vl_pstore(l, i, ptr);

            Py_DECREF(pkey);
        }

        return l;
    } else if (PyDict_Check(obj)) {
        /* Dictionary -> hash */
        h = vh_create();

        while (PyDict_Next(obj, &pos, &pkey, &pvalue)) {
            repr = PyObject_Str(pkey);
            key = PyString_AsString(repr);
            ptr = v_from_object(pvalue);

            if (v_isscalar(ptr))
                vh_store(h, key, (vscalar *) ptr);
            else
                vh_pstore(h, key, ptr);

            Py_DECREF(repr);
        }

        return h;
    }

    repr = PyObject_Repr(obj);
    s = vs_screate(PyString_AsString(repr));
    Py_DECREF(repr);

    v_exception("can't convert %s to Vars object", vs_sgetref(s));
    return s;
}

/* Convert image gray data to Python string object */
static PyObject *vi_gray_object(vimage *image) {
    unsigned char *data = vi_get_graydata(image);
    int height = vi_get_height(image);
    int width = vi_get_width(image);
    PyObject *obj = PyString_FromStringAndSize(data, width * height);
    Py_INCREF(obj);
    return obj;
}

/* Convert image RGB data to Python string object */
static PyObject *vi_rgb_object(vimage *image) {
    unsigned char *data = vi_get_rgbdata(image);
    int height = vi_get_height(image);
    int width = vi_get_width(image);
    PyObject *obj = PyString_FromStringAndSize(data, width * height * 3);
    Py_INCREF(obj);
    return obj;
}
%}

%include "vars-defs.i"

%extend v_scalar {
    char *__str__() {return vs_sget(self);}
}
%pythoncode %{
## Extra Vars python code.

def v_thaw_object(file):
    "Thaw a Python object from a Vars freeze file."
    ptr = v_thaw_file(file)
    return v_to_object(ptr)

def v_freeze_object(obj, file):
    "Freeze a Python object to a Vars freeze file."
    ptr = v_from_object(obj)
    return v_freeze_file(ptr, file)

# Convert Vars image to PIL image.
def _create_image(image):
    import Image, ImageOps

    if image.has_rgb():
        mode = "RGB"
        data = vi_rgb_object(image)
    else:
        mode = "L"
        data = vi_gray_object(image)

    size = image.get_size()
    im = Image.fromstring(mode, size, data)
    return ImageOps.flip(im)

v_image.image = _create_image
%}
