/* Common test stuff */

#ifndef TEST_H
#define TEST_H

#include <stdio.h>
#include <string.h>
#include <vars.h>
#include <math.h>

/*
 * You must use this at the start of the tests.  If your test source file
 * is called 'test-XXX.c', set name as XXX.
 */
#define TEST_START(name)                                                \
        test_name = name;                                               \
        v_debug_env();                                                  \
        sprintf(input, "%s/test-%s.in", SRCDIR, name);                  \
        sprintf(output, "%s/test-%s.out", BUILDDIR, name);              \
        printf("\nRUN: test-%s\n", name)

/*
 * Use this macro to perform tests.  The 'desc' argument is a short
 * description of the test.  The 'passcond' argument is a C condition
 * which should be zero only if the test failed.
 */
#define TEST(desc, passcond)                                            \
        printf("TEST: %s... ", desc);                                   \
        fflush(stdout);                                                 \
        if (passcond) {                                                 \
                printf("ok\n");                                         \
        } else {                                                        \
                printf("failed\n");                                     \
                code = 1;                                               \
        }

/*
 * Like the previous macro, except abort the rest of the tests if it fails.
 */
#define TEST_NOFAIL(desc, passcond)                                     \
        TEST(desc, passcond);                                           \
        if (code) TEST_ABORT

#define TEST_FAIL(msg)                                                  \
        {                                                               \
                printf("FAIL: %s\n", msg);                              \
                code = 1;                                               \
                TEST_ABORT;                                             \
        }

/*
 * You must use this at the end of the tests, to return the appropriate code.
 */
#define TEST_FINISH \
        v_exit(code);

/*
 * If you want to refer to an input file, use this macro.  Make sure the
 * input file is called 'test-XXX.in'.
 */
#define TEST_INPUT \
        input

/*
 * Similarly, use this for any output files, called 'test-XXX.out'.
 */
#define TEST_OUTPUT \
        output

/*
 * If you need more than one input file, then make 'test-XXX.in' a
 * directory, and use this macro to get the name of files in it.  Also
 * add the directory name to the DIRS variable in Makefile.am.
 */
#define TEST_INPUT_FILE(name) \
        (sprintf(infile, "%s.%s", TEST_INPUT, name), infile)

/*
 * Same as TEST_INPUT_FILE, but for output files.
 */
#define TEST_OUTPUT_FILE(name) \
        (sprintf(outfile, "%s.%s", TEST_OUTPUT, name), outfile)

/*
 * If tests can't be done (e.g. due to a feature missing on the machine),
 * use this macro to skip them.
 */
#define TEST_SKIP(reason)                                               \
        {                                                               \
                printf("SKIP: %s\n", reason);                           \
                TEST_FINISH;                                            \
        }

/*
 * If tests can't continue, due to previous failures, use this macro to
 * abort them.
 */
#define TEST_ABORT                                                      \
        {                                                               \
                printf("ABORT: can't continue tests\n");                \
                TEST_FINISH;                                            \
        }

/*
 * Use this macro to enable debugging of the test program.
 */
#define TEST_DEBUG \
        debug_flag = 1

/*
 * You can use this for debugging tests -- it prints a Vars variable to
 * stdout (only if debugging is enabled).
 */
#define TEST_PRINT(var) \
        if (debug_flag) v_print(var, stdout)

#define BUFFER_SIZE 80

/* Name of the test */
static char *test_name;

/* Input name */
static char input[BUFFER_SIZE];

/* Output name */
static char output[BUFFER_SIZE];

/* Sub-directory input filename */
static char infile[BUFFER_SIZE];

/* Sub-directory output filename */
static char outfile[BUFFER_SIZE];

/* Whether debugging tests */
static int debug_flag = 0;

/* Test exit code */
static int code = 0;

/* Scribble buffer */
static char buf[BUFSIZ];

#endif
