#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include <iiimp-data.h>

#include "iiimp-dataP.h"
#include "print-misc.h"


IIIMP_annotation *
iiimp_annotation_new(
    IIIMP_data_s *		data_s,
    IIIMP_card32		id,
    IIIMP_annotation_value *	value)
{
    IIIMP_annotation *		anno;
    IIIMP_annotation_value *	anno_v;

    anno = (IIIMP_annotation *)malloc(sizeof (IIIMP_annotation));
    if (NULL == anno) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    anno->nbyte = (4 + 4);
    anno->id = id;
    anno->value = value;
    anno->next = NULL;
    anno->value_nbyte = 0;

    for (anno_v = value; NULL != anno_v; anno_v = anno_v->next) {
	anno->value_nbyte += anno_v->nbyte;
	anno->nbyte += anno_v->nbyte;
    }

    return anno;
}


void
iiimp_annotation_delete(IIIMP_data_s * data_s, IIIMP_annotation * anno)
{
    if (NULL == anno) return;
    switch (anno->id) {
    case IIIMP_ANNOTATION_TEXT_INPUT_BEFORE_CONVERSION:
    case IIIMP_ANNOTATION_READING_OF_THE_TEXT:
	iiimp_annotation_text_delete(data_s, anno->value);
	break;
    case IIIMP_ANNOTATION_TYPE_OF_SPEECH:
    case IIIMP_ANNOTATION_TYPE_OF_CLAUSE:
	iiimp_annotation_string_delete(data_s, anno->value);
	break;
    default:
	break;
    }
    free(anno);
    return;
}


void
iiimp_annotation_pack(
    IIIMP_data_s *	data_s,
    IIIMP_annotation *	m,
    size_t *		nbyte,
    uchar_t **		ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->id, rest, p, data_s->byte_swap);

    switch (m->id) {
    case IIIMP_ANNOTATION_TEXT_INPUT_BEFORE_CONVERSION:
    case IIIMP_ANNOTATION_READING_OF_THE_TEXT:
	iiimp_annotation_text_pack(data_s, m->value, &rest, &p);
	break;
    case IIIMP_ANNOTATION_TYPE_OF_SPEECH:
    case IIIMP_ANNOTATION_TYPE_OF_CLAUSE:
	iiimp_annotation_string_pack(data_s, m->value, &rest, &p);
	break;
    default:
	break;
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


void
iiimp_annotation_list_pack(
    IIIMP_data_s *	data_s,
    IIIMP_annotation *	m,
    size_t *		nbyte,
    uchar_t **		ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    for (; NULL != m; m = m->next) {
	iiimp_annotation_pack(data_s, m, &rest, &p);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


IIIMP_annotation *
iiimp_annotation_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_annotation *	anno;
    size_t		rest;
    const uchar_t *	p;
    int			len;
    IIIMP_card32	anno_id;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    GETU32(anno_id, rest, p, data_s->byte_swap);
    GETU32(len, rest, p, data_s->byte_swap);

    if ((rest < len) || (0 != (len & 0x03))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    anno = (IIIMP_annotation *)malloc(sizeof (IIIMP_annotation));
    if (NULL == anno) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    anno->id = anno_id;
    anno->next = NULL;

    switch (anno->id) {
    case IIIMP_ANNOTATION_TEXT_INPUT_BEFORE_CONVERSION:
    case IIIMP_ANNOTATION_READING_OF_THE_TEXT:
	anno->value = iiimp_annotation_text_unpack(data_s, &rest, &p, len);
	break;
    case IIIMP_ANNOTATION_TYPE_OF_SPEECH:
    case IIIMP_ANNOTATION_TYPE_OF_CLAUSE:
	anno->value = iiimp_annotation_string_unpack(data_s, &rest, &p, len);
	break;
    default:
	anno->value = NULL;
	data_s->status = IIIMP_DATA_INVALID;
	break;
    }

    if (NULL == anno->value) {
	free(anno);
	return NULL;
    }

    *nbyte -= (nbyte_max - rest);
    *ptr = p;

    return anno;
}


void
iiimp_annotation_print(
    IIIMP_data_s *	data_s,
    IIIMP_annotation *	m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp,
		  "\tannotation: %s: start_index=%d end_index=%d length=%d\n",
		  annotation_id_string_get(m->id),
		  m->value->start_index, m->value->end_index, m->value->length);

    switch (m->id) {
    case IIIMP_ANNOTATION_TEXT_INPUT_BEFORE_CONVERSION:
    case IIIMP_ANNOTATION_READING_OF_THE_TEXT:
	iiimp_text_print(data_s, m->value->v.text);
	break;
    case IIIMP_ANNOTATION_TYPE_OF_SPEECH:
    case IIIMP_ANNOTATION_TYPE_OF_CLAUSE:
	iiimp_string_print(data_s, m->value->v.string);
	break;
    }
}


/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
