#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#include <iiimp-data.h>
#include <iiimp-opcode.h>

#include "iiimp-dataP.h"

IIIMP_open *
iiimp_open_new(
    IIIMP_data_s *	        data_s,
    IIIMP_card32                ns_id,
    IIIMP_card32                flags,
    IIIMP_utf8string *	        filename)
{
    IIIMP_open *	data;
    size_t 		nbyte = 0;

    data = (IIIMP_open *) malloc(sizeof(IIIMP_open));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    nbyte = 4 + 4 + 4 + filename->nbyte;

    data->nbyte = nbyte;
    data->ns_id = ns_id;
    data->flags = flags;
    data->filename = filename;

    return data;
}

IIIMP_read *
iiimp_read_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        size)
{
    IIIMP_read *        data;
 
    data = (IIIMP_read *) malloc(sizeof(IIIMP_read));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->size = size;

    return data;
}

IIIMP_write *
iiimp_write_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        size,
    void *              object)
{
    IIIMP_write *	data;

    data = (IIIMP_write *)malloc(sizeof (IIIMP_write));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }
    data->nbyte = 4 + 4 + 4 + size + PAD(size);
    data->ns_id = ns_id;
    data->size = size;
    data->object = object;

    return data;
}

IIIMP_close *
iiimp_close_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32        ns_id)
{
    IIIMP_close *	data;

    data = (IIIMP_close *)malloc(sizeof (IIIMP_close));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }
    data->nbyte = 4 + 4;
    data->ns_id = ns_id;

    return data;
}

IIIMP_seek *
iiimp_seek_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        offset,
    IIIMP_card32        whence)
{
    IIIMP_seek *     data;

    data = (IIIMP_seek *)malloc(sizeof (IIIMP_seek));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + 4 + 4;
    data->ns_id = ns_id;
    data->offset = offset;
    data->whence = whence;

    return data;
}

IIIMP_opendir *
iiimp_opendir_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  dirname)
{
    IIIMP_opendir *     data;
    data = (IIIMP_opendir *)malloc(sizeof (IIIMP_opendir));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + dirname->nbyte;
    data->ns_id = ns_id;
    data->dirname = dirname;

    return data;
}

IIIMP_closedir *
iiimp_closedir_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id)
{
    IIIMP_closedir *     data;
    data = (IIIMP_closedir *)malloc(sizeof (IIIMP_closedir));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4;
    data->ns_id = ns_id;

    return data;
}

IIIMP_mkdir *
iiimp_mkdir_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        mode,
    IIIMP_utf8string *  path)
{
    IIIMP_mkdir *     data;

    data = (IIIMP_mkdir *)malloc(sizeof (IIIMP_mkdir));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->mode = mode;
    data->path = path;

    return data;
}

IIIMP_rmdir *
iiimp_rmdir_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  path)
{
    IIIMP_rmdir *     data;

    data = (IIIMP_rmdir *)malloc(sizeof (IIIMP_rmdir));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->path = path;

    return data;
}

IIIMP_symlink *
iiimp_symlink_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  dest,
    IIIMP_utf8string *  src)
{
    IIIMP_symlink *     data;

    data = (IIIMP_symlink *)malloc(sizeof (IIIMP_symlink));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + dest->nbyte + src->nbyte;
    data->ns_id = ns_id;
    data->dest = dest;
    data->src = src;

    return data;
}

IIIMP_stat *
iiimp_stat_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  path)
{
    IIIMP_stat *     data;

    data = (IIIMP_stat *)malloc(sizeof (IIIMP_stat));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->path = path;

    return data;
}

IIIMP_lstat *
iiimp_lstat_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  path)
{
    IIIMP_lstat *     data;

    data = (IIIMP_lstat *)malloc(sizeof (IIIMP_lstat));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->path = path;

    return data;
}

IIIMP_fstat *
iiimp_fstat_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id)
{
    IIIMP_fstat *     data;

    data = (IIIMP_fstat *)malloc(sizeof (IIIMP_fstat));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4;
    data->ns_id = ns_id;

    return data;
}

IIIMP_creat *
iiimp_creat_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        mode,
    IIIMP_utf8string *  path)
{
    IIIMP_creat *     data;

    data = (IIIMP_creat *)malloc(sizeof (IIIMP_creat));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->mode = mode;
    data->path = path;

    return data;
}

IIIMP_lseek *
iiimp_lseek_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        offset,
    IIIMP_card32        whence)
{
    IIIMP_lseek *     data;

    data = (IIIMP_lseek *)malloc(sizeof (IIIMP_lseek));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + 4;
    data->ns_id = ns_id;
    data->offset = offset;
    data->whence = whence;

    return data;
}

IIIMP_unlink *
iiimp_unlink_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  path)
{
    IIIMP_unlink *     data;

    data = (IIIMP_unlink *)malloc(sizeof (IIIMP_unlink));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->path = path;

    return data;
}

IIIMP_rename *
iiimp_rename_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  old_name,
    IIIMP_utf8string *  new_name)
{
    IIIMP_rename *     data;

    data = (IIIMP_rename *)malloc(sizeof (IIIMP_rename));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + old_name->nbyte + new_name->nbyte;
    data->ns_id = ns_id;
    data->old_name = old_name;
    data->new_name = new_name;

    return data;
}

IIIMP_fcntl *
iiimp_fcntl_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        cmd,
    IIIMP_card32        arg)
{
    IIIMP_fcntl *     data;

    data = (IIIMP_fcntl *)malloc(sizeof (IIIMP_fcntl));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + 4;
    data->ns_id = ns_id;
    data->cmd = cmd;
    data->arg = arg;

    return data;
}

IIIMP_truncate *
iiimp_truncate_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  path,
    IIIMP_card32        length)
{
    IIIMP_truncate *     data;

    data = (IIIMP_truncate *)malloc(sizeof (IIIMP_truncate));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->path = path;
    data->length = length;

    return data;
}

IIIMP_ftruncate *
iiimp_ftruncate_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        length)
{
    IIIMP_ftruncate *     data;
    data = (IIIMP_ftruncate *)malloc(sizeof (IIIMP_ftruncate));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->length = length;

    return data;
}

void
iiimp_open_delete(IIIMP_data_s * data_s, IIIMP_open *m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->filename);
    free(m);
    return;
}

void
iiimp_read_delete(IIIMP_data_s * data_s, IIIMP_read * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_write_delete(IIIMP_data_s * data_s, IIIMP_write * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_close_delete(IIIMP_data_s * data_s, IIIMP_close * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_seek_delete(IIIMP_data_s * data_s, IIIMP_seek * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_opendir_delete(IIIMP_data_s * data_s, IIIMP_opendir * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->dirname);
    free(m);
    return;
}

void
iiimp_closedir_delete(IIIMP_data_s * data_s, IIIMP_closedir * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_mkdir_delete(IIIMP_data_s * data_s, IIIMP_mkdir * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_rmdir_delete(IIIMP_data_s * data_s, IIIMP_rmdir * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_symlink_delete(IIIMP_data_s * data_s, IIIMP_symlink * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->dest);
    iiimp_utf8string_delete(data_s, m->src);
    free(m);
    return;
}

void
iiimp_stat_delete(IIIMP_data_s * data_s, IIIMP_stat * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_fstat_delete(IIIMP_data_s * data_s, IIIMP_fstat * m)
{
    if (NULL == m) return;
    free(m);
    return;
}


void
iiimp_lstat_delete(IIIMP_data_s * data_s, IIIMP_lstat * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_creat_delete(IIIMP_data_s * data_s, IIIMP_creat * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_lseek_delete(IIIMP_data_s * data_s, IIIMP_lseek * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_unlink_delete(IIIMP_data_s * data_s, IIIMP_unlink * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_rename_delete(IIIMP_data_s * data_s, IIIMP_rename * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->old_name);
    iiimp_utf8string_delete(data_s, m->new_name);
    free(m);
    return;
}

void
iiimp_fcntl_delete(IIIMP_data_s * data_s, IIIMP_fcntl * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_truncate_delete(IIIMP_data_s * data_s, IIIMP_truncate * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_ftruncate_delete(IIIMP_data_s * data_s, IIIMP_ftruncate * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_open_pack(
    IIIMP_data_s *	data_s,
    IIIMP_open *        m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->flags, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->filename, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_read_pack(
    IIIMP_data_s *	data_s,
    IIIMP_read *        m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->size, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_write_pack(
    IIIMP_data_s *	data_s,
    IIIMP_write *       m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;
    int                 i, count, object_size;

    object_size = m->size + PAD((m->size) % 4);
    count = (object_size / 4);

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->size, rest, p, data_s->byte_swap);

    for (i=0; i<count; i++) {
	PUTU32(*((IIIMP_card32 *)m->object + i), rest, p, data_s->byte_swap);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_close_pack(
    IIIMP_data_s *	data_s,
    IIIMP_close *       m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_seek_pack(
    IIIMP_data_s *      data_s,
    IIIMP_seek *        m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->offset, rest, p, data_s->byte_swap);
    PUTU32(m->whence, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_opendir_pack(
    IIIMP_data_s *      data_s,
    IIIMP_opendir *     m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->dirname, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_closedir_pack(
    IIIMP_data_s *      data_s,
    IIIMP_closedir *    m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_mkdir_pack(
    IIIMP_data_s *      data_s,
    IIIMP_mkdir *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->mode, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_rmdir_pack(
    IIIMP_data_s *      data_s,
    IIIMP_rmdir *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_symlink_pack(
    IIIMP_data_s *      data_s,
    IIIMP_symlink *     m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->dest, &rest, &p);
    iiimp_utf8string_pack(data_s, m->src, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_stat_pack(
    IIIMP_data_s *      data_s,
    IIIMP_stat *        m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_lstat_pack(
    IIIMP_data_s *      data_s,
    IIIMP_lstat *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_fstat_pack(
    IIIMP_data_s *      data_s,
    IIIMP_fstat *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_creat_pack(
    IIIMP_data_s *      data_s,
    IIIMP_creat *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->mode, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_lseek_pack(
    IIIMP_data_s *      data_s,
    IIIMP_lseek *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->offset, rest, p, data_s->byte_swap);
    PUTU32(m->whence, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_unlink_pack(
    IIIMP_data_s *      data_s,
    IIIMP_unlink *      m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_rename_pack(
    IIIMP_data_s *      data_s,
    IIIMP_rename *      m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->old_name, &rest, &p);
    iiimp_utf8string_pack(data_s, m->new_name, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_fcntl_pack(
    IIIMP_data_s *      data_s,
    IIIMP_fcntl *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->cmd, rest, p, data_s->byte_swap);
    PUTU32(m->arg, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_truncate_pack(
    IIIMP_data_s *      data_s,
    IIIMP_truncate *    m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->length, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_ftruncate_pack(
    IIIMP_data_s *      data_s,
    IIIMP_ftruncate *   m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->length, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

IIIMP_open *
iiimp_open_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_open *        data;
    size_t              rest;
    const uchar_t *     p;
  
    rest = nbyte_max;
    p = *ptr;
    
    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }
    
    data = (IIIMP_open *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->filename = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->flags, rest, p, data_s->byte_swap);
    data->filename = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_read *
iiimp_read_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_read *        data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_read *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->size, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_write *
iiimp_write_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_write *       data;
    IIIMP_card32 *      buf;
    int                 i, count, object_size;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    object_size = (rest - 4 - 4 - 4);
    count = (object_size / 4);

    data = (IIIMP_write *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->object = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->size, rest, p, data_s->byte_swap);

    buf = (IIIMP_card32 *)malloc(data->size);
    if (NULL == buf) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    for (i=0; i<count; i++) {
	GETU32(*(buf + i), rest, p, data_s->byte_swap);
    }
    data->object = (void *)buf;

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_close *
iiimp_close_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_close *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < ( 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_close *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_seek *
iiimp_seek_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_seek *        data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_seek *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->offset, rest, p, data_s->byte_swap);
    GETU32(data->whence, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_opendir *
iiimp_opendir_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_opendir *     data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_opendir *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->dirname = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->dirname = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_closedir *
iiimp_closedir_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_closedir *    data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_closedir *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_mkdir *
iiimp_mkdir_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_mkdir *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < ( 4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_mkdir *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->mode, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_rmdir *
iiimp_rmdir_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_rmdir *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_rmdir *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_symlink *
iiimp_symlink_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_symlink *     data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_symlink *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->dest = NULL;
    data->src = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->dest = iiimp_utf8string_unpack(data_s, &rest, &p, rest);
    data->src = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_stat *
iiimp_stat_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_stat *        data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_stat *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_lstat *
iiimp_lstat_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_lstat *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_lstat *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_fstat *
iiimp_fstat_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_fstat *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_fstat *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_creat *
iiimp_creat_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_creat *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_creat *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->mode, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_lseek *
iiimp_lseek_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_lseek *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_lseek *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->offset, rest, p, data_s->byte_swap);
    GETU32(data->whence, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_unlink *
iiimp_unlink_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_unlink *      data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_unlink *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_rename *
iiimp_rename_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_rename *      data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_rename *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->old_name = NULL;
    data->new_name = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->old_name = iiimp_utf8string_unpack(data_s, &rest, &p, rest);
    data->new_name = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_fcntl *
iiimp_fcntl_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_fcntl *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_fcntl *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->cmd, rest, p, data_s->byte_swap);
    GETU32(data->arg, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_truncate *
iiimp_truncate_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_truncate *    data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_truncate *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->length, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_ftruncate *
iiimp_ftruncate_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_ftruncate *        data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_ftruncate *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->length, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

void
iiimp_open_print(
    IIIMP_data_s *   data_s,
    IIIMP_open *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tflags = %d\n", m->flags);
    (void)fprintf(data_s->print_fp, "\tfilename = ");
    iiimp_utf8string_print(data_s, m->filename);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_read_print(
    IIIMP_data_s *   data_s,
    IIIMP_read *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tsize = %d\n", m->size);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_write_print(
    IIIMP_data_s *    data_s,
    IIIMP_write *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tsize = %d\n", m->size);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_close_print(
    IIIMP_data_s *    data_s,
    IIIMP_close *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_seek_print(
    IIIMP_data_s *    data_s,
    IIIMP_seek *      m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\toffset = %d\n", m->offset);
    (void)fprintf(data_s->print_fp, "\twhence = %d\n", m->whence);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_opendir_print(
    IIIMP_data_s *      data_s,
    IIIMP_opendir *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->dirname);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_closedir_print(
    IIIMP_data_s *      data_s,
    IIIMP_closedir *    m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_mkdir_print(
    IIIMP_data_s *      data_s,
    IIIMP_mkdir *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tmode = %d\n", m->mode);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_rmdir_print(
    IIIMP_data_s *    data_s,
    IIIMP_rmdir *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_symlink_print(
    IIIMP_data_s *      data_s,
    IIIMP_symlink *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fputc('\n', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "\tdest = ");
    iiimp_utf8string_print(data_s, m->dest);
    (void)fputc('\n', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "\tsrc = ");
    iiimp_utf8string_print(data_s, m->src);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_stat_print(
    IIIMP_data_s *   data_s,
    IIIMP_stat *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_lstat_print(
    IIIMP_data_s *    data_s,
    IIIMP_lstat *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_fstat_print(
    IIIMP_data_s *    data_s,
    IIIMP_fstat *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_creat_print(
    IIIMP_data_s *    data_s,
    IIIMP_creat *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_lseek_print(
    IIIMP_data_s *    data_s,
    IIIMP_lseek *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\toffset = %d\n", m->offset);
    (void)fprintf(data_s->print_fp, "\twhence = %d\n", m->whence);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_unlink_print(
    IIIMP_data_s *     data_s,
    IIIMP_unlink *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_rename_print(
    IIIMP_data_s *     data_s,
    IIIMP_rename *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\told name = ");
    iiimp_utf8string_print(data_s, m->old_name);
    (void)fputc('\n', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "\tnew name = ");
    iiimp_utf8string_print(data_s, m->new_name);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_fcntl_print(
    IIIMP_data_s *    data_s,
    IIIMP_fcntl *     m)
{   
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tcmd = %d\n", m->cmd);
    (void)fprintf(data_s->print_fp, "\targ = %d\n", m->arg);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_truncate_print(
    IIIMP_data_s *       data_s,
    IIIMP_truncate *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tlength = %d\n", m->length);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_ftruncate_print(
    IIIMP_data_s *        data_s,
    IIIMP_ftruncate *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tlength = %d\n", m->length);
    (void)fputc('\n', data_s->print_fp);
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
