/*  Inti-GConf: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gconf/changeset.h
//! @brief GConfChangeSet C++ wrapper interface.
//!
//! Provides ChangeSet, an object that represents a set of configuration changes to be made.

#ifndef INTI_GCONF_CHANGESET_H
#define INTI_GCONF_CHANGESET_H

#ifndef GCONF_GCONF_CHANGESET_H
#include <gconf/gconf-changeset.h>
#endif

#ifndef INTI_MEMORY_HANDLER_H
#include <inti/memoryhandler.h>
#endif
#ifndef INTI_SLOT_H
#include <inti/base.h>
#endif

#ifndef INTI_SLOT_H
#include <inti/slot.h>
#endif

#ifndef CPP_VECTOR_H
#include <vector>
#endif

namespace Inti {

namespace G {
class Error;
}

namespace GConf {

class Schema;
class Value;

//! @class ChangeSet changeset.h inti/gconf/chnageset.h
//! @brief A GConfChangeSet C++ wrapper class.
//!
//! A ChangeSet allows you to collect a set of changes to configuration keys (set/unset operations). 
//! You can then commit all the changes at once. This is convenient for something like a preferences
//! dialog; you can collect all the pending changes in a ChangeSet, then when the user clicks
//! <EM>apply</EM> send them all to the configuration database. The ChangeSet allows you to avoid
//! sending every preferences setting when "apply" is clicked; you only have to send the settings
//! the user changed.

class ChangeSet : public ReferencedBase, public MemoryHandler
{
	ChangeSet(const ChangeSet&);
	ChangeSet& operator=(const ChangeSet&);

	GConfChangeSet *cs_;
	
	static void destroy_notify(void *data);

public:
	typedef Slot2<void, String&, const Value*> ForeachSlot;
	//!< Signature for the foreach method called to iterate over each change in a ChangeSet.
	//!<
	//!< <B>Example:</B> Method signature for ForeachSlot.
	//!< @code
	//!< void method(const String& key, const Value *value);
	//!<
	//!< // key: A key whose value has been changed.
	//!< // value: The new value for <EM>key</EM>, or null if the value was unset.
	//!< @endcode
	//!<
	//!< You must not call GConf::Change::set_remove() during the iteration, because
	//!< you'll confuse the internal data structures and cause memory corruption.

//! @name Constructors
//! @{

	ChangeSet();
	//!< Construct a new ChangeSet.

	explicit ChangeSet(GConfChangeSet *cs);
	//!< Constructs a ChangeSet with an exisitng GConfChangeSet.

	~ChangeSet();

//! @}
//! @name Accessors
//! @{

	GConfChangeSet* gconf_change_set() const { return cs_; }
	//!< Get a pointer to the GConfChangeSet structure.

//! @}
//! @name Methods
//! @{

	virtual void ref();
	//!< Increases the reference count of a ChangeSet by one.

	virtual void unref();
	//!< Decreases the reference count of a ChangeSet by one; if the reference count reaches 0,
	//!< the ChangeSet is destroyed.

	void clear();
	//!< Clears all changes from a ChangeSet, so that committing the change set would have no effect.

	unsigned int size() const;
	//!< Returns the number of changes in a ChangeSet (a Change stores a key/value pair).

	void remove(const String& key);
	//!< Removes a change from a ChangeSet.
	//!< @param key The key to remove from the change set.
	//!<
	//!< <BR>The key given as the key argument will not be modified if this change set is committed.
	//!< If key is not in the change set, this function has no effect.

	void foreach(ForeachSlot *each);

	bool check_value(const String& key, Value *value = 0);
	//!< Checks whether the change set contains the given key;
	//!< @param key The key to check.
	//!< @param value  A pointer to an empty Value object.
	//!< @return true if the key is in the ChangeSet.
	//!< 
	//!< <BR>If you just want to check for a value, and don't care what it is, 
	//!< <EM>value</EM> can be null. To get the value pass a pointer to an empty
	//!< Value object. This method copies the key's value into your value object.
	//!<
	//!< <B>Example:</B> An example of check_value() in action.
	//!< @code
	//!< GConf::Value value;
	//!< if (check_value("/foo", &value) && value.is_set())
	//!< {
	//!<    // do something with the value
	//!<    ...
	//!< }
	//!< @endcode

	void set(const String& key, const Value& value);
	//!< Adds a "set" operation to a change set.
	//!< @param key The key to change.
	//!< @param value The value to change the key to.
	//!<
	//!< <BR>This method is similar to gconf_engine_set(), except that no errors can occur
	//!< (errors occur later, when you try to commit the change set).

	void unset(const String& key);
	//!< Adds an "unset" operation to a ChangeSet. 
	//!< @param key The key to change.
	//!<
	//!< This function schedules a gconf_engine_unset().

	void set_int(const String& key, int value);
	//!< Adds a "set" operation; takes an int argument, so you can avoid creating a Value.
	//!< @param key The key to set.
	//!< @param value The new value of <EM>key</EM>.

	void set_bool(const String& key, bool value);
	//!< Adds a "set" operation; takes a bool argument, so you can avoid creating a Value.
	//!< @param key The key to set.
	//!< @param value The new value of <EM>key</EM>.

	void set_float(const String& key, double value);
	//!< Adds a "set" operation; takes a double argument, so you can avoid creating a Value.
	//!< @param key The key to set.
	//!< @param value The new value of <EM>key</EM>.

	void set_string(const String& key, const String& value);
	//!< Adds a "set" operation; takes a String argument, so you can avoid creating a Value.
	//!< @param key The key to set.
	//!< @param value The new value of <EM>key</EM>.

	void set_schema(const String& key, const Schema& value);
	//!< Adds a "set" operation; takes a Schema argument, so you can avoid creating a Value.
	//!< @param key The key to set.
	//!< @param value The new value of <EM>key</EM>.

	void set_list(const String& key, const std::vector<int>& list);
 	//!< Adds a "set" operation; takes a vector of int, so you can avoid creating a list Value.
	//!< @param key The key you want set.
	//!< @param list A reference to a vector of int that contains the list elements.

	void set_list(const String& key, const std::vector<bool>& list);
 	//!< Adds a "set" operation; takes a vector of bool, so you can avoid creating a list Value.
	//!< @param key The key you want set.
	//!< @param list A reference to a vector of bool that contains the list elements.

	void set_list(const String& key, const std::vector<double>& list);
 	//!< Adds a "set" operation; takes a vector of double, so you can avoid creating a list Value.
	//!< @param key The key you want set.
	//!< @param list A reference to a vector of double that contains the list elements.

	void set_list(const String& key, const std::vector<String>& list);
 	//!< Adds a "set" operation; takes a vector of String, so you can avoid creating a list Value.
	//!< @param key The key you want set.
	//!< @param list A reference to a vector of String that contains the list elements.

	void set_list(const String& key, const std::vector<Schema>& list);
 	//!< Adds a "set" operation; takes a vector of Schema, so you can avoid creating a list Value.
	//!< @param key The key you want set.
	//!< @param list A reference to a vector of Schema that contains the list elements.

	void set_pair(const String& key, int car_data, int cdr_data);
 	//!< Adds a "set" operation; takes two ints, so you can avoid creating a pair Value.

	void set_pair(const String& key, int car_data, bool cdr_data);
 	//!< Adds a "set" operation; takes an int and a bool, so you can avoid creating a pair Value.

	void set_pair(const String& key, int car_data, double cdr_data);
 	//!< Adds a "set" operation; takes an int and a double, so you can avoid creating a pair Value.

	void set_pair(const String& key, int car_data, const String& cdr_data);
 	//!< Adds a "set" operation; takes an int and a String, so you can avoid creating a pair Value.

	void set_pair(const String& key, int car_data, const Schema& cdr_data);
 	//!< Adds a "set" operation; takes an int and a Schema, so you can avoid creating a pair Value.

	void set_pair(const String& key, bool car_data, int cdr_data);
 	//!< Adds a "set" operation; takes a bool and an int, so you can avoid creating a pair Value.

	void set_pair(const String& key, bool car_data, bool cdr_data);
 	//!< Adds a "set" operation; takes two bools, so you can avoid creating a pair Value.

	void set_pair(const String& key, bool car_data, double cdr_data);
 	//!< Adds a "set" operation; takes a bool and a double, so you can avoid creating a pair Value.

	void set_pair(const String& key, bool car_data, const String& cdr_data);
 	//!< Adds a "set" operation; takes a bool and a String, so you can avoid creating a pair Value.

	void set_pair(const String& key, bool car_data, const Schema& cdr_data);
 	//!< Adds a "set" operation; takes a bool and a Schema, so you can avoid creating a pair Value.

	void set_pair(const String& key, double car_data, int cdr_data);
 	//!< Adds a "set" operation; takes a double and a int, so you can avoid creating a pair Value.

	void set_pair(const String& key, double car_data, bool cdr_data);
 	//!< Adds a "set" operation; takes a double and a bool, so you can avoid creating a pair Value.

	void set_pair(const String& key, double car_data, double cdr_data);
 	//!< Adds a "set" operation; takes two doubles, so you can avoid creating a pair Value.

	void set_pair(const String& key, double car_data, const String& cdr_data);
 	//!< Adds a "set" operation; takes a double and a String, so you can avoid creating a pair Value.

	void set_pair(const String& key, double car_data, const Schema& cdr_data);
 	//!< Adds a "set" operation; takes a double and a Schema, so you can avoid creating a pair Value.

	void set_pair(const String& key, const String& car_data, int cdr_data);
 	//!< Adds a "set" operation; takes a String and an int, so you can avoid creating a pair Value.

	void set_pair(const String& key, const String& car_data, bool cdr_data);
 	//!< Adds a "set" operation; takes a String and a bool, so you can avoid creating a pair Value.

	void set_pair(const String& key, const String& car_data, double cdr_data);
 	//!< Adds a "set" operation; takes a String and a double, so you can avoid creating a pair Value.

	void set_pair(const String& key, const String& car_data, const String& cdr_data);
 	//!< Adds a "set" operation; takes two Strings, so you can avoid creating a pair Value.

	void set_pair(const String& key, const String& car_data, const Schema& cdr_data);
 	//!< Adds a "set" operation; takes a String and a Schema, so you can avoid creating a pair Value.

	void set_pair(const String& key, const Schema& car_data, int cdr_data);
 	//!< Adds a "set" operation; takes a Schema and a int, so you can avoid creating a pair Value.

	void set_pair(const String& key, const Schema& car_data, bool cdr_data);
 	//!< Adds a "set" operation; takes a Schema and a bool, so you can avoid creating a pair Value.

	void set_pair(const String& key, const Schema& car_data, double cdr_data);
 	//!< Adds a "set" operation; takes a Schema and a double, so you can avoid creating a pair Value.

	void set_pair(const String& key, const Schema& car_data, const String& cdr_data);
 	//!< Adds a "set" operation; takes a Schema and a String, so you can avoid creating a pair Value.

	void set_pair(const String& key, const Schema& car_data, const Schema& cdr_data);
 	//!< Adds a "set" operation; takes two Schemas, so you can avoid creating a pair Value.

//! @}
};

} // namespace GConf

} // namespace Inti

#endif // INTI_GCONF_CHANGESET_H

