#!/usr/bin/tclsh
#-----------------------------------------------------------------------#
#        IPAcco - Cisco IP accounting collector and visualizer          #
#                          Version 0.2
#             Copyright (c) 2004-2005 Dmitriy Stepanenko
#-----------------------------------------------------------------------#
# This source file is subject the license, that is bundled with this    #
# package in the file LICENSE.                                          #
# If you did not receive a copy of the Mudropolk license, please send a #
# note to mpolk@kt-privat.donetsk.ua so I can mail you a copy.          #
#-----------------------------------------------------------------------#
# Author: Dmitriy Stepanenko aka Mudropolk <mpolk@kt-privat.donetsk.ua> #
#-----------------------------------------------------------------------#
# $Id: ipacco-datapump.tcl,v 1.1 2005/04/14 11:43:55 mpolk Exp $
# Script retrieving IP-accounting data and dumping it to the database.  #
# Also packs day data to the week storage, week - to month and so on.   #
#-----------------------------------------------------------------------#

package require mysqltcl
package require math
package require cmdline
package require Tclx

# Parsing the command line and getting arguments
proc ParseCmdLine {} {
    global argv argv0 tcl_platform
    global RunMode ConfFile ConfVars
    set ConfVars {
        {DBServer "localhost"}
        {DBName "ipacco"}
        {DBUser "ipacco-pump"}
        {DBPassword "ipacco=pump"}
        {Router "cisco"}
        {LogFile ""}
        {DebugLevel 1}
    }
    foreach Var $ConfVars {
        global [lindex $Var 0]
        set [lindex $Var 0]  [lindex $Var 1]
    };#foreach
    
    set ValidOptions {
        {c.arg  ""  "Use the specified configuration file."}
        {r          "Run continously, i.e. perform data
                      processing cycles until being interrupted."}
        {d          "Run in daemon mode. Similar to -r except that
                      all console output is inhibited."}
    }
    if {[catch {array set Options [::cmdline::getoptions argv $ValidOptions]} \
              Msg]} {
        puts $Msg
        exit 1
    };#if
    
    set RunMode OnePass
    if {$Options(r)} {set RunMode ContinousRun}
    if {$Options(d)} {set RunMode Daemon}
    
    if {$Options(c) != ""} {
        set ConfFile $Options(c)
        if {! [file readable $ConfFile]} {
            print "Cannot open config file \"$ConfFile\"!"
            exit 1
        };#if
    } elseif {$tcl_platform(platform) == "windows"} {
        set ConfFile "[file rootname $argv0].cfg"
    } else {
        set ConfFile [file join "etc/ipacco" "[file tail [file rootname $argv0]].cfg"]
    };#if
};#ParseCmdLine


# Loading the configuration file
proc LoadSettings {} {
    global ConfFile ConfVars
    foreach Var $ConfVars {
        global [lindex $Var 0]
    };#foreach
    
    if {! [file exists $ConfFile]} {
        print "\nNo config file found - using defaults:"
    } else {
        if {! [file readable $ConfFile]} {
            print "\nCannot open config file \"$ConfFile\"!\a"
            exit 1
        };#if
        
        print "\nLoading settings from the config file \"$ConfFile\":"
        set CFStream [open $ConfFile RDONLY]
        while {[gets $CFStream Line] >= 0} {
            set Line [string trim $Line]
            if {$Line == "" || \
                [string index $Line 0] == "#" || \
                [string index $Line 0] == ";"} {
                continue
            };#if
            
            set MatchFound 0
            foreach Var $ConfVars {
                set Pattern "[lindex $Var 0]\\s*=\\s*(.*)"
                if {[regexp -nocase $Pattern $Line Match Value]} {
                    set [lindex $Var 0] $Value
                    set MatchFound 1
                    break
                };#if
            };#foreach
        
            if {! $MatchFound} {
                print "\nUnknown parameter:\n$Line\a"
                exit 1
            };#if
        };#while
        close $CFStream
    };#if

    foreach Var $ConfVars {
        if {[lindex $Var 0] != "DBPassword"} {
            print "  [lindex $Var 0] = [set [lindex $Var 0]]"
        } else {
            print "  [lindex $Var 0] = **********"
        };#if
    };#foreach
};#LoadSettings


# Loading the specified temporal parameter from the database
proc LoadTemporalSetting {ParamName} {
    global DB
    set Row [mysqlsel $DB "SELECT IntValue, CharValue \
                           FROM Settings WHERE Name = '$ParamName'" \
                      -flatlist] 
    set Result [lindex $Row 0]
    if {[string match -nocase "min*" [lindex $Row 1]]} {
        set Result [expr {$Result * 60}]
    } elseif {[string match -nocase "hour*" [lindex $Row 1]]} {
        set Result [expr {$Result * 3600}]
    } elseif {[string match -nocase "day*" [lindex $Row 1]]} {
        set Result [expr {$Result * 86400}]
    } elseif {[string match -nocase "week*" [lindex $Row 1]]} {
        set Result [expr {$Result * 604800}]
    } elseif {[string match -nocase "month*" [lindex $Row 1]]} {
        set Result [expr {$Result * 2592000}]
    } elseif {[string match -nocase "year*" [lindex $Row 1]]} {
        set Result [expr {$Result * 31536000}]
    } else {
        error "$ParamName units not specified!"
    };#if
    
    return $Result
};#LoadTemporalSettinng


# Procedure unloading IP accounting data from the router to the database
proc UnloadDataFromRouter {} {
    global DB Router Time DebugLevel
    set DataLine " *(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\
                   +(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\
                   +(\\d+) +(\\d+)"
    set AgeLine1 "Accounting data age is\\s+(\\d+):(\\d+)"
    set AgeLine2 "Accounting data age is\\s+(\\d+w)?(\\d+d)?(\\d+h)?(\\d+)?"
    set AgeLine3 "Accounting data age is\\s+(\\d+)"

    mysqlexec $DB "START TRANSACTION"
    mysqlexec $DB "INSERT INTO DayReadouts (Time) VALUES ('$Time')"
    set ReadoutID [mysqlinsertid $DB]
    
    exec rsh $Router -n clear ip accounting
    set Pipe [open "| rsh $Router -n show ip accounting checkpoint" r]
    fconfigure $Pipe -translation lf
    set NRecs 0
    while {! [eof $Pipe]} {
        set Line [gets $Pipe]
        if {$DebugLevel >= 2} {
            print $Line
        };#if
        set Age 0
        if {[regexp $DataLine $Line Match Src1 Src2 Src3 Src4 \
                                          Dst1 Dst2 Dst3 Dst4 \
                                          Packets Bytes]} {
            set Src $Src1.$Src2.$Src3.$Src4
            set Dst $Dst1.$Dst2.$Dst3.$Dst4
            if {$DebugLevel >= 3} {
                print "$Src => $Dst: $Packets packets, $Bytes bytes"
            };#if
            mysqlexec $DB \
                 "INSERT INTO DayData (Src1, Src2, Src3, Src4, Dst1, Dst2, Dst3, Dst4, \
                                       Packets, Bytes, ReadoutID) \
                  VALUES ($Src1, $Src2, $Src3, $Src4, $Dst1, $Dst2, $Dst3, $Dst4, \
                          $Packets, $Bytes, $ReadoutID)"
	    incr NRecs
        } elseif {[regexp $AgeLine1 $Line Match Hours Minutes]} {
            set Age [expr {$Hours * 60 + $Minutes}]
        } elseif {[regexp $AgeLine2 $Line Match Weeks Days Hours Minutes]} {
            if {$Weeks != ""} {
                incr Age [expr {[string range $Weeks 0 end-1] * 60 * 24 * 7}]
            }
            if {$Days != ""} {
                incr Age [expr {[string range $Days 0 end-1] * 60 * 24}]
            }
            if {$Hours != ""} {
                incr Age [expr {[string range $Hours 0 end-1] * 60}]
            }
            if {$Minutes != ""} {
                incr Age $Minutes
            }
        } elseif {[regexp $AgeLine3 $Line Match Minutes]} {
            set Age $Minutes
        };#if
        
        if {$Age} {
            log "Data age = $Age"
	    log "$NRecs records"
            mysqlexec $DB "UPDATE DayReadouts SET Age = $Age WHERE ID = $ReadoutID"
        };#if
    };#while
    catch {close $Pipe}
    
    mysqlexec $DB "COMMIT"
};#UnloadDataFromRouter


# Procedure packing day data to the week table
proc PackData {From To} {
    global DB DebugLevel
    
    set OutputViewGrid [LoadTemporalSetting "${To}ViewGrid"]
    set LastDayReadoutTime \
        [lindex [mysqlsel $DB "SELECT MAX(Time) FROM DayReadouts" -flatlist] 0]
    
    set PrevTime 0
    set rsReadouts [mysqlquery $DB "SELECT ID, Time, Age \
                                    FROM ${From}Readouts \
                                    WHERE Packed = 0 AND \
                                        Time <= '$LastDayReadoutTime'"]
    while {[set Readout [mysqlnext $rsReadouts]] != ""} {
        set ReadoutID [lindex $Readout 0]
        set EndTime [clock scan [lindex $Readout 1] -gmt 1]
        set Age [expr {[lindex $Readout 2] * 60}]
        set StartTime [expr {$EndTime - $Age}]
        if {abs($StartTime - $PrevTime) <= 60} {
            set Age [expr {$EndTime - $PrevTime}]
            set StartTime $PrevTime
        } elseif {$Age == 0} {
            set Age 60
            set StartTime [expr {$EndTime - $Age}]
        };#if
        set PrevTime $EndTime
        
        log "[clock format $StartTime -format {%D %T} -gmt 1] -\
             [clock format $EndTime -format {%D %T} -gmt 1]"
              
        foreach T [array names PackedReadouts] {
            set PackedReadouts($T) [list [lindex $PackedReadouts($T) 0] 0]
        };#foreach
              
        set T0 [expr {int($StartTime / $OutputViewGrid) * $OutputViewGrid}]
        set T [expr {$T0 + $OutputViewGrid}]
        while {$T0 <= $EndTime} {
            set Quotient [expr {([math::min $T $EndTime] - \
                                 [math::max $T0 $StartTime] + 0.0) / $Age}]

            set InsertionFlag " "
            if {[array get PackedReadouts $T] != ""} {
                set PackedReadoutID [lindex $PackedReadouts($T) 0]
            } else {
                set Tstr [clock format $T -format {%Y-%m-%d %H:%M} -gmt 1]
                set Query [mysqlquery $DB "SELECT ID FROM ${To}Readouts \
                                           WHERE Time = '$Tstr'"]
                set Row [mysqlnext $Query]
                if {$Row != ""} {
                    set PackedReadoutID [lindex $Row 0]
                } else {
                    mysqlexec $DB "START TRANSACTION"
                    mysqlexec $DB "INSERT INTO ${To}Readouts (Time, Age) \
                                   VALUES('$Tstr', [expr {$OutputViewGrid / 60}])"
                    set PackedReadoutID [mysqlinsertid $DB]
                    # Create an all-zero data record which could be a time mark
                    # for graphing the periods where no interesting data
                    # present (i.e. all data are filtered out)
                    mysqlexec $DB \
                        "INSERT INTO ${To}Data \
                             (ReadoutID, \
                              Src1, Src2, Src3, Src4, \
                              Dst1, Dst2, Dst3, Dst4, \
                              Bytes, Packets) \
                         VALUES 
                             ($PackedReadoutID, \
                              0, 0, 0, 0, 0, 0, 0, 0, 0, 0)"
                    mysqlexec $DB "COMMIT"
                    set InsertionFlag "+"
                };#if
                mysqlendquery $Query
            };#if
            
            set PackedReadouts($T) [list $PackedReadoutID $Quotient]
            log "  ([clock format $T0 -format {%D %T} -gmt 1]) - \
                    [clock format $T  -format {%D %T} -gmt 1] \
                    $Quotient/1 $InsertionFlag$PackedReadoutID"

            set T0 $T
            incr T $OutputViewGrid
        };#//while

        mysqlexec $DB "START TRANSACTION"
        set rsData [mysqlquery $DB "SELECT Src1, Src2, Src3, Src4, \
                                        Dst1, Dst2, Dst3, Dst4, Bytes, Packets \
                                FROM ${From}Data \
                                WHERE ReadoutID = $ReadoutID"]
        while {[set Row [mysqlnext $rsData]] != ""} {
            set Src1 [lindex $Row 0]
            set Src2 [lindex $Row 1]
            set Src3 [lindex $Row 2]
            set Src4 [lindex $Row 3]
            set Dst1 [lindex $Row 4]
            set Dst2 [lindex $Row 5]
            set Dst3 [lindex $Row 6]
            set Dst4 [lindex $Row 7]
            set Bytes [lindex $Row 8]
            set Packets [lindex $Row 9]
	    if {$DebugLevel >= 4} {
        	print "    $Src1.$Src2.$Src3.$Src4 => $Dst1.$Dst2.$Dst3.$Dst4 $Bytes"
	    };#if
              
            foreach T [array names PackedReadouts] {
                set PackedReadoutID [lindex $PackedReadouts($T) 0]
                set Quotient [lindex $PackedReadouts($T) 1]
        	if {$DebugLevel >= 5} {
            	    print "      $PackedReadoutID $Quotient/1"
		};#if
                if {$Quotient == 0} \
                    continue
                    
                set B [expr {round($Bytes * $Quotient)}]
                if {$B == 0} \
                    continue
                set P [expr {round($Packets * $Quotient)}]
                set Query "UPDATE ${To}Data SET \
                               Bytes = Bytes + $B, \
                               Packets = Packets + $P \
                           WHERE ReadoutID = $PackedReadoutID AND \
                               Src1 = $Src1 AND Src2 = $Src2 AND \
                               Src3 = $Src3 AND Src4 = $Src4 AND \
                               Dst1 = $Dst1 AND Dst2 = $Dst2 AND \
                               Dst3 = $Dst3 AND Dst4 = $Dst4"
                if {[mysqlexec $DB $Query] == 0} {
                    mysqlexec $DB \
                        "INSERT INTO ${To}Data \
                             (ReadoutID, \
                              Src1, Src2, Src3, Src4, \
                              Dst1, Dst2, Dst3, Dst4, \
                              Bytes, Packets) \
                         VALUES 
                             ($PackedReadoutID, \
                              $Src1, $Src2, $Src3, $Src4, \
                              $Dst1, $Dst2, $Dst3, $Dst4, \
                              $B, $P)"
                };#if
            };#foreach
        };#while

        mysqlexec $DB "UPDATE ${From}Readouts SET Packed = 1 WHERE ID = $ReadoutID"
        mysqlexec $DB "COMMIT"
        #unset PackedReadouts
    };#while
    
    mysqlendquery $rsReadouts
};#PackData


# Procedure cutting old data from the specified table
proc CutData {ViewType} {
    global DB
    
    set ViewWindowSize [LoadTemporalSetting "${ViewType}ViewWindow"]
    set T0 [clock format [expr {[clock seconds] - $ViewWindowSize}] \
                  -format {%Y-%m-%d %H:%M}]
    log "Cutting readouts before $T0"
    
    set rsReadouts [mysqlquery $DB "SELECT ID, Time \
                                    FROM ${ViewType}Readouts \
                                    WHERE Time < '$T0' AND Packed = 1"]
    while {[set Readout [mysqlnext $rsReadouts]] != ""} {
        set ReadoutID [lindex $Readout 0]
        set Time [clock scan [lindex $Readout 1]]
        log -nonewline "[clock format $Time -format {%D %T}]... "

        mysqlexec $DB "START TRANSACTION"
        set RecordsDeleted [mysqlexec $DB "DELETE FROM ${ViewType}Data \
                                           WHERE ReadoutID = $ReadoutID"]
        mysqlexec $DB "DELETE FROM ${ViewType}Readouts \
                       WHERE ID = $ReadoutID"
        log "$RecordsDeleted recs deleted"
        mysqlexec $DB "COMMIT"
    };#while
    
    mysqlendquery $rsReadouts
};#CutData


# Writing a string both to stdout but only if not in daemon mode
proc print {args} {
    global RunMode LogStream
    set NoNewLine 0
    set CopyToLog 0
    foreach arg $args {
        if {$arg == "-nonewline"} {
            set NoNewLine 1
            continue
        };#if
        if {$arg == "-log"} {
            set CopyToLog 1
            continue
        };#if
        if {$RunMode != "Daemon"} {puts -nonewline $arg}
        if {$CopyToLog && ($LogStream != "")} {puts -nonewline $LogStream $arg}
    };#foreach

    if {! $NoNewLine} {
        if {$RunMode != "Daemon"} {puts ""}
        if {$CopyToLog && ($LogStream != "")} {puts $LogStream ""}
    };#if
};#print


# Writing a string both to stdout and log
proc log {args} {
    eval print -log $args
};#log


# Processing keyboard interruptions (or kill -INT)
proc ProcessBreak {} {
    global Waiting Interrupted
    set Interrupted 1
    set Waiting 0
};#ProcessBreak


# A single cycle of tha actual work
proc WorkCycle {} {
    global DB DBServer DBUser DBPassword DBName
    global Time Interrupted
    
    set Time [clock format [clock seconds] -format "%Y-%m-%d %H:%M:%S"]
    log "\n$Time"
    
    print -nonewline "Connecting to database..."
    set DB [mysqlconnect -host $DBServer -user $DBUser -password $DBPassword]
    mysqluse $DB $DBName
    
    log "\nUnloading data from the router..."
    UnloadDataFromRouter
    if {$Interrupted} return
    
    log "Packing day data..."
    PackData Day Week
    if {$Interrupted} return
    log "Cutting day data..."
    CutData Day
    if {$Interrupted} return
    
    log "Packing week data..."
    PackData Week Month
    if {$Interrupted} return
    log "Cutting week data..."
    CutData Week
    if {$Interrupted} return
    
    log "Packing month data..."
    PackData Month Year
    if {$Interrupted} return
    log "Cutting month data..."
    CutData Month
    if {$Interrupted} return
    
    log "Cutting year data..."
    CutData Year
    
    mysqlclose $DB
};#WorkCycle


# Here we start
fconfigure stdout -buffering none
ParseCmdLine
print "IPAcco Data Pump V0.2"
LoadSettings

set Interrupted 0
set Waiting 0
signal trap {SIGINT SIGTERM} {ProcessBreak}

set DB [mysqlconnect -host $DBServer -user $DBUser -password $DBPassword]
mysqluse $DB $DBName
set CyclePeriod [LoadTemporalSetting "DayViewGrid"]
mysqlclose $DB

while {! $Interrupted} {
    if {$RunMode != "OnePass"} {
        set NextRun [expr {([clock seconds] + $CyclePeriod - 1) / $CyclePeriod \
                     * $CyclePeriod}]
        set TimeToWait [expr {$NextRun - [clock seconds]}]
        if {$TimeToWait > 0} {
            if {$DebugLevel > 0} {
                print -nonewline \
                    "\nWaiting until [clock format $NextRun -format %H:%M:%S]..."
            };#if
            set Waiting 1
            after [expr {$TimeToWait * 1000}] {set Waiting 0}
            vwait Waiting
            if {$DebugLevel > 0} {
                print ""
            };#if
        };#if
    };#if
    
    if {$Interrupted} break;

    set LogStream ""
    if {$LogFile != ""} {
        set LogStream [open $LogFile a]
        fconfigure $LogStream -buffering line
    };#if
    
    if {[catch {WorkCycle} Msg] == 1} {
        log $Msg
    };#if
    
    if {$LogStream != ""} {
	close $LogStream
    };#if
    set LogStream ""
    
    if {$RunMode == "OnePass"} {
        break
    };#if
};#while