/* 
 * Copyright (c) 1999 - 2002 Roman V. Palagin <romanp@unshadow.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 * 	$Id: ipacctd.c,v 1.46 2003/08/22 08:41:51 romanp Exp $
 */

static char rcs_id[]=
    "@(#) $Id: ipacctd.c,v 1.46 2003/08/22 08:41:51 romanp Exp $";

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/resource.h>
#include <sys/ioctl.h>
#include <sys/un.h>

#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <netdb.h>
#include <fcntl.h>
#include <paths.h>
#include <signal.h>
#include <syslog.h>
#include <stdarg.h>
#include <err.h>

#include "ipacctd.h"
#include "pathnames.h"

#define VERSION	        "$Revision: 1.46 $"

/* this one from BSDI's kernel */
#define IPV4_HASH(addr1, addr2, port1, port2, nbuckets) \
    ((((addr1) ^ (addr2)) + (port1) + (port2)) & ((nbuckets - 1)))

#define	MAX_PKT_SIZE	0xffff
#define	MAX_NRECORDS	30000
#define NBUCKETS        (8192*2)	/* must be power of 2 */

#undef MAX
#define MAX(a,b)  	(((a) > (b)) ? (a) : (b))

/* hash record */
struct ip_hash {
    struct acct 	r;
    time_t 		stime;   
    u_int32_t 		pkt;
    u_int32_t 		bytes;
    struct ip_hash 	*next;
};

u_short 	bindport;
struct 		ip_hash *ip_acct[NBUCKETS];
time_t 		start_time;
struct 		rusage ru0;

char 		pidfile[FILENAME_MAX + 1];
char 		savefile[FILENAME_MAX + 1];
char		socketpath[FILENAME_MAX + 1];

u_short 	flag_nodaemon;
u_short 	flag_divert = 1;
u_short 	flag_verbose;
u_short 	flag_when;
u_short		flag_cisco;
u_short		flag_server;
u_short 	debug;

u_int	        max_nrecords;
u_int		nrecords;
u_int		pkt_threshold;
u_int		bytes_threshold;

int 		signalpipe[2];

int		main(int, char **);
static void 	ip_hash_dump(int, u_int32_t *, u_int32_t *);
static void 	ip_hash_debug(struct rusage *);
static struct 	ip_hash *ip_hash_add(struct ip *, struct acct *);
static void 	ip_hash_clear(void);
static void 	ip_acct_save(int fd);
static void	ip_acct_mk_record(struct ip *ip, struct acct *);
static void 	prt_rusage(int, struct rusage *);
static void 	usage(void);
static void 	fd_printf(int, const char *, ...);
static int	create_unix_socket(void);

/* signal stuff */
__sighandler_t	catch_signal;
static void 	terminate(void);
static void 	save_ip_acct(int fd);
static void 	reapchild(void);
static void 	dump_int_state(void);

int 
main(int argc, char **argv)
{
	char 	packet[MAX_PKT_SIZE]; 
	struct 	sockaddr_in ds;		/* divert socket address */
	int 	spkt;			/* divert socket */
	int	sunx = -1;		/* unix socket */
	int 	op, maxsock, sock_bsize, size_ds = sizeof(ds);
	fd_set 	allsocks;
	struct 	ip *ip;
	struct 	acct rec;
    
	bzero(&ds, size_ds);
	max_nrecords = MAX_NRECORDS;
 
	while ((op = getopt(argc, argv, "b:cdf:hm:p:r:stvwV?")) != -1) {
		switch (op) {
		case 'b': /* set socket buffers size */
			sock_bsize = atoi(optarg);
			break;
		case 'c':
			flag_cisco = 1;
			break;
		case 'd': /* don't become daemon */
			flag_nodaemon = 1;
			break;
		case 'f': /* accounting save file */
			strncpy(savefile, optarg, FILENAME_MAX);
			break;
		case '?':
		case 'h':  /* print usage */
			usage();
			break;
		case 'm': /* maximum number of records in hash */
			max_nrecords = atoi(optarg);
			break;
		case 'p': /* bind port */
			bindport = atoi(optarg);
			ds.sin_port = htons(bindport);
			break;
		case 'r': /* pid file */
			strncpy(pidfile, optarg, FILENAME_MAX);
			break;
		case 's':
			flag_server = 1;
			break;
		case 't': /* ipfw rule is 'tee' rule */
			flag_divert = 0;
			break;
		case 'v': /* also log IP protocol and source/dest port 
			     (if applicable) 
			  */
			flag_verbose = 1;
			break;
		case 'w': /* print time when first packet seen */
			flag_when = 1;
			break;
		case 'V': /* print RCS Id */
			warnx("%s", rcs_id);
			exit (0);
		default:
			usage();
		}
	}
	
	if (!ds.sin_port) {
		warnx("you must specify port");
		usage();
	}

	/* signal handling */
	if (signal(SIGTERM, catch_signal) == SIG_ERR)
		err(1, "signal(SIGTERM)");
	if (!flag_server)
		if (signal(SIGHUP, catch_signal) == SIG_ERR)
			err(1, "signal(SIGHUP)");
	if (signal(SIGUSR2, catch_signal) == SIG_ERR)
		err(1, "signal(SIGUSR2)");
	if (signal(SIGCHLD, catch_signal) == SIG_ERR)
		err(1, "signal(SIGCHLD)");
	if (signal(SIGPIPE, SIG_IGN) == SIG_ERR)
		err(1, "signal(SIGPIPE)");

	if (pipe(signalpipe) == -1)
		err(1, "pipe");
	maxsock = signalpipe[0];

	if (!savefile[0])
		sprintf(savefile, "/var/log/ipacct.%i", bindport);
	if (!pidfile[0])
		sprintf(pidfile, "%sipacctd.%i", _PATH_VARRUN, bindport);
	
	ds.sin_family = AF_INET;

	if ((spkt = socket(PF_INET, SOCK_RAW, IPPROTO_DIVERT)) == -1)
		err(1, "socket(IP_DIVERT)");
	
	if ((bind(spkt, (struct sockaddr *)&ds, sizeof(ds))) == -1)
		err(1, "bind");
	maxsock = MAX(maxsock, spkt);

	/* clear our file mode creation mask */
	umask(0);

	if (flag_server) {
		sunx = create_unix_socket();
		if (sunx == -1)
			err(1, "create_unix_socket");
		if (listen(sunx, 5) == -1) {
			unlink(socketpath);
			err(1, "listen");
		}
		maxsock = MAX(maxsock, sunx);
	}

	/* clear ip_acct hash */
	bzero(ip_acct, sizeof(struct ip_hash *) * NBUCKETS);

	FD_ZERO(&allsocks);
	FD_SET(spkt, &allsocks);
	FD_SET(signalpipe[0], &allsocks);
	if (flag_server)
		FD_SET(sunx, &allsocks);

	if (!flag_nodaemon) {
		/* become daemon */
		daemon(0, 1);
		/* change current dir to /var/tmp so core goes here :) */
		chdir(_PATH_VARTMP);
	}

	/* open syslog */
	openlog("ipacctd", LOG_PID | LOG_CONS, LOG_DAEMON);

	/* write out pid to pid file */
	{
		FILE *f;

		if ((f = fopen(pidfile, "w")) == NULL) 
			syslog(LOG_ERR, "(port %i) can't open %s : %m",
			    bindport, pidfile);
		else {
			fprintf(f, "%d\n", getpid());
			fchmod(fileno(f), S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
			fclose(f);
		}
	}

	/* remember when show begin */
	(void)time(&start_time);

	syslog(LOG_NOTICE, "(port %i) accounting started "
	    "(%s, using \'%s\' socket, %s statistic, "
	    "threshold = %d, socket buffer = %d)",
	    bindport, VERSION,
	    flag_divert ? "divert" : "tee",
	    flag_verbose ? "verbose" : "normal",
	    max_nrecords, sock_bsize);

	/* set socket buffers size (if specified) */
	if (sock_bsize) {
		if (setsockopt(spkt, SOL_SOCKET, SO_RCVBUF, 
		    &sock_bsize, sizeof(int)) == -1)
			syslog(LOG_WARNING, "(port %d) setsockopt(SO_RCVBUF): %m",
			    bindport);
		if (setsockopt(spkt, SOL_SOCKET, SO_SNDBUF, 
		    &sock_bsize, sizeof(int)) == -1)
			syslog(LOG_WARNING, "(port %d) setsockopt(SO_SNDBUF): %m",
			    bindport);
	}
	
	for (;;) {
		u_int len;
		fd_set readable;

		readable = allsocks;

		if (select(maxsock + 1, &readable, NULL, NULL, NULL) == -1) {
			if(errno != EINTR) {
				syslog(LOG_ERR, "(port %i) select: %m", bindport);
				exit(1);
			}
			continue;
		}
		/* check for packet available */
		if (FD_ISSET(spkt, &readable)) {
			if ((len = recvfrom(spkt, packet, MAX_PKT_SIZE, 0,
			    (struct sockaddr *)&ds, &size_ds)) == -1) {
				syslog(LOG_ERR, "(port %i) receive: %m", bindport);
			} else {
				if (flag_divert)
					if ((len = sendto(spkt, packet, len, 0,
					    (struct sockaddr *)&ds, size_ds)) == -1)
						syslog(LOG_ERR, "(port %i) send: %m", bindport);
				ip = (struct ip*)packet;
				ip_acct_mk_record(ip, &rec);
				ip_hash_add(ip, &rec);
			}
		}
		/* process any posted signals */
		if (FD_ISSET(signalpipe[0], &readable)) {
			int n;
	  
			if (ioctl(signalpipe[0], FIONREAD, &n) == -1) {
				syslog(LOG_ERR, "(port %i) ioctl(FIONREAD): %m", bindport);
				continue;
			}
			while (--n >= 0) {
				char c;

				if (read(signalpipe[0], &c, 1) != 1) {
					syslog(LOG_ERR, "(port %i) read from signalpipe: %m", bindport);
					break;
				}
				switch (c) {
				case 'T':	/* SIGTERM */
					close(spkt);
					terminate();
					break;
				case 'H': 	/* SIGHUP */
					save_ip_acct(-1);
					break;
				case 'C':	/* SIGCHLD */
					reapchild();
					break;
				case 'U': 	/* SIGUSR2 */
					dump_int_state();
					break;
				default:
					syslog(LOG_WARNING, "(port %d) uknown signal received - ignored",
					    bindport);
					break;
				}
			}
		}
		/* connection arrived on unix socket */
		if (flag_server && FD_ISSET(sunx, &readable)) {
			int newfd;

			newfd = accept(sunx, NULL, 0);
			if (newfd == -1)
				syslog(LOG_ERR, "(port %d) accept: %m", bindport);
			else
				save_ip_acct(newfd);
		}
	}
}

static void
ip_acct_mk_record(struct ip *ip, struct acct *rec)
{
	char *packet;

	packet = (char *)ip;

	bzero(rec, sizeof(*rec));
	bcopy(&ip->ip_src.s_addr, rec, 2 * sizeof(u_int32_t));

	if (flag_verbose) {
		if ((ip->ip_p == IPPROTO_TCP) ||
		    (ip->ip_p == IPPROTO_UDP)) 
			bcopy(packet + (ip->ip_hl << 2), &rec->s_port, 2 * sizeof(u_short)); 
		else if (ip->ip_p == IPPROTO_ICMP) { 
			/* save icmp message type as s_port */
			rec->s_port = htons((*(u_short*)(packet + (ip->ip_hl << 2))) & 0xff);
			/* save icmp message sub-code as d_port */
			rec->d_port = htons(((*(u_short*)(packet + (ip->ip_hl << 2))) & 0xff00) >> 8);
		}
		rec->ip_p = ip->ip_p;
	}
}

static void 
ip_acct_save(int fd)
{
	u_int32_t pkt, bytes;
	char save_path[FILENAME_MAX + 1 + 256];
	time_t now;

	now = time(NULL);
	/* if savefile contains any strftime format chars - expand it now. */
	strftime(save_path, sizeof(save_path), savefile, localtime(&now));

	if ((fd == -1) &&
	    (fd = open(save_path, O_WRONLY | O_APPEND | O_CREAT, 0600)) == -1) {
		syslog(LOG_ERR,"(port %i) can't open %s: %m", bindport, save_path);
		_exit(1);
	}

	ip_hash_dump(fd, &pkt, &bytes);

	close(fd);

	syslog(LOG_NOTICE,"(port %i) accounting dumped to %s"
	    " (%u packets, %u bytes)", bindport, 
	    flag_server ? socketpath : save_path, pkt, bytes);
}

static void 
ip_hash_clear(void)
{
	register int i;
	register struct ip_hash *iph, *nxt;

	/* Walk down throug *next and free all memory */
	for (i = 0; i < NBUCKETS; i++) {
		for (iph = ip_acct[i]; iph; iph = nxt) {
			nxt = iph->next;
			free(iph);
		}
	}
	/* clear ip_acct hash */
	bzero(ip_acct, sizeof(struct ip_hash *) * NBUCKETS);
	/* clear threshold vars */
	nrecords = pkt_threshold = bytes_threshold = 0;
}

static void 
ip_hash_dump(int fd, u_int32_t *pkt, u_int32_t *bytes)
{
	register int i;
	register struct ip_hash *iph;
	struct protoent *p;

	*pkt = *bytes = 0;

	/* Walk down through *next and dump all entries */
	for (i = 0; i < NBUCKETS; i++) {
		for (iph = ip_acct[i]; iph; iph = iph->next) {
			if (flag_cisco)
				fd_printf(fd, " ");
			fd_printf(fd, "%s\t", inet_ntoa(iph->r.s_addr));
			if (flag_verbose)
				fd_printf(fd, "%d\t", ntohs(iph->r.s_port));
			fd_printf(fd, "%s\t", inet_ntoa(iph->r.d_addr));
			if (flag_verbose) {
				fd_printf(fd, "%d\t", ntohs(iph->r.d_port));
				p = getprotobynumber(iph->r.ip_p);
				fd_printf(fd, (p)?"%s\t":"%d\t",
				    (p)?p->p_name:(u_short)iph->r.ip_p);
			}
			if (flag_cisco)
				fd_printf(fd, "%u\t%u", iph->pkt, iph->bytes);
			else
				fd_printf(fd, "%u\t%u", iph->bytes, iph->pkt);
			if (flag_when)
				fd_printf(fd, "\t%lu\n", iph->stime);
			else
				fd_printf(fd, "\n");
			(*pkt) += iph->pkt;
			(*bytes) += iph->bytes;
		}
	}
	if (pkt_threshold) {
		fd_printf(fd, "Accounting threshold exceeded for "
		    "%d packet(s) and %d byte(s)\n",
		    pkt_threshold, bytes_threshold);
		syslog(LOG_WARNING, "(port %d) accounting threshold exceeded for "
		    "%d packet(s) and %d byte(s)\n",
		    bindport, pkt_threshold, bytes_threshold);
	}
}

/*
 * Add ip packet 'ip' to hash.
 * return pointer to struct ip_hash
 */
static struct ip_hash *
ip_hash_add(struct ip *ip, struct acct *r)
{
	register struct ip_hash *iph, *top;
	int idx;

	idx = IPV4_HASH(r->s_addr.s_addr, r->d_addr.s_addr, 
	    r->s_port, r->d_port, NBUCKETS);

	for (top = iph = ip_acct[idx]; ;iph = (*(top = iph)).next) {
		if (!iph) {
			if (nrecords >= max_nrecords) {
				pkt_threshold++;
				bytes_threshold += ntohs(ip->ip_len);
				return (NULL);
			}
			iph = malloc(sizeof(struct ip_hash));
			if (iph == NULL) {
				syslog(LOG_ERR, "(port %d) malloc: %m", bindport);
				return (NULL);
			}
			nrecords++;
			if (!top) 
				ip_acct[idx] = iph;
			else 
				top->next = iph;
			bcopy(r, &iph->r, sizeof(struct acct));
			iph->stime = time(NULL);
			iph->bytes = ntohs(ip->ip_len);
			iph->pkt = 1;
			iph->next = NULL;
			break;
		} else if (!bcmp(&iph->r, r, sizeof(struct acct))) {
			iph->bytes += ntohs(ip->ip_len);
			iph->pkt++;
			break;
		}
	}
	return (iph);
}

/*
  Dump current status of daemon
*/
static void 
ip_hash_debug (struct rusage *r)
{
	register struct ip_hash *iph;
	register u_int idx;
	u_int32_t r_in_hash[NBUCKETS], pkts, bytes;
	int t_buckets = 0;
	int fd;
	time_t dump_time = time(NULL);

	if ((fd = open(_PATH_IPACCTDDUMP, O_WRONLY|O_APPEND|
	    O_CREAT|O_NOFOLLOW, 0644)) == -1) {
		syslog(LOG_ERR,"(port %i) can't open %s: %m", 
		    bindport, _PATH_IPACCTDDUMP);
		_exit(1);
	}

	/* print information about process */
	fd_printf(fd, "\n\n\tIPacctd status dump on %s\n", ctime(&dump_time));
	fd_printf(fd, "IPacctd version %s\n", rcs_id);
	fd_printf(fd, "Compiled on %s %s\n", __DATE__, __TIME__);
	fd_printf(fd, "Started at %s\n", ctime(&start_time));
	if (r) {
		fd_printf(fd, "Resource usage:\n");
		prt_rusage(fd, r);
	}

	fd_printf(fd, "\nHash usage statistic:\n");
	bzero(r_in_hash, sizeof(r_in_hash));

	for (idx = 0; idx < NBUCKETS; idx++) {
		for (iph = ip_acct[idx]; iph; r_in_hash[idx]++, iph=iph->next)
		    ;
	}

	for (idx = 0; idx < NBUCKETS; idx++) {
		if (r_in_hash[idx]) {
			t_buckets++;
			fd_printf(fd, "\t%05i:%i\n", idx, r_in_hash[idx]);
		}
	}

	fd_printf(fd, "\n\tTotal number of records in hash: %i "
	    "( ~%i bytes, %i buckets in use)\n\n", 
	    nrecords, nrecords * sizeof (struct ip_hash), t_buckets);
	fd_printf(fd, "Current accounting records:\n");
	flag_when = 1;
	ip_hash_dump(fd, &pkts, &bytes);
	fd_printf(fd, "\n\tTotal number of packets accounted: %i (%i bytes)\n\n",
	    pkts, bytes);
	close(fd);
	syslog(LOG_NOTICE, "(port %i) IPacctd dump finished to %s",
	    bindport, _PATH_IPACCTDDUMP);
}

void 
flag_signal(char c)
{
	if (write(signalpipe[1], &c, 1) == -1)
		syslog(LOG_ERR, "(port %i) write to signalpipe: %m", bindport);
}

void
catch_signal(int sig)
{
	switch (sig) {
	case SIGHUP:
		flag_signal('H');
		break;
	case SIGTERM:
		flag_signal('T');
		break;
	case SIGUSR2:
		flag_signal('U');
		break;
	case SIGCHLD:
		flag_signal('C');
		break;
	default:
		flag_signal('?');
		break;
	}
}

void 
save_ip_acct(int fd)
{
	pid_t pid;

	pid = fork();

	switch (pid) {
	case -1: /* error */
		syslog(LOG_ERR,"(port %i) fork: %m", bindport);
		return;
	case 0: /* child */
		ip_acct_save(fd);
		_exit(0);
	default:
		if (fd != -1)
			close(fd);
		ip_hash_clear();
	}
}

void 
terminate(void)
{
	if (flag_server)
		ip_acct_save(-1);
	unlink(pidfile);
	syslog(LOG_NOTICE,"(port %i) accounting stopped", bindport);
	exit(0);
}

void 
dump_int_state(void)
{
	pid_t pid;
	struct rusage *ru;

	if (getrusage(RUSAGE_SELF, &ru0) == -1) {
		syslog(LOG_ERR,"(port %i) getrusage: %m", bindport);
		ru = NULL;
	}
	else ru = &ru0;

	pid = fork();

	switch (pid) {
	case -1: /* error */
		syslog(LOG_ERR,"(port %i) fork: %m", bindport);
		return;
	case 0: /* child */
		ip_hash_debug(ru);
		_exit(0);
	default: /* parent */
		return;
	}
}

void 
reapchild(void)
{
	int status;

	for(;;) {
		if (wait3(&status, WNOHANG, NULL) <= 0)
			break;
	}
}

static void 
prt_rusage(int fd, struct rusage *r)
{
	fd_printf(fd, "\t%d.%01du + %d.%01ds time\n", 
	    r->ru_utime.tv_sec, r->ru_utime.tv_usec/100000,
	    r->ru_stime.tv_sec, r->ru_stime.tv_usec/100000);
	fd_printf(fd, "\t%d maxrss\n", r->ru_maxrss);
	fd_printf(fd, "\t%d + %d flts\n", r->ru_minflt, r->ru_majflt);
	fd_printf(fd, "\t%d swaps\n", r->ru_nswap);
	fd_printf(fd, "\t%di + %do ops\n", r->ru_inblock, r->ru_oublock);
	fd_printf(fd, "\t%d signals\n", r->ru_nsignals);
	fd_printf(fd, "\t%d + %d csw\n", r->ru_nvcsw, r->ru_nivcsw);
}

static void
fd_printf(int fd, const char *fmt, ...)
{
	char *res;
	va_list va;
	int nchars;

	va_start(va, fmt);
	vasprintf(&res, fmt, va);
	if (res == NULL) {
		syslog(LOG_ERR, "(port %d) fd_printf: vasprintf() failed",
		    bindport); 
		_exit(1);
	} else {
		nchars = write(fd, res, strlen(res));
		if (nchars != strlen(res)) {
			syslog(LOG_ERR, "(port %d) fd_printf: short write",
			    bindport);
			_exit(1);
		}
		free(res);
	}
}

static int
create_unix_socket(void)
{
	struct 	sockaddr_un sun;
	int	s;
	int	rc;
	struct	stat sbuf;

	/* make dir _PATH_UPREFIX, owned by root and mode 0700 */
	rc = stat(_PATH_UPREFIX, &sbuf);
	if (rc == -1) {
		if (errno == ENOENT) {
			/* path doesn't exist, try to create it */
			rc = mkdir(_PATH_UPREFIX, 0700);
			if (rc == -1)
				/* 
				 * smth. wrong, including situation when
				 * path was created beetween stat() and mkdir()
				 */
				return (-1);
			else goto bind_socket;
		} else {
			/* stat failed - smth. going wrong... */
			return (-1);
		}
	}

	/* 
	 *  stat() was succeeded - check that path is dir,
	 *  owned by root and permissions is 0700
	 */
	if (!S_ISDIR(sbuf.st_mode)) {
		errno = ENOTDIR;
		return (-1);
	}
	if ((sbuf.st_uid != 0) || ((sbuf.st_mode & 0777) !=  0700)) {
		errno = EACCES;
		return (-1);
	}
 bind_socket:

	/* bind socket to name */
	snprintf(socketpath, sizeof(socketpath), _PATH_UPREFIX "/ipacct.%d", bindport);
	if ((s = socket(PF_LOCAL, SOCK_STREAM, 0)) == -1)
		return (-1);
	bzero(&sun, sizeof(sun));
	sun.sun_family = AF_UNIX;
	strlcpy(sun.sun_path, socketpath, sizeof(sun.sun_path));
	unlink(sun.sun_path);
	if (bind(s, (struct sockaddr *)&sun, sizeof(sun)) == -1)
		return (-1);

	return (s);
}

static void 
usage(void)
{
	fprintf(stderr, "usage: ipacctd [-cdhtwvV?] [-b receive_buffer_size]"
	    "[-f dumpfile] [-m threshold] [-r pidfile] -p ipfw_port\n");
	exit(0);
}
