/*
    Copyright (C) 2002  Frank W. Josellis

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111, USA.

*/

/* ipaddr.c: IP address decomposer.
 *
 * Input is of the form "<ip_address> [<ip_netmask>]" as dotted quads
 * or in the slash notation "<ip_address>/<mask>".
 *
 * Output lists the corresponding netmask, network address, host portion,
 * broadcast address, and CIDR notation of the network address.
 */ 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>


void help(char *prog){
  fprintf(stderr,"usage: %s <ip_address> [<ip_netmask>]\n"
	  "       %s <ip_address>/<mask>\n\n"
	  "Note: <ip_address> and <ip_netmask> stand for a dotted quad,\n"
	  "      <mask> denotes an integer (0,...,32).\n\n", prog, prog);
  exit(2);
}


int main(int argc, char *argv[])
{
   struct in_addr full_ip, network_ip, local_ip, bcast_ip, netmask_ip;
   unsigned int full_num, network_num, local_num, bcast_num, netmask_num; 
   short len;
   char *ptr[2];

   if (argc < 2 || argc > 3) help(argv[0]);

   if(argc == 2){
     ptr[0] = strtok(argv[1],"/");
     ptr[1] = strtok(NULL,"/");
   }
   else {
     ptr[0] = argv[1];
     ptr[1] = argv[2];
   }

   if(!inet_aton(ptr[0], &full_ip)){
     fprintf(stderr,"ERROR: invalid IP address: %s\n", ptr[0]);
     exit(1);
   }
   full_num = full_ip.s_addr;
   
   if(ptr[1]){
     if(argc == 3){
       if(!inet_aton(ptr[1], &netmask_ip)){
	 fprintf(stderr,"ERROR: invalid netmask: %s\n", ptr[1]);
	 exit(1);
       }
       {
	 unsigned int N = ntohl(netmask_ip.s_addr);

	 len = 0;
	 if(N)
	   while(len++ < 32 && ~(N ^ ~(0xffffffff<<(32 - len))));

	 if(len > 32){
	   fprintf(stderr,"ERROR: invalid netmask: %s\n", ptr[1]);
	   exit(1);
	 }
       }
     }
     else {
       if((len = atoi(ptr[1])) < 0 || len > 32){
	 fprintf(stderr,"ERROR: invalid netmask: %d\n", len);
	 exit(1);
       } 
       netmask_ip.s_addr = len ? htonl(0xffffffff<<(32 - len)) : 0x0 ;
     }

     netmask_num     = netmask_ip.s_addr;    
     network_num     = full_num & netmask_num;
     local_ip.s_addr = full_num & ~netmask_num;

     network_ip.s_addr = network_num;
     bcast_ip.s_addr   = network_num | ~netmask_num;
   } 
   else { 
     /* That is, no netmask was given - use library. */
     network_num = ntohl(inet_netof(full_ip));
     network_ip  = inet_makeaddr(htonl(network_num), 0);

     local_num = ntohl(inet_lnaof(full_ip));
     local_ip  = inet_makeaddr(0, htonl(local_num));
     
     bcast_ip  = inet_makeaddr(htonl(network_num), INADDR_BROADCAST);
     bcast_num = bcast_ip.s_addr;
     
     len = 32;
     while(bcast_num<<(32 - len) != network_num) len--;
     netmask_ip.s_addr = htonl(0xffffffff<<(32 - len));
   }
   
   printf(" IP address:\t%s\n", inet_ntoa(full_ip));
   printf(" Netmask:\t%s\n", inet_ntoa(netmask_ip));
   printf(" Network:\t%s\n", inet_ntoa(network_ip));
   printf(" Host portion:\t%s\n", inet_ntoa(local_ip));
   printf(" Broadcast:\t%s\n", inet_ntoa(bcast_ip));
   printf(" CIDR:\t\t%s/%d\n", inet_ntoa(network_ip), len);

   return(0);
}

