/**
 * Copyright 2003 Wayne Schroeder
 * Released under the GPL version 2
 */

#include "ipsectrace.h"
#include <stdio.h>
#include <inttypes.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <netinet/in.h>

void nomem()
{
	fprintf(stderr,"Error, failed to allocate memory.  Exiting...\n\n");
	exit(1);
}

void processPacket( u_char * param, const struct pcap_pkthdr * pkt, const u_char * data )
{
	ethernet_header_t ethPart;
	ip_header_t ipPart;
	Stats * s = (Stats *) param;
	int processed = 0;
	u_int16_t ipLen;
	char sipStr[30];
	char dipStr[30];
	char *hashKey;
	u_int32_t sip;
	u_int32_t dip;
	StreamStats  * streamStats;
	bool freeKey = false;

	if ( pkt->caplen < sizeof(ethernet_header_t) + sizeof(ip_header_t) )
	{
		fprintf(stderr,"Snaplen not large enough to read entire eth + ip header information.\n");
		return;
	}
	
	memcpy(&ethPart,data,sizeof(ethernet_header_t));
	processed += sizeof(ethernet_header_t);

	memcpy(&ipPart,data+processed,sizeof(ip_header_t));
	processed += sizeof(ip_header_t);
	
	ipLen = ntohs(ipPart.length);
	s->totalBytes += ipLen;

	sip = ntohl(ipPart.sip);
	dip = ntohl(ipPart.dip);

	sprintf(sipStr,"%u.%u.%u.%u", sip>>24, sip>>16 & 0xFF , sip>>8 & 0xFF, sip & 0xFF );
	sprintf(dipStr,"%u.%u.%u.%u", dip>>24, dip>>16 & 0xFF , dip>>8 & 0xFF, dip & 0xFF );

	/* now I have the ip's in dotted quad strings, lets generate the key for the hash */
	if ( ! ( hashKey = ( char * ) malloc(40) ) )
		nomem();

	sprintf(hashKey,"%s -> %s", sipStr, dipStr);

	streamStats = (*(s->streamStats))[hashKey];
	//fprintf(stderr,"streamStats: 0x%08x\n",streamStats);

	if ( ! streamStats )
	{
		if ( ! ( streamStats = (StreamStats *) malloc(sizeof(StreamStats)) ) )
			nomem();

		(*(s->streamStats))[hashKey] = streamStats;
		memcpy(&(streamStats->firstPacketTime),&(pkt->ts),sizeof(timeval));
		streamStats->totalBytes = 0;
	}
	else
		freeKey = true;
		
	memcpy(&(streamStats->lastPacketTime),&(pkt->ts),sizeof(timeval));
	streamStats->totalBytes += ipLen;

	if ( freeKey )
		free( hashKey );
}

int main( int argc, char ** argv )
{
	char * toOpen = NULL;
	pcap_t * pc = NULL;
	char ebuf[PCAP_ERRBUF_SIZE];
	struct bpf_program fp;

	if ( argc >= 2 )
		toOpen = argv[1];

	/* open file or device */
	
	if ( toOpen )
		pc = pcap_open_offline(toOpen,ebuf);
	
	/*
	if ( ! pc )
		pc = pcap_open_live(toOpen,68,0,10,ebuf);
	 */
	
	if ( ! pc )
	{
		fprintf(stderr,"Could not open as file or interface, I quit.\n");
		exit(1);
	}

	do
	{
		/* compile the filter for ipsec traffic */

		if ( pcap_compile( pc, &fp, "esp or ah", 1, 0 ) == -1 )
		{
			fprintf(stderr,"error compiling the built in esp filter!\n");
			break;
		}

		if ( pcap_setfilter( pc, &fp ) == -1 )
		{
			fprintf(stderr,"error attaching filter to open pcap session!\n");
			break;
		}
		
		Stats s;
		s.totalBytes = 0;
		s.streamStats = new StreamStatsHash;

		pcap_loop( pc, -1, processPacket, (u_char *) &s );
		
		fprintf(stdout,"Total Bytes: %lu\n\n",s.totalBytes);
		
		fprintf(stderr,"start seconds, end seconds, stream, total bytes, total seconds\n");

		char * key;
		StreamStats * ss;

		for ( StreamStatsHash::iterator it1 = s.streamStats->begin(); it1 != s.streamStats->end(); it1++ )
		{
			key = (*it1).first;
			ss = (*it1).second;

			fprintf(stdout,"%lu, %lu, %s, %lu, %lu\n", ss->firstPacketTime.tv_sec, ss->lastPacketTime.tv_sec, key, ss->totalBytes,
				ss->lastPacketTime.tv_sec - ss->firstPacketTime.tv_sec );
		}

		for ( StreamStatsHash::iterator it1 = s.streamStats->begin(); it1 != s.streamStats->end();  it1 = s.streamStats->begin() )
		{
			key = (*it1).first;
			ss = (*it1).second;
			s.streamStats->erase(it1);
			free( key );
			free( ss );
		}

		delete s.streamStats;



	}
	while (0);

	if ( pc )
		pcap_close(pc);

	pcap_freecode( &fp );

	return 0;
}
