/*
 * ----------------------------------------------------------------
 * Night Light Asynchrounous DNS Resolver
 * ----------------------------------------------------------------
 * Copyright (C) 1997-2003 Jonas Kvinge <jonas@night-light.net>
 * All rights reserved.
 *
 * This code is a result of thousands of hours of work by
 * Jonas Kvinge <jonas@night-light.net>
 *
 * You may not create derivative works based on this code.
 *
 * Modified source code or binaries compiled from modified source
 * code distributed in any shape or form without the authors
 * permission is expressly forbidden.
 *
 * This code is provided without warranty of any kind.
 *
 * Under no circumstances and under no legal contract or otherwise
 * shall Jonas Kvinge be liable to you or any other person for any
 * damages, computer failure, malfunction or any other damages or
 * losses.
 *
 * Last modified by:
 * Jonas Kvinge (25.06.2001)
 *
 */

#define ARES_TCP_C

#define NEED_SYS_TYPES_H 1		/* Extra types */
#define NEED_SYS_PARAM_H 1		/* Some systems need this */
#define NEED_LIMITS_H 0			/* Kernel limits */
#define NEED_STDARG_H 1			/* va_list, etc */
#define NEED_ERRNO_H 1			/* errno */
#define NEED_CTYPE_H 1			/* isdigit(), etc */
#define NEED_NETINET_IN_H 1		/* in_addr, sockaddr_in, etc */
#define NEED_ARPA_INET_H 1		/* inet_ntoa(), inet_aton(), etc */
#define NEED_STDIO_H 1			/* Standard C UNIX functions */
#define NEED_STDLIB_H 1			/* malloc(), exit(), atoi(), etc */
#define NEED_TIME_H 1			/* time(), etc */
#define NEED_SYSCTL_H 0			/* sysctl(), etc */
#define NEED_SYS_STAT_H 0		/* chmod(), mkdir(), etc */
#define NEED_SYS_UIO_H 1		/* iovec, etc */
#define NEED_FCNTL_H 1			/* open(), creat(), fcntl(), etc */
#define NEED_SYS_IOCTL_H 1		/* ioctl(), etc */
#define NEED_SYS_FILIO_H 1		/* Solaris need this for ioctl(), etc */
#define NEED_UNISTD_H 1			/* Unix standard functions */
#define NEED_STRING_H 1			/* C string functions */
#define NEED_SIGNAL_H 0			/* Signal functions */
#define NEED_SYS_SOCKET_H 1		/* Socket functions */
#define NEED_NETDB_H 0			/* Network database functions */
#define NEED_ARPA_NAMESER_H 1		/* Nameserver definitions */
#define NEED_GETUSERPW_HEADERS 0 	/* Functions to retrive system passwords */

#include "includes.h"

#include "dns.h"

#include "ares_io.h"
#include "ares_int.h"
#include "ares_tcp.h"
#include "ares_parse.h"

/* VARIABLES - JONAS (05.07.2000) */

const char ARESTCP_VERSION[] = "0.5.3d";

/* ARES_TCP_CONNECT FUNCTION - JONAS (25.06.2000) */

void ares_tcp_connect(struct Ares_Server_Struct *Server) {

  assert(Server != NULL);

  if (!Ares_IsConnectProc(Server->TCP_Flags)) {

    Ares_SetConnectProc(Server->TCP_Flags);
    Server->TCP_Time = NOW;

  }

  if (!Ares_IsSocket(Server->TCP_Flags)) {

    Server->TCP_Socket = socket(AF_INET, SOCK_STREAM, 0);
    if (Server->TCP_Socket <= ERROR) {
      ares_tcp_disconnect(Server, "Ares: Server %s TCP: Unable to create socket: [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
    Ares_SetSocket(Server->TCP_Flags);

  }

  if (!Ares_IsSockOPT(Server->TCP_Flags)) {

    signed long int Result = 0;
    unsigned long int Flags = 0;

#if defined(NBLOCK_SYSV)
    Flags = 1;
    Result = ioctl(Server->TCP_Socket, FIONBIO, &Flags);
    if (Result <= ERROR) {
      ares_tcp_disconnect(Server, "Ares: Server %s TCP: Unable to set socket in non-blocking mode using ioctl(): [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
#else
    Result = fcntl(Server->TCP_Socket, F_GETFL, &Flags);
    if (Result <= ERROR) {
      ares_tcp_disconnect(Server, "Ares: Server %s TCP: Unable to get socket flags using fcntl(): [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
#if defined(NBLOCK_BSD)
    Flags |= O_NDELAY;
#elif defined(NBLOCK_POSIX)
    Flags |= O_NONBLOCK;
#else
    #warning "This system does not support non-blocking sockets?"
    Flags |= O_NONBLOCK;
#endif
    Result = fcntl(Server->TCP_Socket, F_SETFL, &Flags);
    if (Result <= ERROR) {
      ares_tcp_disconnect(Server, "Ares: Server %s TCP: Unable to set socket in non-blocking mode using fcntl(): [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
#endif

    Ares_SetSockOPT(Server->TCP_Flags);

  }

  if ((ARES_SOCKKEEPALIVEOPT == TRUE) && (!Ares_IsKeepAliveOPT(Server->TCP_Flags))) {

    signed long int Result = 0;
    unsigned long int OPT = 1;

    Result = setsockopt(Server->TCP_Socket, SOL_SOCKET, SO_KEEPALIVE, &OPT, sizeof(OPT));
    if (Result <= ERROR) {
      ares_tcp_disconnect(Server, "Ares: Server %s TCP: Unable to set Keep Alive option for socket: [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }

    Ares_SetKeepAliveOPT(Server->TCP_Flags);

  }

  if (!Ares_IsBound(Server->TCP_Flags)) {

    signed long int Result = 0;
    struct sockaddr_in SockAddr = { 0 };

    memset(&SockAddr, 0, sizeof(SockAddr));

    SockAddr.sin_family = AF_INET;
    SockAddr.sin_addr.s_addr = INADDR_ANY;
    SockAddr.sin_port = 0;

    Result = bind(Server->TCP_Socket, (struct sockaddr *) &SockAddr, sizeof(SockAddr));
    if (Result <= ERROR) {
      ares_tcp_disconnect(Server, "Ares: Server %s TCP: Unable to bind socket: [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
    Ares_SetBound(Server->TCP_Flags);

  }

  if (!Ares_IsConnect(Server->TCP_Flags)) {

    signed long int Result = 0;
    struct sockaddr_in SockAddr = { 0 };

    memset(&SockAddr, 0, sizeof(SockAddr));

    SockAddr.sin_family = AF_INET;
    SockAddr.sin_addr = Server->InAddr;
    SockAddr.sin_port = Server->PortN;

    Result = connect(Server->TCP_Socket, (struct sockaddr *) &SockAddr, sizeof(SockAddr));
    if (Result <= ERROR) {
      if (errno == EINPROGRESS) {
        Ares_SetConnecting(Server->TCP_Flags);
        return;
      }
      else {
        ares_tcp_disconnect(Server, "Ares: Unable to connect to nameserver %s using TCP: [%d] %s", Server->HostIPS, errno, strerror(errno));
        return;
      }
    }
    Ares_SetConnected(Server->TCP_Flags);

  }

  if (!Ares_IsConnected(Server->TCP_Flags)) {

    signed long int Result = 0;
    int OPT = 0;
    getsockopt_optlen_type OPTLen = sizeof(OPT);

    Result = getsockopt(Server->TCP_Socket, SOL_SOCKET, SO_ERROR, &OPT, &OPTLen);
    if (Result <= ERROR) {
      ares_tcp_disconnect(Server, "Ares: Unable to get connection result for %s using TCP: [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
    if (OPT != SUCCESS) {
      ares_tcp_disconnect(Server, "Ares: Unable to connect to nameserver %s using TCP: [%d] %s", Server->HostIPS, OPT, strerror(OPT));
      return;
    }

    Ares_ClearConnecting(Server->TCP_Flags);
    Ares_SetConnected(Server->TCP_Flags);

    sysprint(BITMASK_DEBUG_ARES, "Ares: Successfully connected TCP socket to nameserver %s.", Server->HostIPS);

  }

  Ares_ClearConnectProc(Server->TCP_Flags);


}

/* ARES_TCP_DISCONNECT FUNCTION - JONAS (25.06.2000) */

void ares_tcp_disconnect(struct Ares_Server_Struct *Server, const char *LinePT, ...) {

  char Line[LINELEN+1] = "";
  va_list Args = { 0 };
  struct Ares_SendQ_Struct *SendQ = NULL;

  assert(Server != NULL);
  assert(LinePT != NULL);

  va_start(Args, LinePT);
  vsnprintf(Line, LINELEN+1, LinePT, Args);
  va_end(Args);

  if (Ares_IsSocket(Server->TCP_Flags)) {
    close(Server->TCP_Socket);
    Server->TCP_Socket = FD_NONE;
  }
  Server->TCP_Flags = 0;

  while (Server->TCPSend_Head != NULL) {
    SendQ = Server->TCPSend_Head;
    Server->TCPSend_Head = SendQ->Next;
    free(SendQ);
  }

  sysprint(BITMASK_DEBUG_ARES, "%s", Line);

}

/* ARES_TCP_READ FUNCTION - JONAS (25.06.2000) */

void ares_tcp_read(struct Ares_Server_Struct *Server) {

  signed long int Result = 0;

  assert(Server != NULL);
  assert(Ares_IsSocket(Server->TCP_Flags));
  assert(Server->TCP_Socket != FD_NONE);

  if (Ares_IsConnectProc(Server->TCP_Flags)) { ares_tcp_connect(Server); }
  if (!Ares_IsSocket(Server->TCP_Flags)) { return; }

  if (Server->TempBufferLen < 2) {
    Result = recv(Server->TCP_Socket, Server->TempBuffer + Server->TempBufferLen, 2 - Server->TempBufferLen, MSG_NOSIGNAL);
    if (Result <= ERROR) {
      if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR)) { return; }
      ares_tcp_disconnect(Server, "Ares: Read error to nameserver %s TCP: [%d]: %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
    if (Result == 0) {
      ares_tcp_disconnect(Server, "Ares: EOF to nameserver %s TCP", Server->HostIPS);
      return;
    }
    Server->TempBufferLen += Result;
    if (Server->TempBufferLen == 2) {
      Server->BufferLen = Server->TempBuffer[0] << 8 | Server->TempBuffer[1];
      Server->Buffer = malloc(Server->BufferLen);
      if (Server->Buffer == NULL) { return; }
    }
  }

  if (Server->TempBufferLen == 2) {
    Result = recv(Server->TCP_Socket, Server->Buffer + Server->BufferPos, Server->BufferLen - Server->BufferPos, MSG_NOSIGNAL);
    if (Result <= ERROR) {
      if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR)) { return; }
      ares_tcp_disconnect(Server, "Ares: Read error to nameserver %s TCP: [%d]: %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
    if (Result == 0) {
      ares_tcp_disconnect(Server, "Ares: EOF to nameserver %s TCP", Server->HostIPS);
      return;
    }
    Server->BufferPos += Result;
    if (Server->BufferPos == Server->BufferLen) {
      ares_parse(Server->Buffer, Server->BufferLen, Server, TRUE);
      FREE(Server->Buffer);
      Server->BufferLen = 0;
      Server->BufferPos = 0;
      memset(Server->TempBuffer, 0, 2);
      Server->TempBufferLen = 0;
    }
  }

}

/* ARES_TCP_WRITE FUNCTION - JONAS (25.06.2000) */

void ares_tcp_write(struct Ares_Server_Struct *Server) {

  struct Ares_SendQ_Struct *SendQ = NULL;
  struct iovec *Vector = NULL;
  unsigned long int Count = 0;
  signed long int Result = 0;

  assert(Server != NULL);
  assert(Ares_IsSocket(Server->TCP_Flags));
  assert(Server->TCP_Socket != FD_NONE);

  if (Ares_IsConnectProc(Server->TCP_Flags)) { ares_tcp_connect(Server); }
  if (!Ares_IsSocket(Server->TCP_Flags)) { return; }

  Count = 0;
  for (SendQ = Server->TCPSend_Head ; SendQ != NULL ; SendQ = SendQ->Next) { Count++; }

  Vector = malloc(Count * sizeof(struct iovec));
  if (Vector == NULL) { return; }

  Count = 0;
  for (SendQ = Server->TCPSend_Head ; SendQ ; SendQ = SendQ->Next) {
    Vector[Count].iov_base = (char *) SendQ->Buffer;
    Vector[Count].iov_len = SendQ->Len;
    Count++;
  }

  Result = writev(Server->TCP_Socket, Vector, Count);
  FREE(Vector);
  if (Result <= ERROR) {
    if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR) || (errno == ENOMEM) || (errno == ENOBUFS)) { return; }
    ares_tcp_disconnect(Server, "Ares: Write error to nameserver %s TCP: [%d]: %s", Server->HostIPS, errno, strerror(errno));
    return;
  }

  while (Result > 0) {
    SendQ = Server->TCPSend_Head;
    if (Server->TCPSend_Head == NULL) { break; }
    if (Result >= SendQ->Len) {
      Result -= SendQ->Len;
      Server->TCPSend_Head = SendQ->Next;
      free(SendQ);
    }
    else {
      SendQ->Buffer += Result;
      SendQ->Len -= Result;
      break;
    }
  }

}

