/*
 * ----------------------------------------------------------------
 * Night Light IRC Proxy Client I/O System
 * ----------------------------------------------------------------
 * Copyright (C) 1997-2003 Jonas Kvinge <jonas@night-light.net>
 * All rights reserved.
 *
 * This code is a result of thousands of hours of work by
 * Jonas Kvinge <jonas@night-light.net>
 *
 * You may not create derivative works based on this code.
 *
 * Modified source code or binaries compiled from modified source
 * code distributed in any shape or form without the authors
 * permission is expressly forbidden.
 *
 * This code is provided without warranty of any kind.
 *
 * Under no circumstances and under no legal contract or otherwise
 * shall Jonas Kvinge be liable to you or any other person for any
 * damages, computer failure, malfunction or any other damages or
 * losses.
 *
 * Last modified by:
 * Jonas Kvinge (31.07.2001)
 *
 */

#define CLIENT_IO_C

#define NEED_SYS_TYPES_H 1		/* Extra types */
#define NEED_SYS_PARAM_H 1		/* Some systems need this */
#define NEED_LIMITS_H 0			/* Kernel limits */
#define NEED_STDARG_H 1			/* va_list, etc */
#define NEED_ERRNO_H 1			/* errno */
#define NEED_CTYPE_H 0			/* isdigit(), etc */
#define NEED_NETINET_IN_H 1		/* in_addr, sockaddr_in, etc */
#define NEED_ARPA_INET_H 1		/* inet_ntoa(), inet_aton(), etc */
#define NEED_STDIO_H 1			/* Standard C UNIX functions */
#define NEED_STDLIB_H 1			/* malloc(), exit(), atoi(), etc */
#define NEED_TIME_H 1			/* time(), etc */
#define NEED_SYSCTL_H 0			/* sysctl(), etc */
#define NEED_SYS_STAT_H 0		/* chmod(), mkdir(), etc */
#define NEED_SYS_UIO_H 0		/* iovec, etc */
#define NEED_FCNTL_H 1			/* open(), creat(), fcntl(), etc */
#define NEED_SYS_IOCTL_H 0		/* ioctl(), etc */
#define NEED_SYS_FILIO_H 0		/* Solaris need this for ioctl(), etc */
#define NEED_UNISTD_H 1			/* Unix standard functions */
#define NEED_STRING_H 1			/* C string functions */
#define NEED_SIGNAL_H 0			/* Signal functions */
#define NEED_SYS_SOCKET_H 1		/* Socket functions */
#define NEED_NETDB_H 1			/* Network database functions */
#define NEED_ARPA_NAMESER_H 0		/* Nameserver definitions */
#define NEED_GETUSERPW_HEADERS 0 	/* Functions to retrive system passwords */

#include "includes.h"

#include "irc.h"

#if ARES
  #include "ares_io.h"
#endif

#include "access_conf.h"

#include "listen_io.h"

#include "client_io.h"
#include "client_auth.h"

#include "conn_io.h"
#include "conn_sendq.h"

#include "chan.h"
#include "chan_user.h"

/* VARIABLES - JONAS (31.07.2001) */

struct Client_Struct *Client_Head = NULL;
struct Client_Struct *Client_Tail = NULL;
unsigned long int G_Clients = 0;

extern struct AccessConf_Struct *Allow_Head;
extern struct AccessConf_Struct *Deny_Head;

extern unsigned short int Access_CheckFirst;
extern unsigned short int Access_Default;

extern struct Conn_Struct *Conn_Head;

/* CLIENT_ADD - JONAS (01.08.2001) */

struct Client_Struct *client_add(struct Listen_Struct *ListenS, const char *const HostIPSPT, const unsigned long int HostIPH, const unsigned long long int HostIPN, const unsigned long int PortH, const unsigned long int PortN, const signed long int FD) {

  struct Client_Struct *ClientS = NULL;
  struct Client_Struct *ClientS_NEW = NULL;

  assert(HostIPSPT != NULL);

  ClientS_NEW = malloc(sizeof(struct Client_Struct));
  if (ClientS_NEW == NULL) {
    aerrno = AEMALLOC;
    return(NULL);
  }
  memset(ClientS_NEW, 0, sizeof(struct Client_Struct));

  ClientS_NEW->HostIPS = strdup(HostIPSPT);
  if (ClientS_NEW->HostIPS == NULL) {
    free(ClientS_NEW->HostIPS);
    free(ClientS_NEW);
    aerrno = AEMALLOC;
    return(NULL);
  }

  ClientS_NEW->HostIPH = HostIPH;
  ClientS_NEW->HostIPN = HostIPN;
  ClientS_NEW->PortH = PortH;
  ClientS_NEW->PortN = PortN;
  ClientS_NEW->FD = FD;
  ClientS_NEW->Time = NOW;
  ClientS_NEW->ListenS = ListenS;

  Client_SetSocket(ClientS_NEW);

  if (Client_Head == NULL) {
    Client_Head = ClientS_NEW;
    Client_Tail = ClientS_NEW;
  }
  else {
    ClientS = Client_Tail;
    ClientS->Next = ClientS_NEW;
    ClientS_NEW->Prev = ClientS;
    Client_Tail = ClientS_NEW;
  }

  G_Clients++;

  client_resolve(ClientS_NEW);

  aerrno = AESUCCESS;
  return(ClientS_NEW);

}

/* CLIENT_REM FUNCTION - JONAS (01.08.2001) */

void client_rem(struct Client_Struct *ClientS) {

  assert(ClientS != NULL);

  if (ClientS->Prev == NULL) { Client_Head = ClientS->Next; }
  else { ClientS->Prev->Next = ClientS->Next; }

  if (ClientS->Next == NULL) { Client_Tail = ClientS->Prev; }
  else { ClientS->Next->Prev = ClientS->Prev; }

  free(ClientS->HostName);
  free(ClientS->HostIPS);
  free(ClientS->Nick);
  free(ClientS->User);
  free(ClientS->Host);
  free(ClientS->Pass);
  free(ClientS->RecvBuffer);
  free(ClientS->SendBuffer);
  free(ClientS);

  G_Clients--;

}

/* CLIENT_RESOLVE FUNCTION - JONAS (01.07.2000) */

void client_resolve(struct Client_Struct *ClientS) {

  #if !ARES
    struct hostent *HostEnt = NULL;
    unsigned long int HostEntLen = sizeof(struct hostent);
  #endif

  assert(ClientS != NULL);

  if (!Host_IsHostIPToName(ClientS->ResolveFlags)) {
    sysprint(BITMASK_DEBUG_USERIO, "Attempting to resolve client IP-address %s to hostname...", ClientS->HostIPS);
    Host_SetResolving(ClientS->ResolveFlags);
    #if ARES
      ares_gethostbyaddr(&ClientS->HostIPN, sizeof(ClientS->HostIPN), AF_INET, &client_hostiptoname, ClientS);
      return;
    #else
      HostEnt = gethostbyaddr(&ClientS->HostIPN, sizeof(ClientS->HostIPN), AF_INET);
      client_hostiptoname(ClientS, errno, HostEnt);
    #endif
  }

  if (!Host_IsHostNameToIP(ClientS->ResolveFlags)) {
    sysprint(BITMASK_DEBUG_USERIO, "Attempting to resolve client hostname %s (PTR on IP-Address %s) to IP-address...", ClientS->HostName, ClientS->HostIPS);
    Host_SetResolving(ClientS->ResolveFlags);
    #if ARES
      ares_gethostbyname(ClientS->HostName, AF_INET, &client_hostnametoip, ClientS);
      return;
    #else
      HostEnt = gethostbyname(ClientS->HostName);
      client_hostnametoip(ClientS, errno, HostEnt);
    #endif
  }

  Host_SetResolved(ClientS->ResolveFlags);

  sysprint(BITMASK_MAIN, "New client from %s resolved to %s.", ClientS->HostIPS, ClientS->HostName);
  ClientS->Time = NOW;

  if (G_Clients > MAXCLIENTS) { client_close(ClientS, "The maximum number of clients has been reached."); return; }
  if (client_accessconfcheck(ClientS) == FALSE) { return; }

  client_notice(ClientS, "%s v%s", LONGNAME, VERSION);
  client_notice(ClientS, "%s", COPYRIGHT);

}

/* CLIENT_HOSTIPTONAME FUNCTION - JONAS (01.03.2000) */

void client_hostiptoname(void *ArgPT, unsigned short int ErrNo, struct hostent *HostEnt) {

  struct Client_Struct *ClientS = ArgPT;

  assert(ClientS != NULL);

  Host_ClearResolving(ClientS->ResolveFlags);
  if (!Client_IsSocket(ClientS)) { return; } /* The user has already closed the connection */
  Host_SetHostIPToName(ClientS->ResolveFlags);

  if (HostEnt == NULL) {
    sysprint(BITMASK_DEBUG_USERIO, "Unable to resolve client IP-address %s to hostname: [%d] %s", ClientS->HostIPS, ErrNo, ares_strerror(ErrNo));
    ClientS->HostName = strrealloc(ClientS->HostName, HOST_UNRESOLVED);
    Host_SetHostNameToIP(ClientS->ResolveFlags);
  }
  else {
    ClientS->HostName = strrealloc(ClientS->HostName, HostEnt->h_name);
    if (aerrno != AESUCCESS) {
      client_close(ClientS, "Memory allocation failure: [%d] %s", errno, strerror(errno));
      return;
    }
    Host_SetHostIPToName(ClientS->ResolveFlags);
    sysprint(BITMASK_DEBUG_USERIO, "Resolved client IP-address %s to hostname %s.", ClientS->HostIPS, ClientS->HostName);
  }

  #if ARES
    client_resolve(ClientS);
  #endif

}

/* CLIENT_HOSTNAMETOIP FUNCTION - JONAS (01.03.2000) */

void client_hostnametoip(void *ArgPT, unsigned short int ErrNo, struct hostent *HostEnt) {

  struct Client_Struct *ClientS = ArgPT;
  unsigned long int HostIPN = 0;
  char *HostIPPT = NULL;

  assert(ClientS != NULL);

  Host_ClearResolving(ClientS->ResolveFlags);
  if (!Client_IsSocket(ClientS)) { return; } /* The user has already closed the connection */
  Host_SetHostNameToIP(ClientS->ResolveFlags);

  if (HostEnt == NULL) {
    sysprint(BITMASK_DEBUG_USERIO, "Unable to resolve client hostname %s (PTR on IP-Address %s) to IP-address: [%d] %s", ClientS->HostName, ClientS->HostIPS, ErrNo, ares_strerror(ErrNo));
    ClientS->HostName = strrealloc(ClientS->HostName, HOST_UNRESOLVED);
  }
  else {
    memcpy(&HostIPN, HostEnt->h_addr, HostEnt->h_length);
    if (ClientS->HostIPN == HostIPN) { sysprint(BITMASK_DEBUG_USERIO, "Resolved client hostname %s (PTR on IP-Address %s) to correct IP-address %s.", ClientS->HostName, ClientS->HostIPS, ClientS->HostIPS); }
    else {
      struct in_addr INAddr;
      memset(&INAddr, 0, sizeof(INAddr));
      INAddr.s_addr = HostIPN;
      HostIPPT = inet_ntoa(INAddr);
      sysprint(BITMASK_DEBUG_USERIO, "Resolved client hostname %s (PTR on IP-Address %s) to a different IP-address %s.", ClientS->HostName, ClientS->HostIPS, HostIPPT);
      ClientS->HostName = strrealloc(ClientS->HostName, HOST_UNRESOLVED);
    }
  }

  #if ARES
    client_resolve(ClientS);
  #endif

}

/* CLIENT_ACCESSCONFCHECK FUNCTION - JONAS (31.07.2001) */

unsigned short int client_accessconfcheck(struct Client_Struct *ClientS) {

  struct AccessConf_Struct *AllowConf = NULL;
  struct AccessConf_Struct *DenyConf = NULL;

  for (AllowConf = Allow_Head ; AllowConf != NULL ; AllowConf = AllowConf->Next) {
    if ((ClientS->HostName != NULL) && (strwm(AllowConf->Host, ClientS->HostName) == TRUE)) { break; }
    if (strwm(AllowConf->Host, ClientS->HostIPS) == TRUE) { break; }
  }

  for (DenyConf = Deny_Head ; DenyConf != NULL ; DenyConf = DenyConf->Next) {
    if ((ClientS->HostName != NULL) && (strwm(DenyConf->Host, ClientS->HostName) == TRUE)) { break; }
    if (strwm(DenyConf->Host, ClientS->HostIPS) == TRUE) { break; }
  }

  if (Access_CheckFirst == ACCESS_ALLOW) {
    if (AllowConf != NULL) {
      sysprint(BITMASK_DEBUG_USERIO, "Allowing connection from %s(%s): Matching ALLOW rule: \"%s\".", ClientS->HostName, ClientS->HostIPS, AllowConf->Host);
      return(TRUE);
    }
    if (DenyConf != NULL) {
      sysprint(BITMASK_DEBUG_USERIO, "Refusing connection from %s(%s): Matching DENY rule: \"%s\".", ClientS->HostName, ClientS->HostIPS, DenyConf->Host);
#if CLIENT_SHOWDENYREASON
      client_close(ClientS, "Connection denied: Your host is matching DENY rule \"%s\": %s", DenyConf->Host, DenyConf->Reason);
#else
      client_close(ClientS, "Connection denied from your host: %s", DenyConf->Reason);
#endif
      return(FALSE);
    }
  }
  else {
    if (DenyConf != NULL) {
      sysprint(BITMASK_DEBUG_USERIO, "Refusing connection from %s(%s): Matching DENY rule: \"%s\".", ClientS->HostName, ClientS->HostIPS, DenyConf->Host);
#if CLIENT_SHOWDENYREASON
      client_close(ClientS, "Connection denied: Your host is matching DENY rule \"%s\": %s", DenyConf->Host, DenyConf->Reason);
#else
      client_close(ClientS, "Connection denied from your host: %s", DenyConf->Reason);
#endif
      return(FALSE);
    }
    if (AllowConf != NULL) {
      sysprint(BITMASK_DEBUG_USERIO, "Allowing connection from %s(%s): Matching ALLOW rule: \"%s\".", ClientS->HostName, ClientS->HostIPS, AllowConf->Host);
      return(TRUE);
    }
  }

  if (Access_Default == ACCESS_ALLOW) {
    sysprint(BITMASK_DEBUG_USERIO, "Allowing connection from %s(%s): No matching rule, default = ALLOW.", ClientS->HostName, ClientS->HostIPS);
    return(TRUE);
  }
  else {
    sysprint(BITMASK_DEBUG_USERIO, "Refusing connection from %s(%s): No matching rule, default = DENY.", ClientS->HostName, ClientS->HostIPS);
#if CLIENT_SHOWDENYREASON
    client_close(ClientS, "Connection denied: No matching rule for your host, defaulting to DENY.");
#else
    client_close(ClientS, "Connection denied from your host: %s", ACCESS_CONF_NOREASON);
#endif
    return(FALSE);
  }

}

/* CLIENT_FDS FUNCTION - JONAS (22.07.2001) */

void client_fds(fd_set *ReadFDS, fd_set *WriteFDS, unsigned long int *FDS) {

  struct Client_Struct *ClientS = NULL;
  struct Client_Struct *ClientS_DEL = NULL;
  time_t Duration = 0;

  for (ClientS = Client_Head ; ClientS != NULL ;) {

    if ((Host_IsResolved(ClientS->ResolveFlags)) && (Client_IsSocket(ClientS))) {
      if (Client_IsVerified(ClientS)) {
        if (Client_IsSentPing(ClientS)) {
          Duration = (NOW - ClientS->SentPingTime);
          if (Duration >= CLIENT_PINGTIMEOUT) { client_close(ClientS, "No response from client in %s, closing connection to client.", strduration(Duration)); }
        }
        else {
          Duration = (NOW - ClientS->LastRecvTime);
          if (Duration >= CLIENT_IDLETIMEBEFOREPING) {
            ClientS->SentPingTime = NOW;
            Client_SetSentPing(ClientS);
            client_addsend(ClientS, "PING :ircproxy");
          }
        }
      }
      else {
        Duration = (NOW - ClientS->Time);
        if (Duration >= CLIENT_VERIFYTIMEOUT) { client_close(ClientS, "No response from client in %s while waiting for PASS/NICK/USER, closing connection to client.", strduration(Duration)); }
      }
    }

    if (Client_IsSentError(ClientS)) {
      Duration = (NOW - ClientS->SentErrorTime);
      if (Duration >= CLIENT_ERRORTIMEOUT) { client_cleanup(ClientS, "Disconnecting client %s (%s): Timeout while closing connection.", ClientS->HostName, ClientS->HostIPS); }
    }

    if ((Client_IsSocket(ClientS)) && (Host_IsResolved(ClientS->ResolveFlags))) {
      FD_SET(ClientS->FD, ReadFDS);
      if (ClientS->SendBuffer != NULL) { FD_SET(ClientS->FD, WriteFDS); }
    }

    if ((!Client_IsSocket(ClientS)) && (!Host_IsResolving(ClientS->ResolveFlags))) {
      ClientS_DEL = ClientS;
      ClientS = ClientS->Next;
      client_rem(ClientS_DEL);
      continue;
    }
    ClientS = ClientS->Next;

  }

}

/* CLIENT_IO FUNCTION - JONAS (01.07.2000) */

void client_io(fd_set *ReadFDS, fd_set *WriteFDS, unsigned long int *FDS) {

  struct Client_Struct *ClientS = NULL;

  for (ClientS = Client_Head ; ClientS != NULL ; ClientS = ClientS->Next) {

    assert(*FDS >= 0);
    if (*FDS <= 0) { return; }

    if (Client_IsSocket(ClientS)) {
      if (FD_ISSET(ClientS->FD, ReadFDS)) { *FDS = *FDS - 1; }
      if (FD_ISSET(ClientS->FD, WriteFDS)) { *FDS = *FDS - 1; }
    }

    if (Client_IsSocket(ClientS)) {
      if (FD_ISSET(ClientS->FD, ReadFDS)) { client_recv(ClientS); }
    }
    if (Client_IsSocket(ClientS)) {
      if (FD_ISSET(ClientS->FD, WriteFDS)) { client_send(ClientS); }
    }

  }

}

/* CLIENT_RECV FUNCTION - JONAS (01.07.2000) */

void client_recv(struct Client_Struct *ClientS) {

  signed long int Result = 0;
  unsigned long int OldLen = 0;
  unsigned long int NewLen = 0;
  char RecvBuffer[RECVBUFFERLEN+1] = "";
  char *RecvBufferPT = NULL;

  assert(ClientS != NULL);

  ClientS->LastRecvTime = NOW;

  do {
    memset(&RecvBuffer, 0, sizeof(RecvBuffer));
    Result = recv(ClientS->FD, RecvBuffer, RECVBUFFERLEN, MSG_NOSIGNAL);
    if (Result <= ERROR) {
      if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR)) { break; }
      client_cleanup(ClientS, "Read error to client %s (%s): [%d] %s", ClientS->HostName, ClientS->HostIPS, errno, strerror(errno));
      return;
    }
    if (Result == 0) {
      if (ClientS->RecvBuffer != NULL) { client_parse(ClientS); }
      client_cleanup(ClientS, "EOF to client %s (%s).", ClientS->HostName, ClientS->HostIPS);
      return;
    }
    if (ClientS->RecvBuffer == NULL) { OldLen = 0; }
    else { OldLen = strlen(ClientS->RecvBuffer); }
    NewLen = OldLen + Result + 1;
    RecvBufferPT = realloc(ClientS->RecvBuffer, NewLen);
    if (RecvBufferPT == NULL) {
      client_close(ClientS, "Memory allocation failure: [%d] %s", errno, strerror(errno));
      return;
    }
    ClientS->RecvBuffer = RecvBufferPT;
    RecvBufferPT += OldLen;
    strcpy(RecvBufferPT, RecvBuffer);
  }
  while (Result >= RECVBUFFERLEN);

  if (Client_IsSentError(ClientS)) { return; }

  client_parse(ClientS);

}

/* CLIENT_SEND FUNCTION - JONAS (01.07.2000) */

void client_send(struct Client_Struct *ClientS) {

  char *SendBufferPT = NULL;
  char SendBuffer[SENDBUFFERLEN+1] = "";
  unsigned long int SendLen = 0;
  unsigned long int SentLen = 0;
  signed long int Result = 0;

  assert(ClientS != NULL);

  for (SendBufferPT = ClientS->SendBuffer ; *SendBufferPT != '\0' ; SendBufferPT += SentLen) {
    SendLen = strlen(SendBufferPT);
    if (SendLen > SENDBUFFERLEN) { SendLen = SENDBUFFERLEN; }
    memset(&SendBuffer, 0, sizeof(SendBuffer));
    strncpy(SendBuffer, SendBufferPT, SendLen);
    Result = send(ClientS->FD, SendBufferPT, SendLen, MSG_NOSIGNAL);
    if (Result <= ERROR) {
      if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR) || (errno == ENOMEM) || (errno == ENOBUFS)) {
        unsigned long int Len = 0;
        sysprint(BITMASK_MAIN, "Write error to client %s (%s): [%d] %s", ClientS->HostName, ClientS->HostIPS, errno, strerror(errno));
        /*
         * EAGAIN/EWOULDBLOCK - THE SOCKET IMPLEMENTATION CAN'T HANDLE MORE DATA.
         * EINTR - INTERRUPTED BY A SIGNAL.
         * ENOMEM - NO MEMORY LEFT.
         * ENOBUFS - NO BUFFER SPACE AVAILABLE.
         *
         * COPY WHATS LEFT TO THE THE START OF THE SENDBUFFER, REALLOCATE THE MEMORY AND BAIL OUT - JONAS (01.12.1999)
         *
         */
        Len = strlen(SendBufferPT) + 1;
        memmove(ClientS->SendBuffer, SendBufferPT, Len);
        SendBufferPT = realloc(ClientS->SendBuffer, Len);
        assert(SendBufferPT != NULL);
        ClientS->SendBuffer = SendBufferPT;
        return;
      }
      client_cleanup(ClientS, "Write error to client %s (%s): [%d] %s", ClientS->HostName, ClientS->HostIPS, errno, strerror(errno));
      return;
    }
    SentLen = Result;
    assert(SentLen = SendLen);
  }

  FREE(ClientS->SendBuffer);

  if (Client_IsSentError(ClientS)) { client_cleanup(ClientS, "Successfully closed connection to client %s (%s).", ClientS->HostName, ClientS->HostIPS); }

}

/* CLIENT_ADDSEND FUNCTION - JONAS (01.07.2000) */

void client_addsend(struct Client_Struct *ClientS, const char *const LinePT, ...) {

  char Line[IRCMSGLEN+1] = "";
  char LineCRLF[IRCMSGCRLFLEN+1] = "";
  va_list Args = { 0 };
  unsigned long int Len = 0;
  unsigned long int OldLen = 0;
  unsigned long int NewLen = 0;
  char *SendBufferPT = NULL;

  assert(ClientS != NULL);
  assert(LinePT != NULL);

  va_start(Args, LinePT);
  vsnprintf(Line, IRCMSGLEN+1, LinePT, Args);
  va_end(Args);

  snprintf(LineCRLF, IRCMSGCRLFLEN+1, "%s\r\n", Line);

  Len = strlen(LineCRLF);
  if (ClientS->SendBuffer == NULL) { OldLen = 0; }
  else { OldLen = strlen(ClientS->SendBuffer); }
  NewLen = OldLen + Len + 1;
  SendBufferPT = realloc(ClientS->SendBuffer, NewLen);
  if (SendBufferPT == NULL) { return; }
  ClientS->SendBuffer = SendBufferPT;
  SendBufferPT = SendBufferPT + OldLen;
  strcpy(SendBufferPT, LineCRLF);

}

/* CLIENT_PARSE FUNCTION - JONAS (18.07.2001) */

void client_parse(struct Client_Struct *ClientS) {

  char *BufferPT = NULL;
  char *LinePT = NULL;
  unsigned long int Count = 0;
  unsigned long int Len = 0;
  char *TempPT = NULL;

  BufferPT = ClientS->RecvBuffer;

  for (Count = 1 ; BufferPT != NULL ; Count++) {
    TempPT = strchr(BufferPT, '\n');
    if (TempPT == NULL) {
      if (Count == 1) { return; }
      Len = strlen(BufferPT) + 1;
      memmove(ClientS->RecvBuffer, BufferPT, Len);
      TempPT = realloc(ClientS->RecvBuffer, Len);
      if (TempPT == NULL) {
        client_close(ClientS, "Memory allocation failure: [%d] %s", errno, strerror(errno));
        return;
      }
      ClientS->RecvBuffer = TempPT;
      return;
    }
    LinePT = strtok(BufferPT, "\n");
    BufferPT = strtok(NULL, "\0");
    if (LinePT == NULL) { continue; }
    Len = strlen(LinePT);
    if (LinePT[Len-1] == '\r') { LinePT[Len-1] = '\0'; }
    client_parse_message(ClientS, LinePT);
    if (ClientS->RecvBuffer == NULL) { return; }
  }
  FREE(ClientS->RecvBuffer);

}

/* CLIENT_PARSE_MESSAGE FUNCTION - JONAS (17.07.2001) */

void client_parse_message(struct Client_Struct *ClientS, char *MessagePT) {

  char *PrefixPT = NULL;
  char *LinePT = NULL;
  char *CommandPT = NULL;
  char **ParamsPT = NULL;
  char **TempPT = NULL;
  unsigned short int Index = 0;
  unsigned short int Params = 0;
  time_t Duration = 0;

  Duration = (NOW - ClientS->LineTime);
  if (Duration >= CLIENT_LINEFLOODTTL) { ClientS->Lines = 0; }

  ++ClientS->Lines;
  if (ClientS->Lines >= CLIENT_MAXLINES) {
    client_close(ClientS, ACCESS_CONF_USERFLOOD);
    access_conf_adddeny(ClientS->HostName, ACCESS_CONF_USERFLOOD);
    return;
  }
  ClientS->LineTime = NOW;

  while (MessagePT[0] == ' ') { MessagePT++; }
  if (MessagePT[0] == '\0') { return; }

  if (MessagePT[0] == ':') {
    MessagePT++;
    PrefixPT = MessagePT;
    MessagePT = strchr(MessagePT, ' ');
    if (MessagePT == NULL) { return; }
    MessagePT[0] = '\0';
    MessagePT++;
    while (MessagePT[0] == ' ') { MessagePT++; }
    if (MessagePT[0] == '\0') { return; }
  }
  else {
    if (ClientS->User == NULL) { PrefixPT = "*"; }
    else { PrefixPT = ClientS->User; }
  }

  LinePT = strdup(MessagePT);
  if (LinePT == NULL) {
    client_close(ClientS, "Memory allocation failure: [%d] %s", errno, strerror(errno));
    return;
  }

  CommandPT = MessagePT;
  MessagePT = strchr(MessagePT, ' ');
  if (MessagePT != NULL) {
    MessagePT[0] = '\0';
    MessagePT++;
    while (MessagePT[0] == ' ') { MessagePT++; }
    if (MessagePT[0] == '\0') { return; }

    FOREVERLOOP {
      if (MessagePT[0] == ':') {
        MessagePT++;
        ++Params;
        TempPT = realloc(ParamsPT, (sizeof(char *) * Params));
        if (TempPT == NULL) {
          client_close(ClientS, "Memory allocation failure: [%d] %s", errno, strerror(errno));
          free(ParamsPT);
          return;
        }
        ParamsPT = TempPT;
        ParamsPT[Index] = MessagePT;
        break;
      }
      else {
        ++Params;
        TempPT = realloc(ParamsPT, (sizeof(char *) * Params));
        if (TempPT == NULL) {
          client_close(ClientS, "Memory allocation failure: [%d] %s", errno, strerror(errno));
          free(ParamsPT);
          return;
        }
        ParamsPT = TempPT;
        ParamsPT[Index] = MessagePT;
        ++Index;
        MessagePT = strchr(MessagePT, ' ');
        if (MessagePT == NULL) { break; }
        MessagePT[0] = '\0';
        MessagePT++;
        while (MessagePT[0] == ' ') { MessagePT++; }
        if (MessagePT[0] == '\0') { break; }
      }
    }
  }

  strupper(CommandPT);

  client_parse_command(ClientS, LinePT, CommandPT, ParamsPT, Params);

  free(LinePT);
  free(ParamsPT);

}

/* CLIENT_PARSE_COMMAND FUNCTION - JONAS (17.07.2001) */

void client_parse_command(struct Client_Struct *ClientS, char *LinePT, char *CommandPT, char **ParamsPT, const unsigned short int Params) {

  if (!Client_IsVerified(ClientS)) {
    if (strcasecmp(CommandPT, "PASS") == FALSE) {
      if (Params < 1) { client_notice(ClientS, "%s :Not enough parameters.", CommandPT); return; }
      ClientS->Pass = strrealloc(ClientS->Pass, ParamsPT[0]);
#if 0
      client_notice(ClientS, "Got password.");
#endif
    }
    else if (strcasecmp(CommandPT, "NICK") == FALSE) {
      if (Params < 1) { client_notice(ClientS, "%s :Not enough parameters.", CommandPT); return; }
      ClientS->Nick = strrealloc(ClientS->Nick, ParamsPT[0]);
#if 0
      client_notice(ClientS, "Got nickname.");
#endif
      if (ClientS->Pass == NULL) { client_notice(ClientS, "You need to send the PASS command!"); }
    }
    else if (strcasecmp(CommandPT, "USER") == FALSE) {
      if (Params < 4) { client_notice(ClientS, "%s :Not enough parameters.", CommandPT); return; }
      ClientS->User = strrealloc(ClientS->User, ParamsPT[0]);
#if 0
      client_notice(ClientS, "Got username.");
#endif
      if (ClientS->Pass == NULL) { client_notice(ClientS, "You need to send the PASS command!"); }
    }
    else { client_addsend(ClientS, ":%s 451 * :You have not registered.", IRCP_USER_HOST(ClientS)); }

    if ((ClientS->User != NULL) && (ClientS->Pass != NULL)) {

      struct Conn_Struct *ConnS = NULL;

      assert(geteuid() != 0);
      client_authcheck(ClientS->User, ClientS->Pass);
      assert(geteuid() != 0);
      if (aerrno != AESUCCESS) {
        client_close(ClientS, "USER / PASS is incorrect.");
        return;
      }

      sysprint(BITMASK_MAIN, "Client %s (%s) sucessfully authenticated as \"%s\".", ClientS->HostName, ClientS->HostIPS, ClientS->User);
      client_noticeallbutone(ClientS, "Client %s (%s) authenticated as \"%s\".", ClientS->HostName, ClientS->HostIPS, ClientS->User);

      Client_SetVerified(ClientS);
      client_welcome(ClientS);

      for (ConnS = Conn_Head ; ConnS != NULL ; ConnS = ConnS->Next) {
        if ((strcmp(ClientS->User, ConnS->User) == FALSE) && (Conn_IsAttached(ConnS))) { client_attach(ClientS, ConnS); break; }
      }
      return;
    }
    return;
  }

  if (strcasecmp(CommandPT, "PCONNLIST") == FALSE) {

    struct Conn_Struct *ConnS = NULL;
    struct ConnServer_Struct *ConnServer = NULL;
    unsigned short int Count = 0;

    for (ConnS = Conn_Head ; ConnS != NULL ; ConnS = ConnS->Next) {

      if (strcmp(ConnS->User, ClientS->User) != FALSE) { continue; }
      ++Count;

      client_notice(ClientS, "\2Connection: %s Host: %s\2", ConnS->Name, ConnS->Host);

      if (Host_IsResolving(ConnS->ResolveFlags)) { client_notice(ClientS, "--- Status: Resolving local hostname %s to IP-address.", ConnS->HostName); }
      else if (Host_IsResolving(ConnS->ServerResolveFlags)) { client_notice(ClientS, "--- Status: Resolving server nostname %s to IP-address.", ConnS->ServerHostName); }
      else if (Conn_IsConnecting(ConnS)) { client_notice(ClientS, "--- Status: Connecting to %s(%s):%ld.", ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH); }
      else if (Conn_IsWelcome(ConnS)) { client_notice(ClientS, "--- Status: Connected to %s(%s):%ld \"%s\".", ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, ConnS->ServerName); }
      else if (Conn_IsConnected(ConnS)) { client_notice(ClientS, "--- Status: Socket connected to %s(%s):%ld.", ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH); }
      else { client_notice(ClientS, "--- Status: Disconnected."); }

      if (ConnS->Server_Head == NULL) { client_notice(ClientS, "--- No servers!"); }
      else {
        for (ConnServer = ConnS->Server_Head ; ConnServer != NULL ; ConnServer = ConnServer->Next) {
          client_notice(ClientS, "--- Server: %s Port: %ld", ConnServer->Host, ConnServer->Port);
        }
      }

    }

    if (Count == 0) {
      client_notice(ClientS, "You have no connections.");
    }

    return;

  }
  else if (strcasecmp(CommandPT, "PCONNECT") == FALSE) {

    struct Conn_Struct *ConnS = NULL;

    if (Params < 1) {
      if (ClientS->ConnS == NULL) { client_notice(ClientS, "Usage: PCONNECT <Connection name> [Connection server]."); return; }
      ConnS = ClientS->ConnS;
    }
    else { ConnS = conn_get(ParamsPT[0]); }

    if (ConnS == NULL) { client_notice(ClientS, "No such connection, use /PCONNLIST to list your connections."); return; }

    if (strcmp(ConnS->User, ClientS->User) != FALSE) { client_notice(ClientS, "Permission denied, your not the owner of connection %s.", ConnS->Name); return; }

    if (Params >= 2) {
      for (ConnS->ConnServerTry = ConnS->Server_Head ; ConnS->ConnServerTry != NULL ; ConnS->ConnServerTry = ConnS->ConnServerTry->Next) {
        if (strwm(ParamsPT[1], ConnS->ConnServerTry->Host) == TRUE) { break; }
      }
      if (ConnS->ConnServerTry == NULL) { client_notice(ClientS, "No such server for that connection."); return; }
    }
    if (Host_IsResolving(ConnS->ResolveFlags)) {
      if (ConnS->ConnServerTry == NULL) { client_notice(ClientS, "Connection %s: Resolving local hostname %s to IP-address.", ConnS->Name, ConnS->HostName); return; }
      ares_cancelquery(NULL, ConnS);
      return;
    }
    else if (Host_IsResolving(ConnS->ServerResolveFlags)) {
      if (ConnS->ConnServerTry == NULL) { client_notice(ClientS, "Connection %s: Resolving server hostname %s to IP-address.", ConnS->Name, ConnS->ServerHostName); return; }
      ares_cancelquery(NULL, ConnS);
      return;
    }
    else if (Conn_IsConnecting(ConnS)) {
      if (ConnS->ConnServerTry == NULL) { client_notice(ClientS, "%s: Connecting to %s(%s):%ld.", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH); return; }
      conn_disconnect(ConnS, "Connection %s: Connection attempt to %s(%s):%ld cancelled by user %s(%s).", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, ClientS->HostName, ClientS->HostIPS);
      return;
    }
    else if (Conn_IsWelcome(ConnS)) {
      if ((ConnS->ConnServerTry == NULL) || (strcasecmp(ConnS->ServerHostName, ConnS->ConnServerTry->Host) == FALSE)) { client_notice(ClientS, "Connection %s: Already connected to %s, use PQUIT to quit.", ConnS->Name, ConnS->ServerHostName); return; }
      conn_quit(ConnS, "Changing server to %s.", ConnS->ConnServerTry->Host);
      return;
    }
    else if (Conn_IsConnected(ConnS)) {
      if (ConnS->ConnServerTry == NULL) { client_notice(ClientS, "Connection %s: Socket connected to %s(%s):%ld.", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH); return; }
      conn_disconnect(ConnS, "Connection %s attempt to %s(%s):%ld cancelled by user %s(%s).", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, ClientS->HostName, ClientS->HostIPS);
      return;
    }

    conn_connect(ConnS);
    return;

  }
  else if (strcasecmp(CommandPT, "PQUIT") == FALSE) {

    struct Conn_Struct *ConnS = NULL;

    if (Params >= 1) {
      ConnS = conn_get(ParamsPT[0]);
      if (ConnS == NULL) { client_notice(ClientS, "No such connection, eventually use /PCONNLIST to list your connections."); return; }
    }
    else {
      ConnS = ClientS->ConnS;
      if (ConnS == NULL) { client_notice(ClientS, "Usage: PQUIT <Connection name>"); return; }
    }

    if (strcmp(ConnS->User, ClientS->User) != FALSE) { client_notice(ClientS, "Permission denied, your not the owner of connection %s.", ConnS->Name); return; }

    if (Conn_IsConnect(ConnS)) {
      if (Conn_IsWelcome(ConnS)) {
        if (Conn_IsSentQuit(ConnS)) { conn_disconnect(ConnS, "Connection %s to server %s aborted by user %s (%s[%s]).", ConnS->Name, ConnS->ServerHostName, ClientS->User, ClientS->HostName, ClientS->HostIPS); return; }
        else { conn_quit(ConnS, "%s (%s[%s]) requested QUIT.", ClientS->User, ClientS->HostName, ClientS->HostIPS); }
      }
      else { conn_disconnect(ConnS, "Connection %s to server %s aborted by user %s (%s[%s]).", ConnS->Name, ConnS->ServerHostName, ClientS->User, ClientS->HostName, ClientS->HostIPS); return; }
    }
    else { client_notice(ClientS, "Connection %s is not connected.", ConnS->Name); return; }

    return;

  }
  else if ((strcasecmp(CommandPT, "PATTACH") == FALSE) || (strcasecmp(CommandPT, "PRESUME") == FALSE)) {

    struct Conn_Struct *ConnS = NULL;

    if (ClientS->ConnS != NULL) { client_notice(ClientS, "You are already attached to %s (%s)", ClientS->ConnS->Name, ClientS->ConnS->ServerHostName); return; }
    if (Params < 1) { client_notice(ClientS, "Usage: PATTACH <Connection name>"); return; }
    ConnS = conn_get(ParamsPT[0]);
    if (ConnS == NULL) { client_notice(ClientS, "No such connection."); return; }
    if (strcmp(ConnS->User, ClientS->User) != FALSE) { client_notice(ClientS, "Permission denied, your not the owner of connection %s.", ConnS->Name); return; }
    client_attach(ClientS, ConnS);

  }
  else if (strcasecmp(CommandPT, "PDETACH") == FALSE) {
    if (ClientS->ConnS == NULL) { client_notice(ClientS, "You have no attached connection."); return; }
    Conn_ClearAttached(ClientS->ConnS);
    client_detach(ClientS);
  }
  else if (strcasecmp(CommandPT, "QUIT") == FALSE) {
    client_close(ClientS, "Client sent QUIT.");
  }
  else if (strcasecmp(CommandPT, "PING") == FALSE) {
    if (Params < 1) { client_notice(ClientS, "%s :Not enough parameters.", CommandPT); return; }
    client_addsend(ClientS, "PONG :%s", ParamsPT[0]);
  }
  else if (strcasecmp(CommandPT, "PONG") == FALSE) {
    Client_ClearSentPing(ClientS);
  }
  else if (strcasecmp(CommandPT, "QUOTE") == FALSE) {

    char *DumbPT = NULL;

    if (ClientS->ConnS == NULL) { client_notice(ClientS, "QUOTE: No attached connection."); return; }
    if (!Conn_IsConnected(ClientS->ConnS)) { client_notice(ClientS, "Connection %s not established.", ClientS->ConnS->Name); return; }

    DumbPT = strchr(LinePT, ' ');
    DumbPT++;
    conn_addsendq(ClientS->ConnS, "%s", DumbPT);
  }
  else if (strcasecmp(CommandPT, "READLOG") == FALSE) {

    FILE *FilePT = NULL;
    struct Conn_Struct *ConnS = NULL;
    unsigned long int Count = 0;

    if (Params >= 1) {
      ConnS = conn_get(ParamsPT[0]);
      if (ConnS == NULL) { client_notice(ClientS, "No such connection, eventually use /PCONNLIST to list your connections."); return; }
    }
    else {
      ConnS = ClientS->ConnS;
      if (ConnS == NULL) { client_notice(ClientS, "Usage: READLOG <Connection name>"); return; }
    }

    if (strcmp(ConnS->User, ClientS->User) != FALSE) { client_notice(ClientS, "Permission denied, your not the owner of connection %s.", ConnS->Name); return; }
    if (ConnS->FileName == NULL) { client_notice(ClientS, "Logfile missing for connection %s.", ConnS->Name); return; }
#if !WIN32
    if (Conn_IsLogHomeDir(ConnS)) { sysseteuidbyuser(ConnS->User); }
#endif

    FilePT = fopen(ConnS->FileName, "r");
    if (FilePT == NULL) {
      sysprint(BITMASK_ERROR, "Unable to open %s for reading: [%d] %s", ConnS->FileName, errno, strerror(errno));
      client_notice(ClientS, "Unable to open %s for reading: [%d] %s", ConnS->FileName, errno, strerror(errno));
#if !WIN32
      if (Conn_IsLogHomeDir(ConnS)) { sysseteuidnormal(); }
#endif
      return;
    }

    FOREVERLOOP {

      char Line[LINELEN+1] = "";
      char *DumbPT = NULL;

      memset(&Line, 0, LINELEN+1);
      DumbPT = fgets(Line, LINELEN, FilePT);
      if (DumbPT == NULL) { break; }

      Count++;

      client_addsend(ClientS, ":%s PRIVMSG %s :%s", IRCP_USER_HOST(ClientS), IRCP_USER_NICK(ClientS), Line);

    }
    if (Count == 0) { client_notice(ClientS, "Logfile for %s is empty.", ConnS->Name); }
    fclose(FilePT);

#if !WIN32
    if (Conn_IsLogHomeDir(ConnS)) { sysseteuidnormal(); }
#endif

  }
  else if (strcasecmp(CommandPT, "ERASELOG") == FALSE) {

    FILE *FilePT = NULL;
    struct Conn_Struct *ConnS = NULL;
    unsigned long int Count = 0;

    if (Params >= 1) {
      ConnS = conn_get(ParamsPT[0]);
      if (ConnS == NULL) { client_notice(ClientS, "No such connection, eventually use /PCONNLIST to list your connections."); return; }
    }
    else {
      ConnS = ClientS->ConnS;
      if (ConnS == NULL) { client_notice(ClientS, "Usage: ERASELOG <Connection name>"); return; }
    }

    if (strcmp(ConnS->User, ClientS->User) != FALSE) { client_notice(ClientS, "Permission denied, your not the owner of connection %s.", ConnS->Name); return; }
    if (ConnS->FileName == NULL) { client_notice(ClientS, "Logfile missing for connection %s.", ConnS->Name); return; }
#if !WIN32
    if (Conn_IsLogHomeDir(ConnS)) { sysseteuidbyuser(ConnS->User); }
#endif

    FilePT = fopen(ConnS->FileName, "r");
    if (FilePT == NULL) {
      sysprint(BITMASK_ERROR, "Unable to open %s for reading: [%d] %s", ConnS->FileName, errno, strerror(errno));
      client_notice(ClientS, "Unable to open %s for reading: [%d] %s", ConnS->FileName, errno, strerror(errno));
#if !WIN32
      if (Conn_IsLogHomeDir(ConnS)) { sysseteuidnormal(); }
#endif
      return;
    }
    FOREVERLOOP {
      char Line[LINELEN+1] = "";
      char *DumbPT = NULL;
      memset(&Line, 0, LINELEN+1);
      DumbPT = fgets(Line, LINELEN, FilePT);
      if (DumbPT == NULL) { break; }
      Count++;
    }
    fclose(FilePT);
    if (Count == 0) {
      client_notice(ClientS, "Logfile for %s is empty.", ConnS->Name);
#if !WIN32
      if (Conn_IsLogHomeDir(ConnS)) { sysseteuidnormal(); }
#endif
      return;
    }

    FilePT = fopen(ConnS->FileName, "w+");
    if (FilePT == NULL) {
      sysprint(BITMASK_ERROR, "Unable to open %s for writing: [%d] %s", ConnS->FileName, errno, strerror(errno));
      client_notice(ClientS, "Unable to open %s for writing: [%d] %s", ConnS->FileName, errno, strerror(errno));
#if !WIN32
      if (Conn_IsLogHomeDir(ConnS)) { sysseteuidnormal(); }
#endif
      return;
    }
    fclose(FilePT);
#if !WIN32
    if (Conn_IsLogHomeDir(ConnS)) { sysseteuidnormal(); }
#endif

    client_notice(ClientS, "Logfile for %s erased.", ConnS->Name);

  }
  else {
    if (ClientS->ConnS == NULL) { client_notice(ClientS, "Unknown proxy command (No attached connection)."); }
    else {
      if (Conn_IsWelcome(ClientS->ConnS)) {
        conn_addsendq(ClientS->ConnS, "%s", LinePT);
      }
      else { client_notice(ClientS, "Unknown proxy command (Attached connection not established)."); }
    }
  }

}

/* CLIENT_NOTICE FUNCTION - JONAS (01.07.2000) */

void client_notice(struct Client_Struct *ClientS, const char *const LinePT, ...) {

  char Line[IRCNOTICELEN+1] = "";
  va_list Args = { 0 };

  assert(LinePT != NULL);

  va_start(Args, LinePT);
  vsnprintf(Line, IRCNOTICELEN+1, LinePT, Args);
  va_end(Args);

  if (!Client_IsSentError(ClientS)) { client_addsend(ClientS, ":%s NOTICE %s :%s", IRCP_USER_HOST(ClientS), IRCP_USER_NICK(ClientS), Line); }

}

/* CLIENT_NOTICEALLUSER FUNCTION - JONAS (01.07.2000) */

void client_noticealluser(const char *UserPT, const char *const LinePT, ...) {

  char Line[IRCNOTICELEN+1] = "";
  va_list Args = { 0 };
  struct Client_Struct *ClientS = NULL;

  assert(LinePT != NULL);

  va_start(Args, LinePT);
  vsnprintf(Line, IRCNOTICELEN+1, LinePT, Args);
  va_end(Args);

  sysprint(BITMASK_DEBUG_USERIO, "%s", Line);

  for (ClientS = Client_Head ; ClientS != NULL ; ClientS = ClientS->Next) {
    if ((Client_IsVerified(ClientS)) && (!Client_IsSentError(ClientS)) && (strcmp(ClientS->User, UserPT) == FALSE)) { client_addsend(ClientS, ":%s NOTICE %s :%s", IRCP_USER_HOST(ClientS), IRCP_USER_NICK(ClientS), Line); }
  }

}

/* CLIENT_NOTICEALLBUTONE FUNCTION - JONAS (01.07.2000) */

void client_noticeallbutone(struct Client_Struct *ClientO, const char *const LinePT, ...) {

  char Line[IRCNOTICELEN+1] = "";
  va_list Args = { 0 };
  struct Client_Struct *UserL = NULL;

  assert(LinePT != NULL);

  va_start(Args, LinePT);
  vsnprintf(Line, IRCNOTICELEN+1, LinePT, Args);
  va_end(Args);

  sysprint(BITMASK_DEBUG_USERIO, "%s", Line);

  for (UserL = Client_Head ; UserL != NULL ; UserL = UserL->Next) {
    if ((UserL != ClientO) && (Client_IsVerified(UserL)) && (!Client_IsSentError(UserL)) && (strcmp(UserL->User, ClientO->User) == FALSE)) { client_addsend(UserL, ":%s NOTICE %s :%s", IRCP_USER_HOST(ClientO), IRCP_USER_NICK(ClientO), Line); }
  }

}

/* CLIENT_NOTICEALL FUNCTION - JONAS (01.07.2000) */

void client_noticeall(const char *const LinePT, ...) {

  char Line[IRCNOTICELEN+1] = "";
  va_list Args = { 0 };
  struct Client_Struct *ClientS = NULL;

  assert(LinePT != NULL);

  va_start(Args, LinePT);
  vsnprintf(Line, IRCNOTICELEN+1, LinePT, Args);
  va_end(Args);

  for (ClientS = Client_Head ; ClientS != NULL ; ClientS = ClientS->Next) {
    if ((Client_IsVerified(ClientS)) && (!Client_IsSentError(ClientS))) { client_addsend(ClientS, ":%s NOTICE %s :%s", IRCP_USER_HOST(ClientS), IRCP_USER_NICK(ClientS), Line); }
  }

}

/* CLIENT_CLOSE FUNCTION - JONAS (09.06.2001) */

void client_close(struct Client_Struct *ClientS, const char *const LinePT, ...) {

  char Line[LINELEN+1] = "";
  va_list Args = { 0 };

  assert(ClientS != NULL);
  assert(LinePT != NULL);
  assert(!Client_IsSentError(ClientS));

  va_start(Args, LinePT);
  vsnprintf(Line, LINELEN+1, LinePT, Args);
  va_end(Args);

  sysprint(BITMASK_MAIN, "Closing connection to client %s (%s): %s", ClientS->HostName, ClientS->HostIPS, Line);
  if (Client_IsVerified(ClientS)) { client_noticeallbutone(ClientS, "Closing connection to client %s (%s): %s", ClientS->HostName, ClientS->HostIPS, Line); }

  Client_SetSentError(ClientS);
  ClientS->SentErrorTime = time(NULL);
  client_addsend(ClientS, "ERROR :%s", Line);

}

/* CLIENT_CLEANUP FUNCTION - JONAS (01.07.2000) */

void client_cleanup(struct Client_Struct *ClientS, const char *const LinePT, ...) {

  char Line[LINELEN+1] = "";
  va_list Args = { 0 };

  assert(ClientS != NULL);
  assert(LinePT != NULL);

  va_start(Args, LinePT);
  vsnprintf(Line, LINELEN+1, LinePT, Args);
  va_end(Args);

  sysprint(BITMASK_MAIN, "%s", Line);
  if (Client_IsVerified(ClientS)) { client_noticeallbutone(ClientS, "%s", Line); }

  if (Client_IsSocket(ClientS)) {
    close(ClientS->FD);
    ClientS->FD = FD_NONE;
  }
  ClientS->Flags = 0;

  FREE(ClientS->RecvBuffer);
  FREE(ClientS->SendBuffer);

  if (ClientS->ConnS != NULL) { client_detach(ClientS); }

}

/* CLIENT_CLOSEALL FUNCTION - JONAS (09.06.2001) */

unsigned short int client_closeall(const char *const MessagePT, ...) {

  char Message[LINELEN+1] = "";
  va_list Args = { 0 };
  unsigned short int Count = 0;
  struct Client_Struct *ClientS = NULL;

  assert(MessagePT != NULL);

  va_start(Args, MessagePT);
  vsnprintf(Message, LINELEN+1, MessagePT, Args);
  va_end(Args);

  for (ClientS = Client_Head ; ClientS != NULL ; ClientS = ClientS->Next) {
    if (!Client_IsSentError(ClientS)) { client_close(ClientS, "%s", Message); }
    ++Count;
  }

  return(Count);

}

/* CLIENT_WELCOME FUNCTION - JONAS (01.07.2000) */

void client_welcome(struct Client_Struct *ClientS) {

  struct Conn_Struct *P_ConnS = NULL;
  struct Client_Struct *P_ClientS = NULL;
  char Line1[LINELEN+1] = "";
  char Line2[LINELEN+1] = "";
  char *TempPT = NULL;
  char *ConnsPT = NULL;
  char *ClientsPT = NULL;
  unsigned short int ConnsCount = 0;
  unsigned short int UsersCount = 0;
  unsigned short int Len = 0;

  client_notice(ClientS, "Authenticated as %s!", ClientS->User);
  client_notice(ClientS, "Commands are: PCONNLIST, PATTACH, PDETACH, PCONNECT, PQUIT, READLOG and ERASELOG.");

  for (P_ConnS = Conn_Head , memset(&Line1, 0, LINELEN+1) , memset(&Line2, 0, LINELEN+1) ; P_ConnS != NULL ; P_ConnS = P_ConnS->Next) {

    if (strcmp(P_ConnS->User, ClientS->User) != FALSE) { continue; }

    ++ConnsCount;

    if (Conn_IsWelcome(P_ConnS)) { snprintf(Line1, LINELEN+1, "%s(Connected %s:%ld)", P_ConnS->Name, P_ConnS->ServerHostName, P_ConnS->ServerPortH); }
    else { snprintf(Line1, LINELEN+1, "%s(Disconnected)", P_ConnS->Name); }
    snprintf(Line2, LINELEN+1, ", %s", Line1);

    if (ConnsPT != NULL) {
      Len = strlen(ConnsPT) + strlen(Line2);
      if (Len > IRCNOTICELEN) {
        client_notice(ClientS, "%s.", ConnsPT);
        FREE(ConnsPT);
      }
    }

    if (ConnsPT == NULL) {
      TempPT = straddbuffer(ConnsPT, "Your IRC connections: ", IRCNOTICELEN, 0);
      if (TempPT == NULL) { break; }
      ConnsPT = TempPT;
      TempPT = straddbuffer(ConnsPT, Line1, IRCNOTICELEN, 0);
      if (TempPT == NULL) { FREE(ConnsPT); break; }
      ConnsPT = TempPT;
      continue;
    }

    TempPT = straddbuffer(ConnsPT, Line2, IRCNOTICELEN, 0);
    if (TempPT == NULL) { FREE(ConnsPT); break; }
    ConnsPT = TempPT;

  }
  if (ConnsPT != NULL) { client_notice(ClientS, "%s.", ConnsPT); FREE(ConnsPT); }
  if (ConnsCount <= 0) { client_notice(ClientS, "You have no connections."); }


  for (P_ClientS = Client_Head , memset(&Line1, 0, LINELEN+1) , memset(&Line2, 0, LINELEN+1) ; P_ClientS != NULL ; P_ClientS = P_ClientS->Next) {

    if (!Client_IsVerified(P_ClientS)) { continue; }
    if (strcmp(P_ClientS->User, ClientS->User) != FALSE) { continue; }

    ++UsersCount;

    if (P_ClientS->ConnS == NULL) { snprintf(Line1, LINELEN+1, "%s(%s) Detached", P_ClientS->HostName, P_ClientS->HostIPS); }
    else { snprintf(Line1, LINELEN+1, "%s(%s) Attached=%s", P_ClientS->HostName, P_ClientS->HostIPS, P_ClientS->ConnS->Name); }
    snprintf(Line2, LINELEN+1, ", %s", Line1);

    if (ClientsPT != NULL) {
      Len = strlen(ClientsPT) + strlen(Line2);
      if (Len > IRCNOTICELEN) {
        client_notice(ClientS, "%s.", ClientsPT);
        FREE(ClientsPT);
      }
    }

    if (ClientsPT == NULL) {
      TempPT = straddbuffer(ClientsPT, "Clients from your host: ", IRCNOTICELEN, 0);
      if (TempPT == NULL) { break; }
      ClientsPT = TempPT;
      TempPT = straddbuffer(ClientsPT, Line1, IRCNOTICELEN, 0);
      if (TempPT == NULL) { break; }
      ClientsPT = TempPT;
      continue;
    }

    TempPT = straddbuffer(ClientsPT, Line2, IRCNOTICELEN, 0);
    if (TempPT == NULL) { FREE(ClientsPT); break; }
    ClientsPT = TempPT;

  }
  if (ClientsPT != NULL) {
    client_notice(ClientS, "%s.", ClientsPT);
    FREE(ClientsPT);
  }

}

/* CLIENT_ATTACH FUNCTION - JONAS (01.07.2000) */

void client_attach(struct Client_Struct *ClientS, struct Conn_Struct *ConnS) {

  unsigned short int Count = 0;
  unsigned short int Users = 0;
  unsigned short int Index = 0;
  struct Chan_Struct *ChanS = NULL;
  struct ChanUser_Struct *ChanUser = NULL;
  char *NamesPT = NULL;
  unsigned short int NameIndex = 0;
  unsigned short int NameLen = 0;
  unsigned short int DumbLen = 0;
  unsigned short int UserLen = 0;
  unsigned short int NewLen = 0;
  char *DumbPT = NULL;

  if (ConnS->NumClients >= ConnS->MaxClients) { client_notice(ClientS, "Connection %s has %d attached clients, only %d clients are allowed.", ConnS->Name, ConnS->NumClients, ConnS->MaxClients); return; }

  if (!Conn_IsWelcome(ConnS)) {
    ClientS->ConnS = ConnS;
    ++ConnS->NumClients;
    client_notice(ClientS, "Attached on connection %s: Not connected.", ConnS->Name);
    client_noticeallbutone(ClientS, "Client %s (%s) attached on connection %s.", ClientS->HostName, ClientS->HostIPS, ConnS->Name);
    return;
  }
  if (ConnS->ServerName == NULL) { client_notice(ClientS, "Internal error."); return; }

  for (ChanS = ConnS->Chan_Head ; ChanS != NULL ; ChanS = ChanS->Next) {
    if (!Chan_IsNames(ChanS)) { client_notice(ClientS, "Please wait for %s/NAMES to complete.", ChanS->Chan); return; }
  }

  if (ConnS->NumClients <= 0) {
    if ((ConnS->AutoAway == TRUE) && (Conn_IsAway(ConnS))) { conn_addsendq(ConnS, "AWAY"); }
    if (ConnS->PublicAway == TRUE) {
      for (ChanS = ConnS->Chan_Head ; ChanS != NULL ; ChanS = ChanS->Next) { conn_addsendq(ConnS, "PRIVMSG %s :\1ACTION %s\1", ChanS->Chan, ConnS->PublicAttachMsg); }
    }
    if ((!Conn_IsSentNick(ConnS)) && (!Conn_IsSentAwayNick(ConnS)) && (Conn_IsAwayNick(ConnS)) && (strcasecmp(ConnS->IRCNick, ConnS->Nick) != FALSE) && (strcasecmp(ConnS->Nick, ConnS->AwayNick) != FALSE)) {
      Conn_SetSentNick(ConnS);
      ConnS->NicksIndex = 0;
      conn_addsendq(ConnS, "NICK %s", ConnS->Nick);
    }
  }

  for (Count = 0 ; Count <= 4 ; ++Count) {
    if (ConnS->Welcome[Count] != NULL) { client_addsend(ClientS, ":%s 00%d %s %s", ConnS->ServerName, Count, ConnS->IRCNick, ConnS->Welcome[Count]); }
  }
  for (Index = 0 ; Index < ConnS->ISupportLines ; ++Index) {
    if (ConnS->ISupport[Index] != NULL) { client_addsend(ClientS, ":%s 00%d %s %s", ConnS->ServerName, IRC_GENERIC_RPL_ISUPPORT, ConnS->IRCNick, ConnS->ISupport[Index]); }
  }

  for (ChanS = ConnS->Chan_Head ; ChanS != NULL ; ChanS = ChanS->Next) {
    assert(ChanS->Me != NULL);
    client_addsend(ClientS, ":%s JOIN :%s", ChanS->Me->NUH, ChanS->Chan);
    FREE(NamesPT);
    NameLen = 12;
    NameLen += strlen(ConnS->ServerName);
    NameLen += strlen(ChanS->Me->Nick);
    NameLen += strlen(ChanS->Chan);
    DumbPT = realloc(NamesPT, NameLen);
    if (DumbPT == NULL) {
      client_close(ClientS, "Memory allocation failure: [%d] %s", errno, strerror(errno));
      free(NamesPT);
      return;
    }
    NamesPT = DumbPT;
    sprintf(NamesPT, ":%s 353 %s = %s :", ConnS->ServerName, ChanS->Me->Nick, ChanS->Chan);
    NameIndex = NameLen - 1;
    Count = 0;
    Users = 0;

    for (ChanUser = ChanS->User_Head ; ChanUser != NULL ; ChanUser = ChanUser->Next) {
      Count++;
      UserLen = strlen(ChanUser->Nick);
      if (ChanUser_IsOP(ChanUser)) { UserLen++; }
      else if (ChanUser_IsVoice(ChanUser)) { UserLen++; }
      if (Count > 1) { UserLen++; }
      DumbLen = NameLen + UserLen - 1;
      if (DumbLen > IRCMSGLEN) {
        client_addsend(ClientS, "%s", NamesPT);
        memset(NamesPT, 0, NameLen);
        FREE(NamesPT);
        NameLen = 12;
        NameLen += strlen(ConnS->ServerName);
        NameLen += strlen(ChanS->Me->Nick);
        NameLen += strlen(ChanS->Chan);
        DumbPT = realloc(NamesPT, NameLen);
        if (DumbPT == NULL) {
          client_close(ClientS, "Memory allocation failure: [%d] %s", errno, strerror(errno));
          free(NamesPT);
          return;
        }
        NamesPT = DumbPT;
        sprintf(NamesPT, ":%s 353 %s = %s :", ConnS->ServerName, ChanS->Me->Nick, ChanS->Chan);
        Count = 1;
        NameIndex = NameLen - 1;
        UserLen = strlen(ChanUser->Nick);
        if (ChanUser_IsOP(ChanUser)) { UserLen++; }
        else if (ChanUser_IsVoice(ChanUser)) { UserLen++; }
        DumbLen = NameLen + UserLen - 1;
        assert(DumbLen <= IRCMSGLEN);
      }
      NewLen = NameLen + UserLen;
      DumbPT = realloc(NamesPT, NewLen);
      if (DumbPT == NULL) {
        client_close(ClientS, "Memory allocation failure: [%d] %s", errno, strerror(errno));
        free(NamesPT);
        return;
      }
      NamesPT = DumbPT;
      DumbPT += NameIndex;
      if (Count > 1) {
        *DumbPT = ' ';
        DumbPT++;
      }
      if (ChanUser_IsOP(ChanUser)) {
        *DumbPT = '@';
        DumbPT++;
      }
      else if (ChanUser_IsVoice(ChanUser)) {
        *DumbPT = '+';
        DumbPT++;
      }
      Users++;
      strcpy(DumbPT, ChanUser->Nick);
      NameIndex += UserLen;
      NameLen += UserLen;
    }
    if (Count > 0) {
      client_addsend(ClientS, "%s", NamesPT);
      memset(NamesPT, 0, NameLen);
      FREE(NamesPT);
      NameLen = 0;
    }
    assert(ChanS->Users == Users);
    client_addsend(ClientS, ":%s 366 %s %s :End of NAMES from proxy", ConnS->ServerName, ChanS->Me->Nick, ChanS->Chan);
  }

  Conn_SetAttached(ConnS);
  ++ConnS->NumClients;
  ClientS->ConnS = ConnS;

  client_notice(ClientS, "Attached on connection %s %s(%s):%ld \"%s\"", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, ConnS->ServerName);
  client_noticeallbutone(ClientS, "Client %s (%s) attached on connection %s.", ClientS->HostName, ClientS->HostIPS, ConnS->Name);

}

/* CLIENT_DETACH FUNCTION - JONAS (01.07.2000) */

void client_detach(struct Client_Struct *ClientS) {

  assert(ClientS != NULL);
  assert(ClientS->ConnS != NULL);

  --ClientS->ConnS->NumClients;

  if (Conn_IsWelcome(ClientS->ConnS)) {
    if ((Client_IsSocket(ClientS)) && (Client_IsVerified(ClientS))) {
      client_close(ClientS, "Detached from connection %s %s(%s):%ld \"%s\"", ClientS->ConnS->Name, ClientS->ConnS->ServerHostName, ClientS->ConnS->ServerHostIPS, ClientS->ConnS->ServerPortH, ClientS->ConnS->ServerName);
    }
    if (ClientS->ConnS->NumClients <= 0) {
      if ((ClientS->ConnS->AutoAway == TRUE) && (!Conn_IsAway(ClientS->ConnS))) { conn_addsendq(ClientS->ConnS, "AWAY :%s", ClientS->ConnS->AwayMsg); }
      if (ClientS->ConnS->PublicAway == TRUE) {
        struct Chan_Struct *ChanS = NULL;
        for (ChanS = ClientS->ConnS->Chan_Head ; ChanS != NULL ; ChanS = ChanS->Next) { conn_addsendq(ClientS->ConnS, "PRIVMSG %s :\1ACTION %s\1", ChanS->Chan, ClientS->ConnS->PublicDetachMsg); }
      }
      if ((!Conn_IsSentNick(ClientS->ConnS)) && (!Conn_IsSentAwayNick(ClientS->ConnS)) && (!Conn_IsAwayNick(ClientS->ConnS)) && (strcasecmp(ClientS->ConnS->IRCNick, ClientS->ConnS->AwayNick) != FALSE) && (strcasecmp(ClientS->ConnS->Nick, ClientS->ConnS->AwayNick) != FALSE)) {
        Conn_SetSentAwayNick(ClientS->ConnS);
        ClientS->ConnS->NicksIndex = 0;
        conn_addsendq(ClientS->ConnS, "NICK %s", ClientS->ConnS->AwayNick);
      }
    }
  }
  else {
    if ((Client_IsSocket(ClientS)) && (Client_IsVerified(ClientS))) {
      client_notice(ClientS, "Detached from connection %s: Not connected.", ClientS->ConnS->Name);
    }
  }

  client_noticeallbutone(ClientS, "Client %s (%s) detached from connection %s.", ClientS->HostName, ClientS->HostIPS, ClientS->ConnS->Name);
  ClientS->ConnS = NULL;

}
