/*
 * ----------------------------------------------------------------
 * Night Light IRC Proxy Connection I/O System
 * ----------------------------------------------------------------
 * Copyright (C) 1997-2003 Jonas Kvinge <jonas@night-light.net>
 * All rights reserved.
 *
 * This code is a result of thousands of hours of work by
 * Jonas Kvinge <jonas@night-light.net>
 *
 * You may not create derivative works based on this code.
 *
 * Modified source code or binaries compiled from modified source
 * code distributed in any shape or form without the authors
 * permission is expressly forbidden.
 *
 * This code is provided without warranty of any kind.
 *
 * Under no circumstances and under no legal contract or otherwise
 * shall Jonas Kvinge be liable to you or any other person for any
 * damages, computer failure, malfunction or any other damages or
 * losses.
 *
 * Last modified by:
 * Jonas Kvinge (07.07.2003)
 *
 */

#define CONN_IO_C

#define NEED_SYS_TYPES_H 1		/* Extra types */
#define NEED_SYS_PARAM_H 1		/* Some systems need this */
#define NEED_LIMITS_H 0			/* Kernel limits */
#define NEED_STDARG_H 1			/* va_list, etc */
#define NEED_ERRNO_H 1			/* errno */
#define NEED_CTYPE_H 1			/* isdigit(), etc */
#define NEED_NETINET_IN_H 1		/* in_addr, sockaddr_in, etc */
#define NEED_ARPA_INET_H 1		/* inet_ntoa(), inet_aton(), etc */
#define NEED_STDIO_H 1			/* Standard C UNIX functions */
#define NEED_STDLIB_H 1			/* malloc(), exit(), atoi(), etc */
#define NEED_TIME_H 1			/* time(), etc */
#define NEED_SYSCTL_H 0			/* sysctl(), etc */
#define NEED_SYS_STAT_H 0		/* chmod(), mkdir(), etc */
#define NEED_SYS_UIO_H 0		/* iovec, etc */
#define NEED_FCNTL_H 1			/* open(), creat(), fcntl(), etc */
#define NEED_SYS_IOCTL_H 1		/* ioctl(), etc */
#define NEED_SYS_FILIO_H 1		/* Solaris need this for ioctl(), etc */
#define NEED_UNISTD_H 1			/* Unix standard functions */
#define NEED_STRING_H 1			/* C string functions */
#define NEED_SIGNAL_H 0			/* Signal functions */
#define NEED_SYS_SOCKET_H 1		/* Socket functions */
#define NEED_NETDB_H 1			/* Network database functions */
#define NEED_ARPA_NAMESER_H 0		/* Nameserver definitions */
#define NEED_GETUSERPW_HEADERS 0 	/* Functions to retrive system passwords */

#define NEED_NETDB_H 1

#include "includes.h"

#include "irc.h"

#if ARES
  #include "ares_io.h"
#endif

#include "access_conf.h"

#include "conn_io.h"
#include "conn_sendq.h"
#include "conn_log.h"
#include "conn_parser.h"
#include "conn_ignore.h"

#include "client_io.h"

#include "chan.h"
#include "chan_user.h"

/* VARIABLES - JONAS (31.07.2001) */

struct Conn_Struct *Conn_Head = NULL;
struct Conn_Struct *Conn_Tail = NULL;
unsigned long int G_Conns = 0;

extern struct Client_Struct *Client_Head;

/* CONN_ADD FUNCTION - JONAS (22.07.2001) */

struct Conn_Struct *conn_add(const char *const NamePT, const char *const NickPT, const char *const AwayNickPT, const char *const UserPT, const char *const HostPT, const char *const ModePT, const char *const InfoPT, const char *const ChansPT, const unsigned short int AutoConnect, const unsigned short int Logging, const unsigned short int AutoAway, const unsigned short int PublicAway, const unsigned short int RegainNick, const unsigned short int MaxClients, const unsigned short int SendMaxLines, const unsigned short int SendLineTime, const unsigned short int SendMaxBuffer, const unsigned short int SendBufferTime, const char *const AwayMsgPT, const char *const PublicDetachMsgPT, const char *const PublicAttachMsgPT) {

  struct Conn_Struct *ConnS = NULL;
  struct Conn_Struct *ConnS_NEW = NULL;

  assert(NamePT != NULL);
  assert(NickPT != NULL);
  assert(UserPT != NULL);
  assert(HostPT != NULL);
  assert(ModePT != NULL);
  assert(InfoPT != NULL);
  assert(ChansPT != NULL);
  assert(AwayMsgPT != NULL);
  assert(PublicDetachMsgPT != NULL);
  assert(PublicAttachMsgPT != NULL);

  ConnS = conn_get(NamePT);
  if (ConnS != NULL) {
    aerrno = AEEXISTS;
    return(ConnS);
  }

  ConnS_NEW = malloc(sizeof(struct Conn_Struct));
  if (ConnS_NEW == NULL) {
    aerrno = AEMALLOC;
    return(NULL);
  }
  memset(ConnS_NEW, 0, sizeof(struct Conn_Struct));

  ConnS_NEW->Name = strdup(NamePT);
  ConnS_NEW->Nick = strdup(NickPT);
  ConnS_NEW->AwayNick = strdup(AwayNickPT);
  ConnS_NEW->User = strdup(UserPT);
  ConnS_NEW->Host = strdup(HostPT);
  ConnS_NEW->Mode = strdup(ModePT);
  ConnS_NEW->Info = strdup(InfoPT);
  ConnS_NEW->Chans = strdup(ChansPT);
  ConnS_NEW->AwayMsg = strdup(AwayMsgPT);
  ConnS_NEW->PublicDetachMsg = strdup(PublicDetachMsgPT);
  ConnS_NEW->PublicAttachMsg = strdup(PublicAttachMsgPT);

  if (
     (ConnS_NEW->Name == NULL) ||
     (ConnS_NEW->Nick == NULL) ||
     (ConnS_NEW->AwayNick == NULL) ||
     (ConnS_NEW->User == NULL) ||
     (ConnS_NEW->Host == NULL) ||
     (ConnS_NEW->Mode == NULL) ||
     (ConnS_NEW->Info == NULL) ||
     (ConnS_NEW->Chans == NULL) ||
     (ConnS_NEW->AwayMsg == NULL) ||
     (ConnS_NEW->PublicDetachMsg == NULL) ||
     (ConnS_NEW->PublicAttachMsg == NULL)
    ) {
    free(ConnS_NEW->Name);
    free(ConnS_NEW->Nick);
    free(ConnS_NEW->AwayNick);
    free(ConnS_NEW->User);
    free(ConnS_NEW->Host);
    free(ConnS_NEW->Mode);
    free(ConnS_NEW->Info);
    free(ConnS_NEW->Chans);
    free(ConnS_NEW->AwayMsg);
    free(ConnS_NEW->PublicDetachMsg);
    free(ConnS_NEW->PublicAttachMsg);
    free(ConnS_NEW);
    aerrno = AEMALLOC;
    return(NULL);
  }

  ConnS_NEW->AutoConnect = AutoConnect;
  ConnS_NEW->Logging = Logging;
  ConnS_NEW->RegainNick = RegainNick;
  ConnS_NEW->AutoAway = AutoAway;
  ConnS_NEW->PublicAway = PublicAway;
  ConnS_NEW->MaxClients = MaxClients;
  ConnS_NEW->SendMaxLines = SendMaxLines;
  ConnS_NEW->SendLineTime = SendLineTime;
  ConnS_NEW->SendMaxBuffer = SendMaxBuffer;
  ConnS_NEW->SendBufferTime = SendBufferTime;

  conn_initconnection(ConnS_NEW);

  if (Conn_Head == NULL) {
    Conn_Head = ConnS_NEW;
    Conn_Tail = ConnS_NEW;
  }
  else {
    ConnS = Conn_Tail;
    ConnS->Next = ConnS_NEW;
    ConnS_NEW->Prev = ConnS;
    Conn_Tail = ConnS_NEW;
  }

  G_Conns++;

  conn_log(ConnS_NEW, HOSTNAME, "", ""); /* Init the log file */

  aerrno = AESUCCESS;
  return(ConnS_NEW);

}

/* CONN_REM FUNCTION - JONAS (01.08.2001) */

void conn_rem(struct Conn_Struct *ConnS) {

  struct Client_Struct *ClientS = NULL;

  assert(ConnS != NULL);

  if (ConnS->Prev == NULL) { Conn_Head = ConnS->Next; }
  else { ConnS->Prev->Next = ConnS->Next; }

  if (ConnS->Next == NULL) { Conn_Tail = ConnS->Prev; }
  else { ConnS->Next->Prev = ConnS->Prev; }

  for (ClientS = Client_Head ; ClientS != NULL ; ClientS = ClientS->Next) {
    if (ClientS->ConnS == ConnS) {
      if (!Client_IsSentError(ClientS)) { client_notice(ClientS, "You are automatically detached from connection %s: Connection removed.", ConnS->Name); }
      client_detach(ClientS);
    }
  }

  conn_remignores(ConnS);
  conn_initconnection(ConnS);

  free(ConnS->Name);
  free(ConnS->Nick);
  free(ConnS->AwayNick);
  free(ConnS->User);
  free(ConnS->Host);
  free(ConnS->Mode);
  free(ConnS->Info);
  free(ConnS->Chans);
  free(ConnS->AwayMsg);
  free(ConnS->PublicDetachMsg);
  free(ConnS->PublicAttachMsg);
  conn_remservers(ConnS);
  free(ConnS);

  G_Conns--;

}

/* CONN_GET - JONAS (06.10.2000) */

struct Conn_Struct *conn_get(const char *const NamePT) {

  struct Conn_Struct *ConnS = NULL;

  assert(NamePT != NULL);

  for (ConnS = Conn_Head ; ConnS != NULL ; ConnS = ConnS->Next) {
    if (strcasecmp(ConnS->Name, NamePT) == FALSE) {
      aerrno = AESUCCESS;
      return(ConnS);
    }
  }
  aerrno = AENOMATCH;
  return(NULL);

}

/* CONN_ADDSERVER - JONAS (31.07.2001) */

struct ConnServer_Struct *conn_addserver(struct Conn_Struct *ConnS, const char *const HostPT, const unsigned long int Port, const char *const PassPT) {

  struct ConnServer_Struct *ConnServer = NULL;
  struct ConnServer_Struct *ConnServer_NEW = NULL;

  assert(ConnS != NULL);
  assert(HostPT != NULL);
  assert(PassPT != NULL);

  ConnServer = conn_getserver(ConnS, HostPT);
  if (ConnServer != NULL) {
    aerrno = AEEXISTS;
    return(ConnServer);
  }

  ConnServer_NEW = malloc(sizeof(struct ConnServer_Struct));
  if (ConnServer_NEW == NULL) {
    aerrno = AEMALLOC;
    return(NULL);
  }
  memset(ConnServer_NEW, 0, sizeof(struct ConnServer_Struct));

  ConnServer_NEW->Host = strdup(HostPT);
  ConnServer_NEW->Port = Port;

  if (ConnServer_NEW->Host == NULL) {
    free(ConnServer_NEW->Host);
    free(ConnServer_NEW);
    aerrno = AEMALLOC;
    return(NULL);
  }

  if (ConnS->Server_Head == NULL) {
    ConnS->Server_Head = ConnServer_NEW;
    ConnS->Server_Tail = ConnServer_NEW;
  }
  else {
    ConnServer = ConnS->Server_Tail;
    ConnServer->Next = ConnServer_NEW;
    ConnServer_NEW->Prev = ConnServer;
    ConnS->Server_Tail = ConnServer_NEW;
  }

  ConnS->NumServers++;

  aerrno = AESUCCESS;
  return(ConnServer_NEW);

}

/* CONN_REMSERVER FUNCTION - JONAS (31.07.2001) */

void conn_remserver(struct Conn_Struct *ConnS, struct ConnServer_Struct *ConnServer) {

  assert(ConnS != NULL);
  assert(ConnServer != NULL);

  if (ConnServer->Prev == NULL) { ConnS->Server_Head = ConnServer->Next; }
  else { ConnServer->Prev->Next = ConnServer->Next; }

  if (ConnServer->Next == NULL) { ConnS->Server_Tail = ConnServer->Prev; }
  else { ConnServer->Next->Prev = ConnServer->Prev; }

  free(ConnServer->Host);
  free(ConnServer);

  if (ConnS->ConnServerTry == ConnServer) { ConnS->ConnServerTry = NULL; }

  ConnS->NumServers--;

}

/* CONN_GETSERVER FUNCTION - JONAS (31.07.2001) */

struct ConnServer_Struct *conn_getserver(struct Conn_Struct *ConnS, const char *const HostPT) {

  struct ConnServer_Struct *ConnServer = NULL;

  assert(ConnS != NULL);
  assert(HostPT != NULL);

  for (ConnServer = ConnS->Server_Head ; ConnServer != NULL ; ConnServer = ConnServer->Next) {
    if (strcasecmp(ConnServer->Host, HostPT) == FALSE) {
      aerrno = AESUCCESS;
      return(ConnServer);
    }
  }
  aerrno = AENOMATCH;
  return(NULL);

}

/* CONN_REMSERVERS FUNCTION - JONAS (31.07.2001) */

void conn_remservers(struct Conn_Struct *ConnS) {

  while (ConnS->Server_Head != NULL) { conn_remserver(ConnS, ConnS->Server_Head); }

}

/* CONN_INITCONNECTION FUNCTION - JONAS (01.08.2001) */

void conn_initconnection(struct Conn_Struct *ConnS) {

  unsigned short int Index = 0;

  assert(ConnS != NULL);

  FREE(ConnS->FileName);
  ConnS->ResolveFlags = 0;
  FREE(ConnS->HostName);
  FREE(ConnS->HostIPS);
  ConnS->HostIPH = 0;
  ConnS->HostIPN = 0;
  ConnS->PortH = 0;
  ConnS->PortN = 0;

  ConnS->ServerResolveFlags = 0;
  FREE(ConnS->ServerHostName);
  FREE(ConnS->ServerHostIPS);
  ConnS->ServerHostIPH = 0;
  ConnS->ServerHostIPN = 0;
  ConnS->ServerPortH = 0;
  ConnS->ServerPortN = 0;

  ConnS->ConnectionFlags = 0;
  ConnS->FD = FD_NONE;
  FREE(ConnS->ServerName);
  FREE(ConnS->IRCNick);
  for (Index = 0 ; Index <= 4 ; ++Index) { FREE(ConnS->Welcome[Index]); }
  for (Index = 0 ; Index < ConnS->ISupportLines ; ++Index) { free(ConnS->ISupport[Index]); }
  FREE(ConnS->ISupport);
  ConnS->ISupportLines = 0;
  conn_initsendq(ConnS);
  FREE(ConnS->RecvBuffer);
  FREE(ConnS->SendBuffer);
  ConnS->NicksIndex = 0;
  chan_remall(ConnS);

}

/* CONN_CONNECT FUNCTION - JONAS (17.07.2001) */

void conn_connect(struct Conn_Struct *ConnS) {

  signed long int Result = 0;
  #if !ARES
    struct hostent *HostEnt = NULL;
    unsigned long int HostEntLen = sizeof(struct hostent);
  #endif

  assert(ConnS != NULL);

  if (!Conn_IsConnectProc(ConnS)) {

    Conn_SetConnectProc(ConnS);
    ConnS->ConnectProcTime = NOW;

    if (ConnS->ConnServerTry == NULL) {

      struct ConnServer_Struct *ConnServer = NULL;

      for (ConnServer = ConnS->Server_Head ; ConnServer != NULL ; ConnServer = ConnServer->Next) {
        if ((ConnS->ConnServerTry == NULL) || (ConnServer->Tries < ConnS->ConnServerTry->Tries)) { ConnS->ConnServerTry = ConnServer; }
      }
      if (ConnS->ConnServerTry == NULL) { conn_disconnect(ConnS, "Connection %s: No servers to try.", ConnS->Name); return; }
    }

    ++ConnS->ConnServerTry->Tries;
    ConnS->ServerHostName = strrealloc(ConnS->ServerHostName, ConnS->ConnServerTry->Host);
    ConnS->ServerName = strrealloc(ConnS->ServerName, ConnS->ConnServerTry->Host);
    ConnS->ServerPortH = ConnS->ConnServerTry->Port;
    ConnS->ServerPortN = htons(ConnS->ConnServerTry->Port);
    ConnS->ConnServerTry = NULL;

  }

  if (!Host_IsHostToIP(ConnS->ResolveFlags)) {

    struct in_addr INAddr;

    memset(&INAddr, 0, sizeof(INAddr));
    Result = inet_aton(ConnS->Host, &INAddr);
    if (Result == 0) {
      ConnS->HostName = strrealloc(ConnS->HostName, ConnS->Host);
      if (aerrno != AESUCCESS) {
        conn_disconnect(ConnS, "Connection %s: Memory allocation failure: [%d] %s", ConnS->Name, errno, strerror(errno));
        return;
      }
      Host_SetResolving(ConnS->ResolveFlags);
      client_noticealluser(ConnS->User, "Connection %s: Resolving local hostname %s to IP-address.", ConnS->Name, ConnS->HostName);
      #if ARES
        ares_gethostbyname(ConnS->HostName, AF_INET, &conn_hosttoip, ConnS);
        return;
      #else
        HostEnt = gethostbyname(ConnS->HostName);
        conn_hosttoip(ConnS, errno, HostEnt);
      #endif
    }
    else {
      ConnS->HostIPN = INAddr.s_addr;
      ConnS->HostIPH = ntohl(ConnS->HostIPN);
      ConnS->HostIPS = strrealloc(ConnS->HostIPS, ConnS->Host);
      if (aerrno != AESUCCESS) {
        conn_disconnect(ConnS, "Connection %s: Memory allocation failure: [%d] %s", ConnS->Name, errno, strerror(errno));
        return;
      }
      ConnS->HostName = strrealloc(ConnS->HostName, ConnS->Host);
      if (aerrno != AESUCCESS) {
        conn_disconnect(ConnS, "Connection %s: Memory allocation failure: [%d] %s", ConnS->Name, errno, strerror(errno));
        return;
      }
      Host_SetHostToIP(ConnS->ResolveFlags);
    }

  }

  Host_SetResolved(ConnS->ResolveFlags);

  if (!Host_IsHostToIP(ConnS->ServerResolveFlags)) {

    struct in_addr INAddr;

    memset(&INAddr, 0, sizeof(INAddr));
    Result = inet_aton(ConnS->ServerHostName, &INAddr);
    if (Result == 0) {
      Host_SetResolving(ConnS->ServerResolveFlags);
      client_noticealluser(ConnS->User, "Connection %s: Resolving server hostname %s to IP-address.", ConnS->Name, ConnS->ServerHostName);
      #if ARES
        ares_gethostbyname(ConnS->ServerHostName, AF_INET, &conn_serverhosttoip, ConnS);
        return;
      #else
        HostEnt = gethostbyname(ConnS->ServerHostName);
        conn_serverhosttoip(ConnS, errno, HostEnt);
      #endif
    }
    else {
      ConnS->ServerHostIPS = strrealloc(ConnS->ServerHostIPS, ConnS->ServerHostName);
      if (aerrno != AESUCCESS) {
        conn_disconnect(ConnS, "Connection %s: Memory allocation failure: [%d] %s", ConnS->Name, errno, strerror(errno));
        return;
      }
      Host_SetHostToIP(ConnS->ServerResolveFlags);
      ConnS->ServerHostIPN = INAddr.s_addr;
      ConnS->ServerHostIPH = ntohl(ConnS->ServerHostIPN);
    }

  }

  Host_SetResolved(ConnS->ServerResolveFlags);

  if (!Conn_IsSocket(ConnS)) {

#if !WIN32
    sysseteuidbyuser(ConnS->User);
#endif
    Result = socket(AF_INET, SOCK_STREAM, 0);
#if !WIN32
    sysseteuidnormal();
#endif
    if (Result <= ERROR) {
      conn_disconnect(ConnS, "Connection %s: Unable to create socket: [%d] %s", ConnS->Name, errno, strerror(errno));
      return;
    }
    ConnS->FD = Result;
    Conn_SetSocket(ConnS);

  }

  if (!Conn_IsSockOPT(ConnS)) {

    unsigned long int Flags = 0;

#if defined(NBLOCK_SYSV)
    Flags = 1;
    Result = ioctl(ConnS->FD, FIONBIO, &Flags);
    if (Result <= ERROR) {
      conn_disconnect(ConnS, "Connection %s: Unable to set socket in non-blocking mode using ioctl(): [%d] %s", ConnS->Name, errno, strerror(errno));
      return;
    }
#else
    Result = fcntl(ConnS->FD, F_GETFL);
    if (Result <= ERROR) {
      conn_disconnect(ConnS, "Connection %s: Unable to get socket flags using fcntl(): [%d] %s", ConnS->Name, errno, strerror(errno));
      return;
    }
    Flags = Result;
#if defined(NBLOCK_BSD)
    Flags |= O_NDELAY;
#elif defined(NBLOCK_POSIX)
    Flags |= O_NONBLOCK;
#else
    #warning "This system does not support non-blocking sockets?"
    Flags |= O_NONBLOCK;
#endif
    Result = fcntl(ConnS->FD, F_SETFL, Flags);
    if (Result <= ERROR) {
      conn_disconnect(ConnS, "Connection %s: Unable to set socket in non-blocking mode using fcntl(): [%d] %s", ConnS->Name, errno, strerror(errno));
      return;
    }
#endif

    Conn_SetSockOPT(ConnS);

  }

  if ((CONN_SOCKKEEPALIVEOPT == TRUE) && (!Conn_IsKeepAliveOPT(ConnS))) {

    unsigned long int OPT = 1;

    Result = setsockopt(ConnS->FD, SOL_SOCKET, SO_KEEPALIVE, &OPT, sizeof(OPT));
    if (Result <= ERROR) {
      conn_disconnect(ConnS, "Connection %s: Unable to set Keep Alive option for socket: [%d] %s", ConnS->Name, errno, strerror(errno));
      return;
    }

    Conn_SetKeepAliveOPT(ConnS);

  }

  if (!Conn_IsBound(ConnS)) {

    struct sockaddr_in SockAddr = { 0 };

    memset(&SockAddr, 0, sizeof(SockAddr));
    SockAddr.sin_family = AF_INET;
    SockAddr.sin_addr.s_addr = ConnS->HostIPN;
    SockAddr.sin_port = 0;
    Result = bind(ConnS->FD, (struct sockaddr *) &SockAddr, sizeof(SockAddr));
    if (Result <= ERROR) {
      conn_disconnect(ConnS, "Connection %s: Unable to bind socket to %s(%s): [%d] %s", ConnS->Name, ConnS->HostName, ConnS->HostIPS, errno, strerror(errno));
      return;
    }
    Conn_SetBound(ConnS);

  }

  if (!Conn_IsConnect(ConnS)) {

    struct sockaddr_in SockAddr = { 0 };

    client_noticealluser(ConnS->User, "Connection %s: Connecting to server %s(%s):%ld.", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH);

    memset(&SockAddr, 0, sizeof(SockAddr));
    SockAddr.sin_family = AF_INET;
    SockAddr.sin_addr.s_addr = ConnS->ServerHostIPN;
    SockAddr.sin_port = ConnS->ServerPortN;
    Result = connect(ConnS->FD, (struct sockaddr *) &SockAddr, sizeof(SockAddr));
    if (Result <= ERROR) {
      if (errno == EINPROGRESS) {
        Conn_SetConnecting(ConnS);
        ConnS->ConnectTime = NOW;
        conn_logon(ConnS);
        return;
      }
      else {
        conn_disconnect(ConnS, "Connection %s: Unable to connect to %s(%s):%ld: [%d] %s", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, errno, strerror(errno));
        return;
      }
    }
    else {
      Conn_SetConnected(ConnS);
      ConnS->ConnectTime = NOW;
      ConnS->ConnectedTime = NOW;
      conn_logon(ConnS);
      client_noticealluser(ConnS->User, "Connection %s: Socket connected to %s(%s):%ld.", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH);
    }

  }

  if (!Conn_IsConnected(ConnS)) {

    int Error = 0;
    getsockopt_optlen_type ErrorLen = sizeof(Error);

    Result = getsockopt(ConnS->FD, SOL_SOCKET, SO_ERROR, &Error, &ErrorLen);
    if (Result <= ERROR) {
      conn_disconnect(ConnS, "Connection %s: Unable to get connection result for %s(%s):%ld: [%d] %s", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, errno, strerror(errno));
      return;
    }
    if (Error != SUCCESS) {
      conn_disconnect(ConnS, "Connection %s: Unable to connect to %s(%s):%ld: [%d] %s", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, Error, strerror(Error));
      return;
    }

    client_noticealluser(ConnS->User, "Connection %s: Socket connected to %s(%s):%ld.", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH);
    Conn_ClearConnecting(ConnS);
    Conn_SetConnected(ConnS);
    ConnS->ConnectedTime = NOW;

  }

  if (!Conn_IsSockName(ConnS)) {

    struct sockaddr_in SockAddr = { 0 };
    getsockname_namelen_type SockAddrLen = sizeof(SockAddr);

    memset(&SockAddr, 0, sizeof(SockAddr));
    Result = getsockname(ConnS->FD, (struct sockaddr *) &SockAddr, &SockAddrLen);
    if (Result <= ERROR) {
      conn_disconnect(ConnS, "Connection %s: Unable to get socket name for %s(%s):%ld: [%d] %s", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, errno, strerror(errno));
      return;
    }

    ConnS->PortN = SockAddr.sin_port;
    ConnS->PortH = ntohs(ConnS->PortN);
    Conn_SetSockName(ConnS);

  }

  Conn_ClearConnectProc(ConnS); /* MISSION COMPLETE! */

}

/* CONN_DISCONNECT FUNCTION - JONAS (01.07.2000) */

void conn_disconnect(struct Conn_Struct *ConnS, const char *const LinePT, ...) {

  char Line[IRCNOTICELEN+1] = "";
  va_list Args = { 0 };
  struct Client_Struct *ClientS = NULL;

  assert(ConnS != NULL);
  assert(LinePT != NULL);

  va_start(Args, LinePT);
  vsnprintf(Line, IRCNOTICELEN+1, LinePT, Args);
  va_end(Args);

  client_noticealluser(ConnS->User, "%s", Line);

  if (Conn_IsWelcome(ConnS)) {
    for (ClientS = Client_Head ; ClientS != NULL ; ClientS = ClientS->Next) {
      if (ClientS->ConnS == ConnS) { client_close(ClientS, "Disconnected from attached connection %s server %s(%s):%ld.", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH); }
    }
  }
  if (Conn_IsSocket(ConnS)) {
    close(ConnS->FD);
    ConnS->FD = FD_NONE;
  }

  conn_initconnection(ConnS);

  if (ConnS->ConnServerTry != NULL) { conn_connect(ConnS); }

}

/* CONN_HOSTTOIP FUNCTION - JONAS (01.03.2000) */

void conn_hosttoip(void *ArgPT, unsigned short int ErrNo, struct hostent *HostEnt) {

  struct Conn_Struct *ConnS = ArgPT;
  char *HostIPPT = NULL;
  struct in_addr INAddr;

  assert(ConnS != NULL);

  Host_ClearResolving(ConnS->ResolveFlags);

  if (HostEnt == NULL) {
    conn_disconnect(ConnS, "Connection %s: Unable to resolve hostname %s to IP-address: [%d] %s", ConnS->Name, ConnS->HostName, ErrNo, ares_strerror(ErrNo));
    return;
  }
  memcpy(&ConnS->HostIPN, HostEnt->h_addr, HostEnt->h_length);
  memset(&INAddr, 0, sizeof(INAddr));
  INAddr.s_addr = ConnS->HostIPN;
  HostIPPT = inet_ntoa(INAddr);
  ConnS->HostIPS = strrealloc(ConnS->HostIPS, HostIPPT);
  if (aerrno != AESUCCESS) {
    conn_disconnect(ConnS, "Connection %s: Memory allocation failure: [%d] %s", ConnS->Name, errno, strerror(errno));
    return;
  }
  ConnS->HostIPH = ntohl(ConnS->HostIPN);

  client_noticealluser(ConnS->User, "Connection %s: Resolved hostname %s to IP-address %s.", ConnS->Name, ConnS->HostName, ConnS->HostIPS);

  Host_SetHostToIP(ConnS->ResolveFlags);

  #if ARES
    conn_connect(ConnS);
  #endif

}

/* CONN_SERVERHOSTTOIP FUNCTION - JONAS (01.03.2000) */

void conn_serverhosttoip(void *ArgPT, unsigned short int ErrNo, struct hostent *HostEnt) {

  struct Conn_Struct *ConnS = ArgPT;
  char *HostIPPT = NULL;
  struct in_addr INAddr;

  assert(ConnS != NULL);

  Host_ClearResolving(ConnS->ServerResolveFlags);

  if (HostEnt == NULL) {
    conn_disconnect(ConnS, "Connection %s: Unable to resolve server hostname %s to IP-Address: [%d] %s", ConnS->Name, ConnS->ServerHostName, ErrNo, ares_strerror(ErrNo));
    return;
  }
  memcpy(&ConnS->ServerHostIPN, HostEnt->h_addr, HostEnt->h_length);
  memset(&INAddr, 0, sizeof(INAddr));
  INAddr.s_addr = ConnS->ServerHostIPN;
  HostIPPT = inet_ntoa(INAddr);
  ConnS->ServerHostIPS = strrealloc(ConnS->ServerHostIPS, HostIPPT);
  if (aerrno != AESUCCESS) {
    conn_disconnect(ConnS, "Connection %s: Memory allocation failure: [%d] %s", ConnS->Name, errno, strerror(errno));
    return;
  }

  client_noticealluser(ConnS->User, "Connection %s: Resolved server hostname %s to IP-address %s.", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS);

  Host_SetHostToIP(ConnS->ServerResolveFlags);

  #if ARES
    conn_connect(ConnS);
  #endif

}

/* CONN_LOGON FUNCTION - JONAS (01.07.2000) */

void conn_logon(struct Conn_Struct *ConnS) {

  char *NickPT = ConnS->Nick;
  char *InfoPT = ConnS->Info;

  assert(ConnS != NULL);

  if (ConnS->NumClients <= 0) {
    NickPT = ConnS->AwayNick;
    Conn_SetSentAwayNick(ConnS);
  }
  else {
    NickPT = ConnS->Nick;
    Conn_SetSentNick(ConnS);
  }

#if !USER_CONF
  if (strcmp(ConnS->Info, "0") == FALSE) {
    InfoPT = sysgetnamefromuser(ConnS->User);
    if (InfoPT == NULL) { InfoPT = "0"; }
  }
#endif

  conn_addsendq(ConnS, "NICK %s", NickPT);
  conn_addsendq(ConnS, "USER %s 0 0 :%s", ConnS->User, InfoPT);

}

/* CONN_FDS FUNCTION - JONAS (22.07.2001) */

void conn_fds(fd_set *ReadFDS, fd_set *WriteFDS, unsigned long int *FDS) {

  struct Conn_Struct *ConnS = NULL;
  struct Conn_Struct *ConnS_DEL = NULL;
  time_t Duration = 0;

  for (ConnS = Conn_Head ; ConnS != NULL ;) {

    if (NOW > ConnS->IgnoreExpireTime) { conn_ignoreexpire(ConnS); }

    FAKELOOP {

      if (!Conn_IsConnect(ConnS)) { break; }

      if (ConnS->NumSendQs > 0) { conn_flushsendq(ConnS); }

      if (!Conn_IsWelcome(ConnS)) {
        Duration = (NOW - ConnS->ConnectTime);
        if (Duration >= CONN_CONNECTTIMEOUT) { conn_disconnect(ConnS, "Connection %s: Connection attempt to server %s(%s):%ld timed out.", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH); break; }
        break;
      }

      if (Conn_IsSentQuit(ConnS)) {
        Duration = (NOW - ConnS->SentQuitTime);
        if (Duration >= CONN_QUITTIMEOUT) { conn_disconnect(ConnS, "Connection %s: Timeout while waiting for server %s(%s):%ld to close connection.", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH); break; }
        break;
      }

      if (!Conn_IsWelcome(ConnS)) { break; } /* Don't PING the connection if we haven't received 001 --Jonas */

      /* LAG CHECK */

      if (Conn_IsSentPing(ConnS)) {
        Duration = (NOW - ConnS->SentPingTime);
        if (Duration >= CONN_PINGTIMEOUT) { conn_quit(ConnS, "No response from server in %s", strduration(Duration)); }
      }
      else {
        Duration = (NOW - ConnS->LastRecvTime);
        if (Duration >= CONN_IDLETIMEBEFOREPING) {
          Conn_SetSentPing(ConnS);
          ConnS->SentPingTime = NOW;
          conn_addsendq(ConnS, "PING :PROXY");
        }
      }

      /* REGAIN NICK */

      if ((ConnS->RegainNick == TRUE) && (!Conn_IsSentISON(ConnS)) && (!Conn_IsSentNick(ConnS)) && (!Conn_IsSentAwayNick(ConnS)) && (strcasecmp(((Conn_IsAwayNick(ConnS)) ? ConnS->AwayNick : ConnS->Nick), ConnS->IRCNick) != FALSE) && (ConnS->NumClients <= 0)) {
        Duration = (NOW - ConnS->SentISONTime);
        if (Duration >= CONN_ISONTIME) {
          Conn_SetSentISON(ConnS);
          ConnS->SentISONTime = NOW;
          conn_addsendq(ConnS, "ISON :%s", ((Conn_IsAwayNick(ConnS)) ? ConnS->AwayNick : ConnS->Nick));
        }
      }

    }

    if ((Conn_IsSocket(ConnS)) && (Host_IsResolved(ConnS->ResolveFlags))) {
      FD_SET(ConnS->FD, ReadFDS);
      if (ConnS->SendBuffer != NULL) { FD_SET(ConnS->FD, WriteFDS); }
    }

    if ((Conn_IsRemove(ConnS)) && (!Conn_IsConnectProc(ConnS)) && (!Conn_IsSocket(ConnS)) && (!Host_IsResolving(ConnS->ResolveFlags)) && (!Host_IsResolving(ConnS->ServerResolveFlags))) {
      ConnS_DEL = ConnS;
      ConnS = ConnS->Next;
      conn_rem(ConnS_DEL);
      continue;
    }

    if ((ConnS->AutoConnect == TRUE) && (!Conn_IsRemove(ConnS)) && (!Conn_IsConnectProc(ConnS)) && (!Conn_IsSocket(ConnS))) {
      Duration = NOW - ConnS->ConnectProcTime;
      if (Duration >= CONN_INTERVAL) { conn_connect(ConnS); }
    }

    ConnS = ConnS->Next;
    continue;

  }

}

/* CONN_IO FUNCTION - JONAS (01.07.2000) */

void conn_io(fd_set *ReadFDS, fd_set *WriteFDS, unsigned long int *FDS) {

  struct Conn_Struct *ConnS = NULL;

  assert(ReadFDS != NULL);
  assert(WriteFDS != NULL);

  for (ConnS = Conn_Head ; ConnS != NULL ; ConnS = ConnS->Next) {

    assert(*FDS >= 0);
    if (*FDS <= 0) { return; }

    if (Conn_IsSocket(ConnS)) {
      if (FD_ISSET(ConnS->FD, ReadFDS)) { *FDS = *FDS - 1; }
      if (FD_ISSET(ConnS->FD, WriteFDS)) { *FDS = *FDS - 1; }
    }

    if (Conn_IsSocket(ConnS)) {
      if (FD_ISSET(ConnS->FD, ReadFDS)) { conn_recv(ConnS); }
    }
    if (Conn_IsSocket(ConnS)) {
      if (FD_ISSET(ConnS->FD, WriteFDS)) { conn_send(ConnS); }
    }

  }

}

/* CONN_RECV FUNCTION - JONAS (01.07.2000) */

void conn_recv(struct Conn_Struct *ConnS) {

  signed long int Result = 0;
  unsigned long int OldLen = 0;
  unsigned long int NewLen = 0;
  char RecvBuffer[RECVBUFFERLEN+1] = "";
  char *RecvBufferPT = NULL;

  assert(ConnS != NULL);
  assert(Conn_IsSocket(ConnS));
  assert(ConnS->FD != FD_NONE);

  if ((Conn_IsConnectProc(ConnS)) && (Host_IsResolved(ConnS->ResolveFlags))) {
    conn_connect(ConnS);
    if (!Conn_IsSocket(ConnS)) { return; }
  }

  ConnS->LastRecvTime = NOW;

  do {
    memset(&RecvBuffer, 0, sizeof(RecvBuffer));
    Result = recv(ConnS->FD, RecvBuffer, RECVBUFFERLEN, MSG_NOSIGNAL);
    if (Result <= ERROR) {
      if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR)) { break; }
      conn_disconnect(ConnS, "Connection %s: Read error to %s(%s):%ld: [%d] %s", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, errno, strerror(errno));
      return;
    }
    if (Result == 0) {
      if (ConnS->RecvBuffer != NULL) { conn_parser(ConnS); }
      conn_disconnect(ConnS, "Connection %s: EOF to %s(%s):%ld.", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH);
      return;
    }

    if (ConnS->RecvBuffer == NULL) { OldLen = 0; }
    else { OldLen = strlen(ConnS->RecvBuffer); }
    NewLen = OldLen + Result + 1;
    RecvBufferPT = realloc(ConnS->RecvBuffer, NewLen);
    if (RecvBufferPT == NULL) {
      conn_disconnect(ConnS, "Connection %s: Memory allocation failure: [%d] %s", ConnS->Name, errno, strerror(errno));
      return;
    }
    ConnS->RecvBuffer = RecvBufferPT;
    RecvBufferPT += OldLen;
    strcpy(RecvBufferPT, RecvBuffer);
  }
  while (Result >= RECVBUFFERLEN);

  if (Conn_IsSentQuit(ConnS)) { return; }

  conn_parser(ConnS);

}

/* CONN_SEND FUNCTION - JONAS (01.07.2000) */

void conn_send(struct Conn_Struct *ConnS) {

  char *SendBufferPT = NULL;
  char SendBuffer[SENDBUFFERLEN+1] = "";
  unsigned long int SendLen = 0;
  unsigned long int SentLen = 0;
  signed long int Result = 0;

  assert(ConnS != NULL);
  assert(Conn_IsSocket(ConnS));
  assert(ConnS->FD != FD_NONE);
  assert(ConnS->SendBuffer != NULL);

  if ((Conn_IsConnectProc(ConnS)) && (Host_IsResolved(ConnS->ResolveFlags))) {
    conn_connect(ConnS);
    if (!Conn_IsSocket(ConnS)) { return; }
  }

  for (SendBufferPT = ConnS->SendBuffer ; *SendBufferPT != 0 ; SendBufferPT += SentLen) {
    SendLen = strlen(SendBufferPT);
    if (SendLen > SENDBUFFERLEN) { SendLen = SENDBUFFERLEN; }
    memset(&SendBuffer, 0, sizeof(SendBuffer));
    strncpy(SendBuffer, SendBufferPT, SendLen);
    Result = send(ConnS->FD, SendBufferPT, SendLen, MSG_NOSIGNAL);
    if (Result <= ERROR) {
      if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR) || (errno == ENOMEM) || (errno == ENOBUFS)) {
        unsigned long int Len = 0;
        client_noticealluser(ConnS->User, "Connection %s: Write error to %s(%s):%ld: [%d] %s", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, errno, strerror(errno));
        /*
         * EAGAIN/EWOULDBLOCK -- THE SOCKET IMPLEMENTATION CAN'T HANDLE MORE DATA.
         * EINTR - INTERRUPTED BY A SIGNAL.
         * ENOMEM - NO MEMORY LEFT.
         * ENOBUFS - NO BUFFER SPACE AVAILABLE.
         *
         * COPY WHATS LEFT TO THE THE START OF THE SENDBUFFER, REALLOCATE THE MEMORY AND BAIL OUT - JONAS (01.12.1999)
         *
         */
        Len = strlen(SendBufferPT) + 1;
        memmove(ConnS->SendBuffer, SendBufferPT, Len);
        SendBufferPT = realloc(ConnS->SendBuffer, Len);
        assert(SendBufferPT != NULL);
        ConnS->SendBuffer = SendBufferPT;
        return;
      }
      conn_disconnect(ConnS, "Connection %s: Write error to %s(%s):%ld: [%d] %s", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, errno, strerror(errno));
      return;
    }
    SentLen = Result;
    assert(SentLen = SendLen);
  }

  FREE(ConnS->SendBuffer);

}

/* CONN_ADDSEND FUNCTION - JONAS (01.07.2000) */

void conn_addsend(struct Conn_Struct *ConnS, const char *const LinePT) {

  char LineCRLF[IRCMSGCRLFLEN+1] = "";
  unsigned long int Len = 0;
  unsigned long int OldLen = 0;
  unsigned long int NewLen = 0;
  char *SendBufferPT = NULL;

  assert(ConnS != NULL);
  assert(LinePT != NULL);
  assert(Conn_IsSocket(ConnS));

  snprintf(LineCRLF, IRCMSGCRLFLEN+1, "%s\r\n", LinePT);

  Len = strlen(LineCRLF);
  if (ConnS->SendBuffer == NULL) { OldLen = 0; }
  else { OldLen = strlen(ConnS->SendBuffer); }
  NewLen = OldLen + Len + 1;
  SendBufferPT = realloc(ConnS->SendBuffer, NewLen);
  if (SendBufferPT == NULL) { return; }
  ConnS->SendBuffer = SendBufferPT;
  SendBufferPT = SendBufferPT + OldLen;
  strcpy(SendBufferPT, LineCRLF);

}

/* CONN_QUIT FUNCTION - JONAS (09.06.2001) */

void conn_quit(struct Conn_Struct *ConnS, const char *const MessagePT, ...) {

  char Message[IRCQUITMSGLEN+1] = "";
  va_list Args = { 0 };

  assert(ConnS != NULL);
  assert(Conn_IsWelcome(ConnS));
  assert(MessagePT != NULL);

  va_start(Args, MessagePT);
  vsnprintf(Message, IRCQUITMSGLEN+1, MessagePT, Args);
  va_end(Args);

  if (!Conn_IsSentQuit(ConnS)) {
    Conn_SetSentQuit(ConnS);
    ConnS->SentQuitTime = time(NULL);
    conn_initsendq(ConnS);
    conn_initflushl(ConnS);
    conn_initflushb(ConnS);
    conn_addsendq(ConnS, "QUIT :%s", Message);
    client_noticealluser(ConnS->User, "Connection %s: Quitting %s(%s):%ld \"%s\": %s", ConnS->Name, ConnS->ServerHostName, ConnS->ServerHostIPS, ConnS->ServerPortH, ConnS->ServerName, Message);
  }

}

/* CONN_CLOSEALL FUNCTION - JONAS (09.06.2001) */

unsigned short int conn_closeall(const char *const MessagePT, ...) {

  char Message[LINELEN+1] = "";
  va_list Args = { 0 };
  unsigned short int Count = 0;
  struct Conn_Struct *ConnS = NULL;

  assert(MessagePT != NULL);

  va_start(Args, MessagePT);
  vsnprintf(Message, LINELEN+1, MessagePT, Args);
  va_end(Args);

  for (ConnS = Conn_Head ; ConnS != NULL ; ConnS = ConnS->Next) {
    if (!Conn_IsRemove(ConnS)) {
      if ((Host_IsResolving(ConnS->ResolveFlags)) || (Host_IsResolving(ConnS->ServerResolveFlags))) { ares_cancelquery(NULL, ConnS); }
      if (Conn_IsSocket(ConnS)) {
        if (Conn_IsWelcome(ConnS)) {
          if (!Conn_IsSentQuit(ConnS)) { conn_quit(ConnS, "%s", Message); }
        }
        else {
          conn_disconnect(ConnS, "Connection %s: %s", ConnS->Name, Message);
        }
      }
    }
    Conn_SetRemove(ConnS);
    ++Count;
  }

  return(Count);

}

