/*
 * ----------------------------------------------------------------
 * Night Light Memory Allocation Functions (memcalls.c)
 * ----------------------------------------------------------------
 * Copyright (C) 1997-2003 Jonas Kvinge <jonas@night-light.net>
 * All rights reserved.
 *
 * This code is a result of thousands of hours of work by
 * Jonas Kvinge <jonas@night-light.net>
 *
 * You may not create derivative works based on this code.
 *
 * Modified source code or binaries compiled from modified source
 * code distributed in any shape or form without the authors
 * permission is expressly forbidden.
 *
 * This code is provided without warranty of any kind.
 *
 * Under no circumstances and under no legal contract or otherwise
 * shall Jonas Kvinge be liable to you or any other person for any
 * damages, computer failure, malfunction or any other damages or
 * losses.
 *
 * Last modified by:
 * Jonas Kvinge (26.03.2002)
 *
 */

#define MEMCALLS_C

#define NEED_SYS_TYPES_H 1		/* Extra types */
#define NEED_SYS_PARAM_H 1		/* Some systems need this */
#define NEED_LIMITS_H 0			/* Kernel limits */
#define NEED_STDARG_H 1			/* va_list, etc */
#define NEED_ERRNO_H 1			/* errno */
#define NEED_CTYPE_H 1			/* isdigit(), etc */
#define NEED_NETINET_IN_H 0		/* in_addr, sockaddr_in, etc */
#define NEED_ARPA_INET_H 0		/* inet_ntoa(), inet_aton(), etc */
#define NEED_STDIO_H 1			/* Standard C UNIX functions */
#define NEED_STDLIB_H 1			/* malloc(), exit(), atoi(), etc */
#define NEED_TIME_H 1			/* time(), etc */
#define NEED_SYSCTL_H 0			/* sysctl(), etc */
#define NEED_SYS_STAT_H 0		/* chmod(), mkdir(), etc */
#define NEED_SYS_UIO_H 0		/* iovec, etc */
#define NEED_FCNTL_H 1			/* open(), creat(), fcntl(), etc */
#define NEED_SYS_IOCTL_H 0		/* ioctl(), etc */
#define NEED_SYS_FILIO_H 0		/* Solaris need this for ioctl(), etc */
#define NEED_UNISTD_H 1			/* Unix standard functions */
#define NEED_STRING_H 1			/* C string functions */
#define NEED_SIGNAL_H 0			/* Signal functions */
#define NEED_SYS_SOCKET_H 0		/* Socket functions */
#define NEED_NETDB_H 0			/* Network database functions */
#define NEED_ARPA_NAMESER_H 0		/* Nameserver definitions */
#define NEED_GETUSERPW_HEADERS 0 	/* Functions to retrive system passwords */

#include "includes.h"

#if !MEMDEBUG
  #error "memcalls.h included, but MEMDEBUG not defined TRUE."
#endif

#ifdef MALLOC_REDEFINE
  #error "Sanity check: malloc() can't be redefined in here!"
#endif

/* VARIABLES - JONAS (21.09.2001) */

struct MemTable_Struct *MemTable_Head = NULL;
struct MemTable_Struct *MemTable_Tail = NULL;
unsigned long int G_Allocations = 0;
struct MemTable_Struct *MemHashTable[MALLOC_HASH_SIZE] = { NULL };
size_t MemPrefixSize = (const size_t) sizeof(MALLOC_PREFIX);
size_t MemPostfixSize = (const size_t) sizeof(MALLOC_POSTFIX);
unsigned short int IgnoreWildPTR = FALSE;

/* MEMTABLE_ADD - JONAS (21.09.2001) */

void memtable_add(const void *const PTR, const char *const FilePT, const unsigned short int Line, const char *const FunctionPT, const size_t Size) {

  struct MemTable_Struct *MemTableS = NULL;
  struct MemTable_Struct *MemTableS_NEW = NULL;
  unsigned long int Hash = 0;

  assert(PTR != NULL);
  assert(FilePT != NULL);
  assert(FunctionPT != NULL);

  MemTableS_NEW = malloc(sizeof(struct MemTable_Struct));
  if (MemTableS_NEW == NULL) {
    return;
  }
  memset(MemTableS_NEW, 0, sizeof(struct MemTable_Struct));

  MemTableS_NEW->PTR = (const void *const) PTR;
  MemTableS_NEW->File = (const char *const) FilePT;
  MemTableS_NEW->Line = (const unsigned short int) Line;
  MemTableS_NEW->Function = (const char *const) FunctionPT;
  MemTableS_NEW->Size = Size;
  MemTableS_NEW->Time = NOW;

  if (MemTable_Head == NULL) {
    MemTable_Head = MemTableS_NEW;
    MemTable_Tail = MemTableS_NEW;
  }
  else {
    MemTableS = MemTable_Tail;
    MemTableS->Next = MemTableS_NEW;
    MemTableS_NEW->Prev = MemTableS;
    MemTable_Tail = MemTableS_NEW;
  }

  Hash = MallocHash(PTR);

  if (MemHashTable[Hash] == NULL) { MemHashTable[Hash] = MemTableS_NEW; }
  else {
    for (MemTableS = MemHashTable[Hash] ; (MemTableS->NextH != NULL) ; MemTableS = MemTableS->NextH);
    MemTableS->NextH = MemTableS_NEW;
    MemTableS_NEW->PrevH = MemTableS;
  }

  G_Allocations++;

  sysprint(BITMASK_DEBUG_MEMORY, "%s() - Pointer: %p - File: %s - Line: %d - Function: %s", __FUNCTION__, PTR, FilePT, Line, FunctionPT);

}

/* MEMTABLE_REM - JONAS (21.09.2001) */

void memtable_rem(struct MemTable_Struct *MemTableS) {

  struct MemTable_Struct *MemTableH = NULL;
  unsigned long int Hash = 0;

  assert(MemTableS != NULL);

  if (MemTableS->Prev == NULL) { MemTable_Head = MemTableS->Next; }
  else { MemTableS->Prev->Next = MemTableS->Next; }

  if (MemTableS->Next == NULL) { MemTable_Tail = MemTableS->Prev; }
  else { MemTableS->Next->Prev = MemTableS->Prev; }

  Hash = MallocHash(MemTableS->PTR);
  for (MemTableH = MemHashTable[Hash] ; ((MemTableH != NULL) && (MemTableH != MemTableS)) ; MemTableH = MemTableH->NextH);
  assert(MemTableH == MemTableS);

  if (MemTableH->PrevH != NULL) { MemTableH->PrevH->NextH = MemTableH->NextH; }
  if (MemTableH->NextH != NULL) { MemTableH->NextH->PrevH = MemTableH->PrevH; }

  if (MemHashTable[Hash] == MemTableH) { MemHashTable[Hash] = MemTableH->NextH; }

  G_Allocations--;

  sysprint(BITMASK_DEBUG_MEMORY, "%s() - Pointer: %p - File: %s - Line: %d - Function: %s", __FUNCTION__, MemTableS->PTR, MemTableS->File, MemTableS->Line, MemTableS->Function);

  free(MemTableS);

}

/* MEMTABLE_GET - JONAS (21.09.2001) */

struct MemTable_Struct *memtable_get(const void *const PTR) {

  struct MemTable_Struct *MemTableH = MemHashTable[MallocHash(PTR)];

  while ((MemTableH != NULL) && (PTR != MemTableH->PTR)) { MemTableH = MemTableH->NextH; }

  return(MemTableH);

}

/* MEMALLOC - JONAS (21.09.2001) */

void *memalloc(const char *const FilePT, const unsigned short int Line, const char *const FunctionPT, const size_t MemSize) {

  void *MemPTR = NULL;
  void *PrefixPTR = NULL;
  void *PostfixPTR = NULL;
  size_t RealSize = MemSize + MemPrefixSize + MemPostfixSize;

  sysprint(BITMASK_DEBUG_MEMORY, "%s() - File: %s - Line: %d - Function: %s - Size: %ld", __FUNCTION__, FilePT, Line, FunctionPT, (PRINT_SIZE_T) MemSize);

  if (MemSize > MALLOC_LIMIT) {
    sysprint(BITMASK_ERROR, "Memory request to large from function %s() (%s:%d) (%ld bytes)", FunctionPT, FilePT, Line, (PRINT_SIZE_T) MemSize);
    return(NULL);
  }

  PrefixPTR = malloc(RealSize);
  if (PrefixPTR == NULL) {
    sysprint(BITMASK_ERROR, "Memory allocation failure in function %s() (%s:%d): %ld bytes: [%d] %s", FunctionPT, FilePT, Line, (PRINT_SIZE_T) MemSize, errno, strerror(errno));
    return(NULL);
  }
  memset(PrefixPTR, 0, RealSize);

  PostfixPTR = PrefixPTR + MemPrefixSize + MemSize;
  MemPTR = PrefixPTR + MemPrefixSize;

  strcpy(PrefixPTR, MALLOC_PREFIX);
  strcpy(PostfixPTR, MALLOC_POSTFIX);

  memtable_add(MemPTR, FilePT, Line, FunctionPT, MemSize);

  return(MemPTR);

}

/* MEMREALLOC - JONAS (26.03.2002) */

void *memrealloc(const char *const FilePT, const unsigned short int Line, const char *const FunctionPT, void *const OldPTR, const size_t MemSize) {

  struct MemTable_Struct *MemTableS = NULL;
  void *NewPTR = NULL;

  void *OldPrefixPTR = NULL;
  void *OldPostfixPTR = NULL;

  void *NewPrefixPTR = NULL;
  void *NewPostfixPTR = NULL;

  size_t RealSize = 0;

  sysprint(BITMASK_DEBUG_MEMORY, "%s() - File: %s - Line: %d - Function: %s - Size: %ld", __FUNCTION__, FilePT, Line, FunctionPT, (PRINT_SIZE_T) MemSize);

  if (OldPTR != NULL) {

    MemTableS = memtable_get(OldPTR);
    if (MemTableS == NULL) {
      if (IgnoreWildPTR == FALSE) { sysprint(BITMASK_ERROR, "Wild pointer %p from function %s() (%s:%d) passed to function realloc().", OldPTR, FunctionPT, FilePT, Line); }
    }
    else {

      if (MemTableS->Size == MemSize) { return(OldPTR); }

      OldPrefixPTR = OldPTR - MemPrefixSize;
      OldPostfixPTR = OldPTR + MemTableS->Size;

      if (strncmp(OldPrefixPTR, MALLOC_PREFIX, MALLOC_PREFIX_LEN) != 0) {
        sysprint(BITMASK_ERROR, "Magic memory prefix \"%s\" corrupt on pointer %p from function %s() (%s:%d) passed to function realloc().", MALLOC_PREFIX, OldPTR, FunctionPT, FilePT, Line);
        abort();
      }
      if (strncmp(OldPostfixPTR, MALLOC_POSTFIX, MALLOC_POSTFIX_LEN) != 0) {
        sysprint(BITMASK_ERROR, "Magic memory postfix \"%s\" corrupt on pointer %p from function %s() (%s:%d) passed to function realloc().", MALLOC_POSTFIX, OldPTR, FunctionPT, FilePT, Line);
        abort();
      }
      memtable_rem(MemTableS);
    }
  }

  if (MemSize > MALLOC_LIMIT) {
    sysprint(BITMASK_ERROR, "Memory request to large from function %s (%s:%d) (%ld bytes)", FunctionPT, FilePT, Line, (PRINT_SIZE_T) MemSize);
    return(NULL);
  }

  if (MemSize == 0) {
    if (OldPrefixPTR != NULL) { free(OldPrefixPTR); }
    return(NULL);
  }

  if (OldPostfixPTR != NULL) { memset(OldPostfixPTR, 0, MemPostfixSize); }

  RealSize += MemPrefixSize;
  RealSize += MemSize;
  RealSize += MemPostfixSize;

  NewPrefixPTR = realloc(OldPrefixPTR, RealSize);
  if (NewPrefixPTR == NULL) {
    if (RealSize > 0) { sysprint(BITMASK_ERROR, "Memory allocation failure in function %s (%s:%d): %ld bytes: [%d] %s", FunctionPT, FilePT, Line, (PRINT_SIZE_T) MemSize, errno, strerror(errno)); }
    return(NULL);
  }

  NewPostfixPTR = NewPrefixPTR + MemPrefixSize + MemSize;
  NewPTR = NewPrefixPTR + MemPrefixSize;

  strcpy(NewPrefixPTR, MALLOC_PREFIX);
  strcpy(NewPostfixPTR, MALLOC_POSTFIX);

  memtable_add(NewPTR, FilePT, Line, FunctionPT, MemSize);

  return(NewPTR);

}

/* MEMCALLOC - JONAS (21.09.2001) */

void *memcalloc(const char *const FilePT, const unsigned short int Line, const char *const FunctionPT, const size_t Elements, const size_t Size) {

  size_t MemSize = (Size * Elements);
  void *PTR = memalloc(FilePT, Line, FunctionPT, MemSize);

  return(PTR);

}

/* MEMFREE - JONAS (21.09.2001) */

signed short int memfree(const char *const FilePT, const unsigned short int Line, const char *const FunctionPT, void *const MemPTR) {

  struct MemTable_Struct *MemTableS = NULL;
  void *PrefixPTR = NULL;
  void *PostfixPTR = NULL;
  size_t TotalSize = 0;

  sysprint(BITMASK_DEBUG_MEMORY, "%s() - File: %s - Line: %d - Function: %s - Pointer: %p", __FUNCTION__, FilePT, Line, FunctionPT, MemPTR);

  if (MemPTR == NULL) { return(SUCCESS); }

  MemTableS = memtable_get(MemPTR);
  if (MemTableS == NULL) {
    if (IgnoreWildPTR == FALSE) { sysprint(BITMASK_ERROR, "Wild pointer %p from function %s (%s:%d) passed to function free().", MemPTR, FunctionPT, FilePT, Line); }
    return(ERROR);
  }

  PrefixPTR = MemPTR - MemPrefixSize;
  PostfixPTR = MemPTR + MemTableS->Size;
  TotalSize = MemPrefixSize + MemTableS->Size + MemPostfixSize;

  if (strncmp(PrefixPTR, MALLOC_PREFIX, MALLOC_PREFIX_LEN) != 0) {
    sysprint(BITMASK_ERROR, "Magic memory prefix \"%s\" corrupt on pointer %p from function %s (%s:%d) passed to function free().", MALLOC_PREFIX, MemPTR, FunctionPT, FilePT, Line);
    abort();
  }

  if (strncmp(PostfixPTR, MALLOC_POSTFIX, MALLOC_POSTFIX_LEN) != 0) {
    sysprint(BITMASK_ERROR, "Magic memory postfix \"%s\" corrupt on pointer %p from function %s (%s:%d) passed to function free().", MALLOC_POSTFIX, MemPTR, FunctionPT, FilePT, Line);
    abort();
  }

  memtable_rem(MemTableS);

  memset(PrefixPTR, MALLOC_REDBYTE, TotalSize);
  free(PrefixPTR);

  return(SUCCESS);

}
