/*
 * ----------------------------------------------------------------
 * Night Light Asynchrounous DNS Resolver
 * ----------------------------------------------------------------
 * Copyright (C) 1997-2003 Jonas Kvinge <jonas@night-light.net>
 * All rights reserved.
 *
 * This code is a result of thousands of hours of work by
 * Jonas Kvinge <jonas@night-light.net>
 *
 * You may not create derivative works based on this code.
 *
 * Modified source code or binaries compiled from modified source
 * code distributed in any shape or form without the authors
 * permission is expressly forbidden.
 *
 * This code is provided without warranty of any kind.
 *
 * Under no circumstances and under no legal contract or otherwise
 * shall Jonas Kvinge be liable to you or any other person for any
 * damages, computer failure, malfunction or any other damages or
 * losses.
 *
 * Last modified by:
 * Jonas Kvinge (17.07.2001)
 *
 */

#define ARES_IO_C

#define NEED_SYS_TYPES_H 1		/* Extra types */
#define NEED_SYS_PARAM_H 1		/* Some systems need this */
#define NEED_LIMITS_H 0			/* Kernel limits */
#define NEED_STDARG_H 1			/* va_list, etc */
#define NEED_ERRNO_H 1			/* errno */
#define NEED_CTYPE_H 1			/* isdigit(), etc */
#define NEED_NETINET_IN_H 1		/* in_addr, sockaddr_in, etc */
#define NEED_ARPA_INET_H 1		/* inet_ntoa(), inet_aton(), etc */
#define NEED_STDIO_H 1			/* Standard C UNIX functions */
#define NEED_STDLIB_H 1			/* malloc(), exit(), atoi(), etc */
#define NEED_TIME_H 1			/* time(), etc */
#define NEED_SYSCTL_H 0			/* sysctl(), etc */
#define NEED_SYS_STAT_H 0		/* chmod(), mkdir(), etc */
#define NEED_SYS_UIO_H 0		/* iovec, etc */
#define NEED_FCNTL_H 1			/* open(), creat(), fcntl(), etc */
#define NEED_SYS_IOCTL_H 0		/* ioctl(), etc */
#define NEED_SYS_FILIO_H 0		/* Solaris need this for ioctl(), etc */
#define NEED_UNISTD_H 1			/* Unix standard functions */
#define NEED_STRING_H 1			/* C string functions */
#define NEED_SIGNAL_H 0			/* Signal functions */
#define NEED_SYS_SOCKET_H 1		/* Socket functions */
#define NEED_NETDB_H 1			/* Network database functions */
#define NEED_ARPA_NAMESER_H 1		/* Nameserver definitions */
#define NEED_GETUSERPW_HEADERS 0 	/* Functions to retrive system passwords */

#include "includes.h"

#include "dns.h"

#include "ares_io.h"
#include "ares_int.h"
#include "ares_query.h"
#include "ares_parse.h"
#include "ares_tcp.h"
#include "ares_udp.h"

/* VARIABLES - JONAS (05.07.2000) */

const char ARES_IO_VERSION[] = "0.5.3d";

struct Ares_Server_Struct *Ares_Server_Head = NULL;
struct Ares_Server_Struct *Ares_Server_Tail = NULL;

struct Ares_Host_Struct *Ares_Host_Head = NULL;
struct Ares_Host_Struct *Ares_Host_Tail = NULL;

const char *Ares_StrError[] = {

  "No error condition.",
  "No data for requested type.",
  "Query was misformatted.",
  "General failure.",
  "Domain name not found.",
  "Server does not support requested operation.",
  "Server refused query.",
  "Misformatted query.",
  "Misformatted domain name.",
  "Unsupported address family.",
  "Misformatted reply.",
  "Could not contact servers.",
  "Timeout while contacting servers.",
  "End of file.",
  "Error reading file.",
  "Out of memory.",
  "Resolver destruction.",
  "Query cancelled."

};

extern struct Ares_Query_Struct *Ares_Query_Head;

/* ARES_STRERROR FUNCTION - JONAS (25.06.2000) */

const char *ares_strerror(unsigned long int Code) {

  assert((Code >= 0) && (Code < (sizeof(Ares_StrError) / sizeof(*Ares_StrError))));
  return(Ares_StrError[Code]);

}

/* ARES_READRESOLVCONF FUNCTION - JONAS (25.06.2000) */

void ares_readresolvconf(void) {

  struct Ares_Server_Struct *Server = NULL;
  struct Ares_Server_Struct *Server_NEW = NULL;
  FILE *FilePT = NULL;
  signed long int Result = 0;
  struct in_addr InAddr;

  FilePT = fopen(RESOLVCONFFILE, "r");
  if (FilePT == NULL) {
    sysprint(BITMASK_ERROR, "Ares: Unable to open %s: [%d] %s", RESOLVCONFFILE, errno, strerror(errno));
    exit(0);
  }

  FOREVERLOOP {

    char Line[LINELEN+1] = "";
    char *LinePT = NULL;

    char *TempPT = NULL;

    char *OptionPT = NULL;
    char *ValuePT = NULL;

    memset(&Line, 0, LINELEN+1);
    TempPT = fgets(Line, LINELEN, FilePT);
    if (TempPT == NULL) { break; }

    LinePT = Line;

    while ((TempPT = strchr(LinePT, '\r')) != NULL) { *TempPT = '\0'; }
    while ((TempPT = strchr(LinePT, '\n')) != NULL) { *TempPT = '\0'; }
    while ((TempPT = strchr(LinePT, '\t')) != NULL) { *TempPT = ' '; }

    while ((LinePT[0] == ' ') || (LinePT[0] == '\t')) { ++LinePT; }
    if ((LinePT[0] == '\0') || (LinePT[0] == '#') || (LinePT[0] == ';')) { continue; }

    OptionPT = LinePT;
    StrMovePastToken(LinePT, ' ');
    if (LinePT == NULL) { continue; }

    ValuePT = LinePT;
    StrMovePastToken(LinePT, ' ');

    if (strcasecmp(OptionPT, "nameserver") == FALSE) {

      memset(&InAddr, 0, sizeof(InAddr));
      Result = inet_aton(ValuePT, &InAddr);
      if (Result == 0) { continue; }
      if (InAddr.s_addr == INADDR_NONE) { continue; }

      Server_NEW = malloc(sizeof(struct Ares_Server_Struct));
      if (Server_NEW == NULL) {
        fclose(FilePT);
        return;
      }
      memset(Server_NEW, 0, sizeof(struct Ares_Server_Struct));

      Server_NEW->HostIPS = strdup(ValuePT);
      Server_NEW->InAddr = InAddr;
      Server_NEW->PortH = NAMESERVER_PORT;
      Server_NEW->PortN = htons(Server_NEW->PortH);

      if (Ares_Server_Head == NULL) {
        Ares_Server_Head = Server_NEW;
        Ares_Server_Tail = Server_NEW;
      }
      else {
        Server = Ares_Server_Tail;
        Server->Next = Server_NEW;
        Server_NEW->Prev = Server;
        Ares_Server_Tail = Server_NEW;
      }

      sysprint(BITMASK_DEBUG_ARES, "Ares: Nameserver \"%s\" from \"%s\".", Server_NEW->HostIPS, RESOLVCONFFILE);

    }
  }
  fclose(FilePT);

  if (Ares_Server_Head == NULL) {

    Server = malloc(sizeof(struct Ares_Server_Struct));
    if (Server == NULL) { return; }
    memset(Server, 0, sizeof(struct Ares_Server_Struct));

    Ares_Server_Head = Server;
    Ares_Server_Tail = Server;

    Server->HostIPS = strdup("127.0.0.1");
    Server->InAddr.s_addr = htonl(INADDR_LOOPBACK);
    Server->PortH = NAMESERVER_PORT;
    Server->PortN = htons(Server->PortH);

    sysprint(BITMASK_MAIN, "Ares: No valid nameservers found in \"%s\", defaulting to \"%s\".", RESOLVCONFFILE, Server->HostIPS);

  }

}

/* ARES_READHOSTS FUNCTION - JONAS (25.06.2000) */

void ares_readhosts(void) {

  struct Ares_Host_Struct *Host = NULL;
  struct Ares_Host_Struct *Host_NEW = NULL;

  FILE *FilePT = NULL;

  FilePT = fopen(HOSTSFILE, "r");
  if (FilePT == NULL) {
    sysprint(BITMASK_ERROR, "Ares: Unable to open %s: [%d] %s", HOSTSFILE, errno, strerror(errno));
    exit(0);
  }

  FOREVERLOOP {

    char Line[LINELEN+1] = "";
    char *LinePT = NULL;

    char *TempPT = NULL;

    char *HostIPSPT = NULL;
    char *HostNamePT = NULL;

    memset(&Line, 0, LINELEN+1);
    TempPT = fgets(Line, LINELEN, FilePT);
    if (TempPT == NULL) { break; }

    LinePT = Line;

    while ((TempPT = strchr(LinePT, '\r')) != NULL) { *TempPT = '\0'; }
    while ((TempPT = strchr(LinePT, '\n')) != NULL) { *TempPT = '\0'; }
    while ((TempPT = strchr(LinePT, '\t')) != NULL) { *TempPT = ' '; }

    while ((LinePT[0] == ' ') || (LinePT[0] == '\t')) { ++LinePT; }
    if ((LinePT[0] == '\0') || (LinePT[0] == '#') || (LinePT[0] == ';')) { continue; }

    HostIPSPT = LinePT;
    StrMovePastToken(LinePT, ' ');

    while ((LinePT != NULL) && (LinePT[0] != '\0')) {

      HostNamePT = LinePT;
      StrMovePastToken(LinePT, ' ');

      Host_NEW = malloc(sizeof(struct Ares_Host_Struct));
      if (Host_NEW == NULL) {
        fclose(FilePT);
        return;
      }
      memset(Host_NEW, 0, sizeof(struct Ares_Host_Struct));

      Host_NEW->HostIPS = strdup(HostIPSPT);
      Host_NEW->HostName = strdup(HostNamePT);

      if (Ares_Host_Head == NULL) {
        Ares_Host_Head = Host_NEW;
        Ares_Host_Tail = Host_NEW;
      }
      else {
        Host = Ares_Host_Tail;
        Host->Next = Host_NEW;
        Host_NEW->Prev = Host;
        Ares_Host_Tail = Host_NEW;
      }

      sysprint(BITMASK_DEBUG_ARES, "Ares: Host \"%s\"=\"%s\" from \"%s\".", Host_NEW->HostName, Host_NEW->HostIPS, HOSTSFILE);

    }

  }

  fclose(FilePT);

}

/* ARES_START FUNCTION - JONAS (25.06.2000) */

void ares_start(void) {

  ares_readresolvconf();
  ares_readhosts();

}

/* ARES_STOP FUNCTION - JONAS (25.06.2000) */

void ares_stop(void) {

  struct Ares_Server_Struct *Server = NULL;
  struct Ares_Host_Struct *Host = NULL;

  while (Ares_Query_Head != NULL) { ares_query_rem(Ares_Query_Head); }

  while (Ares_Server_Head != NULL) {
    Server = Ares_Server_Head;
    Ares_Server_Head = Server->Next;
    ares_tcp_disconnect(Server, "Ares: Disconnected from %s TCP: Stopping DNS resolver.", Server->HostIPS);
    ares_udp_disconnect(Server, "Ares: Disconnected from %s UDP: Stopping DNS resolver.", Server->HostIPS);
    free(Server->HostIPS);
    free(Server);
  }
  Ares_Server_Tail = NULL;

  while (Ares_Host_Head != NULL) {
    Host = Ares_Host_Head;
    Ares_Host_Head = Host->Next;
    free(Host->HostName);
    free(Host->HostIPS);
    free(Host);
  }
  Ares_Host_Tail = NULL;

}

/* ARES_FDS FUNCTION - JONAS (17.07.2001) */

void ares_fds(fd_set *ReadFDS, fd_set *WriteFDS, unsigned long int *FDS) {

  struct Ares_Server_Struct *Server = NULL;
  struct Ares_Query_Struct *Query = NULL;
  struct Ares_Query_Struct *Query_PRV = NULL;
  struct Ares_Query_Struct *Query_DEL = NULL;
  time_t Duration = 0;

  assert(ReadFDS != NULL);
  assert(WriteFDS != NULL);

  for (Server = Ares_Server_Head ; Server != NULL ; Server = Server->Next) {

    if ((Server->ConnectTCP == TRUE) && (!Ares_IsConnectProc(Server->TCP_Flags)) && (!Ares_IsConnected(Server->TCP_Flags))) {
      Duration = NOW - Server->TCP_Time;
      if (Duration >= ARES_TCP_CONNECTWAIT) { ares_tcp_connect(Server); }
    }
    else {
      if (Ares_IsSocket(Server->TCP_Flags)) {
        assert(Server->TCP_Socket > 0);
        FD_SET(Server->TCP_Socket, ReadFDS);
        if (Server->TCPSend_Head != NULL) { FD_SET(Server->TCP_Socket, WriteFDS); }
      }
    }

    if ((!Ares_IsConnectProc(Server->UDP_Flags)) && (!Ares_IsConnected(Server->UDP_Flags))) {
      Duration = NOW - Server->UDP_Time;
      if (Duration >= ARES_UDP_CONNECTWAIT) { ares_udp_connect(Server); }
    }
    else {
      if (Ares_IsSocket(Server->UDP_Flags)) {
        assert(Server->UDP_Socket > 0);
        FD_SET(Server->UDP_Socket, ReadFDS);
        if (Server->UDPSend_Head != NULL) { FD_SET(Server->UDP_Socket, WriteFDS); }
      }
    }

  }

  for (Query = Ares_Query_Head ; Query != NULL ;) {

    if (NOW >= Query->Server_Timeout) { ares_query_server(Query); }

    if ((NOW >= Query->Timeout) || (Query->Tries > ARES_TRIES)) {
      Query_DEL = Query;
      Query = Query->Next;
      ares_query_callback(Query_DEL, ARES_ETIMEOUT);
      ares_query_rem(Query_DEL);
    }
    else {
      Query_PRV = Query;
      Query = Query->Next;
    }

  }

}

/* ARES_IO FUNCTION - JONAS (25.06.2000) */

void ares_io(fd_set *ReadFDS, fd_set *WriteFDS, unsigned long int *FDS) {

  struct Ares_Server_Struct *Server = NULL;

  assert(ReadFDS != NULL);
  assert(WriteFDS != NULL);

  for (Server = Ares_Server_Head ; Server != NULL ; Server = Server->Next) {

    assert(*FDS >= 0);
    if (*FDS <= 0) { return; }

    if (Ares_IsSocket(Server->TCP_Flags)) {
      if (FD_ISSET(Server->TCP_Socket, ReadFDS)) { *FDS = *FDS - 1; }
      if (FD_ISSET(Server->TCP_Socket, WriteFDS)) { *FDS = *FDS - 1; }
    }

    if (Ares_IsSocket(Server->TCP_Flags)) {
      if (FD_ISSET(Server->TCP_Socket, ReadFDS)) { ares_tcp_read(Server); }
    }
    if (Ares_IsSocket(Server->TCP_Flags)) {
      if (FD_ISSET(Server->TCP_Socket, WriteFDS)) { ares_tcp_write(Server); }
    }



    if (Ares_IsSocket(Server->UDP_Flags)) {
      if (FD_ISSET(Server->UDP_Socket, ReadFDS)) { *FDS = *FDS - 1; }
      if (FD_ISSET(Server->UDP_Socket, WriteFDS)) { *FDS = *FDS - 1; }
    }

    if (Ares_IsSocket(Server->UDP_Flags)) {
      if (FD_ISSET(Server->UDP_Socket, ReadFDS)) { ares_udp_read(Server); }
    }
    if (Ares_IsSocket(Server->UDP_Flags)) {
      if (FD_ISSET(Server->UDP_Socket, WriteFDS)) { ares_udp_write(Server); }
    }
    

  }

}

/* ARES_GETHOSTBYNAME FUNCTION - JONAS (25.06.2000) */

void ares_gethostbyname(const char *NamePT, const signed long int Family, Ares_Callback Callback, void *ArgPT) {

  struct Ares_Host_Struct *Host = NULL;

  assert(NamePT != NULL);

  if (Family != AF_INET) { Callback(ArgPT, ARES_ENOTIMP, NULL); return; }

  for (Host = Ares_Host_Head ; Host != NULL ; Host = Host->Next) {
    if (strcasecmp(Host->HostName, NamePT) == FALSE) {

      signed long int Result = 0;
      struct in_addr InAddr;
      struct hostent *HostEnt = NULL;

      memset(&InAddr, 0, sizeof(InAddr));
      Result = inet_aton(Host->HostIPS, &InAddr);
      if (Result == 0) { continue; }
      if (InAddr.s_addr == INADDR_NONE) { continue; }

      HostEnt = malloc(sizeof(struct hostent));
      HostEnt->h_addr_list = malloc(2 * sizeof(char *));
      if (HostEnt->h_addr_list == NULL) {
        free(HostEnt);
        Callback(ArgPT, ARES_ENOMEM, NULL);
        return;
      }
      HostEnt->h_addr_list[0] = (char *) &InAddr;
      HostEnt->h_addr_list[1] = NULL;
      HostEnt->h_name = strdup(Host->HostName);
      if (HostEnt->h_name == NULL) {
        free(HostEnt->h_addr_list);
        free(HostEnt);
        Callback(ArgPT, ARES_ENOMEM, NULL);
        return;
      }
      HostEnt->h_aliases = NULL;
      HostEnt->h_addrtype = AF_INET;
      HostEnt->h_length = sizeof(struct in_addr);
      Callback(ArgPT, ARES_SUCCESS, HostEnt);
      free(HostEnt->h_name);
      free(HostEnt->h_addr_list);
      free(HostEnt);
      return;
    }
  }

  if (Ares_Server_Head == NULL) { Callback(ArgPT, ARES_ESERVFAIL, NULL); return; }

  ares_query_add(C_IN, T_A, NamePT, 0, 0, Callback, ArgPT);

}

/* ARES_GETHOSTBYADDR FUNCTION - JONAS (25.06.2000) */

void ares_gethostbyaddr(const void *AddrPT, const unsigned long int AddrLen, const unsigned long int Family, Ares_Callback Callback, void *ArgPT) {

  char HostIPS[HOSTLEN+1] = "";
  char Name[HOSTLEN+1] = "";
  signed long int B1 = 0;
  signed long int B2 = 0;
  signed long int B3 = 0;
  signed long int B4 = 0;
  struct in_addr InAddr;
  unsigned long int HostH = 0;
  struct Ares_Host_Struct *Host = NULL;

  assert(ArgPT != NULL);

  if (Family != AF_INET) { Callback(ArgPT, ARES_ENOTIMP, NULL); return; }

  memcpy(&InAddr, AddrPT, sizeof(InAddr));
  HostH = ntohl(InAddr.s_addr);
  B1 = (HostH >> 24);
  B2 = ((HostH >> 16) & 0xff);
  B3 = ((HostH >> 8) & 0xff);
  B4 = (HostH & 0xff);
  snprintf(Name, HOSTLEN+1, "%ld.%ld.%ld.%ld.in-addr.arpa", B4, B3, B2, B1);
  snprintf(HostIPS, HOSTLEN+1, "%ld.%ld.%ld.%ld", B1, B2, B3, B4);

  for (Host = Ares_Host_Head ; Host != NULL ; Host = Host->Next) {
    if (strcasecmp(Host->HostIPS, HostIPS) == FALSE) {

      struct hostent *HostEnt = NULL;

      HostEnt = malloc(sizeof(struct hostent));
      HostEnt->h_addr_list = malloc(2 * sizeof(char *));
      if (HostEnt->h_addr_list == NULL) {
        free(HostEnt);
        Callback(ArgPT, ARES_ENOMEM, NULL);
        return;
      }
      HostEnt->h_addr_list[0] = (char *) &InAddr;
      HostEnt->h_addr_list[1] = NULL;
      HostEnt->h_name = strdup(Host->HostName);
      if (HostEnt->h_name == NULL) {
        free(HostEnt->h_addr_list);
        free(HostEnt);
        Callback(ArgPT, ARES_ENOMEM, NULL);
        return;
      }
      HostEnt->h_aliases = NULL;
      HostEnt->h_addrtype = AF_INET;
      HostEnt->h_length = sizeof(struct in_addr);
      Callback(ArgPT, ARES_SUCCESS, HostEnt);
      free(HostEnt->h_name);
      free(HostEnt->h_addr_list);
      free(HostEnt);
      return;
    }
  }

  if (Ares_Server_Head == NULL) { Callback(ArgPT, ARES_ESERVFAIL, NULL); return; }

  ares_query_add(C_IN, T_PTR, Name, InAddr.s_addr, AddrLen, Callback, ArgPT);

}

/* ARES_CANCELQUERY FUNCTION - JONAS (25.06.2000) */

void ares_cancelquery(Ares_Callback Callback, const void *const ArgPT) {

  struct Ares_Query_Struct *Query = NULL;
  struct Ares_Query_Struct *Query_DEL = NULL;

  assert(ArgPT != NULL);

  for (Query = Ares_Query_Head ; Query != NULL ;) {

    if (((ArgPT == NULL) || (Query->Arg == ArgPT)) && ((Callback == NULL) || (Query->Callback == Callback))) {
      sysprint(BITMASK_DEBUG_ARES, "Ares: Cancelling query - ID: %ld, Class: %d, Type: %d, Name: %s.", Query->ID, Query->Class, Query->Type, Query->Name);
      Query_DEL = Query;
      Query = Query->Next;
      ares_query_callback(Query_DEL, ARES_ECANCELLED);
      ares_query_rem(Query_DEL);
      continue;
    }
    Query = Query->Next;
  }

}
