/*
 * ----------------------------------------------------------------
 * Night Light Asynchrounous DNS Resolver
 * ----------------------------------------------------------------
 * Copyright (C) 1997-2003 Jonas Kvinge <jonas@night-light.net>
 * All rights reserved.
 *
 * This code is a result of thousands of hours of work by
 * Jonas Kvinge <jonas@night-light.net>
 *
 * You may not create derivative works based on this code.
 *
 * Modified source code or binaries compiled from modified source
 * code distributed in any shape or form without the authors
 * permission is expressly forbidden.
 *
 * This code is provided without warranty of any kind.
 *
 * Under no circumstances and under no legal contract or otherwise
 * shall Jonas Kvinge be liable to you or any other person for any
 * damages, computer failure, malfunction or any other damages or
 * losses.
 *
 * Last modified by:
 * Jonas Kvinge (30.07.2001)
 *
 */

#define ARES_PARSE_C

#define NEED_SYS_TYPES_H 1		/* Extra types */
#define NEED_SYS_PARAM_H 1		/* Some systems need this */
#define NEED_LIMITS_H 0			/* Kernel limits */
#define NEED_STDARG_H 1			/* va_list, etc */
#define NEED_ERRNO_H 1			/* errno */
#define NEED_CTYPE_H 1			/* isdigit(), etc */
#define NEED_NETINET_IN_H 1		/* in_addr, sockaddr_in, etc */
#define NEED_ARPA_INET_H 1		/* inet_ntoa(), inet_aton(), etc */
#define NEED_STDIO_H 1			/* Standard C UNIX functions */
#define NEED_STDLIB_H 1			/* malloc(), exit(), atoi(), etc */
#define NEED_TIME_H 1			/* time(), etc */
#define NEED_SYSCTL_H 0			/* sysctl(), etc */
#define NEED_SYS_STAT_H 0		/* chmod(), mkdir(), etc */
#define NEED_SYS_UIO_H 0		/* iovec, etc */
#define NEED_FCNTL_H 1			/* open(), creat(), fcntl(), etc */
#define NEED_SYS_IOCTL_H 0		/* ioctl(), etc */
#define NEED_SYS_FILIO_H 0		/* Solaris need this for ioctl(), etc */
#define NEED_UNISTD_H 1			/* Unix standard functions */
#define NEED_STRING_H 1			/* C string functions */
#define NEED_SIGNAL_H 0			/* Signal functions */
#define NEED_SYS_SOCKET_H 1		/* Socket functions */
#define NEED_NETDB_H 1			/* Network database functions */
#define NEED_ARPA_NAMESER_H 1		/* Nameserver definitions */
#define NEED_GETUSERPW_HEADERS 0 	/* Functions to retrive system passwords */

#include "includes.h"

#include "dns.h"

#include "ares_io.h"
#include "ares_int.h"
#include "ares_query.h"
#include "ares_parse.h"

/* VARIABLES - JONAS (30.07.2001) */

extern struct Ares_Query_Struct *Ares_Query_Array[];

/* ARES_PARSE FUNCTION - JONAS (25.06.2000) */

void ares_parse(const unsigned char *BufferPT, const signed long int BufferLen, struct Ares_Server_Struct *Ares_Server, const unsigned short int TCP) {

  unsigned long int ID = 0;
  signed long int TC = 0;
  signed short int RCode = NONE;
  struct Ares_Query_Struct *Query = NULL;
  unsigned short int QDCount = 0;
  unsigned short int ANCount = 0;
  signed long int Result = 0;

  if (BufferLen < HFIXEDSZ) { return; } /* DOESN'T EVEN CONTAIN A HEADER, CAN'T BE ANY GOOD --- JONAS (25.06.2000) */

  ID = DNS_HEADER_QID(BufferPT);
  if (ID > ARES_QUERY_MAX) { return; }
  Query = Ares_Query_Array[ID];
  if (Query == NULL) { return; }

  TC = DNS_HEADER_TC(BufferPT);
  RCode = DNS_HEADER_RCODE(BufferPT);

  if ((TC == TRUE) && (TCP == FALSE)) {
    Query->TCP = TRUE;
    ares_query_server(Query);
    return;
  }

  QDCount = DNS_HEADER_QDCOUNT(BufferPT);
  ANCount = DNS_HEADER_ANCOUNT(BufferPT);
  if (QDCount != 1) {
     ares_query_callback(Query, ARES_EBADRESP);
     ares_query_rem(Query);
     return;
  }

  if (Query->Type == T_A) { Result = ares_parse_a(Query, BufferPT, BufferLen, ANCount); }
  else if (Query->Type == T_PTR) { Result = ares_parse_ptr(Query, BufferPT, BufferLen, ANCount); }
  else { Result = ARES_EBADRESP; }

  if (Result != ARES_SUCCESS) {
    ares_query_callback(Query, Result);
    ares_query_rem(Query);
    return;
  }

  RCode = DNS_HEADER_RCODE(BufferPT);
  ANCount = DNS_HEADER_ANCOUNT(BufferPT);

  switch (RCode) {
    case NOERROR:
      ares_query_callback(Query, ((ANCount > 0) ? ARES_SUCCESS : ARES_ENODATA));
      break;
    case FORMERR:
      ares_query_callback(Query, ARES_EFORMERR);
      break;
    case SERVFAIL:
      ares_query_callback(Query, ARES_ESERVFAIL);
      break;
    case NXDOMAIN:
      ares_query_callback(Query, ARES_ENOTFOUND);
      break;
    case NOTIMP:
      ares_query_callback(Query, ARES_ENOTIMP);
      break;
    case REFUSED:
      ares_query_callback(Query, ARES_EREFUSED);
      break;
    default:
      ares_query_callback(Query, ARES_EBADRESP);
      break;
  }

  ares_query_rem(Query);

}

/* ARES_PARSE_A FUNCTION - JONAS (25.06.2000) */

signed short int ares_parse_a(struct Ares_Query_Struct *Query, const unsigned char *BufferPT, const unsigned long int BufferLen, unsigned short int ANCount) {

  signed long int Result = 0;
  signed short int Count = 0;
  unsigned long int Len = 0;
  signed short int RR_Type = 0;
  signed short int RR_Class = 0;
  signed short int RR_Len = 0;
  signed short int NAddrs = 0;
  signed short int NAliases = 0;
  const unsigned char *WildBufferPT = NULL;
  char *HostNamePT = NULL;
  char *RR_NamePT = NULL;
  char *RR_DataPT = NULL;
  char **AliasesPT = NULL;
  struct in_addr *InAddrs = NULL;
  struct hostent *HostEnt = NULL;
  const unsigned char *MaxBufferPosPT = BufferPT + BufferLen;

  assert(Query != NULL);
  assert(BufferPT != NULL);

  WildBufferPT = BufferPT + HFIXEDSZ;
  Result = ares_parse_nmexp(WildBufferPT, BufferPT, BufferLen, &HostNamePT, &Len);
  if (Result != ARES_SUCCESS) { return(Result); }
  if (WildBufferPT + Len + QFIXEDSZ > MaxBufferPosPT) {
    free(HostNamePT);
    return(ARES_EBADRESP);
  }
  WildBufferPT += Len + QFIXEDSZ;

  InAddrs = malloc(ANCount * sizeof(struct in_addr));
  if (InAddrs == NULL) {
    free(HostNamePT);
    return(ARES_ENOMEM);
  }
  AliasesPT = malloc((ANCount + 1) * sizeof(char *));
  if (AliasesPT == NULL) {
    free(HostNamePT);
    free(InAddrs);
    return(ARES_ENOMEM);
  }
  NAddrs = 0;
  NAliases = 0;

  for (Count = 0 ; Count < ANCount ; Count++) {

    Result = ares_parse_nmexp(WildBufferPT, BufferPT, BufferLen, &RR_NamePT, &Len);
    if (Result != ARES_SUCCESS) { break; }
    assert(RR_NamePT != NULL);

    WildBufferPT += Len;
    if (WildBufferPT + RRFIXEDSZ > MaxBufferPosPT) {
      Result = ARES_EBADRESP;
      break;
    }

    RR_Type = DNS_RR_TYPE(WildBufferPT);
    RR_Class = DNS_RR_CLASS(WildBufferPT);
    RR_Len = DNS_RR_LEN(WildBufferPT);
    WildBufferPT += RRFIXEDSZ;

    if ((RR_Class == C_IN) && (RR_Type == T_A) && (RR_Len == sizeof(struct in_addr)) && (strcasecmp(RR_NamePT, HostNamePT) == FALSE)) {
      free(RR_NamePT);
      RR_NamePT = NULL;
      memcpy(&InAddrs[NAddrs], WildBufferPT, sizeof(struct in_addr));
      NAddrs++;
      Result = ARES_SUCCESS;
    }
    else if ((RR_Class == C_IN) && (RR_Type == T_CNAME)) {

      AliasesPT[NAliases] = RR_NamePT;
      NAliases++;

      Result = ares_parse_nmexp(WildBufferPT, BufferPT, BufferLen, &RR_DataPT, &Len);
      if (Result != ARES_SUCCESS) { break; }
      free(HostNamePT);
      HostNamePT = RR_DataPT;
    }
    else { free(RR_NamePT); }

    WildBufferPT += RR_Len;
    if (WildBufferPT > MaxBufferPosPT) {
      Result = ARES_EBADRESP;
      break;
    }
  }

  if ((Result == ARES_SUCCESS) && (NAddrs == 0)) { Result = ARES_ENODATA; }
  if (Result == ARES_SUCCESS) {
    AliasesPT[NAliases] = NULL;
    HostEnt = malloc(sizeof(struct hostent));
    if (HostEnt != NULL) {
      HostEnt->h_addr_list = malloc((NAddrs + 1) * sizeof(char *));
      if (HostEnt->h_addr_list != NULL) {
        HostEnt->h_name = HostNamePT;
        HostEnt->h_aliases = AliasesPT;
        HostEnt->h_addrtype = AF_INET;
        HostEnt->h_length = sizeof(struct in_addr);
        for (Count = 0 ; Count < NAddrs ; Count++) { HostEnt->h_addr_list[Count] = (char *) &InAddrs[Count]; }
        HostEnt->h_addr_list[NAddrs] = NULL;
        Query->HostEnt = HostEnt;
        return(ARES_SUCCESS);
      }
      free(HostEnt);
    }
    Result = ARES_ENOMEM;
  }
  for (Count = 0 ; Count < NAliases ; Count++) { free(AliasesPT[Count]); }
  free(AliasesPT);
  free(InAddrs);
  free(HostNamePT);

  return(Result);

}

/* ARES_PARSE_PTR FUNCTION - JONAS (25.06.2000) */

signed short int ares_parse_ptr(struct Ares_Query_Struct *Query, const unsigned char *BufferPT, const unsigned long int BufferLen, unsigned short int ANCount) {

  signed long int Result = 0;
  signed short int Count = 0;
  signed long int Len = 0;
  signed short int RR_Type = 0;
  signed short int RR_Class = 0;
  signed short int RR_Len = 0;
  const unsigned char *WildBufferPT = NULL;
  char *PTRNamePT = NULL;
  char *HostNamePT = NULL;
  char *RR_NamePT = NULL;
  char *RR_DataPT = NULL;
  struct hostent *HostEnt = NULL;
  const unsigned char *MaxBufferPosPT = BufferPT + BufferLen;

  assert(Query != NULL);
  assert(BufferPT != NULL);

  WildBufferPT = BufferPT + HFIXEDSZ;
  Result = ares_parse_nmexp(WildBufferPT, BufferPT, BufferLen, &PTRNamePT, &Len);
  if (Result != ARES_SUCCESS) { return(Result); }
  if (WildBufferPT + Len + QFIXEDSZ > MaxBufferPosPT) {
    free(PTRNamePT);
    return(ARES_EBADRESP);
  }
  WildBufferPT += Len + QFIXEDSZ;

  for (Count = 0 ; Count < ANCount ; Count++) {

    Result = ares_parse_nmexp(WildBufferPT, BufferPT, BufferLen, &RR_NamePT, &Len);
    if (Result != ARES_SUCCESS) { break; }
    WildBufferPT += Len;
    if (WildBufferPT + RRFIXEDSZ > MaxBufferPosPT) {
      Result = ARES_EBADRESP;
      break;
    }
    RR_Type = DNS_RR_TYPE(WildBufferPT);
    RR_Class = DNS_RR_CLASS(WildBufferPT);
    RR_Len = DNS_RR_LEN(WildBufferPT);
    WildBufferPT += RRFIXEDSZ;

    if ((RR_Class == C_IN) && (RR_Type == T_PTR) && (strcasecmp(RR_NamePT, PTRNamePT) == FALSE)) {

      Result = ares_parse_nmexp(WildBufferPT, BufferPT, BufferLen, &RR_DataPT, &Len);
      if (Result != ARES_SUCCESS) { break; }
      free(HostNamePT);
      HostNamePT = RR_DataPT;

    }

    else if ((RR_Class == C_IN) && (RR_Type == T_CNAME)) {

      Result = ares_parse_nmexp(WildBufferPT, BufferPT, BufferLen, &RR_DataPT, &Len);
      if (Result != ARES_SUCCESS) { break; }
      free(PTRNamePT);
      PTRNamePT = RR_DataPT;

    }

    free(RR_NamePT);
    WildBufferPT += RR_Len;
    if (WildBufferPT > MaxBufferPosPT) {
      Result = ARES_EBADRESP;
      break;
    }
  }

  if ((Result == ARES_SUCCESS) && (HostNamePT == NULL)) { Result = ARES_ENODATA; }
  if (Result == ARES_SUCCESS) {
    HostEnt = malloc(sizeof(struct hostent));
    if (HostEnt != NULL) {
      HostEnt->h_addr_list = malloc((2 * sizeof(char *)));
      if (HostEnt->h_addr_list != NULL) {
        HostEnt->h_addr_list[0] = malloc(sizeof(Query->Addr));
        if (HostEnt->h_addr_list[0] != NULL) {
          HostEnt->h_aliases = malloc(sizeof(char *));
          if (HostEnt->h_aliases != NULL) {
            HostEnt->h_name = HostNamePT;
            HostEnt->h_aliases[0] = NULL;
            HostEnt->h_addrtype = AF_INET;
            HostEnt->h_length = sizeof(Query->Addr);
            memcpy(HostEnt->h_addr_list[0], &Query->Addr, sizeof(Query->Addr));
            HostEnt->h_addr_list[1] = NULL;
            Query->HostEnt = HostEnt;
            free(PTRNamePT);
            return(ARES_SUCCESS);
          }
          free(HostEnt->h_addr_list[0]);
        }
        free(HostEnt->h_addr_list);
      }
      free(HostEnt);
    }
    Result = ARES_ENOMEM;

  }

  free(HostNamePT);
  free(PTRNamePT);

  return(Result);

}

/* ARES_PARSE_NMEXP FUNCTION - JONAS (25.06.2000) */

signed short int ares_parse_nmexp(const unsigned char *EncodedPT, const unsigned char *BufferPT, const unsigned long int BufferLen, char **DecodedPT, unsigned long int *EncodedLenPT) {

  signed long int Result = 0;
  unsigned long int Len = 0;
  signed long int indir = 0;
  char *Char1 = NULL;
  const unsigned char *Char2 = NULL;

  assert(EncodedPT != NULL);
  assert(BufferPT != NULL);
  assert(DecodedPT != NULL);
  assert(EncodedLenPT != NULL);

  Result = ares_parse_nmlen(EncodedPT, BufferPT, BufferLen);
  if (Result <= ERROR) { return(ARES_EBADNAME); }
  Len = Result;

  *DecodedPT = malloc(Len + 1);
  if (*DecodedPT == NULL) { return(ARES_ENOMEM); }
  Char1 = *DecodedPT;

  Char2 = EncodedPT;
  while (*Char2 != 0) {
    if ((*Char2 & INDIR_MASK) == INDIR_MASK) {
      if (indir == 0) {
        *EncodedLenPT = Char2 + 2 - EncodedPT;
        indir = 1;
      }
      Char2 = BufferPT + ((*Char2 & ~INDIR_MASK) << 8 | *(Char2 + 1));
    }
    else {
      Len = *Char2;
      Char2++;
      while (Len--) {
        if ((*Char2 == '.') || (*Char2 == '\\')) { *Char1++ = '\\'; }
        *Char1++ = *Char2;
        Char2++;
      }
      *Char1++ = '.';
    }
  }
  if (indir == 0) { *EncodedLenPT = Char2 + 1 - EncodedPT; }

  if (Char1 > *DecodedPT) { *(Char1 - 1) = 0; }

  return(ARES_SUCCESS);

}

/* ARES_NMLEN FUNCTION - JONAS (25.06.2000) */

signed short int ares_parse_nmlen(const unsigned char *EncodedPT, const unsigned char *BufferPT, const signed long int Len) {

  signed long int n = 0;
  signed long int offset = 0;
  signed long int indir = 0;

  assert(EncodedPT != NULL);
  assert(BufferPT != NULL);

  if (EncodedPT == BufferPT + Len) { return(ERROR); }

  while (*EncodedPT) {

    if ((*EncodedPT & INDIR_MASK) == INDIR_MASK) {

      if (EncodedPT + 1 >= BufferPT + Len) { return(ERROR); }
      offset = (*EncodedPT & ~INDIR_MASK) << 8 | *(EncodedPT + 1);
      if (offset > Len) { return(ERROR); }
      EncodedPT = BufferPT + offset;

      if (++indir > Len) { return(ERROR); }
    }
    else {

      offset = *EncodedPT;
      if ((EncodedPT + offset) >= (BufferPT + Len)) { return(ERROR); }
      EncodedPT++;
      while (offset--) {
        n += (*EncodedPT == '.' || *EncodedPT == '\\') ? 2 : 1;
        EncodedPT++;
      }
      n++;
    }
  }

  return((n) ? n - 1 : n);

}


