/*
 * ----------------------------------------------------------------
 * Night Light Asynchrounous DNS Resolver
 * ----------------------------------------------------------------
 * Copyright (C) 1997-2003 Jonas Kvinge <jonas@night-light.net>
 * All rights reserved.
 *
 * This code is a result of thousands of hours of work by
 * Jonas Kvinge <jonas@night-light.net>
 *
 * You may not create derivative works based on this code.
 *
 * Modified source code or binaries compiled from modified source
 * code distributed in any shape or form without the authors
 * permission is expressly forbidden.
 *
 * This code is provided without warranty of any kind.
 *
 * Under no circumstances and under no legal contract or otherwise
 * shall Jonas Kvinge be liable to you or any other person for any
 * damages, computer failure, malfunction or any other damages or
 * losses.
 *
 * Last modified by:
 * Jonas Kvinge (30.07.2001)
 *
 */

#define ARES_QUERY_C

#define NEED_SYS_TYPES_H 1		/* Extra types */
#define NEED_SYS_PARAM_H 1		/* Some systems need this */
#define NEED_LIMITS_H 0			/* Kernel limits */
#define NEED_STDARG_H 1			/* va_list, etc */
#define NEED_ERRNO_H 1			/* errno */
#define NEED_CTYPE_H 1			/* isdigit(), etc */
#define NEED_NETINET_IN_H 1		/* in_addr, sockaddr_in, etc */
#define NEED_ARPA_INET_H 1		/* inet_ntoa(), inet_aton(), etc */
#define NEED_STDIO_H 1			/* Standard C UNIX functions */
#define NEED_STDLIB_H 1			/* malloc(), exit(), atoi(), etc */
#define NEED_TIME_H 1			/* time(), etc */
#define NEED_SYSCTL_H 0			/* sysctl(), etc */
#define NEED_SYS_STAT_H 0		/* chmod(), mkdir(), etc */
#define NEED_SYS_UIO_H 0		/* iovec, etc */
#define NEED_FCNTL_H 1			/* open(), creat(), fcntl(), etc */
#define NEED_SYS_IOCTL_H 0		/* ioctl(), etc */
#define NEED_SYS_FILIO_H 0		/* Solaris need this for ioctl(), etc */
#define NEED_UNISTD_H 1			/* Unix standard functions */
#define NEED_STRING_H 1			/* C string functions */
#define NEED_SIGNAL_H 0			/* Signal functions */
#define NEED_SYS_SOCKET_H 1		/* Socket functions */
#define NEED_NETDB_H 1			/* Network database functions */
#define NEED_ARPA_NAMESER_H 1		/* Nameserver definitions */
#define NEED_GETUSERPW_HEADERS 0 	/* Functions to retrive system passwords */

#include "includes.h"

#include "dns.h"

#include "ares_io.h"
#include "ares_int.h"
#include "ares_query.h"
#include "ares_parse.h"

/* VARIABLES - JONAS (30.07.2001) */

struct Ares_Query_Struct *Ares_Query_Head = NULL;
struct Ares_Query_Struct *Ares_Query_Tail = NULL;

unsigned long int Ares_Query_ID = 0;
struct Ares_Query_Struct *Ares_Query_Array[ARES_QUERY_MAX+1] = { NULL };

extern struct Ares_Server_Struct *Ares_Server_Head;

/* ARES_QUERY_ADD FUNCTION - JONAS (30.07.2001) */

void ares_query_add(const unsigned long int Class, const unsigned long int Type, const char *NamePT, const unsigned long int Addr, const unsigned long int AddrLen, Ares_Callback Callback, void *ArgPT) {

  struct Ares_Query_Struct *Query_NEW = NULL;
  struct Ares_Query_Struct *Query = NULL;
  unsigned char *BufferPT = NULL;
  unsigned long int BufferLen = 0;
  signed long int Result = 0;

  assert(NamePT != NULL);

  ++Ares_Query_ID;
  if (Ares_Query_ID >= ARES_QUERY_MAX) { Ares_Query_ID = 1; }
  if (Ares_Query_Array[Ares_Query_ID] != NULL) {
    Callback(ArgPT, ARES_ENOMEM, NULL);
    return;
  }

  sysprint(BITMASK_DEBUG_ARES, "Ares: New query - ID: %ld, Class: %ld, Type: %ld, Name: %s.", Ares_Query_ID, Class, Type, NamePT);


  Result = ares_mkquery(NamePT, Class, Type, Ares_Query_ID, TRUE, &BufferPT, &BufferLen);
  if (Result != SUCCESS) {
    Callback(ArgPT, Result, NULL);
    return;
  }
  if ((BufferLen < HFIXEDSZ) || (BufferLen >= (1 << 16))) {
    Callback(ArgPT, ARES_EBADQUERY, NULL);
    return;
  }


  Query_NEW = malloc(sizeof(struct Ares_Query_Struct));
  if (Query_NEW == NULL) {
    Callback(ArgPT, ARES_ENOMEM, NULL);
    return;
  }
  memset(Query_NEW, 0, sizeof(struct Ares_Query_Struct));

  Query_NEW->Name = strdup(NamePT);
  if (Query_NEW->Name == NULL) {
    free(Query_NEW);
    Callback(ArgPT, ARES_ENOMEM, NULL);
    return;
  }

  Query_NEW->ID = Ares_Query_ID;
  Query_NEW->Class = Class;
  Query_NEW->Type = Type;
  Query_NEW->Addr = Addr;
  Query_NEW->AddrLen = AddrLen;
  Query_NEW->Tries = 0;
  Query_NEW->Timeout = NOW + ARES_TIMEOUT;

  Query_NEW->Buffer = malloc(BufferLen + 2);
  if (Query_NEW->Buffer == NULL) {
    free(Query_NEW->Name);
    free(Query_NEW);
    Callback(ArgPT, ARES_ENOMEM, NULL);
    return;
  }

  Query_NEW->Buffer[0] = ((BufferLen >> 8) & 0xff);
  Query_NEW->Buffer[1] = (BufferLen & 0xff);
  memcpy(Query_NEW->Buffer + 2, BufferPT, BufferLen);
  Query_NEW->BufferLen = BufferLen + 2;
  Query_NEW->Callback = Callback;
  Query_NEW->Arg = ArgPT;

  if ((BufferLen > PACKETSZ) || (ARES_ATCP == TRUE)) { Query_NEW->TCP = TRUE; }
  else { Query_NEW->TCP = FALSE; }

  if (Ares_Query_Head == NULL) {
    Ares_Query_Head = Query_NEW;
    Ares_Query_Tail = Query_NEW;
  }
  else {
    Query = Ares_Query_Tail;
    Query->Next = Query_NEW;
    Query_NEW->Prev = Query;
    Ares_Query_Tail = Query_NEW;
  }

  Ares_Query_Array[Query_NEW->ID] = Query_NEW;

  ares_query_server(Query_NEW);

  free(BufferPT);

}

/* ARES_MKQUERY FUNCTION - JONAS (25.06.2000) */

signed short int ares_mkquery(const char *NamePT, signed long int Class, signed long int Type, unsigned short ID, signed long int RD, unsigned char **BufferPT, signed long int *BufferLenPT) {

  unsigned long int Len = 0;
  const char *DumbPT = NULL;
  const char *WildNamePT = NULL;
  unsigned char *WildBufferPT = NULL;

  assert(NamePT != NULL);
  assert(BufferPT != NULL);
  assert(BufferLenPT != NULL);

  for (DumbPT = NamePT ; *DumbPT != 0 ; DumbPT++) {
    if ((*DumbPT == '\\') && (*(DumbPT + 1) != 0)) { DumbPT++; } /* ESCAPE THE NEXT CHAR */
    Len++;
  }
  if ((*NamePT != 0) && (*(DumbPT - 1) != '.')) { Len++; }

  *BufferLenPT = Len + HFIXEDSZ + QFIXEDSZ + 1;
  *BufferPT = malloc(*BufferLenPT);
  if (*BufferPT == NULL) { return(ARES_ENOMEM); }
  memset(*BufferPT, 0, *BufferLenPT);

  WildBufferPT = *BufferPT;

  DNS_HEADER_SET_QID(WildBufferPT, ID);
  DNS_HEADER_SET_OPCODE(WildBufferPT, QUERY);
  DNS_HEADER_SET_RD(WildBufferPT, RD);
  DNS_HEADER_SET_QDCOUNT(WildBufferPT, 1);

  if (strcmp(NamePT, ".") == FALSE) { NamePT++; }

  WildBufferPT += HFIXEDSZ;
  WildNamePT = NamePT;

  FOREVERLOOP {

   /* THE NAME STARTS WITH A DOT OR WE HAVE TWO DOTS */

   if (*WildNamePT == '.') {
     free(*BufferPT);
     *BufferPT = NULL;
     return(ARES_EBADNAME);
   }

    Len = 0;
    for (DumbPT = WildNamePT ; ((*DumbPT != 0) && (*DumbPT != '.')) ; DumbPT++) {
      if ((*DumbPT == '\\') && (*(DumbPT + 1) != 0)) { DumbPT++; } /* ESCAPE THE NEXT CHAR */
      Len++;
    }
    if (Len > MAXLABEL) {
      free(*BufferPT);
      *BufferPT = NULL;
      return(ARES_EBADNAME);
   }

    *WildBufferPT = Len;
    WildBufferPT++;

    for (; ((*WildNamePT != 0) && (*WildNamePT != '.')) ; WildNamePT++) {
      if ((*WildNamePT == '\\') && (*(WildNamePT + 1) != 0)) { WildNamePT++; } /* ESCAPE THE NEXT CHAR */
      *WildBufferPT = *WildNamePT;
      WildBufferPT++;
    }

    if (*WildNamePT == 0) { break; }
    WildNamePT++;

  }

  *WildBufferPT = 0;
  WildBufferPT++;

  DNS_QUESTION_SET_TYPE(WildBufferPT, Type);
  DNS_QUESTION_SET_CLASS(WildBufferPT, Class);

  return(SUCCESS);

}

/* ARES_QUERY_SERVER FUNCTION - JONAS (17.07.2001) */

void ares_query_server(struct Ares_Query_Struct *Query) {

  assert(Query != NULL);

  ++Query->Tries;

  if (Query->Server == NULL) { Query->Server = Ares_Server_Head; }
  else {
    Query->Server = Query->Server->Next;
    if (Query->Server == NULL) { Query->Server = Ares_Server_Head; }
  }

  Query->Server_Timeout = NOW + ARES_SERVER_TIMEOUT;

  ares_query_send(Query);

}

/* ARES_QUERY_SEND FUNCTION - JONAS (29.07.2001) */

void ares_query_send(struct Ares_Query_Struct *Query) {

  struct Ares_Server_Struct *Server = Query->Server;
  struct Ares_SendQ_Struct *SendQ_NEW = NULL;
  struct Ares_SendQ_Struct *SendQ = NULL;

  assert(Query->Server != NULL);

  SendQ_NEW = malloc(sizeof(struct Ares_SendQ_Struct));
  if (SendQ_NEW == NULL) {
    ares_query_callback(Query, ARES_ENOMEM);
    ares_query_rem(Query);
    return;
  }
  memset(SendQ_NEW, 0, sizeof(struct Ares_SendQ_Struct));

  if (Query->TCP == TRUE) {

    Server->ConnectTCP = TRUE;

    SendQ_NEW->Buffer = Query->Buffer;
    SendQ_NEW->Len = Query->BufferLen;

    if (Server->TCPSend_Head == NULL) { Server->TCPSend_Head = SendQ_NEW; }
    else {
      for (SendQ = Server->TCPSend_Head ; SendQ->Next != NULL ; SendQ = SendQ->Next);
      SendQ->Next = SendQ_NEW;
    }

  }
  else {

    SendQ_NEW->Buffer = Query->Buffer + 2;
    SendQ_NEW->Len = Query->BufferLen - 2;

    if (Server->UDPSend_Head == NULL) { Server->UDPSend_Head = SendQ_NEW; }
    else {
      for (SendQ = Server->UDPSend_Head ; SendQ->Next != NULL ; SendQ = SendQ->Next);
      SendQ->Next = SendQ_NEW;
    }

  }

}

/* ARES_QUERY_CALLBACK FUNCTION - JONAS (29.07.2001) */

void ares_query_callback(struct Ares_Query_Struct *Query, const unsigned short int Result) {

  assert(Query != NULL);

  Query->Callback(Query->Arg, Result, Query->HostEnt);

}

/* ARES_QUERY_REM FUNCTION - JONAS (29.07.2001) */

void ares_query_rem(struct Ares_Query_Struct *Query) {

  assert(Query != NULL);

  if (Query->Prev == NULL) { Ares_Query_Head = Query->Next; }
  else { Query->Prev->Next = Query->Next; }

  if (Query->Next == NULL) { Ares_Query_Tail = Query->Prev; }
  else { Query->Next->Prev = Query->Prev; }

  Ares_Query_Array[Query->ID] = NULL;

  ares_query_init(Query);

  free(Query);

}

/* ARES_QUERY_INIT FUNCTION - JONAS (25.06.2000) */

void ares_query_init(struct Ares_Query_Struct *Query) {

  char **CharPT = NULL;

  FREE(Query->Name);
  FREE(Query->Buffer);
  if (Query->HostEnt != NULL) {
    FREE(Query->HostEnt->h_name);
    for (CharPT = Query->HostEnt->h_aliases ; *CharPT != NULL ; CharPT++) {
      FREE(*CharPT);
    }
    FREE(Query->HostEnt->h_aliases);
    FREE(Query->HostEnt->h_addr_list[0]);
    FREE(Query->HostEnt->h_addr_list);
    FREE(Query->HostEnt);
  }

}
