/*
 * ----------------------------------------------------------------
 * Night Light Asynchrounous DNS Resolver
 * ----------------------------------------------------------------
 * Copyright (C) 1997-2003 Jonas Kvinge <jonas@night-light.net>
 * All rights reserved.
 *
 * This code is a result of thousands of hours of work by
 * Jonas Kvinge <jonas@night-light.net>
 *
 * You may not create derivative works based on this code.
 *
 * Modified source code or binaries compiled from modified source
 * code distributed in any shape or form without the authors
 * permission is expressly forbidden.
 *
 * This code is provided without warranty of any kind.
 *
 * Under no circumstances and under no legal contract or otherwise
 * shall Jonas Kvinge be liable to you or any other person for any
 * damages, computer failure, malfunction or any other damages or
 * losses.
 *
 * Last modified by:
 * Jonas Kvinge (25.06.2001)
 *
 */

#define ARES_UDP_C

#define NEED_SYS_TYPES_H 1		/* Extra types */
#define NEED_SYS_PARAM_H 1		/* Some systems need this */
#define NEED_LIMITS_H 0			/* Kernel limits */
#define NEED_STDARG_H 1			/* va_list, etc */
#define NEED_ERRNO_H 1			/* errno */
#define NEED_CTYPE_H 1			/* isdigit(), etc */
#define NEED_NETINET_IN_H 1		/* in_addr, sockaddr_in, etc */
#define NEED_ARPA_INET_H 1		/* inet_ntoa(), inet_aton(), etc */
#define NEED_STDIO_H 1			/* Standard C UNIX functions */
#define NEED_STDLIB_H 1			/* malloc(), exit(), atoi(), etc */
#define NEED_TIME_H 1			/* time(), etc */
#define NEED_SYSCTL_H 0			/* sysctl(), etc */
#define NEED_SYS_STAT_H 0		/* chmod(), mkdir(), etc */
#define NEED_SYS_UIO_H 0		/* iovec, etc */
#define NEED_FCNTL_H 1			/* open(), creat(), fcntl(), etc */
#define NEED_SYS_IOCTL_H 1		/* ioctl(), etc */
#define NEED_SYS_FILIO_H 1		/* Solaris need this for ioctl(), etc */
#define NEED_UNISTD_H 1			/* Unix standard functions */
#define NEED_STRING_H 1			/* C string functions */
#define NEED_SIGNAL_H 0			/* Signal functions */
#define NEED_SYS_SOCKET_H 1		/* Socket functions */
#define NEED_NETDB_H 0			/* Network database functions */
#define NEED_ARPA_NAMESER_H 1		/* Nameserver definitions */
#define NEED_GETUSERPW_HEADERS 0 	/* Functions to retrive system passwords */

#include "includes.h"

#include "dns.h"

#include "ares_io.h"
#include "ares_int.h"
#include "ares_udp.h"
#include "ares_parse.h"

/* VARIABLES - JONAS (05.07.2000) */

const char ARESUDP_VERSION[] = "0.5.3d";

/* ARES_UDP_CONNECT FUNCTION - JONAS (25.06.2000) */

void ares_udp_connect(struct Ares_Server_Struct *Server) {

  assert(Server != NULL);

  if (!Ares_IsConnectProc(Server->UDP_Flags)) {

    Ares_SetConnectProc(Server->UDP_Flags);
    Server->UDP_Time = NOW;

  }

  if (!Ares_IsSocket(Server->UDP_Flags)) {

    Server->UDP_Socket = socket(AF_INET, SOCK_DGRAM, 0);
    if (Server->UDP_Socket <= ERROR) {
      ares_udp_disconnect(Server, "Ares: Server %s UDP: Unable to create socket: [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }

    Ares_SetSocket(Server->UDP_Flags);

    sysprint(BITMASK_DEBUG_ARES, "Ares: Server %s UDP: Socket created.", Server->HostIPS);

  }

  if (!Ares_IsSockOPT(Server->UDP_Flags)) {

    signed long int Result = 0;
    unsigned long int Flags = 0;

#if defined(NBLOCK_SYSV)
    Flags = 1;
    Result = ioctl(Server->UDP_Socket, FIONBIO, &Flags);
    if (Result <= ERROR) {
      ares_udp_disconnect(Server, "Ares: Server %s UDP: Unable to set socket in non-blocking mode using ioctl(): [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
#else
    Result = fcntl(Server->UDP_Socket, F_GETFL, &Flags);
    if (Result <= ERROR) {
      ares_udp_disconnect(Server, "Ares: Server %s UDP: Unable to get socket flags using fcntl(): [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
#if defined(NBLOCK_BSD)
    Flags |= O_NDELAY;
#elif defined(NBLOCK_POSIX)
    Flags |= O_NONBLOCK;
#else
    #warning "This system does not support non-blocking sockets?"
    Flags |= O_NONBLOCK;
#endif
    Result = fcntl(Server->UDP_Socket, F_SETFL, &Flags);
    if (Result <= ERROR) {
      ares_udp_disconnect(Server, "Ares: Server %s UDP: Unable to set socket in non-blocking mode using fcntl(): [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
#endif

    Ares_SetSockOPT(Server->UDP_Flags);
    sysprint(BITMASK_DEBUG_ARES, "Ares: Server %s UDP: Socket options set.", Server->HostIPS);

  }

  if ((ARES_SOCKKEEPALIVEOPT == TRUE) && (!Ares_IsKeepAliveOPT(Server->UDP_Flags))) {

    signed long int Result = 0;
    unsigned long int OPT = 1;

    Result = setsockopt(Server->UDP_Socket, SOL_SOCKET, SO_KEEPALIVE, &OPT, sizeof(OPT));
    if (Result <= ERROR) { sysprint(BITMASK_DEBUG_ARES, "Ares: Server %s UDP: Unable to set Keep Alive option for socket: [%d] %s", Server->HostIPS, errno, strerror(errno)); }

    Ares_SetKeepAliveOPT(Server->UDP_Flags);

  }

  if (!Ares_IsBound(Server->UDP_Flags)) {

    signed long int Result = 0;
    struct sockaddr_in SockAddr = { 0 };

    memset(&SockAddr, 0, sizeof(SockAddr));

    SockAddr.sin_family = AF_INET;
    SockAddr.sin_addr.s_addr = INADDR_ANY;
    SockAddr.sin_port = 0;

    Result = bind(Server->UDP_Socket, (struct sockaddr *) &SockAddr, sizeof(SockAddr));
    if (Result <= ERROR) {
      ares_udp_disconnect(Server, "Ares: Server %s UDP: Unable to bind socket: [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
    Ares_SetBound(Server->UDP_Flags);

    sysprint(BITMASK_DEBUG_ARES, "Ares: Server %s UDP: Socket sucessfully bound.", Server->HostIPS);

  }

  if (!Ares_IsConnect(Server->UDP_Flags)) {

    signed long int Result = 0;
    struct sockaddr_in SockAddr = { 0 };

    memset(&SockAddr, 0, sizeof(SockAddr));

    SockAddr.sin_family = AF_INET;
    SockAddr.sin_addr = Server->InAddr;
    SockAddr.sin_port = Server->PortN;

    Result = connect(Server->UDP_Socket, (struct sockaddr *) &SockAddr, sizeof(SockAddr));
    if (Result <= ERROR) {
      if (errno == EINPROGRESS) {
        Ares_SetConnecting(Server->UDP_Flags);
        return;
      }
      else {
        ares_udp_disconnect(Server, "Ares: Server %s UDP: Unable to connect: [%d] %s", Server->HostIPS, errno, strerror(errno));
        return;
      }
    }
    Ares_SetConnected(Server->UDP_Flags);

    sysprint(BITMASK_DEBUG_ARES, "Ares: Server %s UDP: Trying to establish connection.", Server->HostIPS);

  }

  if (!Ares_IsConnected(Server->UDP_Flags)) {

    signed long int Result = 0;
    int OPT = 0;
    getsockopt_optlen_type OPTLen = sizeof(OPT);

    Result = getsockopt(Server->UDP_Socket, SOL_SOCKET, SO_ERROR, &OPT, &OPTLen);
    if (Result <= ERROR) {
      ares_udp_disconnect(Server, "Ares: Unable to get connection result for %s using UDP: [%d] %s", Server->HostIPS, errno, strerror(errno));
      return;
    }
    if (OPT != SUCCESS) {
      ares_udp_disconnect(Server, "Ares: Unable to connect to nameserver %s using UDP: [%d] %s", Server->HostIPS, OPT, strerror(OPT));
      return;
    }

    Ares_ClearConnecting(Server->UDP_Flags);
    Ares_SetConnected(Server->UDP_Flags);

    sysprint(BITMASK_DEBUG_ARES, "Ares: Server %s UDP: Socket sucessfully connected.", Server->HostIPS);

  }

  Ares_ClearConnectProc(Server->UDP_Flags);


}

/* ARES_UDP_DISCONNECT FUNCTION - JONAS (25.06.2000) */

void ares_udp_disconnect(struct Ares_Server_Struct *Server, const char *const LinePT, ...) {

  char Line[LINELEN+1] = "";
  va_list Args = { 0 };
  struct Ares_SendQ_Struct *SendQ = NULL;

  assert(Server != NULL);
  assert(LinePT != NULL);

  va_start(Args, LinePT);
  vsnprintf(Line, LINELEN+1, LinePT, Args);
  va_end(Args);

  if (Ares_IsSocket(Server->UDP_Flags)) {
    close(Server->UDP_Socket);
    Server->UDP_Socket = FD_NONE;
  }
  Server->UDP_Flags = 0;

  while (Server->UDPSend_Head != NULL) {
    SendQ = Server->UDPSend_Head;
    Server->UDPSend_Head = SendQ->Next;
    free(SendQ);
  }

  sysprint(BITMASK_DEBUG_ARES, "%s", Line);

}

/* ARES_UDP_READ FUNCTION - JONAS (25.06.2000) */

void ares_udp_read(struct Ares_Server_Struct *Server) {

  signed long int Result = 0;
  unsigned char Buffer[PACKETSZ] = "";

  assert(Server != NULL);
  assert(Ares_IsSocket(Server->UDP_Flags));
  assert(Server->UDP_Socket != FD_NONE);

  memset(&Buffer, 0, sizeof(Buffer));
  Result = recv(Server->UDP_Socket, Buffer, PACKETSZ, MSG_NOSIGNAL);
  if (Result <= ERROR) {
    if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR)) { return; }
    ares_udp_disconnect(Server, "Ares: Read error to nameserver %s UDP: [%d]: %s", Server->HostIPS, errno, strerror(errno));
    return;
  }
  if (Result == 0) {
   ares_udp_disconnect(Server, "Ares: EOF to nameserver %s UDP.", Server->HostIPS);
   return;
  }
  ares_parse(Buffer, Result, Server, FALSE);

}

/* ARES_UDP_WRITE FUNCTION - JONAS (25.06.2000) */

void ares_udp_write(struct Ares_Server_Struct *Server) {

  struct Ares_SendQ_Struct *SendQ = NULL;
  signed long int Result = 0;

  assert(Server != NULL);
  assert(Ares_IsSocket(Server->UDP_Flags));
  assert(Server->UDP_Socket != FD_NONE);

  SendQ = Server->UDPSend_Head;
  Result = send(Server->UDP_Socket, SendQ->Buffer, SendQ->Len, MSG_NOSIGNAL);
  if (Result <= ERROR) {
    if ((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR) || (errno == ENOMEM) || (errno == ENOBUFS)) { return; }
    ares_udp_disconnect(Server, "Ares: Write error to nameserver %s UDP: [%d]: %s", Server->HostIPS, errno, strerror(errno));
    return;
  }

  if (Result == SendQ->Len) {
    Server->UDPSend_Head = SendQ->Next;
    free(SendQ);
    return;
  }
  else {
    SendQ->Buffer += Result;
    SendQ->Len -= Result;
    return;
  }

}

