/*
 * ----------------------------------------------------------------
 * Night Light IRC Proxy Ident Listen System
 * ----------------------------------------------------------------
 * Copyright (C) 1997-2003 Jonas Kvinge <jonas@night-light.net>
 * All rights reserved.
 *
 * This code is a result of thousands of hours of work by
 * Jonas Kvinge <jonas@night-light.net>
 *
 * You may not create derivative works based on this code.
 *
 * Modified source code or binaries compiled from modified source
 * code distributed in any shape or form without the authors
 * permission is expressly forbidden.
 *
 * This code is provided without warranty of any kind.
 *
 * Under no circumstances and under no legal contract or otherwise
 * shall Jonas Kvinge be liable to you or any other person for any
 * damages, computer failure, malfunction or any other damages or
 * losses.
 *
 * Last modified by:
 * Jonas Kvinge (03.07.2003)
 *
 */

#define IDENT_LISTEN_C

#define NEED_SYS_TYPES_H 1		/* Extra types */
#define NEED_SYS_PARAM_H 1		/* Some systems need this */
#define NEED_LIMITS_H 0			/* Kernel limits */
#define NEED_STDARG_H 1			/* va_list, etc */
#define NEED_ERRNO_H 1			/* errno */
#define NEED_CTYPE_H 1			/* isdigit(), etc */
#define NEED_NETINET_IN_H 1		/* in_addr, sockaddr_in, etc */
#define NEED_ARPA_INET_H 1		/* inet_ntoa(), inet_aton(), etc */
#define NEED_STDIO_H 1			/* Standard C UNIX functions */
#define NEED_STDLIB_H 1			/* malloc(), exit(), atoi(), etc */
#define NEED_TIME_H 1			/* time(), etc */
#define NEED_SYSCTL_H 0			/* sysctl(), etc */
#define NEED_SYS_STAT_H 0		/* chmod(), mkdir(), etc */
#define NEED_SYS_UIO_H 0		/* iovec, etc */
#define NEED_FCNTL_H 1			/* open(), creat(), fcntl(), etc */
#define NEED_SYS_IOCTL_H 1		/* ioctl(), etc */
#define NEED_SYS_FILIO_H 1		/* Solaris need this for ioctl(), etc */
#define NEED_UNISTD_H 1			/* Unix standard functions */
#define NEED_STRING_H 1			/* C string functions */
#define NEED_SIGNAL_H 0			/* Signal functions */
#define NEED_SYS_SOCKET_H 1		/* Socket functions */
#define NEED_NETDB_H 1			/* Network database functions */
#define NEED_ARPA_NAMESER_H 0		/* Nameserver definitions */
#define NEED_GETUSERPW_HEADERS 0 	/* Functions to retrive system passwords */

#include "includes.h"

#if ARES
  #include "ares_io.h"
#endif

#include "ident_listen.h"
#include "ident_conn.h"

/* VARIABLES - JONAS (03.07.2003) */

struct IdentListen_Struct *IdentListen_Head = NULL;
struct IdentListen_Struct *IdentListen_Tail = NULL;
unsigned long int G_Ident_Listens = 0;

/* IDENT_LISTEN_ADD FUNCTION - JONAS (03.07.2003) */

struct IdentListen_Struct *ident_listen_add(const char *const HostPT, const unsigned long int Port) {

  struct IdentListen_Struct *IdentListenS = NULL;
  struct IdentListen_Struct *IdentListen_NEW = NULL;

  assert(HostPT != NULL);

  IdentListenS = ident_get(HostPT, Port);
  if (IdentListenS != NULL) {
    aerrno = AEEXISTS;
    return(IdentListenS);
  }

  IdentListen_NEW = malloc(sizeof(struct IdentListen_Struct));
  if (IdentListen_NEW == NULL) {
    aerrno = AEMALLOC;
    return(NULL);
  }

  memset(IdentListen_NEW, 0, sizeof(struct IdentListen_Struct));

  IdentListen_NEW->Host = strdup(HostPT);
  if (IdentListen_NEW->Host == NULL) {
    free(IdentListen_NEW);
    aerrno = AEMALLOC;
    return(NULL);
  }

  IdentListen_NEW->PortH = Port;
  IdentListen_NEW->PortN = htons(Port);

  if (IdentListen_Head == NULL) {
    IdentListen_Head = IdentListen_NEW;
    IdentListen_Tail = IdentListen_NEW;
  }
  else {
    IdentListenS = IdentListen_Tail;
    IdentListenS->Next = IdentListen_NEW;
    IdentListen_NEW->Prev = IdentListenS;
    IdentListen_Tail = IdentListen_NEW;
  }

  G_Ident_Listens++;

  aerrno = AESUCCESS;
  return(IdentListen_NEW);

}

/* IDENT_LISTEN_REM FUNCTION - JONAS (03.07.2003) */

void ident_listen_rem(struct IdentListen_Struct *IdentListenS) {

  assert(IdentListenS != NULL);

  if (IdentListenS->Prev == NULL) { IdentListen_Head = IdentListenS->Next; }
  else { IdentListenS->Prev->Next = IdentListenS->Next; }

  if (IdentListenS->Next == NULL) { IdentListen_Tail = IdentListenS->Prev; }
  else { IdentListenS->Next->Prev = IdentListenS->Prev; }

  ident_listen_init(IdentListenS);

  free(IdentListenS->Host);
  free(IdentListenS);

  G_Ident_Listens--;

}

/* IDENT_LISTEN_INIT FUNCTION - JONAS (03.07.2003) */

void ident_listen_init(struct IdentListen_Struct *IdentListenS) {

  assert(IdentListenS != NULL);

  IdentListenS->Flags = 0;
  FREE(IdentListenS->HostIPS);
  IdentListenS->HostIPH = 0;
  IdentListenS->HostIPN = 0;
  IdentListenS->PortH = 0;
  IdentListenS->PortN = 0;
  IdentListenS->FD = FD_NONE;

}

/* IDENT_LISTEN_GET - JONAS (03.07.2003) */

struct IdentListen_Struct *ident_get(const char *const HostPT, const unsigned long int Port) {

  struct IdentListen_Struct *IdentListenS = NULL;

  assert(HostPT != NULL);

  for (IdentListenS = IdentListen_Head ; IdentListenS != NULL ; IdentListenS = IdentListenS->Next) {
    if ((strcasecmp(IdentListenS->Host, HostPT) == FALSE) && (IdentListenS->PortH == Port)) {
      aerrno = AESUCCESS;
      return(IdentListenS);
    }
  }
  aerrno = AENOMATCH;
  return(NULL);

}

/* IDENT_LISTEN_START - JONAS (03.07.2003) */

void ident_listen_start(struct IdentListen_Struct *IdentListenS) {

  signed long int Result = 0;
  signed long int BindResult = 0;

  struct sockaddr_in SockAddr = { 0 };

#if !ARES
  struct hostent *HostEnt = NULL;
#endif

  unsigned long int Flags = 0;

  assert(IdentListenS != NULL);

  IdentListenS->Time = NOW;
  IdentListenS->Tries++;

  if (!IdentListen_IsResolved(IdentListenS)) {

    struct in_addr INAddr;

    memset(&INAddr, 0, sizeof(INAddr));
    Result = inet_aton(IdentListenS->Host, &INAddr);
    if (Result == 0) {
      IdentListen_SetResolving(IdentListenS);
      #if ARES
        ares_gethostbyname(IdentListenS->Host, AF_INET, &ident_listen_hosttoip, IdentListenS);
        return;
      #else
        HostEnt = gethostbyname(IdentListenS->Host);
        ident_conf_hosttoip(IdentListenS, errno, HostEnt);
      #endif
    }
    else {
      IdentListen_SetResolved(IdentListenS);
      IdentListenS->HostIPN = INAddr.s_addr;
      IdentListenS->HostIPH = ntohl(INAddr.s_addr);
      IdentListenS->HostIPS = strrealloc(IdentListenS->HostIPS, IdentListenS->Host);
      if (aerrno != AESUCCESS) { ident_listen_init(IdentListenS); return; }
    }

  }

  /* CREATE SOCKET */

  Result = socket(AF_INET, SOCK_STREAM, 0);
  if (Result <= ERROR) {
    sysprint(BITMASK_MAIN, "Ident daemon %s(%s):%ld: Unable to create socket: [%d] %s", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH, errno, strerror(errno));
    ident_listen_stop(IdentListenS);
    return;
  }
  IdentListenS->FD = Result;

  IdentListen_SetSocket(IdentListenS);

  /* SET SOCKET IN NON-BLOCKING MODE */

#if defined(NBLOCK_SYSV)
  Flags = 1;
  Result = ioctl(IdentListenS->FD, FIONBIO, &Flags);
  if (Result <= ERROR) {
    sysprint(BITMASK_MAIN, "Ident daemon %s(%s):%ld: Unable to set socket in non-blocking mode using ioctl(): [%d] %s", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH, errno, strerror(errno));
    ident_listen_stop(IdentListenS);
    return;
  }
#else
  Result = fcntl(IdentListenS->FD, F_GETFL, &Flags);
  if (Result <= ERROR) {
    sysprint(BITMASK_MAIN, "Ident daemon %s(%s):%ld: Unable to get socket flags using fcntl(): [%d] %s", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH, errno, strerror(errno));
    ident_listen_stop(IdentListenS);
    return;
  }
#if defined(NBLOCK_BSD)
    Flags |= O_NDELAY;
#elif defined(NBLOCK_POSIX)
    Flags |= O_NONBLOCK;
#else
    #warning "This system does not support non-blocking sockets?"
    Flags |= O_NONBLOCK;
#endif
  Result = fcntl(IdentListenS->FD, F_SETFL, Flags);
  if (Result <= ERROR) {
    sysprint(BITMASK_MAIN, "Ident daemon %s(%s):%ld: Unable to set socket in non-blocking mode using fcntl(): [%d] %s", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH, errno, strerror(errno));
    ident_listen_stop(IdentListenS);
    return;
  }
#endif


  if (IDENTLISTEN_SOCKKEEPALIVE == TRUE) {

    unsigned long int OPT = 1;

    Result = setsockopt(IdentListenS->FD, SOL_SOCKET, SO_KEEPALIVE, &OPT, sizeof(OPT));
    if (Result <= ERROR) {
      sysprint(BITMASK_MAIN, "Ident daemon %s(%s):%ld: Unable to set Keep Alive option for socket: [%d] %s", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH, errno, strerror(errno));
      ident_listen_stop(IdentListenS);
      return;
    }

  }

  if (IDENTLISTEN_SOCKREUSEADDR == TRUE) {

    unsigned long int OPT = 1;

    Result = setsockopt(IdentListenS->FD, SOL_SOCKET, SO_REUSEADDR, &OPT, sizeof(OPT));
    if (Result <= ERROR) {
      sysprint(BITMASK_MAIN, "Ident daemon %s(%s):%ld: Unable to set Reuse Address option for socket: [%d] %s", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH, errno, strerror(errno));
      ident_listen_stop(IdentListenS);
      return;
    }

  }

  /* BIND */

#if !WIN32
  if (IdentListenS->PortH < 1024) { sysseteuid(0); }
#endif

  memset(&SockAddr, 0, sizeof(SockAddr));
  SockAddr.sin_family = AF_INET;
  SockAddr.sin_addr.s_addr = IdentListenS->HostIPN;
  SockAddr.sin_port = IdentListenS->PortN;
  BindResult = bind(IdentListenS->FD, (struct sockaddr *) &SockAddr, sizeof(SockAddr));

#if !WIN32
  if (IdentListenS->PortH < 1024) { sysseteuidnormal(); }
#endif

  if (BindResult <= ERROR) {
    sysprint(BITMASK_MAIN, "Ident daemon %s(%s):%ld: Unable to bind socket to address: [%d] %s", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH, errno, strerror(errno));
    ident_listen_stop(IdentListenS);
    return;
  }

  /* IDENT */  

  Result = listen(IdentListenS->FD, SOMAXCONN);
  if (Result <= ERROR) {
    sysprint(BITMASK_MAIN, "Ident daemon %s(%s):%ld: Unable to listen on address: [%d] %s", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH, errno, strerror(errno));
    ident_listen_stop(IdentListenS);
    return;
  }
  else {
    sysprint(BITMASK_DEBUG_IDENT, "Ident daemon %s(%s):%ld started.", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH);
    IdentListen_SetListening(IdentListenS);
    return;
  }

}

/* IDENT_LISTEN_HOSTTOIP FUNCTION - JONAS (03.07.2003) */

void ident_listen_hosttoip(void *ArgPT, unsigned short int ErrNo, struct hostent *HostEnt) {

  struct IdentListen_Struct *IdentListenS = ArgPT;
  char *HostIPPT = NULL;
  struct in_addr INAddr;

  assert(IdentListenS != NULL);

  if (HostEnt == NULL) {
    sysprint(BITMASK_MAIN, "Ident daemon %s():%ld: Unable to resolve host %s to IP-address: [%d] %s", IdentListenS->Host, IdentListenS->PortH, IdentListenS->Host, ErrNo, ares_strerror(ErrNo));
    ident_listen_init(IdentListenS);
    return;
  }
  memcpy(&IdentListenS->HostIPN, HostEnt->h_addr, HostEnt->h_length);
  memset(&INAddr, 0, sizeof(INAddr));
  INAddr.s_addr = IdentListenS->HostIPN;
  HostIPPT = inet_ntoa(INAddr);
  IdentListenS->HostIPS = strrealloc(IdentListenS->HostIPS, HostIPPT);
  IdentListenS->HostIPH = ntohl(IdentListenS->HostIPN);

  IdentListen_ClearResolving(IdentListenS);
  IdentListen_SetResolved(IdentListenS);

  #if ARES
    ident_listen_start(IdentListenS);
  #endif

}

/* IDENT_LISTEN_STOP - JONAS (03.07.2003) */

void ident_listen_stop(struct IdentListen_Struct *IdentListenS) {

  assert(IdentListenS != NULL);

  if (IdentListen_IsResolving(IdentListenS)) {
    ares_cancelquery(NULL, IdentListenS);
    return;
  }
  if (IdentListen_IsListening(IdentListenS)) { sysprint(BITMASK_DEBUG_IDENT, "Ident daemon %s(%s):%ld stopped.", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH); }
  if (IdentListen_IsSocket(IdentListenS)) {
    close(IdentListenS->FD);
    IdentListenS->FD = FD_NONE;
  }
  ident_listen_init(IdentListenS);

}

/* IDENT_LISTEN_FDS - JONAS (03.07.2003) */

void ident_listen_fds(fd_set *ReadFDS, fd_set *WriteFDS, unsigned long int *FDS) {

  struct IdentListen_Struct *IdentListenS = NULL;

  for (IdentListenS = IdentListen_Head ; IdentListenS != NULL ;) {
    if (IdentListen_IsRemove(IdentListenS)) {
      struct IdentListen_Struct *IdentListenS_DEL = NULL;
      if (IdentListen_IsResolving(IdentListenS)) { continue; }
      ident_listen_stop(IdentListenS);
      IdentListenS_DEL = IdentListenS;
      IdentListenS = IdentListenS->Next;
      ident_listen_rem(IdentListenS_DEL);
      continue;
    }
    if (IdentListen_IsListening(IdentListenS)) { FD_SET(IdentListenS->FD, ReadFDS); }
    else {
      if (!IdentListen_IsResolving(IdentListenS)) {
        unsigned long int Duration = (NOW - IdentListenS->Time);
        if (Duration >= IDENTLISTEN_INTERVAL) { ident_listen_start(IdentListenS); }
      }
    }
    IdentListenS = IdentListenS->Next;
  }

}

/* IDENT_LISTEN_IO - JONAS (03.07.2003) */

void ident_listen_io(fd_set *ReadFDS, fd_set *WriteFDS, unsigned long int *FDS) {

  struct IdentListen_Struct *IdentListenS = NULL;
  signed long int Result = 0;
  unsigned long int FD = 0;
  struct in_addr INAddr;
  struct sockaddr_in SockAddr = { 0 };
  accept_addrlen_type SockAddrLen = sizeof(SockAddr);
  char *HostIPSPT = NULL;
  unsigned long int HostIPH = 0;
  unsigned long long int HostIPN = 0;
  unsigned long int PortH;
  unsigned long int PortN;
  unsigned short int Count = 0;

  for (IdentListenS = IdentListen_Head ; IdentListenS != NULL ; IdentListenS = IdentListenS->Next) {

    assert(*FDS >= 0);
    if (*FDS <= 0) { return; }

    if (!IdentListen_IsListening(IdentListenS)) { continue; }

    if (!FD_ISSET(IdentListenS->FD, ReadFDS)) { continue; }
    *FDS = *FDS - 1;

    for (Count = 0 ;; ++Count) {
      Result = accept(IdentListenS->FD, (struct sockaddr *) &SockAddr, &SockAddrLen);
      if (Result <= ERROR) {
        if (Count == 0) { sysprint(BITMASK_MAIN, "Ident daemon %s(%s):%ld: Unable to accept new incoming connection: [%d] %s", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH, errno, strerror(errno)); }
        break;
      }

      FD = Result;
      memset(&INAddr, 0, sizeof(INAddr));
      memcpy(&HostIPN, &SockAddr.sin_addr, sizeof(SockAddr.sin_addr));
      memcpy(&INAddr.s_addr, &SockAddr.sin_addr, sizeof(SockAddr.sin_addr));
      HostIPSPT = inet_ntoa(INAddr);
      if (HostIPSPT == NULL) {
        close(FD);
        continue;
      }
      HostIPH = ntohl(HostIPN);
      PortN = SockAddr.sin_port;
      PortH = ntohs(SockAddr.sin_port);
      sysprint(BITMASK_DEBUG_IDENT, "Ident daemon %s(%s):%ld: Successfully accepted new incoming connection from \"%s\".", IdentListenS->Host, IdentListenS->HostIPS, IdentListenS->PortH, HostIPSPT);
      ident_conn_add(HostIPSPT, HostIPH, HostIPN, PortH, PortN, FD);
    }
  }
}

/* IDENT_LISTEN_CLOSEALL FUNCTION - JONAS (03.07.2003) */

unsigned short int ident_listen_closeall(const char *const MessagePT, ...) {

  struct IdentListen_Struct *IdentListenS = NULL;
  struct IdentListen_Struct *IdentListenS_DEL = NULL;
  unsigned short int Count = 0;

  for (IdentListenS = IdentListen_Head ; IdentListenS != NULL ;) {
    if (IdentListen_IsResolving(IdentListenS)) {
      ares_cancelquery(NULL, IdentListenS);
      ++Count;
      IdentListen_SetRemove(IdentListenS);
      continue;
    }
    ident_listen_stop(IdentListenS);
    IdentListenS_DEL = IdentListenS;
    IdentListenS = IdentListenS->Next;
    ident_listen_rem(IdentListenS_DEL);
  }

  return(Count);

}

