/*
 * ----------------------------------------------------------------
 * Night Light IRC Proxy Listen I/O System
 * ----------------------------------------------------------------
 * Copyright (C) 1997-2003 Jonas Kvinge <jonas@night-light.net>
 * All rights reserved.
 *
 * This code is a result of thousands of hours of work by
 * Jonas Kvinge <jonas@night-light.net>
 *
 * You may not create derivative works based on this code.
 *
 * Modified source code or binaries compiled from modified source
 * code distributed in any shape or form without the authors
 * permission is expressly forbidden.
 *
 * This code is provided without warranty of any kind.
 *
 * Under no circumstances and under no legal contract or otherwise
 * shall Jonas Kvinge be liable to you or any other person for any
 * damages, computer failure, malfunction or any other damages or
 * losses.
 *
 * Last modified by:
 * Jonas Kvinge (22.07.2001)
 *
 */

#define LISTEN_IO_C

#define NEED_SYS_TYPES_H 1		/* Extra types */
#define NEED_SYS_PARAM_H 1		/* Some systems need this */
#define NEED_LIMITS_H 0			/* Kernel limits */
#define NEED_STDARG_H 1			/* va_list, etc */
#define NEED_ERRNO_H 1			/* errno */
#define NEED_CTYPE_H 1			/* isdigit(), etc */
#define NEED_NETINET_IN_H 1		/* in_addr, sockaddr_in, etc */
#define NEED_ARPA_INET_H 1		/* inet_ntoa(), inet_aton(), etc */
#define NEED_STDIO_H 1			/* Standard C UNIX functions */
#define NEED_STDLIB_H 1			/* malloc(), exit(), atoi(), etc */
#define NEED_TIME_H 1			/* time(), etc */
#define NEED_SYSCTL_H 0			/* sysctl(), etc */
#define NEED_SYS_STAT_H 0		/* chmod(), mkdir(), etc */
#define NEED_SYS_UIO_H 0		/* iovec, etc */
#define NEED_FCNTL_H 1			/* open(), creat(), fcntl(), etc */
#define NEED_SYS_IOCTL_H 1		/* ioctl(), etc */
#define NEED_SYS_FILIO_H 1		/* Solaris need this for ioctl(), etc */
#define NEED_UNISTD_H 1			/* Unix standard functions */
#define NEED_STRING_H 1			/* C string functions */
#define NEED_SIGNAL_H 0			/* Signal functions */
#define NEED_SYS_SOCKET_H 1		/* Socket functions */
#define NEED_NETDB_H 1			/* Network database functions */
#define NEED_ARPA_NAMESER_H 0		/* Nameserver definitions */
#define NEED_GETUSERPW_HEADERS 0 	/* Functions to retrive system passwords */

#include "includes.h"

#if ARES
  #include "ares_io.h"
#endif

#include "listen_io.h"
#include "client_io.h"

/* VARIABLES - JONAS (06.10.2000) */

struct Listen_Struct *Listen_Head = NULL;
struct Listen_Struct *Listen_Tail = NULL;
unsigned long int G_Listens = 0;

extern struct Client_Struct *Client_Head;

/* LISTEN_ADD FUNCTION - JONAS (22.07.2001) */

struct Listen_Struct *listen_add(const char *const HostPT, const unsigned long int Port) {

  struct Listen_Struct *ListenS = NULL;
  struct Listen_Struct *Listen_NEW = NULL;

  assert(HostPT != NULL);

  ListenS = listen_get(HostPT, Port);
  if (ListenS != NULL) {
    aerrno = AEEXISTS;
    return(ListenS);
  }

  Listen_NEW = malloc(sizeof(struct Listen_Struct));
  if (Listen_NEW == NULL) {
    aerrno = AEMALLOC;
    return(NULL);
  }

  memset(Listen_NEW, 0, sizeof(struct Listen_Struct));

  Listen_NEW->Host = strdup(HostPT);
  if (Listen_NEW->Host == NULL) {
    free(Listen_NEW);
    aerrno = AEMALLOC;
    return(NULL);
  }

  Listen_NEW->PortH = Port;
  Listen_NEW->PortN = htons(Port);

  if (Listen_Head == NULL) {
    Listen_Head = Listen_NEW;
    Listen_Tail = Listen_NEW;
  }
  else {
    ListenS = Listen_Tail;
    ListenS->Next = Listen_NEW;
    Listen_NEW->Prev = ListenS;
    Listen_Tail = Listen_NEW;
  }

  G_Listens++;

  aerrno = AESUCCESS;
  return(Listen_NEW);

}

/* LISTEN_REM FUNCTION - JONAS (01.08.2001) */

void listen_rem(struct Listen_Struct *ListenS) {

  struct Client_Struct *ClientS = NULL;

  assert(ListenS != NULL);

  if (ListenS->Prev == NULL) { Listen_Head = ListenS->Next; }
  else { ListenS->Prev->Next = ListenS->Next; }

  if (ListenS->Next == NULL) { Listen_Tail = ListenS->Prev; }
  else { ListenS->Next->Prev = ListenS->Prev; }

  listen_init(ListenS);

  free(ListenS->Host);
  free(ListenS);

  G_Listens--;

  for (ClientS = Client_Head ; ClientS != NULL ; ClientS = ClientS->Next) {
    if (ClientS->ListenS != ListenS) { continue; }
    ClientS->ListenS = NULL;
  }

}

/* LISTEN_INIT FUNCTION - JONAS (01.08.2001) */

void listen_init(struct Listen_Struct *ListenS) {

  assert(ListenS != NULL);

  ListenS->Flags = 0;
  FREE(ListenS->HostIPS);
  ListenS->HostIPH = 0;
  ListenS->HostIPN = 0;
  ListenS->PortH = 0;
  ListenS->PortN = 0;
  ListenS->FD = FD_NONE;

}

/* LISTEN_GET - JONAS (06.10.2000) */

struct Listen_Struct *listen_get(const char *const HostPT, const unsigned long int Port) {

  struct Listen_Struct *ListenS = NULL;

  assert(HostPT != NULL);

  for (ListenS = Listen_Head ; ListenS != NULL ; ListenS = ListenS->Next) {
    if ((strcasecmp(ListenS->Host, HostPT) == FALSE) && (ListenS->PortH == Port)) {
      aerrno = AESUCCESS;
      return(ListenS);
    }
  }
  aerrno = AENOMATCH;
  return(NULL);

}

/* LISTEN_START - JONAS (22.07.2001) */

void listen_start(struct Listen_Struct *ListenS) {

  signed long int Result = 0;
  struct sockaddr_in SockAddr = { 0 };
  unsigned long int Flags = 0;

  assert(ListenS != NULL);

  ListenS->Time = NOW;
  ++ListenS->Tries;

  if (!Listen_IsResolved(ListenS)) {

    struct in_addr INAddr;

    memset(&INAddr, 0, sizeof(INAddr));
    Result = inet_aton(ListenS->Host, &INAddr);
    if (Result == 0) {
      Listen_SetResolving(ListenS);
      #if ARES
        ares_gethostbyname(ListenS->Host, AF_INET, &listen_hosttoip, ListenS);
        return;
      #else
        memset(&HostEnt, 0, sizeof(HostEnt));
        HostEnt = gethostbyname(ListenS->Host);
        listen_conf_hosttoip(ListenS, errno, HostEnt);
      #endif
    }
    else {
      Listen_SetResolved(ListenS);
      ListenS->HostIPN = INAddr.s_addr;
      ListenS->HostIPH = ntohl(INAddr.s_addr);
      ListenS->HostIPS = strrealloc(ListenS->HostIPS, ListenS->Host);
      if (aerrno != AESUCCESS) { listen_init(ListenS); return; }
    }

  }

  /* CREATE SOCKET */

  Result = socket(AF_INET, SOCK_STREAM, 0);
  if (Result <= ERROR) {
    sysprint(BITMASK_MAIN, "Listener %s(%s):%ld: Unable to create socket: [%d] %s", ListenS->Host, ListenS->HostIPS, ListenS->PortH, errno, strerror(errno));
    listen_stop(ListenS);
    return;
  }
  ListenS->FD = Result;
  Listen_SetSocket(ListenS);

  /* SET SOCKET IN NON-BLOCKING MODE */

#if defined(NBLOCK_SYSV)
  Flags = 1;
  Result = ioctl(ListenS->FD, FIONBIO, &Flags);
  if (Result <= ERROR) {
    sysprint(BITMASK_MAIN, "Listener %s(%s):%ld: Unable to set socket in non-blocking mode using ioctl(): [%d] %s", ListenS->Host, ListenS->HostIPS, ListenS->PortH, errno, strerror(errno));
    listen_stop(ListenS);
    return;
  }
#else
  Result = fcntl(ListenS->FD, F_GETFL, &Flags);
  if (Result <= ERROR) {
    sysprint(BITMASK_MAIN, "Listener %s(%s):%ld: Unable to get socket flags using fcntl(): [%d] %s", ListenS->Host, ListenS->HostIPS, ListenS->PortH, errno, strerror(errno));
    listen_stop(ListenS);
    return;
  }
#if defined(NBLOCK_BSD)
    Flags |= O_NDELAY;
#elif defined(NBLOCK_POSIX)
    Flags |= O_NONBLOCK;
#else
    #warning "This system does not support non-blocking sockets?"
    Flags |= O_NONBLOCK;
#endif
  Result = fcntl(ListenS->FD, F_SETFL, Flags);
  if (Result <= ERROR) {
    sysprint(BITMASK_MAIN, "Listener %s(%s):%ld: Unable to set socket in non-blocking mode using fcntl(): [%d] %s", ListenS->Host, ListenS->HostIPS, ListenS->PortH, errno, strerror(errno));
    listen_stop(ListenS);
    return;
  }
#endif

  if (LISTEN_SOCKKEEPALIVE == TRUE) {

    unsigned long int OPT = 1;

    Result = setsockopt(ListenS->FD, SOL_SOCKET, SO_KEEPALIVE, &OPT, sizeof(OPT));
    if (Result <= ERROR) {
      sysprint(BITMASK_MAIN, "Listener %s(%s):%ld: Unable to set Keep Alive option for socket: [%d] %s", ListenS->Host, ListenS->HostIPS, ListenS->PortH, errno, strerror(errno));
      listen_stop(ListenS);
      return;
    }

  }

  if (LISTEN_SOCKREUSEADDR == TRUE) {

    unsigned long int OPT = 1;

    Result = setsockopt(ListenS->FD, SOL_SOCKET, SO_REUSEADDR, &OPT, sizeof(OPT));
    if (Result <= ERROR) {
      sysprint(BITMASK_MAIN, "Listener %s(%s):%ld: Unable to set Reuse Address option for socket: [%d] %s", ListenS->Host, ListenS->HostIPS, ListenS->PortH, errno, strerror(errno));
      listen_stop(ListenS);
      return;
    }

  }

  /* BIND */

  memset(&SockAddr, 0, sizeof(SockAddr));
  SockAddr.sin_family = AF_INET;
  SockAddr.sin_addr.s_addr = ListenS->HostIPN;
  SockAddr.sin_port = ListenS->PortN;
#if !WIN32
  if (ListenS->PortH < 1024) { sysseteuid(0); }
#endif
  Result = bind(ListenS->FD, (struct sockaddr *) &SockAddr, sizeof(SockAddr));
#if !WIN32
  if (ListenS->PortH < 1024) { sysseteuidnormal(); }
#endif
  if (Result <= ERROR) {
    sysprint(BITMASK_MAIN, "Listener %s(%s):%ld: Unable to bind socket to address: [%d] %s", ListenS->Host, ListenS->HostIPS, ListenS->PortH, errno, strerror(errno));
    listen_stop(ListenS);
    return;
  }

  /* LISTEN */  

  Result = listen(ListenS->FD, SOMAXCONN);
  if (Result <= ERROR) {
    sysprint(BITMASK_MAIN, "Listener %s(%s):%ld: Unable to listen on address: [%d] %s", ListenS->Host, ListenS->HostIPS, ListenS->PortH, errno, strerror(errno));
    listen_stop(ListenS);
    return;
  }
  else {
    sysprint(BITMASK_MAIN, "Listening on %s(%s):%ld for incoming connections.", ListenS->Host, ListenS->HostIPS, ListenS->PortH);
    Listen_SetListening(ListenS);
    return;
  }

}

/* LISTEN_HOSTTOIP FUNCTION - JONAS (01.03.2000) */

void listen_hosttoip(void *ArgPT, unsigned short int ErrNo, struct hostent *HostEnt) {

  struct Listen_Struct *ListenS = ArgPT;
  char *HostIPPT = NULL;
  struct in_addr INAddr;

  assert(ListenS != NULL);

  if (HostEnt == NULL) {
    sysprint(BITMASK_MAIN, "Listener %s:%ld: Unable to resolve host %s to IP-address: [%d] %s", ListenS->Host, ListenS->PortH, ListenS->Host, ErrNo, ares_strerror(ErrNo));
    listen_init(ListenS);
    return;
  }
  memcpy(&ListenS->HostIPN, HostEnt->h_addr, HostEnt->h_length);
  memset(&INAddr, 0, sizeof(INAddr));
  INAddr.s_addr = ListenS->HostIPN;
  HostIPPT = inet_ntoa(INAddr);
  ListenS->HostIPS = strrealloc(ListenS->HostIPS, HostIPPT);
  ListenS->HostIPH = ntohl(ListenS->HostIPN);

  Listen_ClearResolving(ListenS);
  Listen_SetResolved(ListenS);

  #if ARES
    listen_start(ListenS);
  #endif

}

/* LISTEN_STOP - JONAS (05.10.2000) */

void listen_stop(struct Listen_Struct *ListenS) {

  assert(ListenS != NULL);

  if (Listen_IsResolving(ListenS)) {
    ares_cancelquery(NULL, ListenS);
    return;
  }

  if (Listen_IsListening(ListenS)) { sysprint(BITMASK_MAIN, "Listener %s(%s):%ld has stopped.", ListenS->Host, ListenS->HostIPS, ListenS->PortH); }
  if (Listen_IsSocket(ListenS)) {
    close(ListenS->FD);
    ListenS->FD = FD_NONE;
  }
  listen_init(ListenS);

}

/* LISTEN_FDS - JONAS (22.07.2001) */

void listen_fds(fd_set *ReadFDS, fd_set *WriteFDS, unsigned long int *FDS) {

  struct Listen_Struct *ListenS = NULL;

  for (ListenS = Listen_Head ; ListenS != NULL ;) {
    if (Listen_IsRemove(ListenS)) {
      struct Listen_Struct *ListenS_DEL = NULL;
      if (Listen_IsResolving(ListenS)) { continue; }
      listen_stop(ListenS);
      ListenS_DEL = ListenS;
      ListenS = ListenS->Next;
      listen_rem(ListenS_DEL);
      continue;
    }
    if (Listen_IsListening(ListenS)) { FD_SET(ListenS->FD, ReadFDS); }
    else {
      if (!Listen_IsResolving(ListenS)) {
        time_t Duration = (NOW - ListenS->Time);
        if (Duration >= LISTEN_INTERVAL) { listen_start(ListenS); }
      }
    }
    ListenS = ListenS->Next;
  }

}

/* LISTEN_IO - JONAS (06.10.2000) */

void listen_io(fd_set *ReadFDS, fd_set *WriteFDS, unsigned long int *FDS) {

  struct Listen_Struct *ListenS = NULL;

  for (ListenS = Listen_Head ; ListenS != NULL ; ListenS = ListenS->Next) {

    unsigned short int Count = 0;

    assert(*FDS >= 0);
    if (*FDS <= 0) { return; }

    if (!Listen_IsListening(ListenS)) { continue; }
    if (!FD_ISSET(ListenS->FD, ReadFDS)) { continue; }
    *FDS = *FDS - 1;
    assert(*FDS >= 0);

    for (Count = 0 ;; ++Count) {

      signed long int Result = 0;
      struct sockaddr_in SockAddr = { 0 };
      accept_addrlen_type SockAddrLen = sizeof(SockAddr);
      unsigned long int FD = 0;
      struct in_addr INAddr;
      unsigned long int HostIPH = 0;
      unsigned long long int HostIPN = 0;
      unsigned long int PortH = 0;
      unsigned long int PortN = 0;
      char *HostIPSPT = NULL;

      Result = accept(ListenS->FD, (struct sockaddr *) &SockAddr, &SockAddrLen);
      if (Result <= ERROR) {
        if (Count == 0) { sysprint(BITMASK_MAIN, "Unable to accept incoming connection on address \"%s(%s):%ld\": [%d] %s", ListenS->Host, ListenS->HostIPS, ListenS->PortH, errno, strerror(errno)); }
        return;
      }

      FD = Result;
      HostIPN = SockAddr.sin_addr.s_addr;
      HostIPH = ntohl(HostIPN);
      PortN = SockAddr.sin_port;
      PortH = ntohs(SockAddr.sin_port);

      memset(&INAddr, 0, sizeof(INAddr));
      INAddr.s_addr = HostIPN;
      HostIPSPT = inet_ntoa(INAddr);
      if (HostIPSPT == NULL) {
        close(FD);
        continue;
      }
      sysprint(BITMASK_MAIN, "Incoming client connection from \"%s:%ld\" on \"%s(%s):%ld\".", HostIPSPT, PortH, ListenS->Host, ListenS->HostIPS, ListenS->PortH);
      client_add(ListenS, HostIPSPT, HostIPH, HostIPN, PortH, PortN, FD);
    }
  }
}

/* LISTEN_CLOSEALL FUNCTION - JONAS (09.06.2001) */

unsigned short int listen_closeall(const char *const MessagePT, ...) {

  struct Listen_Struct *ListenS = NULL;
  struct Listen_Struct *ListenS_DEL = NULL;
  unsigned short int Count = 0;

  for (ListenS = Listen_Head ; ListenS != NULL ;) {
    if (Listen_IsResolving(ListenS)) {
      ares_cancelquery(NULL, ListenS);
      ++Count;
      Listen_SetRemove(ListenS);
      continue;
    }
    listen_stop(ListenS);
    ListenS_DEL = ListenS;
    ListenS = ListenS->Next;
    listen_rem(ListenS_DEL);
  }

  return(Count);

}

