/*
 * @(#)EditorKit.java	1.14 99/04/22
 *
 * Copyright 1997-1999 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */
package javax.swing.text;

import java.io.*;
import javax.swing.Action;
import javax.swing.JEditorPane;

/**
 * Establishes the set of things needed by a text component
 * to be a reasonably functioning editor for some <em>type</em>
 * of text content.  The EditorKit acts as a factory for some
 * kind of policy.  For example, an implementation 
 * of html and rtf can be provided that is replaceable 
 * with other implementations.
 * <p>
 * A kit can safely store editing state as an instance
 * of the kit will be dedicated to a text component.
 * New kits will normally be created by cloning a 
 * prototype kit.  The kit will have it's 
 * <code>setComponent</code> method called to establish
 * it's relationship with a JTextComponent.
 *
 * @author  Timothy Prinzing
 * @version 1.14 04/22/99
 */
public abstract class EditorKit implements Cloneable, Serializable {

    /**
     * Construct an EditorKit.
     */
    public EditorKit() {
    }

    /**
     * Creates a copy of the editor kit.  This
     * allows an implementation to serve as a prototype
     * for others, so that they can be quickly created.
     * <em>In a future release this method will be implemented
     * to use Object.clone</em>
     *
     * @return the copy
     */
    public abstract Object clone();
    /*
    public Object clone() {
	Object o;
	try {
	    o = super.clone();
	} catch (CloneNotSupportedException cnse) {
	    o = null;
	}
	return o;
    }
    */

    /**
     * Called when the kit is being installed into the
     * a JEditorPane.  
     *
     * @param c the JEditorPane
     */
    public void install(JEditorPane c) {
    }

    /**
     * Called when the kit is being removed from the
     * JEditorPane.  This is used to unregister any 
     * listeners that were attached.
     *
     * @param c the JEditorPane
     */
    public void deinstall(JEditorPane c) {
    }

    /**
     * Gets the MIME type of the data that this
     * kit represents support for.
     *
     * @return the type
     */
    public abstract String getContentType();

    /**
     * Fetches a factory that is suitable for producing 
     * views of any models that are produced by this
     * kit.  
     *
     * @return the factory
     */
    public abstract ViewFactory getViewFactory();

    /**
     * Fetches the set of commands that can be used
     * on a text component that is using a model and
     * view produced by this kit.
     *
     * @return the set of actions
     */ 
    public abstract Action[] getActions();

    /**
     * Fetches a caret that can navigate through views
     * produced by the associated ViewFactory.
     *
     * @return the caret
     */
    public abstract Caret createCaret();

    /**
     * Creates an uninitialized text storage model
     * that is appropriate for this type of editor.
     *
     * @return the model
     */
    public abstract Document createDefaultDocument();

    /**
     * Inserts content from the given stream which is expected 
     * to be in a format appropriate for this kind of content
     * handler.
     * 
     * @param in  The stream to read from
     * @param doc The destination for the insertion.
     * @param pos The location in the document to place the
     *   content >= 0.
     * @exception IOException on any I/O error
     * @exception BadLocationException if pos represents an invalid
     *   location within the document.
     */
    public abstract void read(InputStream in, Document doc, int pos) 
	throws IOException, BadLocationException;

    /**
     * Writes content from a document to the given stream
     * in a format appropriate for this kind of content handler.
     * 
     * @param out  The stream to write to
     * @param doc The source for the write.
     * @param pos The location in the document to fetch the
     *   content from >= 0.
     * @param len The amount to write out >= 0.
     * @exception IOException on any I/O error
     * @exception BadLocationException if pos represents an invalid
     *   location within the document.
     */
    public abstract void write(OutputStream out, Document doc, int pos, int len) 
	throws IOException, BadLocationException;

    /**
     * Inserts content from the given stream which is expected 
     * to be in a format appropriate for this kind of content
     * handler.
     * <p>
     * Since actual text editing is unicode based, this would 
     * generally be the preferred way to read in the data.  
     * Some types of content are stored in an 8-bit form however,
     * and will favor the InputStream.
     * 
     * @param in  The stream to read from
     * @param doc The destination for the insertion.
     * @param pos The location in the document to place the
     *   content >= 0.
     * @exception IOException on any I/O error
     * @exception BadLocationException if pos represents an invalid
     *   location within the document.
     */
    public abstract void read(Reader in, Document doc, int pos) 
	throws IOException, BadLocationException;

    /**
     * Writes content from a document to the given stream
     * in a format appropriate for this kind of content handler.
     * <p>
     * Since actual text editing is unicode based, this would 
     * generally be the preferred way to write the data.  
     * Some types of content are stored in an 8-bit form however,
     * and will favor the OutputStream.
     * 
     * @param out  The stream to write to
     * @param doc The source for the write.
     * @param pos The location in the document to fetch the
     *   content >= 0.
     * @param len The amount to write out >= 0.
     * @exception IOException on any I/O error
     * @exception BadLocationException if pos represents an invalid
     *   location within the document.
     */
    public abstract void write(Writer out, Document doc, int pos, int len) 
	throws IOException, BadLocationException;

}
