/*
 * @(#)BezierScroller.java	1.25 99/04/23
 *
 * Copyright (c) 1998, 1999 by Sun Microsystems, Inc. All Rights Reserved.
 * 
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 * 
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 * 
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

package demos.Mix;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.GeneralPath;
import java.awt.geom.PathIterator;
import java.awt.font.FontRenderContext;
import java.awt.font.TextLayout;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;
import java.util.Vector;
import javax.swing.*;
import AnimatingContext;
import DemoSurface;
import DemoPanel;
import CustomControls;


/**
 * Animated Bezier Curve shape with images at the control points.
 * README.txt file scrolling up. Composited Image fading in and out.
 */
public class BezierScroller extends DemoSurface implements AnimatingContext, CustomControls {

    private static String appletStrs[] = 
        {  " ", "Java2Demo",  
           "BezierScroller - Animated Bezier Curve shape with images", 
           "For README.txt file scrolling run in application mode", " " };
    private static final int NUMPTS = 6;
    private static Color greenBlend = new Color(0, 255, 0, 100);
    private static Font font = new Font("serif", Font.PLAIN, 12);
    private static Color blueBlend = new Color(0, 0, 255, 100);
    private static BasicStroke bs = new BasicStroke(3.0f);
    private static Image hotj_img;
    private static BufferedImage img;
    private static final int UP = 0;
    private static final int DOWN = 1;

    private float animpts[] = new float[NUMPTS * 2];
    private float deltas[] = new float[NUMPTS * 2];
    private BufferedReader reader;      
    private int nStrs;          
    private int strH;
    private int yy, ix, iy, imgX;
    private Vector vector, appletVector;
    private float alpha = 0.2f;
    private int alphaDirection;
    private DemoControls controls;
    protected boolean doImage, doShape, doText;
    protected boolean buttonToggle;


    public BezierScroller() {
        setBackground(Color.white);
        doShape = doText = true;
        hotj_img = getImage("HotJava-16.gif");
        Image image = getImage("jumptojavastrip.gif");
        int iw = image.getWidth(this);
        int ih = image.getHeight(this);
        img = new BufferedImage(iw, ih, BufferedImage.TYPE_INT_RGB);
        img.createGraphics().drawImage(image, 0, 0, this);
        controls = new DemoControls(this);
    }


    public String[] getCustomControlsConstraints() {
        return new String[] { BorderLayout.NORTH };
    }


    public Component[] getCustomControls() {
        return new Component[] { (Component) controls };
    }


    public void customControlsThread(int state) {
        if (state == CustomControls.START) {
            controls.start();
        } else if (state == CustomControls.STOP) {
            controls.stop();
        }
    }


    public void animate(float[] pts, float[] deltas, int index, int limit) {
        float newpt = pts[index] + deltas[index];
        if (newpt <= 0) {
            newpt = -newpt;
            deltas[index] = (float) (Math.random() * 4.0 + 2.0);
        } else if (newpt >= (float) limit) {
            newpt = 2.0f * limit - newpt;
            deltas[index] = - (float) (Math.random() * 4.0 + 2.0);
        }
        pts[index] = newpt;
    }


    public void getFile() {
        try {
            File file = new File("README.txt");
            if ((reader = new BufferedReader(new FileReader(file))) != null) {
                getLine();
            }
        } catch (Exception e) { reader = null; }
        if (reader == null) {
            appletVector = new Vector(100);
            for (int i = 0; i < 100; i++) {
                appletVector.addElement(appletStrs[i%appletStrs.length]);
            }
            getLine();
        }
        buttonToggle = true;
    }


    public String getLine() {
        String str = null;
        if (reader != null) {
            try {
                if ((str = reader.readLine()) != null) {
                    if (str.length() == 0) {
                        str = " ";
                    }
                    vector.addElement(str);
                }
            } catch (Exception e) { e.printStackTrace(); reader = null; }
        } else {
            if (appletVector.size() != 0) {
                vector.addElement(str = (String) appletVector.remove(0));
            }
        }
        return str;
    }


    public void reset(int w, int h) {
        for (int i = 0; i < animpts.length; i += 2) {
            animpts[i + 0] = (float) (Math.random() * w);
            animpts[i + 1] = (float) (Math.random() * h);
            deltas[i + 0] = (float) (Math.random() * 6.0 + 4.0);
            deltas[i + 1] = (float) (Math.random() * 6.0 + 4.0);
            if (animpts[i + 0] > w / 2.0f) {
                deltas[i + 0] = -deltas[i + 0];
            }
            if (animpts[i + 1] > h / 2.0f) {
                deltas[i + 1] = -deltas[i + 1];
            }
        }
        FontMetrics fm = getFontMetrics(font);
        strH = fm.getAscent()+fm.getDescent();
        nStrs = h/strH+1;
        vector = new Vector(nStrs);
        ix = (int) (Math.random() * (w - 80));
        iy = (int) (Math.random() * (h - 80));
    }


    public void step(int w, int h) {
        if (doText && vector.size() == 0) {
            getFile();
        }
        if (doText) {
            String s = getLine();
            if (s == null || vector.size() == nStrs && vector.size() != 0) {
                vector.removeElementAt(0);
            }
            yy = (s == null) ? strH : h - vector.size() * strH;
        }
       
        for (int i = 0; i < animpts.length && doShape; i += 2) {
            animate(animpts, deltas, i + 0, w);
            animate(animpts, deltas, i + 1, h);
        }
        if (doImage && alphaDirection == UP) {
            if ((alpha += 0.025) > .99) {
                alphaDirection = DOWN;
                alpha = 1.0f;
            }
        } else if (doImage && alphaDirection == DOWN) {
            if ((alpha -= .02) < 0.01) {
                alphaDirection = UP;
                alpha = 0;
                ix = (int) (Math.random() * (w - 80));
                iy = (int) (Math.random() * (h - 80));
            }
        }
        if (doImage) {
            if ((imgX += 80) == 800) {
                imgX = 0;
            }
        }
    }



    public void drawDemo(int w, int h, Graphics2D g2) {

        if (doText) {
            g2.setColor(Color.lightGray);
            g2.setFont(font);
            float y = yy;
            for (int i = 0; i < vector.size(); i++) {
                y += strH;
                g2.drawString((String)vector.get(i), 1, y);
            }
        }

        if (doShape) {
            float[] ctrlpts = animpts;
            int len = ctrlpts.length;
            float prevx = ctrlpts[len - 2];
            float prevy = ctrlpts[len - 1];
            float curx = ctrlpts[0];
            float cury = ctrlpts[1];
            float midx = (curx + prevx) / 2.0f;
            float midy = (cury + prevy) / 2.0f;
            GeneralPath gp = new GeneralPath(GeneralPath.WIND_NON_ZERO);
            gp.moveTo(midx, midy);
            for (int i = 2; i <= ctrlpts.length; i += 2) {
                float x1 = (midx + curx) / 2.0f;
                float y1 = (midy + cury) / 2.0f;
                prevx = curx;
                prevy = cury;
                if (i < ctrlpts.length) {
                    curx = ctrlpts[i + 0];
                    cury = ctrlpts[i + 1];
                } else {
                    curx = ctrlpts[0];
                    cury = ctrlpts[1];
                }
                midx = (curx + prevx) / 2.0f;
                midy = (cury + prevy) / 2.0f;
                float x2 = (prevx + midx) / 2.0f;
                float y2 = (prevy + midy) / 2.0f;
                gp.curveTo(x1, y1, x2, y2, midx, midy);
            }
            gp.closePath();

            g2.setColor(blueBlend);
            g2.setStroke(bs);
            g2.draw(gp);
            g2.setColor(greenBlend);
            g2.fill(gp);

            PathIterator pi = gp.getPathIterator(null);
            float pts[] = new float[6];
            while ( !pi.isDone() ) {
                if (pi.currentSegment(pts) == pi.SEG_CUBICTO) {
                    g2.drawImage(hotj_img, (int) pts[0], (int) pts[1], this);
                }
                pi.next();
            }
        }

        if (doImage) {
            AlphaComposite ac = AlphaComposite.getInstance(
                                   AlphaComposite.SRC_OVER, alpha);
            g2.setComposite(ac);
            g2.drawImage(img.getSubimage(imgX,0,80,80), ix, iy, this);
        }
    }


    public static void main(String argv[]) {
        final DemoPanel dp = new DemoPanel(new BezierScroller());
        Frame f = new Frame("Java2D Demo - BezierScroller");
        f.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {System.exit(0);}
            public void windowDeiconified(WindowEvent e) { 
                dp.surface.start(); 
            }
            public void windowIconified(WindowEvent e) { 
                dp.surface.stop(); 
            }
        });
        f.add("Center", dp);
        f.pack();
        f.setSize(new Dimension(400,300));
        f.show();
        dp.surface.start();
    }


    static class DemoControls extends JPanel implements ActionListener, Runnable {

        BezierScroller demo;
        JToolBar toolbar;
        JComboBox combo;
        Thread thread;

        public DemoControls(BezierScroller demo) {
            this.demo = demo;
            setBackground(Color.gray);
            add(toolbar = new JToolBar());
            toolbar.setFloatable(false);
            addTool("Image", false);
            addTool("Shape", true);
            addTool("Text", true);
            addMouseListener(new MouseAdapter() {
                public void mouseClicked(MouseEvent e) {
                    if (thread == null) start(); else stop();
                }
            });
        }


        public void addTool(String str, boolean state) {
            JButton b = (JButton) toolbar.add(new JButton(str));
            b.setBackground(state ? Color.green : Color.lightGray);
            b.setSelected(state);
            b.addActionListener(this);
        }


        public void actionPerformed(ActionEvent e) {
            JButton b = (JButton) e.getSource();
            b.setSelected(!b.isSelected());
            b.setBackground(b.isSelected() ? Color.green : Color.lightGray);
            if (b.getText().equals("Image")) {
                demo.doImage = b.isSelected();
            } else if (b.getText().equals("Shape")) {
                demo.doShape = b.isSelected();
            } else {
                demo.doText = b.isSelected();
            }
        }

        public Dimension getPreferredSize() {
            return new Dimension(200,37);
        }


        public void start() {
            if (thread != null) {
                return;
            }
            thread = new Thread(this);
            thread.setPriority(Thread.MIN_PRIORITY);
            thread.setName("Mix.BezierScroller DemoControls Thread");
            thread.start();
        }


        public synchronized void stop() {
            if (thread != null) {
                thread.interrupt();
            }
            thread = null;
            notifyAll();
        }


        public void run() {
            Thread me = Thread.currentThread();
            int i = 0;
            while (thread == me) {
                try {
                    thread.sleep(250);
                } catch (InterruptedException e) { return; }
                if (demo.buttonToggle) {
                    ((JButton) toolbar.getComponentAtIndex(i++%2)).doClick();
                    demo.buttonToggle = false;
                }
            }
            thread = null;
        }
    } // End DemoControls
} // End BezierScroller
