package org.jboss.cache.interceptors;

import org.jboss.cache.*;
import org.jboss.cache.lock.IdentityLock;
import org.jboss.cache.lock.LockingException;
import org.jboss.cache.lock.TimeoutException;
import org.jgroups.blocks.MethodCall;

import javax.transaction.Synchronization;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;
import javax.transaction.Status;
import java.lang.reflect.Method;
import java.util.*;

/**
 * Handles locking. When a TX is associated, we register for TX completion and unlock the locks acquired within the
 * scope of the TX. When no TX is present, we keep track of the locks acquired during the current method and unlock
 * when the method returns
 * @author Bela Ban
 * @version $Id: LockInterceptor.java,v 1.13.2.1 2004/12/30 17:08:24 starksm Exp $
 */
public class LockInterceptor extends Interceptor {
   private TransactionManager tx_mgr=null;
   TransactionTable           tx_table=null;
   private long               lock_acquisition_timeout;

   /** List<Transaction> that we have registered for */
   private List               transactions=Collections.synchronizedList(new ArrayList());


   final static int NONE  = 0;
   final static int READ  = 1;
   final static int WRITE = 2;


   public void setCache(TreeCache cache) {
      super.setCache(cache);
      tx_mgr=cache.getTransactionManager();
      tx_table=cache.getTransactionTable();
      lock_acquisition_timeout=cache.getLockAcquisitionTimeout();
   }



   public Object invoke(MethodCall m) throws Throwable {
      Transaction       tx=null;
      GlobalTransaction gtx=null;
      Object            retval=null;
      Fqn               fqn=null;
      int               lock_type=NONE;
      Method            meth=m.getMethod();
      Object[]          args=m.getArgs();

      /** List<IdentityLock> locks. Locks acquired during the current method; will be released on method return.
       *  This list is only populated when there is no TX, otherwise the TransactionTable maintains the locks
       * (keyed by TX) */
      LinkedList locks=null;

      if(tx_mgr != null && (tx=tx_mgr.getTransaction()) != null && isValid(tx)) { // ACTIVE or PREPARING
         if(!transactions.contains(tx)) {
            gtx=cache.getCurrentTransaction(tx);
            if(gtx == null)
               throw new Exception("failed to get global transaction");
            try {
               OrderedSynchronizationHandler handler=OrderedSynchronizationHandler.getInstance(tx);
               SynchronizationHandler myHandler=new SynchronizationHandler(gtx, tx, cache);
               handler.registerAtTail(myHandler); // needs to be invoked last on TX commit
               transactions.add(tx);
            }
            catch(Exception e) {
               log.error("registration for tx=" + tx + " with transaction manager failed, running without TX", e);
            }
         }
         else {
            gtx=cache.getTransactionTable().get(tx);
         }
      }
      else { // no TX
         locks=new LinkedList();
      }

      // 1. Determine the type of lock (read, write, or none) depending on the method. If no lock is required, invoke
      //    the method, then return immediately
      //    Set the Fqn
      if(meth.equals(TreeCache.putDataMethodLocal) || meth.equals(TreeCache.putDataEraseMethodLocal) ||
            meth.equals(TreeCache.putKeyValMethodLocal)) {
         fqn=(Fqn)args[1];
         lock_type=WRITE;
      }
      else if(meth.equals(TreeCache.removeNodeMethodLocal)) {
         fqn=(Fqn)args[1];
         lock_type=WRITE;
      }
      else if(meth.equals(TreeCache.removeKeyMethodLocal) || meth.equals(TreeCache.removeDataMethodLocal)) {
         fqn=(Fqn)args[1];
         lock_type=WRITE;
      }
      else if(meth.equals(TreeCache.evictKeyValueMethodLocal) || meth.equals(TreeCache.evictNodeMethodLocal)) {
         fqn=(Fqn)args[0];
         lock_type=WRITE;
      }
      else if(meth.equals(TreeCache.addChildMethodLocal)) {
         fqn=(Fqn)args[1];
         lock_type=WRITE;
      }
      else if(meth.equals(TreeCache.getKeyValueMethodLocal)) {
         fqn=(Fqn)args[0];
         lock_type=READ;
      }
      else if(meth.equals(TreeCache.getNodeMethodLocal)) {
         fqn=(Fqn)args[0];
         lock_type=READ;
      }
      else if(meth.equals(TreeCache.getKeysMethodLocal)) {
         fqn=(Fqn)args[0];
         lock_type=READ;
      }
      else if(meth.equals(TreeCache.getChildrenNamesMethodLocal) || meth.equals(TreeCache.releaseAllLocksMethodLocal) ||
            meth.equals(TreeCache.printMethodLocal)) {
         fqn=(Fqn)args[0];
         lock_type=READ;
      }

      if(lock_type == NONE) {
         return super.invoke(m);
      }

      try {
         // 2. Lock the node (must be either read or write if we get here)
         // If no TX: add each acquired lock to the list of locks for this method (locks)
         // If TX: [merge code from TransactionInterceptor]: register with TxManager, on commit/rollback,
         // release the locks for the given TX
         lock(fqn, gtx, lock_type, locks);
         retval=super.invoke(m);
      }
      finally {
         releaseLocks(locks);
      }
      return retval;
   }






   /**
    * Locks a given node.
    * @param fqn
    * @param gtx
    * @param lock_type Guaranteed to be READ or WRITE and *nothing else*
    */
   private void lock(Fqn fqn, GlobalTransaction gtx, int lock_type, LinkedList locks) throws TimeoutException, LockingException {
      Node n, child_node=null;
      Object  child_name;
      Fqn     tmp_fqn=new Fqn();
      int     treeNodeSize;
      Object  owner=gtx != null? gtx : (Object)Thread.currentThread();
      boolean acquired;

      if(fqn == null) {
         log.error("fqn is null - this should not be the case");
         return;
      }

      if((treeNodeSize=fqn.size()) == 0)
         return;
      n=cache.getRoot();

      for(int i=0; i < treeNodeSize; i++) {
         child_name=fqn.get(i);
         tmp_fqn=new Fqn(tmp_fqn, child_name);
         child_node=n.getChild(child_name);

         if(child_node == null) {
            if(log.isTraceEnabled())
               log.trace("failed finding child " + child_name + " of node " + n.getFqn());
            return;
         }

         // Write lock only for destination node in write request.
         if(lock_type == WRITE && i == (treeNodeSize - 1)) {
            acquired=child_node.acquire(owner, lock_acquisition_timeout, Node.LOCK_TYPE_WRITE);
         }
         else {
            acquired=child_node.acquire(owner, lock_acquisition_timeout, Node.LOCK_TYPE_READ);
         }

         if(acquired) {
            if(gtx != null) {
               // add the lock to the list of locks maintained for this transaction
               // (needed for release of locks on commit or rollback)
               cache.getTransactionTable().addLock(gtx, child_node.getImmutableLock());
            }
            else {
               IdentityLock l=child_node.getImmutableLock();
               if(!locks.contains(l))
                  locks.add(l);
            }
         }
         n=child_node;
      }
   }



   private void releaseLocks(LinkedList locks) {
      IdentityLock lock;
      // Release the locks backwards (bottom-up in the tree)
      // Note that the lock could have been released already so don't panic
      if(locks != null) {
         for(ListIterator it=locks.listIterator(locks.size()); it.hasPrevious();) {
            lock=(IdentityLock)it.previous();
            if(log.isTraceEnabled())
               log.trace("releasing lock for " + lock.getFqn() + ": " + lock);
            lock.release(Thread.currentThread());
            it.remove();
         }
      }
   }

   /**
    * Remove all locks held by <tt>tx</tt>, remove the transaction from the transaction table
    * @param gtx
    */
   private void commit(GlobalTransaction gtx) {
      if(log.isTraceEnabled())
         log.trace("committing cache with gtx " + gtx);

      TransactionEntry entry=tx_table.get(gtx);
      if(entry == null) {
         log.error("entry for transaction " + gtx + " not found (maybe already committed)");
         return;
      }

      // Let's do it in stack style, LIFO
      List list=entry.getLocks();
      for(int i=list.size() - 1; i >= 0; i--) {
         IdentityLock lock=(IdentityLock)list.get(i);
         if(log.isTraceEnabled())
            log.trace("releasing lock " + lock);
         lock.release(gtx);
      }

      if(log.isTraceEnabled())
         log.trace("removing local transactions " + entry.getTransactions());
      for(Iterator it=entry.getTransactions().iterator(); it.hasNext();)
         tx_table.remove((Transaction)it.next());
      tx_table.remove(gtx);
   }


   /**
     * Revert all changes made inside this TX: invoke all method calls of the undo-ops
     * list. Then release all locks and remove the TX from the transaction table.
     * <ol>
     * <li>Revert all modifications done in the current TX<li/>
     * <li>Release all locks held by the current TX</li>
     * <li>Remove all temporary nodes created by the current TX</li>
     * </ol>
     *
     * @param tx
     */
   private void rollback(GlobalTransaction tx) {
      List undo_ops;
      TransactionEntry entry=tx_table.get(tx);
      MethodCall undo_op;
      Object retval;
      Fqn node_name;

      if(log.isTraceEnabled())
         log.trace("called to rollback cache with GlobalTransaction=" + tx);

      if(entry == null) {
         log.error("entry for transaction " + tx + " not found (transaction has possibly already been rolled back)");
         return;
      }

      // 1. Revert the modifications by running the undo-op list in reverse. This *cannot* throw any exceptions !
      undo_ops=entry.getUndoOperations();
      for(ListIterator it=undo_ops.listIterator(undo_ops.size()); it.hasPrevious();) {
         undo_op=(MethodCall)it.previous();
         try {
            retval=undo_op.invoke(cache);
            if(retval != null && retval instanceof Throwable) {
               log.error("undo operation failed, error=" + retval);
            }
         }
         catch(Throwable t) {
            log.error("undo operation failed", t);
         }
      }

      // 2. Remove all temporary nodes. Need to do it backwards since node is LIFO.
      for(ListIterator it=entry.getNodes().listIterator(entry.getNodes().size());
          it.hasPrevious();) {
         node_name=(Fqn)it.previous();
         try {
            cache._remove(tx, node_name, false);
         }
         catch(Throwable t) {
            log.error("failed removing node \"" + node_name + "\"", t);
         }
      }


      // 3. Finally, release all locks held by this TX
      // Let's do it in stack style, LIFO
      // Note that the lock could have been released already so don't panic.
      List list=entry.getLocks();
      for(int i=list.size() - 1; i >= 0; i--) {
         IdentityLock lock=(IdentityLock)list.get(i);
         if(log.isTraceEnabled())
            log.trace("releasing lock " + lock);
         lock.release(tx);
      }

      if(log.isTraceEnabled())
         log.trace("removing local transactions " + entry.getTransactions());
      for(Iterator it=entry.getTransactions().iterator(); it.hasNext();)
         tx_table.remove((Transaction)it.next());
      tx_table.remove(tx);
   }



   class SynchronizationHandler implements Synchronization {
      GlobalTransaction gtx=null;
      Transaction       tx=null;
      TreeCache         cache=null;


      SynchronizationHandler(GlobalTransaction gtx, Transaction tx, TreeCache cache) {
         this.gtx=gtx;
         this.cache=cache;
         this.tx=tx;
      }


      /**
       * This method is invoked before the start of the commit or rollback
       * process. We don't do anything because this method handles only the case of cache_mode
       * being LOCAL
       */
      public void beforeCompletion() {
      }


      /**
       * Depending on the status (OK or FAIL), call commit() or rollback() on the CacheLoader
       *
       * @param status
       */
      public void afterCompletion(int status) {
         transactions.remove(tx);
         switch(status) {
            case Status.STATUS_COMMITTED:
               commit(gtx);
               break;

            case Status.STATUS_MARKED_ROLLBACK: // this one is probably not needed
            case Status.STATUS_ROLLEDBACK:
               if(log.isDebugEnabled())
                  log.debug("rolling back transaction");
               rollback(gtx); // roll back locally
               break;
            default:
               rollback(gtx); // roll back locally
               throw new IllegalStateException("failed rolling back transaction: " + status);
         }
      }
   }


}
